/*****************************************************************************
STATISTICAL LINE-LINKER (ANSI C)
================================
 THIS VERSION OF THE LINE-LINKER HAS BEEN DIVIDED INTO 3 MODULES:
 LINKER, GLUER AND SUPPORT. THIS VERSION IS NOW MAINTAINED AS THE
 CURRENT VERSION.


ALGORITHM:
==========
      THE BASIC IDEA IS: LINE LINKING IS PERFORMED IN TWO STEPS.

      This file contains the second step only.

***************************************************************
HISTORY:
 - Oct 6 1993, Fabio Cozman created LINKER1.c; it contains
   only a simplified first step (does not generated
   IMAGE_CHAINS; does not fit lines; connections are single
   pixels.
 - Jan 13 1994, Fabio Cozman created LINKER2.c from LINKER1.c;
   modifications:
   IMAGE_CHAINS is added;
   Elements structure is added (chains and connections are
        Elements now);
   connections are grown and may include more than one pixel;
   straight lines are fit;
   second step is added.
 - Oct 23 1994, Fabio Cozman includes the filter_chain and
   filter_line structures in order to prepare a release
   version. LINKER3.c is already in use (a better version of
   LINKER2.c but almost the same).
 - May 24 1995, Fabio Cozman divides linker3.c into 3 files:
   linker5.c, gluer5.c and linker5-support.c.
**************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "general.h"
#include "linked.h"
#include "linker5.h"
#include "chi-square.h"

/****************************************************************
*****************************************************************
  THE FOLLOWING FUNCTIONS ARE SUPPORT FUNCTIONS,
  USED BOTH IN THE FIRST OR SECOND STEPS.
*****************************************************************
****************************************************************/

/*******************************************************************
 Function that creates a chain structure. 
 It puts 1) the given values  of i in xs,xe and j in ys,ye; 
         2) zeros for start and end connections.
 Returns the pointer to the created chain structure.
Author: Fabio Cozman
Date: Oct 6 93
*******************************************************************/

Chain *create_chain(Elements *chains, int i, int j)
{
Chain *temp_chain;
Line *first_line;

         /**************** ERROR HANDLING ************/
if (chains == NULL) error("create_chain", 0);
if (chains->elements == NULL) error("create_chain", 1);
         /********* Increment total number of chains *********/
increment_total_elements(chains);
         /******** Creates a chain structure *********/
if ( (temp_chain = (Chain *)malloc( sizeof(Chain) ) ) == NULL ) 
  error("create_chain", 3);
         /******** Attaches chain structure in chains structure *****/
chains->elements[ (chains->total) ].id = chains->total;
chains->elements[ (chains->total) ].type = CHAIN; 
chains->elements[ (chains->total) ].data = (Data)temp_chain;
         /******** Fill new chain parameters ****/
temp_chain->xs = temp_chain->xe = i;
temp_chain->ds = NO_DIRECTION;
temp_chain->ys = temp_chain->ye = j;
temp_chain->de = NO_DIRECTION;
temp_chain->first_chain_start = temp_chain->second_chain_start =
  temp_chain->first_chain_end  = temp_chain->second_chain_end = 0;
temp_chain->length = 0;
temp_chain->intensity_summation = 0;
first_line = create_line();
          /******* Initialize line linked_list *****/
temp_chain->lines = (LinkedList *)malloc(sizeof(LinkedList));
temp_chain->lines->id = chains->total;
temp_chain->lines->number_of_nodes = 0;
temp_chain->lines->first_node = temp_chain->lines->last_node = NULL;
insert_unsorted_linked_list(temp_chain->lines, (Data)first_line, TO_START);
return(temp_chain);
}

/******************************************************************
 Function that creates a line and fills the parameters of the line.
Author: Fabio Cozman
Date: Feb 10 94
******************************************************************/

Line *create_line(void)
{
Line *line = NULL;

if ( (line = (Line *)malloc(sizeof(Line))) == NULL ) error("create_line", 1);
line->n = 0;
line->to = 1.0; line->tf = 0.0;
line->a = line->b = line->c = line->d = 0.0;
line->sigma_a = line->sigma_b = 0.0;
line->sigma_c = line->sigma_d = 0.0;
line->cov_ab = line->cov_cd = 0.0;
line->St = line->Stt = line->Sxt = line->Syt = 0.0;
line->Sx = line->Sxx = line->Sy = line->Syy = 0.0;
line->residual_ab = line->residual_cd = 0.0;

return(line);
}

/*****************************************************************
 Function that copy all parameters of one line to another line.
Author: Fabio Cozman
Date: Feb 20 94
*****************************************************************/

void copy_line(Line *line_from, Line *line_to)
{
line_to->n =           line_from->n;
line_to->to =          line_from->to;
line_to->tf =          line_from->tf;
line_to->a =           line_from->a;
line_to->b =           line_from->b;
line_to->sigma_a =     line_from->sigma_a;
line_to->sigma_b =     line_from->sigma_b;
line_to->cov_ab =      line_from->cov_ab;
line_to->c =           line_from->c;
line_to->d =           line_from->d;
line_to->sigma_c =     line_from->sigma_c;
line_to->sigma_d =     line_from->sigma_d;
line_to->cov_cd =      line_from->cov_cd;
line_to->St =          line_from->St;
line_to->Stt =         line_from->Stt;
line_to->Sxt =         line_from->Sxt;
line_to->Syt =         line_from->Syt;
line_to->Sx =          line_from->Sx;
line_to->Sxx =         line_from->Sxx;
line_to->Sy =          line_from->Sy;
line_to->Syy =         line_from->Syy;
line_to->residual_ab = line_from->residual_ab;
line_to->residual_cd = line_from->residual_cd;
}

/****************************************************************
 Function that updates recursively all the parameters of a given
 line, given that a new point of the line (x,y) is given.
Author: Fabio Cozman 
Date: Feb 2 94
****************************************************************/

void update_line(Line *line, int x_in, int y_in, 
		 float variance_of_pixel, int to_end)
{
float t;
float k1, k2, auxiliar;
float da, da2, db, db2, dc, dc2, dd, dd2, innovation;
float x = (float)x_in;
float y = (float)y_in;

/************ ERROR HANDLING ***************/
if (line == NULL) error("update_line", 0);
if (line->n < 0) error("update_line", 1);

       /****** Initialization of variables ******/
(line->n)++;
if (to_end) { (line->tf)++; t = line->tf; }
else        { (line->to)--; t = line->to; }
 
line->St  += t; 
line->Stt += t * t;
line->Sxt += x * t;
line->Syt += y * t;
line->Sx  += x;
line->Sxx += x * x;
line->Sy  += y;
line->Syy += y * y;

       /******* If line has only one point *********/
if (line->n == 1) { line->a = (float)x_in; line->c = (float)y_in; return; }

       /******* If line has only two points ********/
auxiliar = (float)(line->n) * line->Stt - line->St * line->St;
if (line->n == 2) {
line->a = ((line->Sx - x) * t - x * (line->St - t))/(2 * t - line->St);
line->b = (2 * x - line->Sx)/(2 * t - line->St);
line->c = ((line->Sy - y) * t - y * (line->St - t))/(2 * t - line->St);
line->d = (2 * y - line->Sy)/(2 * t - line->St);
line->sigma_a = variance_of_pixel * line->Stt/auxiliar;
line->sigma_b = variance_of_pixel * (float)(line->n)/auxiliar;
line->cov_ab = - variance_of_pixel * line->St/auxiliar;
line->sigma_c = variance_of_pixel * line->Stt/auxiliar;
line->sigma_d = variance_of_pixel * (float)(line->n)/auxiliar;
line->cov_cd = - variance_of_pixel * line->St/auxiliar;
line->residual_ab = line->residual_cd = 0.0;
return;
}
        /******** If line has more than two points ******/
k1 = (line->Stt - line->St * t)/auxiliar;
k2 = (- line->St + (float)(line->n) * t)/auxiliar;

        /**** First line: x = a + bt ******/
innovation = x - line->a - line->b * t;
da = k1 * innovation;
da2 = da * da;
db = k2 * innovation;
db2 = db * db;
line->residual_ab += innovation * innovation +
		     ( da2 + 2 * line->a * da ) * (float)(line->n) + 
		     ( 2 * da * db + 2 * line->b * da + 2 * line->a * db ) *
		                  line->St +
		     ( - 2 * da ) * line->Sx + 
		     ( - 2 * db ) * line->Sxt + 
	             ( 2 * line->b * db + db2 ) * line->Stt;
line->a += da;
line->b += db;
line->sigma_a = variance_of_pixel * line->Stt/auxiliar;
line->sigma_b = variance_of_pixel * (float)(line->n)/auxiliar;
line->cov_ab = - variance_of_pixel * line->St/auxiliar;

        /**** Second line: y = c + dt ******/
innovation = y - line->c - line->d * t;
dc = k1 * innovation;
dc2 = dc * dc;
dd = k2 * innovation;
dd2 = dd * dd;
line->residual_cd += innovation * innovation +
		     ( dc2 + 2 * line->c * dc ) * (float)(line->n) + 
		     ( 2 * dc * dd + 2 * line->d * dc + 2 * line->c * dd ) *
		                 line->St +
		     ( - 2 * dc ) * line->Sy + 
		     ( - 2 * dd ) * line->Syt + 
		     ( 2 * line->d * dd + dd2 ) * line->Stt;
line->c += dc;
line->d += dd;
line->sigma_c = variance_of_pixel * line->Stt/auxiliar;
line->sigma_d = variance_of_pixel * (float)(line->n)/auxiliar;
line->cov_cd = - variance_of_pixel * line->St/auxiliar;
}

/*******************************************************************
 Function that updates the coordinates of pixel in a certain 
 direction.
Author: Fabio Cozman
Date: Oct 6 93
********************************************************************/

void next_one(int *p_next_i, int *p_next_j, int direction)
{
switch (direction) {
 case NORTH:      (*p_next_i)--;                break;
 case SOUTH:      (*p_next_i)++;                break;
 case EAST:                      (*p_next_j)++; break;
 case WEST:                      (*p_next_j)--; break;
 case NORTH_EAST: (*p_next_i)--; (*p_next_j)++; break;
 case NORTH_WEST: (*p_next_i)--; (*p_next_j)--; break;
 case SOUTH_EAST: (*p_next_i)++; (*p_next_j)++; break;
 case SOUTH_WEST: (*p_next_i)++; (*p_next_j)--; break;
 default: printf("\n Error in next_one!");      break;
 }
}

/***************************************************************
 Function that increments the value of total elements in an
 Elements structure.
Author: Fabio Cozman
Date: Jan 29 94
***************************************************************/

void increment_total_elements(Elements *elements)
{ 
if ( elements->total == elements->max )
  printf("\n Maximum number of elements exceeded; you are likely to crash...");
elements->total = ( elements->total < elements->max ? 
		   (elements->total + 1) : (elements->max - 1) );
}

/***************************************************************
 Function that prints nicely a chain. Verbose equal to 0 just
 summarizes the lines on the chain; verbose equal to 1 
 prints a detailed report of lines.
Author: Fabio Cozman
Date: Oct 8 93
***************************************************************/

int print_chain(Elements *chains, int number_chain, int verbose)
{
Chain *temp_chain = NULL;
Line *aux_line = NULL;
NodeLinked *aux_node = NULL;

if (chains == NULL) error("print_chain", 1);

temp_chain = (Chain *)(chains->elements[number_chain].data);
printf("\n Chain %d from (%d,%d [%d]) to (%d,%d [%d]) - length %d, int. %d",
       chains->elements[number_chain].id,
       temp_chain->xs, temp_chain->ys, temp_chain->ds,
       temp_chain->xe, temp_chain->ye, temp_chain->de,
       temp_chain->length, temp_chain->intensity_summation);
printf("\n \t \t [connected to <%d, %d> , <%d, %d> ]", 
       temp_chain->first_chain_start, temp_chain->second_chain_start,
       temp_chain->first_chain_end, temp_chain->second_chain_end);
printf("\n \t Chain contains %d lines. ", temp_chain->lines->number_of_nodes);
if (verbose) {
  aux_node = temp_chain->lines->first_node;
  while (aux_node != NULL) {
    aux_line = (Line *)(aux_node->data);
    print_line(aux_line);
    aux_node = aux_node->child;
  }
}
return(0);
}

/***************************************************************
 Function that prints nicely a chain. 
Author: Fabio Cozman
Date: June 22 94
***************************************************************/

void print_chain_simple(Chain *temp_chain)
{
if (temp_chain == NULL) error("print_chain", 1);

printf("\n Chain from (%d,%d [%d]) to (%d,%d [%d]) - length %d",
       temp_chain->xs, temp_chain->ys, temp_chain->ds,
       temp_chain->xe, temp_chain->ye, temp_chain->de,
       temp_chain->length);
printf("\n \t \t [connected to <%d, %d> , <%d, %d> ]", 
       temp_chain->first_chain_start, temp_chain->second_chain_start,
       temp_chain->first_chain_end, temp_chain->second_chain_end);
printf("\n \t Chain contains %d lines. ", temp_chain->lines->number_of_nodes);
}

/***************************************************************
 Function that prints nicely a FilterChain. 
Author: Fabio Cozman
Date: Oct 23 94
***************************************************************/

void print_filter_chain(FilterChain *filtered_chain)
{
int i;

if (filtered_chain == NULL) error("print_filter_chain", 1);

printf("\n FilterChain from (%d,%d [%d]) to (%d,%d [%d]), length %d",
       filtered_chain->xs, filtered_chain->ys, filtered_chain->ds,
       filtered_chain->xe, filtered_chain->ye, filtered_chain->de,
       filtered_chain->length);
printf(" [connected to <Previous: %d> <Next: %d>] [containing %d lines]", 
       filtered_chain->previous, filtered_chain->next, 
       filtered_chain->number_of_lines);
for (i=0; i<(filtered_chain->number_of_lines); i++) {
  printf("\n \t Line: < %g x + %g y + %g = 0 > [ (%g, %g) to (%g, %g) ]",
	 filtered_chain->filter_lines[i].A, filtered_chain->filter_lines[i].B,
	 filtered_chain->filter_lines[i].C, filtered_chain->filter_lines[i].xo,
	 filtered_chain->filter_lines[i].yo,filtered_chain->filter_lines[i].xf,
	 filtered_chain->filter_lines[i].yf);
}
}

/**************************************************************
 Function that prints nicely a line.
Author: Fabio Cozman
Date: Feb 2 94
**************************************************************/

void print_line(Line *line)
{
/********* ERROR HANDLING *********/
if (line == NULL) error("line", 1);

printf("\n \t \t Line from %g to %g (%d points):",
       line->to, line->tf, line->n);
printf("\n \t \t \t < x = %g + %g t >, < y = %g + %g t >",
       line->a, line->b, line->c, line->d);
printf("\n \t \t \t s_a = %g; s_b = %g; c_ab = %g", 
       line->sigma_a, line->sigma_b, line->cov_ab);
printf("\n \t \t \t s_c = %g; s_d = %g; c_cd = %g",
       line->sigma_c, line->sigma_d, line->cov_cd);
printf("\n \t \t \t St = %g; Stt = %g; Sxt = %g; Syt = %g",
       line->St, line->Stt, line->Sxt, line->Syt);
printf("\n \t \t \t Sx = %g; Sxx = %g",
       line->Sx, line->Sxx);
printf("\n \t \t \t Sy = %g, Syy = %g",
       line->Sy, line->Syy);
printf("\n \t \t \t R_ab = %g; R_cd = %g", 
       line->residual_ab, line->residual_cd);
}

/************************************************************
  Function that draws a line in the image.
  Line is drawn by finding two ending points: this is 
  obtained from the line structure simply by 
    First point:    ( a + b to , c + d to )
    Second point:   ( a + b tf , c + d tf )
Author: Fabio Cozman
Date: Feb 2 94
************************************************************/

void draw_line(FG_Image *image, Line *line, int value)
{
float x1, y1, x2, y2;
/************* ERROR HANDLING ************/
if (image == NULL) error("draw_line", 0);
if (line == NULL) error("draw_line", 1);

if (line->n > 1) {
  x1 = line->a + line->b * line->to;  y1 = line->c + line->d * line->to;
  x2 = line->a + line->b * line->tf;  y2 = line->c + line->d * line->tf;
  fg_i_line(image, x1, y1, x2, y2, (int *)(&value)); 
}
}

/************************************************************
  Function that draws a filter line in the image.
Author: Fabio Cozman
Date: Feb 3 95
************************************************************/

void draw_filter_line(FG_Image *image, FilterLine *filter_line, int value)
{
/************* ERROR HANDLING ************/
if (image == NULL) error("draw_filter_line", 0);
if (filter_line == NULL) error("draw_filter_line", 1);

fg_i_line(image, filter_line->xo, filter_line->yo, 
	  filter_line->xf, filter_line->yf, (int *)(&value)); 
}

/************************************************************
  Function that draws a chain in the image.
Author: Fabio Cozman
Date: Feb 3 95
************************************************************/

void draw_chain(FG_Image *image, Chain *chain, int value)
{
Line *line = NULL;
NodeLinked *node = NULL;

/************* ERROR HANDLING ************/
if (image == NULL) error("draw_chain", 0);
if (chain == NULL) error("draw_chain", 1);

node = chain->lines->first_node;
while (node != NULL) {
  line = (Line *)(node->data);
  draw_line(image, line, value);
  node = node->child;
  }
}

/************************************************************
  Function that draws a filter chain in the image.
Author: Fabio Cozman
Date: Feb 3 95
************************************************************/

void draw_filter_chain(FG_Image *image, FilterChain *filter_chain, int value)
{
int i;

/************* ERROR HANDLING ************/
if (image == NULL) error("draw_filter_chain", 0);
if (filter_chain == NULL) error("draw_filter_chain", 1);

for (i=0; i<filter_chain->number_of_lines; i++)
  draw_filter_line(image, &(filter_chain->filter_lines[i]), value);
}

/***************************************************************
 Function that prints nicely a connection. Verbose has no
 meaning now.
Author: Fabio Cozman
Date: Oct 8 93
***************************************************************/

int print_connection(Elements *connections, int number_connection, int verbose)
{
Connection *temp_connection;
int i;

if (connections == NULL) error("print_connection", 1);
if (connections->elements == NULL) error("print_connection", 2);

temp_connection = 
  (Connection *)(connections->elements[number_connection].data);
printf("\n Connection %d in (%d,%d), size %d, with %d (from %d) chains:\n \t ",
       connections->elements[number_connection].id,
       temp_connection->x, temp_connection->y, 
       temp_connection->size, temp_connection->number_processed_chains,
       temp_connection->number_connected_chains);
for (i=0; i<(temp_connection->number_processed_chains); i++)
  printf(" [ %d ] ", temp_connection->connected_chains[i]);
return(0);
}

/***************************************************************
 Function that prints nicely a connection. 
Author: Fabio Cozman
Date: June 22 94
***************************************************************/

void print_connection_simple(Connection *temp_connection)
{
int i;

if (temp_connection == NULL) error("print_connection", 1);

printf("\n Connection in (%d,%d), size %d, with %d (from %d) chains:\n \t ",
       temp_connection->x, temp_connection->y, 
       temp_connection->size, temp_connection->number_processed_chains,
       temp_connection->number_connected_chains);
for (i=0; i<(temp_connection->number_processed_chains); i++)
  printf(" [ %d ] ", temp_connection->connected_chains[i]);
}

/*************************************************************
 Function that prints an element, deciding whether it is a 
 chain or a connection. Uses the functions SIMPLE to print.
Author: Fabio Cozman
Date: Jun 22 94
*************************************************************/

void print_element(Element *temp_element)
{
Chain *temp_chain = NULL;
Connection *temp_connection = NULL;

switch (temp_element->type) {
 case CHAIN: 
  printf("\n Element %d is a Chain:", temp_element->id);
  temp_chain = (Chain *)(temp_element->data);
  print_chain_simple(temp_chain);
  break;
 case CONNECTION:
  printf("\n Element %d is a Connection:", temp_element->id);
  temp_connection = (Connection *)(temp_element->data);
  print_connection_simple(temp_connection);
  break;
 case GHOST_CONNECTION:
  printf("\n Element %d is a Ghost-Connection:", temp_element->id);
  temp_connection = (Connection *)(temp_element->data);
  print_connection_simple(temp_connection);
  break;
}
}

/***************************************************************
 Function that receives two pixels: (i1, j1), (i2, j2), and 
 returns the position of (i2,j2) relative to (i1,j1). 
 Returns 0 if pixels are not adjacent.
Author: Fabio G. Cozman
Date: Nov 1993
**************************************************************/

int give_direction(int i1, int j1, int i2, int j2)
{
static int cv[] =  { NORTH_WEST, WEST, SOUTH_WEST, NORTH, 0, 
                     SOUTH, NORTH_EAST, EAST, SOUTH_EAST }; 
int position_in_cv = i1 - i2 + 4 * (j1 - j2 + 1);
return(((position_in_cv < 0) || (position_in_cv > 8)) ? 
       0 : cv[position_in_cv] );
}

/****************************************************************
 Function that returns the number of bits that are one in
 a given byte of 8 bits.  Returns -1 in case any problem occurs.
Author: Fabio G. Cozman
Date: Jan 13 1994
*****************************************************************/

int query_number_bits(int binary_number)
{
/* Handy lookup table for #of bits in a byte. From Conrad Poelman;
   /afs/cs/user/conradp/shape/src/occlusions/bitset.h */
static unsigned char bitset[256] = {
    0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8
};
if ((binary_number < 0) || (binary_number > 255)) return(-1);
else return(bitset[binary_number]);
}

/***************************************************************
 Function that returns 1 if the given chain-code indicates that
 the pixel is a connection and 0 if the given chain-code 
 indicates that the pixel is a chain.
Author: Fabio G. Cozman
Date: Jan 13 1994
***************************************************************/

int query_chain_connection(int chain_code)
{
int result = query_number_bits(chain_code);
if (result < 0) return(-1);  /**** Some problem with the chain-code ****/
if (result < 3) return(0);   /**** Pixel is a chain pixel ****/
return(1);                   /**** Pixel is a connection pixel ****/
}

/*****************************************************************
 Function that says, given a chain-code, what is the first
 direction to go in order to find an occupied pixel.
Author: Fabio G. Cozman
Date: Jan 13 1994
*****************************************************************/

int first_available_direction_in_chain_code(int chain_code)
{
if ( chain_code & 1 ) return(NORTH);
if ( chain_code & 2 ) return(SOUTH);
if ( chain_code & 4 ) return(WEST);
if ( chain_code & 8 ) return(EAST);
if ( chain_code & 16 ) return(NORTH_WEST);
if ( chain_code & 32 ) return(NORTH_EAST);
if ( chain_code & 64 ) return(SOUTH_WEST);
if ( chain_code & 128 ) return(SOUTH_EAST);
return(NO_DIRECTION);
}

/****************************************************************
 Function that, given a chain-code, fills a vector of 8 elements
 with all the directions encoded in the chain-code. Returns the
 number of directions in the chain-code.
Author: Fabio Cozman
Date: Feb 20 94
****************************************************************/

int all_available_directions_in_chain_code(int chain_code, 
					   int buf_directions[8])
{
int index = 0;
if ( chain_code & 1 ) { buf_directions[index] = NORTH; index++; }
if ( chain_code & 2 ) { buf_directions[index] = SOUTH; index++; }
if ( chain_code & 4 ) { buf_directions[index] = WEST; index++; }
if ( chain_code & 8 ) { buf_directions[index] = EAST; index++; }
if ( chain_code & 16 ) { buf_directions[index] = NORTH_WEST; index++; }
if ( chain_code & 32 ) { buf_directions[index] = NORTH_EAST; index++; }
if ( chain_code & 64 ) { buf_directions[index] = SOUTH_WEST; index++; }
if ( chain_code & 128 ) { buf_directions[index] = SOUTH_EAST; index++; }
return(index);
}

/**************************************************************
 Function that inverts a given direction.
Author: Fabio Cozman
Date: Sept 9 94
**************************************************************/

int invert_direction(int direction)
{
switch (direction) {
 case (NORTH): return( SOUTH ); break;
 case (SOUTH): return( NORTH ); break;
 case (EAST): return( WEST ); break;
 case (WEST): return( EAST ); break;
 case (NORTH_EAST): return( SOUTH_WEST ); break;
 case (NORTH_WEST): return( SOUTH_EAST ); break;
 case (SOUTH_EAST): return( NORTH_WEST ); break;
 case (SOUTH_WEST): return( NORTH_EAST ); break;
 default: error("invert_direction", 1);
 }
return(NO_DIRECTION);
}

/**************************************************************
 Function that frees CHAINS and CONNECTIONS contents.
Author: Fabio Cozman
Date: Mar 10 94
**************************************************************/

void free_chains_connections(Elements *chains, Elements *connections)
{
int i;
Connection *temp_connection = NULL;
Chain *temp_chain = NULL;

/******* ERROR HANDLING *********/
if (chains == NULL) error("free_chains_connections", 1);
if (connections == NULL) error("free_chains_connections", 2);

for (i=1; i<=connections->total; i++) {
  temp_connection = (Connection *)(connections->elements[i].data);
  free(temp_connection);
}
free(connections->elements);
for (i=1; i<chains->total; i++) {
  temp_chain = (Chain *)(chains->elements[i].data);
  free_linked_list_and_data(temp_chain->lines);
  free(temp_chain);
}
free(chains->elements);
}

/**************************************************************
 Function that frees Elements. Use this instead of 
 free_chains_connections.
Author: Fabio Cozman
Date: Oct 23 94
**************************************************************/

void free_elements(Elements *vector_elements)
{
int i;
Element *temp_element = NULL;
Connection *temp_connection = NULL;
FilterChain *temp_filter_chain = NULL;
Chain *temp_chain = NULL;

/******* ERROR HANDLING *********/
if (vector_elements == NULL) error("free_elements", 1);

for (i=1; i<=vector_elements->total; i++) {
  temp_element = &(vector_elements->elements[i]);
  if (temp_element->type == CHAIN) {
    temp_chain = (Chain *)(temp_element->data);
    free_linked_list_and_data(temp_chain->lines);
    free(temp_chain);
  }
  if ((temp_element->type == CONNECTION) || 
      (temp_element->type == GHOST_CONNECTION) ) {
    temp_connection = (Connection *)(temp_element->data);
    free(temp_connection->connected_chains);
    free(temp_connection);
  } 
  if (temp_element->type == FILTER_CHAIN) {
    temp_filter_chain = (FilterChain *)(temp_element->data);
    free(temp_filter_chain->filter_lines);
    free(temp_filter_chain);
  }
}    
free(vector_elements->elements);
}

/****************************************************************
 Function that transforms all chains in CHAINS in FilterChain
 structures in the vector FILTER_CHAINS.
Author: Fabio Cozman
Date: Oct 23 94
***************************************************************/

void filter_all_chains(Elements *chains, Elements *filter_chains)
{
Chain *chain = NULL;
FilterChain *temp_filter_chain = NULL;
int i;

/************** ERROR HANDLING ************/
if (chains == NULL) error("filter_all_chains", 1);
if (filter_chains == NULL) error("filter_all_chains", 2);

   /*** Basic parameters of FILTER_CHAINS ***/
filter_chains->total = chains->total;
filter_chains->max = chains->max;
filter_chains->elements = (Element *)malloc( (chains->max) * sizeof(Element) );

   /*** Go through all chains in CHAINS ***/
for (i=1; i<=(chains->total); i++) {
       /*** Pick a chain in CHAINS ***/
  chain = (Chain *)(chains->elements[i].data);
       /*** Pick a filter_chain ***/
  temp_filter_chain = (FilterChain *)malloc(sizeof(FilterChain));
  transfer_from_chain_to_filter_chain(chain, temp_filter_chain);
       /*** Attach filtered chain to filter_chains ***/
  filter_chains->elements[i].id = i;
  filter_chains->elements[i].type = FILTER_CHAIN;
  filter_chains->elements[i].data = (Data)temp_filter_chain;
}
}

/****************************************************************
 Function that transforms all chains in CHAINS in FilterChain
 structures in the vector FILTER_CHAINS.
Author: Fabio Cozman
Date: Jan 30 1995
***************************************************************/

void transfer_from_chain_to_filter_chain(Chain *chain,
					 FilterChain *temp_filter_chain)
{
Line *line = NULL;
NodeLinked *node = NULL;
int index_line;

/************ ERROR HANDLING *********/
if (chain == NULL) error("transfer_from_chain_to_filter_chain", 1);
if (temp_filter_chain == NULL) error("transfer_from_chain_to_filter_chain", 2);

       /*** Transfer simple parameters from chain to filter_chain ***/
temp_filter_chain->xs = chain->xs;
temp_filter_chain->ys = chain->ys;
temp_filter_chain->ds = chain->ds;
temp_filter_chain->xe = chain->xe;
temp_filter_chain->ye = chain->ye;
temp_filter_chain->de = chain->de;
temp_filter_chain->length = chain->length;
temp_filter_chain->intensity_summation = chain->intensity_summation;
/*** Transfer pointers to previous and next chains ***/
transfer_pointers_to_previous_and_next_chains(chain, temp_filter_chain);
       /*** Go through all lines in chain ***/
temp_filter_chain->number_of_lines = chain->lines->number_of_nodes;
temp_filter_chain->filter_lines = (FilterLine *)malloc( 
	       (temp_filter_chain->number_of_lines) * sizeof(FilterLine) );
node = chain->lines->first_node; 
index_line = 0; 
while (node != NULL) { /* Loop through the lines */
  line = (Line *)(node->data); /* Pick a line */
  transfer_line_contents(line, 
			 &(temp_filter_chain->filter_lines[index_line]));
  node = node->child;
  index_line++;
  if (index_line > temp_filter_chain->number_of_lines) 
    error("transfer_from_chain_to_filter_chain", 3);
}
}

/****************************************************************
 Function that transforms the flags for previous and next 
 elements in a chain into previous and next indicators in a
 filter_chain.
Author: Fabio Cozman
Date: Oct 24 94
***************************************************************/

void transfer_pointers_to_previous_and_next_chains(Chain *chain, 
					   FilterChain *temp_filter_chain)
{
/******** ERROR HANDLING *********/
if (chain == NULL) error("transfer_pointers_to_previous_and_next_chains", 1);
if (temp_filter_chain == NULL) 
  error("transfer_pointers_to_previous_and_next_chains", 2);

 /**** Beginning of chain *****/
if (chain->first_chain_start == 0) /* Chain is not connected to anything */
  temp_filter_chain->previous = 0;
else {                             /* Chain is connected to a connection */
  if ( (chain->first_chain_start < 0) && (chain->second_chain_start <= 0) )
    temp_filter_chain->previous = chain->first_chain_start;
  else {                           /* Chain is connected to another chain */
    if ( (chain->first_chain_start < 0) && (chain->second_chain_start > 0) )
      temp_filter_chain->previous = chain->second_chain_start;
    else {                    /* Chain connectes to itself (circular chain) */
      if ( (chain->first_chain_start > 0) && (chain->second_chain_start > 0))
	temp_filter_chain->previous = chain->first_chain_start;
      else error("transfer_pointers_to_previous_and_next_chains", 3);
    }
  }
}
 /**** End of chain *****/
if (chain->first_chain_end == 0) /* Chain is not connected to anything */
  temp_filter_chain->next = 0;
else {                             /* Chain is connected to a connection */
  if ( (chain->first_chain_end < 0) && (chain->second_chain_end <= 0) )
    temp_filter_chain->next = chain->first_chain_end;
  else {                           /* Chain is connected to another chain */
    if ( (chain->first_chain_end < 0) && (chain->second_chain_end > 0) )
      temp_filter_chain->next = chain->second_chain_end;
    else {                     /* Chain connectes to itself (circular chain) */
      if ( (chain->first_chain_end > 0) && (chain->second_chain_end > 0))
	temp_filter_chain->next = chain->first_chain_end;
      else error("transfer_pointers_to_previous_and_next_chains", 4);
    }
  }
}
}

/****************************************************************
 Function that transforms the parameters of a Line into 
 the parameters of a FilterLine. If the original line has 
 only one point, it is represented as Y = K (K is a number).
Author: Fabio Cozman
Date: Oct 24 94
***************************************************************/

void transfer_line_contents(Line *line, FilterLine *filtered_line)
{
/********** ERROR HANDLING *********/
if (line == NULL) error("transfer_line_contents", 1);
if (filtered_line == NULL) error("transfer_line_contents", 2);

filtered_line->xo = line->a + line->b * line->to;
filtered_line->yo = line->c + line->d * line->to;
filtered_line->xf = line->a + line->b * line->tf;
filtered_line->yf = line->c + line->d * line->tf;
if ( ( line->b != 0.0 ) && ( line->d != 0.0 ) ) {
  filtered_line->A = line->d;
  filtered_line->B = - line->b;
  filtered_line->C = - line->a * line->d + line->b * line->c;
}
else {
  if ( ( line->b == 0.0 ) && ( line->d != 0.0 ) ) {
    filtered_line->A = 1.0;
    filtered_line->B = 0.0;
    filtered_line->C = - line->a;
      }
  else {
    if ( ( line->b != 0.0 ) && ( line->d == 0.0 ) ) {
      filtered_line->A = 0.0;
      filtered_line->B = 1.0;
      filtered_line->C = - line->c;
    }
  }
}
}



