/*****************************************************************************
STATISTICAL LINE-LINKER (ANSI C)
================================
 THIS VERSION OF THE LINE-LINKER HAS BEEN DIVIDED INTO 3 MODULES:
 LINKER, GLUER AND SUPPORT. THIS VERSION IS NOW MAINTAINED AS THE
 CURRENT VERSION.


ALGORITHM:
==========
      THE BASIC IDEA IS: LINE LINKING IS PERFORMED IN TWO STEPS.

      This file contains the second step only.

***************************************************************
HISTORY:
 - Oct 6 1993, Fabio Cozman created LINKER1.c; it contains
   only a simplified first step (does not generated
   IMAGE_CHAINS; does not fit lines; connections are single
   pixels.
 - Jan 13 1994, Fabio Cozman created LINKER2.c from LINKER1.c;
   modifications:
   IMAGE_CHAINS is added;
   Elements structure is added (chains and connections are
        Elements now);
   connections are grown and may include more than one pixel;
   straight lines are fit;
   second step is added.
 - Oct 23 1994, Fabio Cozman includes the filter_chain and
   filter_line structures in order to prepare a release
   version. LINKER3.c is already in use (a better version of
   LINKER2.c but almost the same).
 - May 24 1995, Fabio Cozman divides linker3.c into 3 files:
   linker5.c, gluer5.c and linker5-support.c.
**************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "general.h"
#include "linked.h"
#include "linker5.h"
#include "chi-square.h"

/****************************************************************
*****************************************************************
 THE FUNCTIONS THAT FOLLOW ARE USED IN THE SECOND STEP:
 CHAIN MERGING AND GAP JUMPING.
*****************************************************************
****************************************************************/

/************************************************************
 Function that tries to jump gaps and connect chains.
 Each chain is separately analysed; for each chain, each
 extremity is analysed: start and end.
 There are two situations which deserve special analysis
 at this point:
  - The extremity of the chain ends in a connection. In this
    case all the chains that end at the connection are 
    considered for continuations of the chain in consideration.
    The chain that best continues the chain in consideration 
    is chosen and assigned as the continuation.
  - The extremity of the chain ends in background. In this
    case, the region around the extremity is searched. The 
    chain that best continues the chain in consideration
    is chosen and assigned as the continuation.
 Chains are connected using Aikake's Information Criterium:
  if the AIC of the line that fits two lines is smaller than 
  the AIC of the two lines separately, then the lines are merged.
 After glueing, the possible situations for the flags in the 
  chains are:
                                    |  First   |    Second
    Gap stops                       |    0     |       0
    Connection stops                |   <0     |       0
    Connection continues with chain |   <0     |      >0
    Closed chain                    |   >0     |      >0
Author: Fabio Cozman
Date: Feb 11 94
*************************************************************/

void glue_chains(FG_Image *image_in, FG_Image *image_out, 
		 FG_Image *image_chains, FG_Subimage bounds, 
		 Elements *chains, Elements *connections, 
		 float variance_of_pixel, int size_exploration)
{
int i;
int *connection_marks;
Chain *temp_chain = NULL;

   /********* Initialize marks ***********/
connection_marks = (int *)(malloc( (connections->total + 2) * sizeof(int) ));
for (i=0; i<(connections->total + 1); i++) connection_marks[i] = 0;
   /********* Examine all chains ***********/
for (i=1; i <= chains->total; i++) {
  temp_chain = (Chain *)(chains->elements[i].data);
  if (temp_chain->length > 1) { /**** Does not analyse too small lines ****/
    if (temp_chain->second_chain_start == 0)  /**** Start is unexplored *****/
      explore_extremity(image_out, image_chains, bounds, chains, connections,
			temp_chain, i, temp_chain->first_chain_start, 
			variance_of_pixel, TO_START, size_exploration,
			connection_marks);
    if (temp_chain->second_chain_end == 0)    /**** End is unexplored ****/
      explore_extremity(image_out, image_chains, bounds, chains, connections,
			temp_chain, i, temp_chain->first_chain_end,
			variance_of_pixel, TO_END, size_exploration,
			connection_marks);
  }
}
free(connection_marks);
}

/***************************************************************
 Function that analyses a given extremity of a chain.
 Picks up the line in the extremity, verifies whether the
 extremity is alone or connected to a connection:
       - if it is alone, build a ghost-connection.
 then analyses all candidate chains.
 If any chain is connected to initial chain, the new chain
 is analysed, until no chain is found.
 OBS:
 1) This function could be changed slightly for a different 
    behavior.
 2) For speed, no error handling in this function.
Author: Fabio Cozman
Date: Mar 5 94
*****************************************************************/

void explore_extremity(FG_Image *image_out, FG_Image *image_chains, 
		       FG_Subimage bounds, 
		       Elements *chains, Elements *connections,
		       Chain *aux_chain, int aux_index, int connection_index, 
		       float variance_of_pixel, int to_end, 
		       int size_exploration, int *connection_marks)
{
int auxiliar, auxiliar_to_end;
Line *aux_line = NULL;
Line *minimum_line = NULL;

/******** ERROR HANDLING *******/
if (aux_index <= 0) error("explore_extremity", 0);

   /********* Create auxiliary line ********/
minimum_line = create_line();
   /********* Explore extremity of chain **********/
while (aux_index > 0) {
  if (connection_index > 0) error("explore_extremity", 1);
  auxiliar = aux_index;
  auxiliar_to_end = to_end;
  if ( to_end == TO_START ) 
    aux_line = (Line *)(aux_chain->lines->first_node->data);
  else 
    aux_line = (Line *)(aux_chain->lines->last_node->data);
  copy_line(aux_line, minimum_line);
  if (connection_index == 0) {  /****** If extremity is alone ******/
    connection_index = explore_gap(image_out, bounds, chains, connections, 
			   aux_chain, aux_index, to_end, size_exploration,
			   connection_marks);
  }
  if (connection_index < 0) /**** Now analyse the connection ****/
    aux_index = continue_connection(image_out, image_chains, bounds, 
                      	    chains, connections, auxiliar, 
			    connection_index, variance_of_pixel, 
			    &to_end, minimum_line, size_exploration);
  else
    aux_index = 0;
  if (aux_index < 0) error("explore_extremity", 2);
  if (aux_index > 0) { /**** If there is a new chain, connect both chains ****/
    if (auxiliar_to_end == TO_END) aux_chain->second_chain_end = aux_index;
    else                           aux_chain->second_chain_start = aux_index;
    aux_chain = (Chain *)(chains->elements[aux_index].data);
    if ( to_end == TO_START ) {
      if (aux_chain->second_chain_start != 0) aux_index = 0;
      connection_index = aux_chain->first_chain_start;
      aux_chain->second_chain_end = auxiliar;
    }
    else {
      if (aux_chain->second_chain_end != 0) aux_index = 0;
      connection_index = aux_chain->first_chain_end;
      aux_chain->second_chain_start = auxiliar;
    }
  }
  else aux_index = 0;
}
free(minimum_line);
}

/**************************************************************
 Function that explores a square region around the 
 extremity of a chain, in order to find a possible chain 
 that can be connected to the given chain (it explores
 both chains and connections in this region).
 If any chain is found in the square region defined by
 SIZE_EXPLORATION, a ghost-connection is created and the
 relevant chains are connected to this ghost-connection.
Author: Fabio Cozman
Date: Mar 6 94
**************************************************************/

int explore_gap(FG_Image *image_out, FG_Subimage bounds, 
		Elements *chains, Elements *connections, 
		Chain *aux_chain, int aux_index, int to_end, 
		int size_exploration, int *connection_marks)
{
int i, j, temp_i, temp_j, is, il, js, jl;
int temp_pixel;
int *data, *mark;
LinkedList *marked_connections = NULL;
LinkedList *extremities_of_connection = NULL;
NodeLinked *node = NULL;

/********* ERROR HANDLING ***********/
if (image_out == NULL) error("explore_gap", 1);
if (chains == NULL) error("explore_gap", 2);
if (connections == NULL) error("explore_gap", 3);

          /****** Initialize list of extremities of connection *******/
extremities_of_connection = (LinkedList *)malloc(sizeof(LinkedList));
extremities_of_connection->id = 0;
extremities_of_connection->number_of_nodes = 0;
extremities_of_connection->first_node = 
  extremities_of_connection->last_node = NULL;

          /****** Initialize list of marked connections *******/
marked_connections = (LinkedList *)malloc(sizeof(LinkedList));
marked_connections->id = 0;
marked_connections->number_of_nodes = 0;
marked_connections->first_node = 
  marked_connections->last_node = NULL;
          /****** Analyse region around starting pixel *****/
if (to_end == TO_END) 
  { temp_i = aux_chain->xe; temp_j = aux_chain->ye; }
else 
  { temp_i = aux_chain->xs; temp_j = aux_chain->ys; }
is = temp_i - size_exploration; 
if (is < (bounds.rs + 1)) is = bounds.rs + 1;
il = temp_i + size_exploration;
if (il > (bounds.re - 1)) il = bounds.re - 1;
js = temp_j - size_exploration; 
if (js < (bounds.cs + 1)) js = bounds.cs + 1;
jl = temp_j + size_exploration;
if (jl > (bounds.ce - 1)) jl = bounds.ce - 1;
for (i = is; i <= il; i++) 
  for (j = js; j <= jl; j++) {
    temp_pixel = fg_i_igetpixel(image_out, i, j);

    if ((temp_pixel > 0) && (temp_pixel != aux_index))
      chain_in_gap_exploration(chains, connections, 
			       extremities_of_connection, temp_pixel, i, j);
    if (temp_pixel < 0)
      connection_in_gap_exploration(chains, connections, marked_connections, 
			    extremities_of_connection, connection_marks,
			    temp_pixel, is, il, js, jl);
  }
if (extremities_of_connection->number_of_nodes == 0) {
  free_linked_list_and_data(marked_connections);
  free_linked_list_and_data(extremities_of_connection);
  return(0);
}
if ((data = (int *)malloc(sizeof(int))) == NULL) error("explore_gap", 4);
*data = aux_index;
insert_unsorted_linked_list(extremities_of_connection, (Data)(data), TO_START);
handle_ghost_connection(connections, aux_chain, 
			extremities_of_connection, temp_i, temp_j);
node = marked_connections->first_node;
while (node != NULL) {
  mark = (int *)(node->data);
  if (*mark >= 0) error("explore_gap", 5);
  connection_marks[ - (*mark) ] = 0;
  node = node->child;
}
if (to_end == TO_END) aux_chain->first_chain_end = - connections->total;
else                  aux_chain->first_chain_start = - connections->total;
free_linked_list_and_data(marked_connections);
free_linked_list_and_data(extremities_of_connection);
return( - (connections->total) );
}

/*********************************************************************
 Function that fills one element of the linked list 
 EXTREMITIES_OF_CONNECTION, when a chain is found in the exploration
 region. 
Author: Fabio Cozman
Date: Mar 6 94
*********************************************************************/

void chain_in_gap_exploration(Elements *chains, Elements *connections, 
			      LinkedList *extremities_of_connection, 
			      int temp_pixel, int i, int j)
{
int *data;
Chain *temp_chain = NULL;

/********* ERROR HANDLING ********/
if (chains == NULL) error("chain_in_gap_exploration", 1);
if (connections == NULL) error("chain_in_gap_exploration", 2);
if (extremities_of_connection == NULL) error("chain_in_gap_exploration", 3);
if (temp_pixel <= 0) error("chain_in_gap_exploration", 4);

if ((data = (int *)malloc(sizeof(int))) == NULL) 
  error("chain_in_gap_exploration", 4);
*data = temp_pixel;

if (temp_pixel > chains->total) error("chain_in_gap_exploration", 5);

temp_chain = (Chain *)(chains->elements[temp_pixel].data);
if ((temp_chain->xs == i) && (temp_chain->ys == j)) {
  insert_unsorted_linked_list(extremities_of_connection,
			      (Data)(data), TO_END);
  if (temp_chain->first_chain_start == 0)
    temp_chain->first_chain_start = - (connections->total + 1);
}
if (temp_chain->length > 1) {
  if ((temp_chain->xe == i) && (temp_chain->ye == j)) {
    insert_unsorted_linked_list(extremities_of_connection,
				(Data)(data), TO_END);
    if (temp_chain->first_chain_end == 0)
      temp_chain->first_chain_end = - (connections->total + 1);
  }
}
}

/*********************************************************************
 Function that fills one element of the linked list 
 EXTREMITIES_OF_CONNECTION, when a connection is found in the 
 exploration region. All the chains that are connected to this
 connection are examined. Chains that are connected outside the
 exploration region are inserted in the linked list (the chains that
 are already in the exploration region will be examined anyway).
 The vector CONNECTION_MARKS is used in order to avoid examining the
 same connection several times.
Author: Fabio Cozman
Date: Mar 6 94
*********************************************************************/

void connection_in_gap_exploration(Elements *chains, Elements *connections, 
				   LinkedList *marked_connections,
				   LinkedList *extremities_of_connection, 
				   int *connection_marks,
				   int temp_pixel, 
				   int is, int il, int js, int jl)
{
int i;
int index_chain, x, y, flag;
int *data, *mark;
Connection *temp_connection = NULL;
Chain *temp_chain = NULL;

/********* ERROR HANDLING ********/
if (chains == NULL) error("connection_in_gap_exploration", 1);
if (connections == NULL) error("connection_in_gap_exploration", 2);
if (marked_connections == NULL) error("connection_in_gap_exploration", 3);
if (extremities_of_connection == NULL) 
  error("connection_in_gap_exploration", 4);
if (connection_marks == NULL) error("connection_in_gap_exploration", 5);
if (temp_pixel >= 0) error("connection_in_gap_exploration", 6);

      /********* If connection is not marked, mark and store it ********/
if (connection_marks[ (- temp_pixel) ] == 1) return;
connection_marks[ (- temp_pixel) ] = 1;
mark = (int *)malloc(sizeof(int));
*mark = temp_pixel;
insert_unsorted_linked_list(marked_connections, (Data)(mark), TO_END);
      /********* Picks the connection *********/
temp_connection = (Connection *)(connections->elements[ (- temp_pixel) ].data);
for (i=0; i<(temp_connection->number_processed_chains); i++) {
  if ( (index_chain = temp_connection->connected_chains[i]) <= 0 ) 
    error("connection_in_gap_exploration", 7);
  temp_chain = (Chain *)(chains->elements[index_chain].data);
  if (temp_chain->first_chain_start == temp_pixel) 
    { x = temp_chain->xs; y = temp_chain->ys; flag = TO_START; }
  else {
    if (temp_chain->first_chain_end == temp_pixel)
      { x = temp_chain->xe; y = temp_chain->ye; flag = TO_END; }
    else { 
      if (temp_chain->first_chain_start > 0) {  /* Detects closed curve */
	if ((temp_chain->second_chain_start <= 0) ||
	    (temp_chain->first_chain_end <= 0) ||   /* Checks consistency */
	    (temp_chain->second_chain_end <= 0))   /*   of closed curve  */ 
	  { error("connection_in_gap_exploration", 8); }
      }
      else
	{ error("connection_in_gap_exploration", 9); } 
    }
  }
  if ( !( (x >= is) && (x <= il) && (y >= js) && (y <= jl) ) ) {
    data = (int *)malloc(sizeof(int));
    *data = index_chain;
    insert_unsorted_linked_list(extremities_of_connection, 
				(Data)(data), TO_END);
      /* The following lines never fire since temp_chain is connected */
    if ((flag == TO_START) && (temp_chain->first_chain_start == 0))
      temp_chain->first_chain_start = - (connections->total + 1);
    if ((flag == TO_END) && (temp_chain->first_chain_end == 0))
      temp_chain->first_chain_end = - (connections->total + 1);
  }
}
}

/********************************************************************
 Function that creates a ghost-connection, i.e., a connection that
 has no corresponding pixels in the image. A ghost-connection has
 size 0 and works only to connect chains that are close but do not
 meet.
Author: Fabio Cozman
Date: Mar 6 94
********************************************************************/

void handle_ghost_connection(Elements *connections, Chain *aux_chain, 
			     LinkedList *extremities_of_connection, 
			     int temp_i, int temp_j)
{
int i;
int total_connections;
Connection *temp_connection = NULL;
NodeLinked *node = NULL;

/******* ERROR_HANDLING *********/
if (connections == NULL) error("handle_ghost_connection", 1);
if (aux_chain == NULL) error("handle_ghost_connection", 2);
if (extremities_of_connection == NULL) error("handle_ghost_connection", 3);
if (extremities_of_connection->number_of_nodes == 0) 
  error("handle_ghost_connection", 4);

          /******* Increment total connections ********/
increment_total_elements(connections);
total_connections = connections->total;
          /******* Creates a connection structure *******/
if ( (temp_connection = (Connection *)(malloc( sizeof(Connection) ))) == NULL )
  error("handle_ghost_connection", 4);
          /******** Attaches connection structure in connections vector ***/
connections->elements[ total_connections ].id = - total_connections;
connections->elements[ total_connections ].type = GHOST_CONNECTION;
connections->elements[ total_connections ].data = (Data)temp_connection;
          /******* Fill new connection parameters ******/
temp_connection->x = temp_i;
temp_connection->y = temp_j;
temp_connection->size = 0;
temp_connection->number_processed_chains = 
  temp_connection->number_connected_chains = 
    extremities_of_connection->number_of_nodes;

if ((temp_connection->connected_chains = 
     (int *)(malloc((temp_connection->number_connected_chains) * sizeof(int))))
     == NULL) error("handle_ghost_connection", 5);
for (i = 0,  node = extremities_of_connection->first_node;
     i < temp_connection->number_connected_chains, node != NULL; i++) {
  temp_connection->connected_chains[i] = *((int *)(node->data));
  node = node->child;
}
}

/**************************************************************
 Function that examine a given chain and a given connection
 (connected to the chain) for possible extensions of the given
 chain. All the other chains that are connected to the given
 connection are examine as possible candidates for extension
 of the given chain. The AIC criterium is used in order to
 select the best option.
Author: Fabio Cozman
Date: Feb 11 94
**************************************************************/

int continue_connection(FG_Image *image_out, FG_Image *image_chains, 
			FG_Subimage bounds, 
			Elements *chains, Elements *connections, 
			int chain_index, int connection_index, 
			float variance_of_pixel, 
			int *p_to_end, Line *minimum_line, 
			int size_exploration)
{
int i;
int index_chain_minimum, index_other_chain, other_to_end, other_to_end_minimum;
int initialize_aic, go;
int x2, y2, d2, n2;
float auxiliar, aic_minimum, aic_composed_line, aic_separated_lines;
Chain *other_chain = NULL;
Connection *connection = NULL;
Line *temp_line = NULL;
Line *other_line = NULL;
Line *composed_line = NULL;

    /******** ERROR HANDLING *******/
if (image_out == NULL) error ("continue_connection", 1);
if (image_chains == NULL) error ("continue_connection", 2);
if (chains == NULL) error("continue_connection", 3);
if (connections == NULL) error("continue_connection", 4);
if (connection_index >= 0) error("continue_connection", 5);
if (minimum_line == NULL) error("continue_connection", 6);

   /******** Initialize variables *********/
index_chain_minimum = 0;
initialize_aic = 1;
go = 0;
aic_minimum = 0.0;
auxiliar = 2 * log( 2 * 3.14159265358 * variance_of_pixel );
composed_line = create_line();
temp_line = create_line();
copy_line(minimum_line, temp_line);
   /******** Pick up connection to which chain is connected ********/
connection = (Connection *)(connections->elements[(- connection_index)].data);
   /******** Go through chains that are connected to the connection *****/
for (i=0; i < connection->number_processed_chains; i++) {
  if ( (index_other_chain = connection->connected_chains[i]) != chain_index ) {
            /***** Picks the other chain *****/
    other_chain = (Chain *)(chains->elements[index_other_chain].data);
    if (other_chain->first_chain_start > 0) /* Detects closed curve */
      {
	if ((other_chain->second_chain_start <= 0) ||
	  (other_chain->first_chain_end <= 0) ||   /* Checks consistency */
	  (other_chain->second_chain_end <= 0))    /*   of closed curve  */ 
	error("continue_connection", 7);
      }
    else {
      if (connection->size == 0) /*** If connection is a ghost-connection ***/
	other_to_end = find_status_jump_gaps(other_chain, connection,
					     connection_index);
      else                       /*** If connection is a normal one ***/
	other_to_end = 
	  find_status_of_other_chain(other_chain, connection_index);
      /**** If the other chain has already been analysed, skip it; 
	otherwise analyse it *******/
      if (((other_to_end == TO_END) && 
	   (other_chain->second_chain_end == 0)) ||
	  ((other_to_end == TO_START) && 
	   (other_chain->second_chain_start == 0))) {
	copy_line(temp_line, composed_line);
	other_line = 
	  x_y_d_n_line(other_chain, &x2, &y2, &d2, &n2, other_to_end);
	/*** Compute AIC of both separated lines: initial and other ****/
	aic_separated_lines = 
	  (composed_line->residual_ab + composed_line->residual_cd + 
	    other_line->residual_ab + 
	      other_line->residual_cd)/variance_of_pixel +
		(float)composed_line->n * auxiliar +
		  (float)other_line->n * auxiliar + 8.0;
	/** Fit a unique line over the points of the two previous lines **/
	residual_pair_of_lines(image_out, image_chains, bounds, composed_line, 
		 x2, y2, d2, n2, variance_of_pixel, *p_to_end);
	/*** Compute AIC of unique line ****/
	aic_composed_line = 
	  (composed_line->residual_ab + 
	     composed_line->residual_cd)/variance_of_pixel + 
	       (float)composed_line->n * auxiliar + 4.0;
	/******** Compare options by comparing AICs ********/
	if ( aic_composed_line < aic_separated_lines ) {
	  if (initialize_aic) 
	    { aic_minimum = aic_composed_line; initialize_aic = 0; go = 1; }
	  else {
	    if ( aic_composed_line < aic_minimum ) go = 1;
	    else go = 0;
	  }
	  if (go == 1) {
	    aic_minimum = aic_composed_line;
	    index_chain_minimum = index_other_chain;
	    other_to_end_minimum = other_to_end;
	    copy_line(composed_line, minimum_line);
	}     }    }   }  } }
free(temp_line);
free(composed_line);
if (other_to_end_minimum == TO_END) *p_to_end = TO_START;
else                                *p_to_end = TO_END;
return(index_chain_minimum);
}

/*****************************************************************
 Function that determines the relevant parameters of a chain in
 the analysis of its connection with neighbors: the relevant
 line, the starting point (x,y) of the line, the number of points
 in the line and the direction of the line.
 Direction of line is NORTH, SOUTH, EAST, WEST, NORTH_WEST, 
 NORTH_EAST, SOUTH_WEST, SOUTH_EAST  in case line has more than
 one point; if line has only one point, direction is set to 
 NO_DIRECTION.
Author: Fabio Cozman
Date: Feb 20 94
*****************************************************************/

Line * x_y_d_n_line(Chain *temp_chain, 
		    int *p_x, int *p_y, int *p_d, int *p_n, int to_end)
{
Line *temp_line;
int temp_i, temp_j;

/*********** ERROR HANDLING **************/
if (temp_chain == NULL) error("x_y_d_n_line", 1); 

  /***** Find position of line and length of line of interest *****/
if (to_end == TO_END) {
  *p_x = temp_i = temp_chain->xe; 
  *p_y = temp_j = temp_chain->ye;
  temp_line = (Line *)(temp_chain->lines->last_node->data);
  *p_n = temp_line->n;
  *p_d = temp_chain->de;
}
else {
  *p_x = temp_i = temp_chain->xs; 
  *p_y = temp_j = temp_chain->ys;
  temp_line = (Line *)(temp_chain->lines->first_node->data);
  *p_n = temp_line->n;
  *p_d = temp_chain->ds;
}
return(temp_line);
}

/*****************************************************************
 Function that discovers whether a chain is connected to a 
 given connection by its start or end; if the chain is not
 connected to given connection, an error is produced. 
 Returns TO_START or TO_END depending on the situation.
Author: Fabio Cozman
Date: Feb 20 94
*****************************************************************/

int find_status_of_other_chain(Chain *other_chain, int connection_index)
{
/****** ERROR HANDLING ********/
if (other_chain == NULL) error("find_status_of_other_chain", 1);

if ( other_chain->first_chain_start == connection_index ) return(TO_START);
else {
  if ( other_chain->first_chain_end == connection_index ) return(TO_END);
  else error("find_status_of_other_chain", 2); 
}
return(-2); /** Impossible! **/
}

/******************************************************************
 Function that discovers whether a chain is connected to a given
 ghost-connection by its start or end. The
 test is simple: it chooses the point that is closest to the 
 given ghost_connection.
 Returns TO_START or TO_END depending on the situation.
Author: Fabio Cozman
Date: Mar 6 94
******************************************************************/

int find_status_jump_gaps(Chain *other_chain, Connection *connection,
			  int index_connection)
{
int xc, yc, auxiliar_x, auxiliar_y, square_distance_start, square_distance_end;

/******* ERROR HANDLING *********/
if (other_chain == NULL) error("find_status_jump_gaps", 1);
if (connection == NULL) error("find_status_jump_gaps", 2);
if (connection->size != 0) error("find_status_jump_gaps", 3);
     /*********** Get position of connection *********/
xc = connection->x; yc = connection->y;
     /*********** Verify start of chain **********/
auxiliar_x = (xc - other_chain->xs); auxiliar_y = (yc - other_chain->ys);
square_distance_start = auxiliar_x * auxiliar_x + auxiliar_y * auxiliar_y;
     /*********** Verify end of chain ***********/
auxiliar_x = (xc - other_chain->xe); auxiliar_y = (yc - other_chain->ye);
square_distance_end = auxiliar_x * auxiliar_x + auxiliar_y * auxiliar_y;
     /**** Make the test ******/
if (square_distance_end >= square_distance_start) {
  if (other_chain->first_chain_start == 0)
    other_chain->first_chain_start = index_connection;
  return(TO_START);
}
else {
  if (other_chain->first_chain_end == 0)
    other_chain->first_chain_end = index_connection;
  return(TO_END);
}
}

/*****************************************************************
 Function that follows a line and updates the contents of 
 composed line. It is based on chain-codes, so it is very fast.
 The line COMPOSED_LINE is extended by following points X, Y on.
 The difficulty is that X, Y do not necessarily agree with the
 endpoints of COMPOSED_LINE; points are corrected for this (if
 no correction is made, the residuals are very large and do not
 reflect true fit).
Author: Fabio Cozman
Date: Feb 20 94
*****************************************************************/

void residual_pair_of_lines(FG_Image *image_out, FG_Image *image_chains, 
			    FG_Subimage bounds, Line *composed_line, 
			    int x, int y, int d, int n, 
			    float variance_of_pixel, int to_end)
{
int i;
int x_go, y_go, x_corrected, y_corrected, x_correction, y_correction;
int chain_code;

     /************* ERROR HANDLING ***********/
if (image_chains == NULL) error("residual_pair_of_lines", 1);
if (composed_line == NULL) error("residual_pair_of_lines", 2);
    /****** Calculate corrections *********/
if (to_end) { 
  x_go = (int)(composed_line->a + composed_line->b * (composed_line->tf + 1));
  y_go = (int)(composed_line->c + composed_line->d * (composed_line->tf + 1));
}
else {
  x_go = (int)(composed_line->a + composed_line->b * (composed_line->to - 1));
  y_go = (int)(composed_line->c + composed_line->d * (composed_line->to - 1));
}
x_correction = x_go - x;
y_correction = y_go - y;
      /****** Follow line through chain codes *******/
for (i=0; i<n; i++) {
  x_corrected = x + x_correction; y_corrected = y + y_correction;
  update_line(composed_line, x_corrected, y_corrected, 
	      variance_of_pixel, to_end);
  if (d != NO_DIRECTION) {
    next_one(&x, &y, d);
    chain_code = fg_i_igetpixel(image_chains, x, y);
    switch (d) {
     case NORTH: chain_code -= 2; break;
     case SOUTH: chain_code -= 1; break;
     case WEST:  chain_code -= 8; break;
     case EAST:  chain_code -= 4; break;
     case NORTH_WEST: chain_code -= 128; break;
     case NORTH_EAST: chain_code -= 64;  break;
     case SOUTH_WEST: chain_code -= 32;  break;
     case SOUTH_EAST: chain_code -= 16;  break;
     default: error("residual_pair_of_lines", 3);
     }
    d = first_available_direction_in_chain_code(chain_code);
 }
}
}
