/*******************************************************
  Here you have fast-and-dirty routines for getting the
  values of a chi-square distribution: tell the tail 
  mass given a value, tell the value given the tail mass. 
  These routines are not very accurate (accuracy up to
  third or fouth decimal place, but are the fastest 
  in the market and use very specific tricks relative 
  to the chi-square. The tail mass is specially 
  inaccurate for degrees_of_freedom < 10 when the mass 
  is larger than 0.85 (a very extreme situation).

  There is also a routine for getting the tail mass
  of a standardized normal distribution. It is accurate
  to the fifth decimal place and EXTREMELY FAST (table
  look-up).

  For high precision, the suggested solution is the 
  slower methods presented in Numerical Recipes in C;
  they rely in the Incomplete Gamma Function.
 
Author: Fabio Cozman
********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "general.h"
#include "chi-square.h"

/********** LOOK-UP tables **********/
  /*** Values  in the list reduced_normal_tail_points
       are A such that P(Z > a) = p, where
       Z is reduced normal variable Z  and  p is
       indicated in the list reduced_normal tail_points ***/
static double reduced_normal_significance_levels[NUMBER_SIGNIFICANCE_LEVELS] =
  { 0.01, 0.05, 0.1, 0.15, 0.2, 0.25, 0.35 };
static double reduced_normal_tail_points[NUMBER_SIGNIFICANCE_LEVELS] =
  { 2.326347874, 1.644853627, 1.281551566, 1.036433389, 0.8416212336,
    0.6744897502, 0.3853204664 };
  /*** Values of the chi_square variable for 1-30 degrees of freedom,
       for the significance levels is the list
       reduced_normal_significance_level. 
       This was obtained through Mathematica, by solving
       Map[FindRoot[GammaRegularized[#/2,x/2] == alpha, {x, 2+#}]&,Range[1,30]]
       and then capturing the output ***/
static double chi_square_table[NUMBER_SIGNIFICANCE_LEVELS][30] =
  { { 6.6349, 9.21034, 11.3449, 13.2767, 15.0863, 16.8119, 18.4753,
      20.0902, 21.666, 23.2093, 24.725, 26.217, 27.6882, 29.1412, 30.5779,
      31.9999, 33.4087, 34.8053, 36.1909, 37.5662, 38.8739, 40.2894, 41.6384,
      42.9798, 44.3141, 45.6417, 46.9629, 48.2782, 49.3753, 50.8916},
    { 3.84146, 5.99146, 7.81473, 9.48773, 11.0705, 12.5916, 14.0671,
      15.5073, 16.919, 18.307, 19.6751, 21.0261, 22.362, 23.6848, 24.9958,
      26.2962, 27.5871, 28.8693, 30.1435, 31.4104, 32.6706, 33.9244, 35.1725,
      36.415, 37.6525, 38.8851, 40.1133, 41.3371, 42.557, 43.773}, 
    { 2.70554, 4.60517, 6.25139, 7.77944, 9.23636, 10.6446, 12.017,
      13.3616, 14.6837, 15.9872, 17.275, 18.5493, 19.8119, 21.0641, 22.3071,
      23.5418, 24.769, 25.9894, 27.2036, 28.412, 29.6151, 30.8133, 32.0069,
      33.1962, 34.3816, 35.5632, 36.7412, 37.9159, 39.0875, 40.256},
    { 2.07225, 3.79424, 5.31705, 6.74488, 8.1152, 9.4461, 10.7479,
      12.0271, 13.288, 14.5339, 15.7671, 16.9893, 18.202, 19.4062, 20.603,
      21.7931, 22.977, 24.1555, 25.3289, 26.4976, 27.662, 28.8225, 29.9792,
      31.1325, 32.2825, 33.4295, 34.5736, 35.715, 36.8538, 37.9903},
    { 1.64237, 3.21888, 4.64163, 5.98862, 7.28928, 8.55806, 9.80325,
      11.0301, 12.2421, 13.442, 14.6314, 15.812, 16.9848, 18.1508, 19.3107,
      20.4651, 21.6146, 22.7595, 23.9004, 25.0375, 26.1711, 27.3015, 28.4288,
      29.5533, 30.6752, 31.7946, 32.9117, 34.0266, 35.1394, 36.2502},
    { 1.3233, 2.77259, 4.10834, 5.38527, 6.62568, 7.8408, 9.03715,
      10.2189, 11.3888, 12.5489, 13.7007, 14.8454, 15.9839, 17.1169, 18.2451,
      19.3689, 20.4887, 21.6049, 22.7178, 23.8277, 24.9348, 26.0393, 27.1413,
      28.2412, 29.3389, 30.4346, 31.5284, 32.6205, 33.7109, 34.7997},
    { 0.873457, 2.09964, 3.28311, 4.43769, 5.57307, 6.69476, 7.80612,
      8.90936, 10.006, 11.0971, 12.1836, 13.2661, 14.3451, 15.4209, 16.494,
      17.5646, 18.633, 19.6993, 20.7638, 21.8265, 22.8876, 23.9473, 25.0055,
      26.0625, 27.1183, 28.173, 29.2266, 30.2791, 31.3308, 32.3815} };
   /*** Values of the distribution function of a standardized normal:
          \int_\infty^x (1/Sqrt[2pi]) Exp[-1/2 x^2] dx ****/
static double normal_table[] =
 {0.5, 0.50399, 0.50798, 0.51197, 0.51595, 0.51994, 0.52392, 0.5279,
  0.53188, 0.53586, 0.53983, 0.5438, 0.54776, 0.55172, 0.55567, 0.55962,
  0.56356, 0.56749, 0.57142, 0.57535, 0.57926, 0.58317, 0.58706, 0.59095,
  0.59483, 0.59871, 0.60257, 0.60642, 0.61026, 0.61409, 0.61791, 0.62172,
  0.62552, 0.6293, 0.63307, 0.63683, 0.64058, 0.64431, 0.64803, 0.65173,
  0.65542, 0.6591, 0.66276, 0.6664, 0.67003, 0.67364, 0.67724, 0.68082,
  0.68439, 0.68793, 0.69146, 0.69497, 0.69847, 0.70194, 0.7054, 0.70884,
  0.71226, 0.71566, 0.71904, 0.7224, 0.72575, 0.72907, 0.73237, 0.73565,
  0.73891, 0.74215, 0.74537, 0.74857, 0.75175, 0.7549, 0.75804, 0.76115,
  0.76424, 0.7673, 0.77035, 0.77337, 0.77637, 0.77935, 0.7823, 0.78524,
  0.78814, 0.79103, 0.79389, 0.79673, 0.79955, 0.80234, 0.80511, 0.80785,
  0.81057, 0.81327, 0.81594, 0.81859, 0.82121, 0.82381, 0.82639, 0.82894,
  0.83147, 0.83398, 0.83646, 0.83891, 0.84134, 0.84375, 0.84614, 0.84849,
  0.85083, 0.85314, 0.85543, 0.85769, 0.85993, 0.86214, 0.86433, 0.8665,
  0.86864, 0.87076, 0.87286, 0.87493, 0.87698, 0.879, 0.881, 0.88298,
  0.88493, 0.88686, 0.88877, 0.89065, 0.89251, 0.89435, 0.89617, 0.89796,
  0.89973, 0.90147, 0.9032, 0.9049, 0.90658, 0.90824, 0.90988, 0.91149,
  0.91309, 0.91466, 0.91621, 0.91774, 0.91924, 0.92073, 0.9222, 0.92364,
  0.92507, 0.92647, 0.92785, 0.92922, 0.93056, 0.93189, 0.93319, 0.93448,
  0.93574, 0.93699, 0.93822, 0.93943, 0.94062, 0.94179, 0.94295, 0.94408,
  0.9452, 0.9463, 0.94738, 0.94845, 0.9495, 0.95053, 0.95154, 0.95254,
  0.95352, 0.95449, 0.95543, 0.95637, 0.95728, 0.95818, 0.95907, 0.95994,
  0.9608, 0.96164, 0.96246, 0.96327, 0.96407, 0.96485, 0.96562, 0.96638,
  0.96712, 0.96784, 0.96856, 0.96926, 0.96995, 0.97062, 0.97128, 0.97193,
  0.97257, 0.9732, 0.97381, 0.97441, 0.975, 0.97558, 0.97615, 0.9767,
  0.97725, 0.97778, 0.97831, 0.97882, 0.97932, 0.97982, 0.9803, 0.98077,
  0.98124, 0.98169, 0.98214, 0.98257, 0.983, 0.98341, 0.98382, 0.98422,
  0.98461, 0.985, 0.98537, 0.98574, 0.9861, 0.98645, 0.98679, 0.98713,
  0.98745, 0.98778, 0.98809, 0.9884, 0.9887, 0.98899, 0.98928, 0.98956,
  0.98983, 0.9901, 0.99036, 0.99061, 0.99086, 0.99111, 0.99134, 0.99158,
  0.9918, 0.99202, 0.99224, 0.99245, 0.99266, 0.99286, 0.99305, 0.99324,
  0.99343, 0.99361, 0.99379, 0.99396, 0.99413, 0.9943, 0.99446, 0.99461,
  0.99477, 0.99492, 0.99506, 0.9952, 0.99534, 0.99547, 0.9956, 0.99573,
  0.99585, 0.99598, 0.99609, 0.99621, 0.99632, 0.99643, 0.99653, 0.99664,
  0.99674, 0.99683, 0.99693, 0.99702, 0.99711, 0.9972, 0.99728, 0.99736,
  0.99744, 0.99752, 0.9976, 0.99767, 0.99774, 0.99781, 0.99788, 0.99795,
  0.99801, 0.99807, 0.99813, 0.99819, 0.99825, 0.99831, 0.99836, 0.99841,
  0.99846, 0.99851, 0.99856, 0.99861, 0.99865, 0.99869, 0.99874, 0.99878,
  0.99882, 0.99886, 0.99889, 0.99893, 0.99896, 0.999, 0.99903, 0.99906,
  0.9991, 0.99913, 0.99916, 0.99918, 0.99921, 0.99924, 0.99926, 0.99929,
  0.99931, 0.99934, 0.99936, 0.99938, 0.9994, 0.99942, 0.99944, 0.99946,
  0.99948, 0.9995, 0.99952, 0.99953, 0.99955, 0.99957, 0.99958, 0.9996,
  0.99961, 0.99962, 0.99964, 0.99965, 0.99966, 0.99968, 0.99969, 0.9997,
  0.99971, 0.99972, 0.99973, 0.99974, 0.99975, 0.99976, 0.99977, 0.99978,
  0.99978, 0.99979, 0.9998, 0.99981, 0.99981, 0.99982, 0.99983, 0.99983,
  0.99984, 0.99985, 0.99985, 0.99986, 0.99986, 0.99987, 0.99987, 0.99988,
  0.99988, 0.99989, 0.99989, 0.9999, 0.9999, 0.9999, 0.99991, 0.99991,
  0.99992, 0.99992, 0.99992, 0.99992, 0.99993, 0.99993, 0.99993, 0.99994,
  0.99994, 0.99994, 0.99994, 0.99995, 0.99995, 0.99995, 0.99995, 0.99995,
  0.99996, 0.99996, 0.99996, 0.99996, 0.99996, 0.99996, 0.99997, 0.99997,
  0.99997};

/*******************************************************************
 Function that returns the limit value for the residual in the
 chi-square test (needed for breaking lines). This function is
 based on tables and formulas from Estatistica, P L O Costa Neto.
 Up to 29 degrees of freedom, the exact value of the limit residual
 is recoved from a table; from 30 degrees of freedom up, the value
 of the residual is approximated by a formula suggested at page 52.
 If the number of degrees of freedom is too small, returns 0.0.
Author: Fabio Cozman
Date: Feb 11 94
********************************************************************/

float chi_square_critical_value(int degrees_of_freedom, int significance_level)
{
float aux;

       /****** ERROR HANDLING *******/
if ((significance_level < 0) || (significance_level > 7))
  error("chi_square_critical_value", 1); 
if (degrees_of_freedom < 1) return(0.0);

if (degrees_of_freedom <  30)
  return(chi_square_table[significance_level][degrees_of_freedom - 1]);
aux = 1.0 - 2.0/(9.0 * (float)degrees_of_freedom) +
  reduced_normal_tail_points[significance_level] *
    sqrt( 2.0/(9.0 * (float)degrees_of_freedom));
return((float)degrees_of_freedom * aux * aux * aux);
}

/******************************************************************
  Function that returns the probability mass in the tail of a 
  chi-square distribution (with n degrees of freedom), where the
  tail is considered after the given value. 
  The function computes  P( X_n^2 > value); it is fast and 
  reasonably accurate (accuracy degrades for degrees_of_freedom < 10
  and smaller values).
Author: Fabio Cozman
Date: Feb 19 94
******************************************************************/

float chi_square_tail_mass(int degrees_of_freedom, float value)
{
float a, b, c;

if (value < 0.0) error("chi_square_tail_mass", 1);
if (value == 0.0) return(1.0);

a = value/(float)degrees_of_freedom;
b = 2.0/(9.0 * (float)degrees_of_freedom);
c = (pow(a, 1.0/3.0) + b - 1.0)/sqrt(b);

return(normal_tail_mass(c));
}

/******************************************************************
 Function that returns the probability mass in the tail of a 
 standardized normal. The tail is considered after the given value.
 Values larger than +/- 4.0 produce a 0.0 return.
Author: Fabio Cozman
Date: Feb 19 94
******************************************************************/

float normal_tail_mass(float value)
{
float v, a, b, r;
int v_int;

if (value > 4.0) return(0.0);
if (value < -4.0) return(1.0);

v = fabs(value * 100.0);
v_int = (int)v;

a = normal_table[ v_int ];
b = normal_table[ v_int + 1 ];

r = a + (b - a) * (v - (float)v_int);

return( value >= 0.0 ? (1 - r) : r );
}



