/****************************************************************************
Functions that implement a Canny operator; temp_imageX
ends up with the edges.
Author: Fabio Cozman
Date: April 21 1995

** First, some of that legal stuff:                                         **
** You may not sell, loan or sub-license all or any part of this program.   **
** You may distribute literal copies of the program provided you receive    **
** no money for it. You have no warranty, express, implied, statutory or    **
** otherwise, and there are no implied warranty of merchantability or       **
** fitness for any particular purpose. The program is provided to you AS IS;**
** no part of this program is warranted to be error free. In no event I am  **
** liable for loss of data or profits, or any damages arising from the use  **
** of the program and its documentation.                                    **
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "general.h"



int canny(FG_Image *inimg, FG_Image *canny_image, FG_Subimage bounds,
          int binarize, int sizeX, int sizeY, 
          float threshold, int number);



/********** PROTOTYPES ***********/
static int gaussian_derivative(FG_Image *inimg, float **gradX,
			       float **gradY, FG_Subimage bounds,
			       float **aux_image,
			       int sizeX, int sizeY);

static void fill_template(float *templateGauss, float *templateDerGauss,
			  int size);

static void  magdir (float **temp_imageX, float **temp_imageY,
		     FG_Subimage bounds, float **aux_image);

static void non_max_sup (FG_Image *canny_image,
			 float ** temp_imageX, float **temp_imageY,
			 FG_Subimage bounds, float **aux_image,
			 int sizeX, int sizeY, float threshold,
			 int number, int binarize);

static float suppress(float **mag, FG_Subimage bounds,
		      float gx, float gy, float thresh,
		      register int i, register int j);

/*****************************************************************
Simplified Canny operator. This function gets an input image
(INIMG) and produces an output image (CANNY_IMAGE) which
contains zeroes for the background and non-zero values for
edges. 

The non-zero values can be of two types: they can
be of 0/1 type, if the variable BINARIZE is set to 1; or they can
be equal to the original value times a constant, if the variable 
BINARIZE is set to zero.
The logic is the following:
 if BINARIZE is 1 -> background is zero, edgels have the value
                     defined by the variable NUMBER
 if BINARIZE is 0 -> background is zero, edgels have the value
                     of the original image multiplied by NUMBER

The function initially performs gaussian smoothing with controllable
size. Notice that, unlike other implementations of smoothing
operations, the gaussian smoothing operates differently in the
X and Y directions. The amount of smoothing is defined by the
variables sizeX and sizeY; the first is equal to 3 times the
standard deviation of the smoothing applied in the X direction and
the second is equal to 3 times the standard deviation of the
smoothing applied in the Y direction. 

The coordinate system is defined as (** pay attention to this! **):
 origin is top left point
 x axis goes from top to bottom         (0,0) ---------------->
 y axis goes from left to right               |
                                              |
                                              |
                                              V

After the image is smoothed, the magnitude at each point is calculated.
If the magnitude is smaller than the given THRESHOLD, the point is not
an edge. Otherwise, the point is an edge only if it is a local
maximum.
**********************************************************************/

int canny(FG_Image *inimg, FG_Image *canny_image, FG_Subimage bounds,
	  int binarize, int sizeX, int sizeY, 
	  float threshold, int number)
{
float **temp_imageX, **temp_imageY, **aux_image;
int i, size_aux;

/*** Initialization and Allocation of Variables ***/
  printf("\n\n Start canny with sizes (%d,%d), threshold %g, and ", 
	 sizeX, sizeY, threshold);
  if (binarize) 
    printf(" binary result (0 and %d).", number);
  else
    printf(" continuous result, multiplied by %d.", number);

  size_aux = bounds.re - bounds.rs + 1;
  aux_image = (float **)(malloc(size_aux * sizeof(float *)));
  temp_imageX = (float **)(malloc(size_aux * sizeof(float *)));
  temp_imageY = (float **)(malloc(size_aux * sizeof(float *)));
  for (i=0; i<size_aux; i++) {
    temp_imageX[i]= (float *)(malloc((bounds.ce-bounds.cs+1) * sizeof(float)));
    temp_imageY[i]= (float *)(malloc((bounds.ce-bounds.cs+1) * sizeof(float)));
    aux_image[i] = (float *)(malloc((bounds.ce-bounds.cs+1) * sizeof(float)));
  }

  printf("\n Smoothing...");
  gaussian_derivative(inimg, temp_imageX, temp_imageY, bounds, aux_image,
		      sizeX, sizeY);

  printf("\n Calculating magnitudes...");
  magdir (temp_imageX, temp_imageY, bounds, aux_image);
  
  printf("\n Suppressing non-maxima...");
  non_max_sup (canny_image, temp_imageX, temp_imageY, bounds, aux_image, 
	       sizeX, sizeY, threshold, number, binarize); 

  for (i=0; i<size_aux; i++) { 
    free(aux_image[i]);
    free(temp_imageX[i]);
    free(temp_imageY[i]);
  }
  free(aux_image);
  free(temp_imageX);
  free(temp_imageY);

  printf("\n Performed Canny operator. \n");

  return(1);
}

/******************************************************
Perform Gaussian Smoothing: initialize template,
and run it twice over image (once in vertical direction,
once in horizontal direction). 
******************************************************/

static int gaussian_derivative(FG_Image *inimg, float **gradX,
			       float **gradY, FG_Subimage bounds, 
			       float **aux_image,
			       int sizeX, int sizeY)
{
int i, j, k;
float *templateGaussX, *templateGaussY, *templateDerGaussX, *templateDerGaussY;
int rs, re, cs, ce;
float valueX, valueY;

/** Establish bounds for performing Canny operator **/
rs = bounds.rs + sizeX;
cs = bounds.cs + sizeY;
re = bounds.re - sizeX;
ce = bounds.ce - sizeY;

/** Allocate the templates for the separable filters **/
templateGaussX = (float *)malloc( (2 * sizeX + 1) * sizeof(float) );
templateDerGaussX = (float *)malloc( (2 * sizeX + 1) * sizeof(float) );
templateGaussY = (float *)malloc( (2 * sizeY + 1) * sizeof(float) );
templateDerGaussY = (float *)malloc( (2 * sizeY + 1) * sizeof(float) );

/** Fill the templates **/
fill_template(templateGaussX, templateDerGaussX, sizeX); 
fill_template(templateGaussY, templateDerGaussY, sizeY); 

/** Perform X smoothed derivative **/
/** First smooth in the Y direction **/
for (i = bounds.rs; i < bounds.re; i++)
  for (j = bounds.cs; j < bounds.ce; j++) {
    if ( (i > rs) && (i < re) && (j > cs) && (j < ce) ) {
      valueY = 0.0; 
      for (k = -sizeY; k <= sizeY; k++) 
	valueY += 
	  templateGaussY[k + sizeY]*((float)fg_i_igetpixel(inimg, i, j+k));
      aux_image[(i-bounds.rs)][(j-bounds.cs)] = valueY;
    }
    else 
      aux_image[(i-bounds.rs)][(j-bounds.cs)] = 
	(float)(fg_i_igetpixel(inimg,i,j));
  }
/** Then take X derivative **/
for (i = bounds.rs; i < bounds.re; i++)
  for (j = bounds.cs; j < bounds.ce; j++) {
    if ( (i > rs) && (i < re) && (j > cs) && (j < ce) ) {
      valueX = 0.0; 
      for (k = -sizeX; k <= sizeX; k++) 
	valueX += templateDerGaussX[k + sizeX] * 
	  aux_image[i+k-bounds.rs][j-bounds.cs];
      gradX[(i-bounds.rs)][(j-bounds.cs)] = valueX;
    }
    else gradX[(i-bounds.rs)][(j-bounds.cs)] = 0.0; 
  }

/** Perform Y smoothed derivative **/
/** First smooth in the X direction **/
for (i = bounds.rs; i < bounds.re; i++)
  for (j = bounds.cs; j < bounds.ce; j++) {
    if ( (i > rs) && (i < re) && (j > cs) && (j < ce) ) {
      valueX = 0.0;
      for (k = -sizeX; k <= sizeX; k++) 
	valueX += 
	  templateGaussX[k + sizeX]*((float)(fg_i_igetpixel(inimg, i+k, j)));
      aux_image[(i-bounds.rs)][(j-bounds.cs)] = valueX;
    }
    else 
      aux_image[(i-bounds.rs)][(j-bounds.cs)] = 
	(float)(fg_i_igetpixel(inimg,i,j));
  }
/** Then take Y derivative **/
for (i = bounds.rs; i < bounds.re; i++)
  for (j = bounds.cs; j < bounds.ce; j++) {
    if ( (i > rs) && (i < re) && (j > cs) && (j < ce) ) {
      valueY = 0.0; 
      for (k = -sizeY; k <= sizeY; k++) 
	valueY += templateDerGaussY[k + sizeY] * 
	  aux_image[i-bounds.rs][j+k-bounds.cs];
      gradY[(i-bounds.rs)][(j-bounds.cs)] = valueY;
    }
    else gradY[(i-bounds.rs)][(j-bounds.cs)] = 0.0; 
  }

free(templateGaussX); 
free(templateDerGaussX);
free(templateGaussY);
free(templateDerGaussY);

return(1);
}

/*********************************************************
Fill Gaussian Template: evaluate derivative of a Gaussian; 
size (of the template) is three times the variance  of
the Gaussian.
Author: Fabio Cozman
Date: April 21 1995
*********************************************************/

static void fill_template(float *templateGauss, float *templateDerGauss, 
			  int size)
{ 
int i, total_size;
float position, std_deviation, multiplicative_constant;

total_size = 2 * size + 1;
multiplicative_constant = 0.0;
std_deviation = (float)size/3.0;

for (i = 0 ; i < total_size; i++) {
  position = (float)(i-size);
  templateGauss[i] = exp(- (position * position)/
			 (2.0 * std_deviation * std_deviation));
  templateDerGauss[i] = (- position/std_deviation) * 
    exp(- (position * position)/(2.0 * std_deviation * std_deviation));
  multiplicative_constant += templateGauss[i];
}

for (i = 0; i < total_size; i++) {
  templateGauss[i] /= multiplicative_constant;
  templateDerGauss[i] /= multiplicative_constant;
}
}

/*****************************************************************
  Calculate magnitude of gradient for each pixel.
  Author: Fabio Cozman
  Date: April 21 1995
*****************************************************************/

static void  magdir (float **temp_imageX, float **temp_imageY, 
		     FG_Subimage bounds, float **aux_image)
{
int i, j;
float x, y;

for (i = bounds.rs; i < bounds.re; i++)
  for (j = bounds.cs; j < bounds.ce; j++) {
    x = temp_imageX[(i-bounds.rs)][(j-bounds.cs)]; 
    y = temp_imageY[(i-bounds.rs)][(j-bounds.cs)];
    aux_image[(i-bounds.rs)][(j-bounds.cs)] = sqrt(x * x + y * y);
  }
}

/*****************************************************************
  Suppress non-maxima.
  Author: Fabio Cozman
  Date: April 21 1995
*****************************************************************/

static void non_max_sup (FG_Image *canny_image, 
			 float **temp_imageX, float **temp_imageY, 
			 FG_Subimage bounds, float **aux_image,
			 int sizeX, int sizeY, float threshold, 
			 int number, int binarize)
{
int i, j;
float drow, dcol, result;
int rs, cs, re, ce;

/** Establish bounds for performing Canny operator **/
rs = bounds.rs + 1;
cs = bounds.cs + 1;
re = bounds.re - 1;
ce = bounds.ce - 1;

for (i = rs; i < re; i++)
  for (j = cs; j < ce; j++) {
    drow = temp_imageX[(i-bounds.rs)][(j-bounds.cs)]; 
    dcol = temp_imageY[(i-bounds.rs)][(j-bounds.cs)];
    result = suppress(aux_image, bounds, dcol, drow, threshold, i, j);
    if (result != 0.0) {
      if (binarize) result = number;
      else result *= (float)number;
    }
    fg_i_iputpixel(canny_image, i, j, (int)result);
  }
}

/*****************************************************************
	Suppress a pixel value if it is not a peak or ridge point
*****************************************************************/
#define N	mag[i+1][j]
#define NE	mag[i+1][j+1]
#define NW	mag[i+1][j-1]
#define S	mag[i-1][j]
#define SE	mag[i-1][j+1]
#define SW	mag[i-1][j-1]
#define E	mag[i][j+1]
#define W	mag[i][j-1]
#define CENTER  mag[i][j]

static float suppress(float **mag, FG_Subimage bounds,
		      float gx, float gy, float thresh, 
		      register int i, register int j)
{
    float mag1, mag2, fraction;
    
    i -= bounds.rs; j -= bounds.cs;
    
    /* applies threshold in magnitude */
    if ( CENTER < thresh ) return(0.0);   

    if (gx==0.0) {
	mag1= N;
	mag2= S;
    }
    else if ( gy==0.0 ){
	mag1= E;
	mag2= W;
    }
    else if (gx * gy > 0.0 ){	/* first & third quadrants */
	gx= fabs(gx);
	gy= fabs(gy);
	if (gx > gy ){			/* first octant */
	    fraction= gy/gx;
	    mag1= fraction * NE  +  (1-fraction) * E;
	    mag2= fraction * SW  +  (1-fraction) * W;
	}
	else{ 				/* second octant */
	    fraction= gx/gy;
	    mag1= fraction * NE  +  (1-fraction) * N;
	    mag2= fraction * SW  +  (1-fraction) * S;
	}
    }
    else{			/* second and fourth quadrants */
	gx= fabs(gx);
	gy= fabs(gy);
	if (gx > gy){			/* third octant */
	    fraction= gy/gx;
	    mag1= fraction * NW  +  (1-fraction) * W;
	    mag2= fraction * SE  +  (1-fraction) * E;
	}
	else {				/* fourth octant */
	    fraction= gx/gy;
	    mag1= fraction * NW  +  (1-fraction) * N;
	    mag2= fraction * SE  +  (1-fraction) * S;
	}
    }

    if (CENTER < mag1 || CENTER < mag2) return(0.0);
    else return(CENTER);
}
