/***************************************************************************
*
*  simul.c -- Program for testing either end of the RATLER system
*
*  Contains: setup_rcp(), initialize_packets(), initialize_values(),
*            display_commands(), display_status(), display_ids(),
*            display_timers(), display_help(), write_command_headings(),
*            write_status_headings(), send_block(), send_query(),
*            setup_display(), get_ids(), get_event(), toggle_timer(),
*            clear_timers(), update_blinker(), main()
*
*  Action:
*    This program was originally written as a testbed for RCP changes.
*    The original code can be found under LIBS:\RCP\SRC\TEST\TESTRCP.
*
*    Since this code was based on the RATLER project code, it was an
*    easy matter to turn this into a simulator which can be used for
*    either the CDS or the vehicle.
*
*    The operator uses this program to send RCP blocks (both manually
*    and automatically on a timed basis), and to parse and display
*    the contents of received RCP blocks.
*
*    This program depends on the fact that in the RTLRBLKS.H file
*    status blocks are grouped together and are consecutive, as
*    are command blocks.  If this were to change, the translation
*    of keystrokes into build_block() calls would have to be
*    re-worked.
*
*    This program may be run on two computers joined by serial link,
*    on one computer with a loopback plug, or on either the vehicle
*    or CDS end of a comm link.  It will run on any DOS computer with
*    a standard COM1 or COM2 serial port (I/O addresses of 3f8 and
*    2f8 respectively).  The same copy of code can be run on both ends,
*    since all blocks can be sent and parsed.
*
*    When started, the user is asked which port to run on.  Enter a 1,
*    a 2, or a return.  Default (CR) is COM1.  The port is automatically
*    initialized to 4800 baud (this is hardwired into the code for
*    RATLER).
*
*    The user is then asked for destination and source RCP ID's.  These
*    must range from 0 through 127.  These are used solely for building
*    blocks.  ID's on incoming blocks are not checked.
*
*    The operator presses keys in order to send command or status blocks.
*    Sent blocks are displayed at the bottom of the screen following
*    the "Out:" prompt.
*
*    Received blocks are displayed at the bottom of the screen following
*    the "In:" prompt.  They are parsed and the values are written to the
*    screen.  If an invalid block is received, it will be noted
*    following the offending block.
*
*    Timers can be used to automatically give keystrokes to the
*    program.  Timer functions are hardcoded into the program.
*    A timer is defined by its interval, initial enable status,
*    the time it was last "run", and the keystroke to be given
*    to the program.  Timer information is displayed following
*    the "Timers:" prompt.  The numbers represent the timers
*    available; those with a leading '+' are enabled.
*
*    The block in the upper right blinks once for every two valid RCP
*    blocks received.
*
*
*  -- KEY GUIDE --
*
*  In general, key presses that involve Shift (e.g., Shift A,
*  or Shift F1) will send blocks related to COMMANDS.
*
*  Key presses that do not involve any other key will send blocks
*  related to STATUS.
*
*  Key presses that involve Alt (e.g., Alt 1, Alt R), are related to
*  functioning of the simul program itself.
*
*  1.  Pressing LOWER-CASE letters will send the corresponding STATUS
*      block ('a' sends lowest-numbered status block, 'b' sends next
*      lowest, etc.).
*
*  2.  Pressing UPPER-CASE letters will send the corresponding COMMAND
*      block ('A' sends lowest-number command block, 'B' sends next
*      lowest, etc.).
*
*  3.  Function keys will send a query for the corresponding STATUS
*      block (F1 sends query for lowest-numbered status block).
*
*  4.  Shifted function keys will send a query for the corresponding
*      COMMAND block (Shift-F1 sends query for lowest-numbered command
*      block).
*
*  5.  Pressing Alt and a numeric key (Alt-1, Alt-2, etc.) will toggle
*      the enable status of the given timer.  A timer displayed with
*      a '+' preceding its number is enabled.
*
*  6.  Alt-0 (zero) will clear all timers.
*
*  7.  Alt-p will toggle the parking brake command (block sent automatically)
*
*  8.  Alt-r will refresh the screen.
*
*  9.  Alt-k will set KILL, Alt-u will set NO_KILL. (block sent automatically)
*
*  10.  Alt-h brings up a help screen.
*
*  10.  Alt-q or Esc exits the program.
*
*  11.  If an unrecognized key is presssed, its unused key code is displayed
*       in order to assist in adding functions later.
*
*
*  Created:  waa, 12-02-93
*  Modified:
*     WAA, 03-30-94 -- Changed key usage to lower-case = sends status,
*       v1.0           upper-case = sends commands.  Included the new
*                      RATLER blocks for testing.  Added RCP ID's
*                      (now asks user for dest and src).
*     WAA, 04-05-94 -- Added the awesome timer stuff.  Copied blinker code
*       v1.5           from SARGE simulator.  Added help screen.
*     WAA, 04-07-94 -- Added ALT-K and ALT-U to change the state of the
*       v1.5a          vehcmd.kill variable.  You still have to send the
*                      block containing this packet manually.  Increased
*                      the keepalive interval from 18 ticks to 2 ticks,
*                      empirically determined by trying to keep the vehicle
*                      alive.  Made the status and command values display
*                      on startup and with every refresh.  Added ALT-P to
*                      toggle parking brake command in vehcmd.parking_brake.
*     WAA, 06-07-94 -- Added third timer for sending send-image-triple
*       v1.5b          commands.  ALT-K, ALT-U, and ALT-P now also send
*                      the command block automatically.  Recompiled to use
*                      the rcpvalue.h file.
*     WAA, 07-20-94 -- Added commands to affect heading, pitch and roll
*       v1.5c          status.
*     WAA, 06-16-95 -- Changed all "fuel" to "voltage" for RATLER.
*       v1.5d
*
***************************************************************************/
#define PROGRAM_NAME "RATLER SIMUL v1.5d"

/* Standard library include files */
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

/* Local library include files */
#include <rcpvalue.h>
extern "C"  /* Need to extern "C" otherwise linker can't find functions */
{
   #include <rcperrs.h>
   #include <rcpproto.h>
}

/* Application-specific include files */
#include "rtlrcp.h"
#include "rtlrblks.h"
#include "comm.h"
#include "struct.h"
#include "textposn.h"

/* Random defines */
#define FIRST_PORT   0  /* First RCP port */
#define OKAY         1

#define MIN_QUERY_LENGTH 5  /* Minimum chars in an RCP query */

#define HIGH_BYTE  0xff00
#define LOW_BYTE   0x00ff

/* For screens */
typedef enum
{
   MIN_SCREEN = 0,

   REGULAR_SCREEN = MIN_SCREEN,
   HELP_SCREEN,

   MAX_SCREEN = HELP_SCREEN
} Screen_type;

/* For communication status display */
#define BLINK_COUNT     2          /* Frequency of updating comm good blk */
#define BLOCK          ''
#define SPACE          ' '
  /* See also BLINKER_POS */

/* Key codes */
#define NUL       0x0000
#define ESC       0x001b

/* Extended sequence key codes */
#define F1        0x3b00
#define F8        0x4200
#define ALT_1     0x7800
#define ALT_0     0x8100
#define ALT_H     0x2300
#define ALT_K     0x2500
#define ALT_P     0x1900
#define ALT_Q     0x1000
#define ALT_R     0x1300
#define ALT_U     0x1600
#define SHIFT_F1  0x5400

#define CTRL_F1   0x5e00
#define ALT_F1    0x6800

/* Text positioning values (local): */
#define FULL_WINDOW          1,1,80,25
#define PROGRAM_NAME_POS     25,1
#define STATUS_HEADING_POS   1,3
#define STATUS_VALUE_POS     20,3
#define COMMAND_HEADING_POS  40,3
#define COMMAND_VALUE_POS    60,3
#define TIMERS_POS           1,21
#define OUTBLOCK_HEADING_POS 1,22
#define OUTBLOCK_POS         6,22
#define INBLOCK_HEADING_POS  1,24
#define INBLOCK_POS          6,24
#define SRC_ID_HEADING_POS   1,2
#define SRC_ID_POS           20,2
#define DEST_ID_HEADING_POS  40,2
#define DEST_ID_POS          60,2
#define BLINKER_POS          79,1

/* Locations for RCP data storage: */
VEHCMD  vehcmd;
VEHSTAT vehstat;

/* Timer stuff */
#define NUM_TIMERS 3
struct TIMER
{
   int     keystroke;
   byte    enabled;
   clock_t interval;   /* ticks, at 18.2 ticks per second */
   clock_t last_reset; /* also start time, in ticks from program start */
} timer[NUM_TIMERS] = { { F1, FALSE,  2, 0 }, /* Keepalive query */
                        { F8, FALSE,  7, 0 }, /* Kitchen sink query */
                        { 'J',FALSE,  7, 3 }  /* Send-image-triple cmd */
                      };

/* Prototypes, this file: */
void clear_timers(void);
void display_commands (int x, int y, VEHCMD *vcp);
void display_help(void);
void display_ids(int dest_id, int src_id);
void display_status (int x, int y, VEHSTAT *vsp);
void display_timers(void);
int  get_event(void);
void get_ids(int *dest_id, int *src_id);
int  initialize_packets (void);
void initialize_values (void);
void send_block(int port, int block_num, int dest_id);
void send_query(int port, int block_num, int dest_id, int src_id);
void setup_display (void);
void toggle_timer (int tnum);
void update_blinker(void);
void write_command_headings (int x, int y);
void write_status_headings (int x, int y);

/*************************************************************************
*
*  setup_rcp()
*
*  Args: none
*
*  Action:
*    Sets up the RCP packets and block function calls.
*
*  Returns:
*    OKAY if all RCP components were successfully initialized
*    !OKAY otherwise.  NOTE that a failure in RCP initialization is fatal.
*
*  Created:  WAA, 12-02-93
*
**************************************************************************/
int setup_rcp(void)
{
   char *fname = "setup_rcp";
   int  dummyint;  /* temporary storage */

   rcp_initialize(1);  /* Set up one set of structures */

   if (add_module (FIRST_PORT, rtlrblks, &dummyint) != RCP_SUCCESS)
   {
      printf ("%s: %s\n", fname, "add_module failed");
      return (!OKAY);
   }

   if (initialize_packets())
   {
      printf ("%s: %s\n", fname, "initialize_cmd failed");
      return (!OKAY);
   }

   /* Load meaningful values into RCP data locations */
   initialize_values();
   return (OKAY);
} /* setup_rcp */

/*************************************************************************
*
*  initialize_packets()
*
*  Args: none
*
*  Action:
*    Sets up all RCP packets contained in the blocks[] array.
*
*  Returns:
*    0 if there were no errors encountered
*    or a positive integer which is the number of errors encountered.
*
*  Created:  WAA, 12-02-93
*
**************************************************************************/
int initialize_packets()
{
   int s; /* Running sum of returned status */
   char *fname = "initialize_packets";  /* For tprint */

   s = 0;
   if (s += set_user_pkt_addr(0, PKT_CMD_SCALE_SPEED, &vehcmd.scale_speed))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_SCALE_SPEED");
   if (s += set_user_pkt_addr(0, PKT_CMD_VEH_MODE, &vehcmd.vehmode))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_VEH_MODE");
   if (s += set_user_pkt_addr(0, PKT_CMD_KILL_STATE, &vehcmd.kill))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_KILL_STATE");
   if (s += set_user_pkt_addr(0, PKT_CMD_PARKING_BRAKE, &vehcmd.parking_brake))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_PARKING_BRAKE");
   if (s += set_user_pkt_addr(0, PKT_CMD_THROTTLE, &vehcmd.left_throttle))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_THROTTLE");
   if (s += set_user_pkt_addr(0, PKT_CMD_THROTTLE1, &vehcmd.right_throttle))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_THROTTLE1");
   if (s += set_user_pkt_addr(0, PKT_CMD_GEAR, &vehcmd.left_gear))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_GEAR");
   if (s += set_user_pkt_addr(0, PKT_CMD_GEAR1, &vehcmd.right_gear))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_GEAR1");
   if (s += set_user_pkt_addr(0, PKT_CMD_PAN0, &vehcmd.pan))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_PAN0");
   if (s += set_user_pkt_addr(0, PKT_CMD_TILT0, &vehcmd.tilt))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_TILT0");
   if (s += set_user_pkt_addr(0, PKT_CMD_PACKAGE0, &vehcmd.package))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_PACKAGE0");
   if (s += set_user_pkt_addr(0, PKT_CMD_VIDEO0_XMITER, &vehcmd.video_power))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_VIDEO0_XMITER");
   if (s += set_user_pkt_addr(0, PKT_CMD_XPOS, &vehcmd.xpos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_XPOS");
   if (s += set_user_pkt_addr(0, PKT_CMD_YPOS, &vehcmd.ypos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_YPOS");
   if (s += set_user_pkt_addr(0, PKT_CMD_HI_RES_THROTTLE0,
       &vehcmd.hi_res_left_throttle))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_HI_RES_THROTTLE0");
   if (s += set_user_pkt_addr(0, PKT_CMD_HI_RES_THROTTLE1,
       &vehcmd.hi_res_right_throttle))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_HI_RES_THROTTLE1");
   if (s += set_user_pkt_addr(0, PKT_CMD_LOGGING, &vehcmd.logging))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_LOGGING");
   if (s += set_user_pkt_addr(0, PKT_CMD_SEND_IMAGE_TRIPLE,
       &vehcmd.send_triple))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_SEND_IMAGE_TRIPLE");


   if (s += set_user_pkt_addr(0, PKT_STAT_VEH_MODE, &vehstat.vehmode))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_VEH_MODE");
   if (s += set_user_pkt_addr(0, PKT_STAT_KILL_STATE, &vehstat.kill))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_KILL_STATE");
   if (s += set_user_pkt_addr(0, PKT_STAT_VIDEO0_XMITER, &vehstat.video_power))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_VIDEO0_XMITER");
   if (s += set_user_pkt_addr(0, PKT_STAT_PARKING_BRAKE, &vehstat.parking_brake))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PARKING_BRAKE");
   if (s += set_user_pkt_addr(0, PKT_STAT_ERROR_CODE, &vehstat.abort_code))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ERROR_CODE");
   if (s += set_user_pkt_addr(0, PKT_STAT_PITCH, &vehstat.left_pitch))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PITCH");
   if (s += set_user_pkt_addr(0, PKT_STAT_ROLL, &vehstat.roll))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ROLL");
   if (s += set_user_pkt_addr(0, PKT_STAT_PAN0, &vehstat.pan))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PAN0");
   if (s += set_user_pkt_addr(0, PKT_STAT_HEADING0, &vehstat.heading))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_HEADING0");
   if (s += set_user_pkt_addr(0, PKT_STAT_SPEED, &vehstat.speed))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_SPEED");
   if (s += set_user_pkt_addr(0, PKT_STAT_VOLTAGE_BATTERY,
      &vehstat.battery_voltage))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_VOLTAGE_BATTERY");
   if (s += set_user_pkt_addr(0, PKT_STAT_INCLINE2, &vehstat.right_pitch))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_INCLINE2");
   if (s += set_user_pkt_addr(0, PKT_STAT_XPOS, &vehstat.xpos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_XPOS");
   if (s += set_user_pkt_addr(0, PKT_STAT_YPOS, &vehstat.ypos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_YPOS");
   if (s += set_user_pkt_addr(0, PKT_STAT_ALTIMETER, &vehstat.altimeter))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ALTIMETER");
   if (s += set_user_pkt_addr(0, PKT_STAT_INC_OD, &vehstat.odometer))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_INC_OD");
   if (s += set_user_pkt_addr(0, PKT_STAT_ABS_OD, &vehstat.accum_odom))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ABS_OD");
   if (s += set_user_pkt_addr(0, PKT_STAT_ENGINE_TEMP, &vehstat.eng_temp))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ENGINE_TEMP");
   if (s += set_user_pkt_addr(0, PKT_STAT_PACKAGE0, &vehstat.package))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PACKAGE0");
   if (s += set_user_pkt_addr(0, PKT_STAT_SCALE_SPEED, &vehstat.scale_speed))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_SCALE_SPEED");
   if (s += set_user_pkt_addr(0, PKT_STAT_TURN_RATE, &vehstat.turn_rate))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_TURN_RATE");
   if (s += set_user_pkt_addr(0, PKT_STAT_LOGGING, &vehstat.logging))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_LOGGING");
   if (s += set_user_pkt_addr(0, PKT_STAT_WHEEL0_ENCODER, &vehstat.wheel0))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_WHEEL0_ENCODER");
   if (s += set_user_pkt_addr(0, PKT_STAT_WHEEL1_ENCODER, &vehstat.wheel1))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_WHEEL1_ENCODER");
   if (s += set_user_pkt_addr(0, PKT_STAT_WHEEL2_ENCODER, &vehstat.wheel2))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_WHEEL2_ENCODER");
   if (s += set_user_pkt_addr(0, PKT_STAT_WHEEL3_ENCODER, &vehstat.wheel3))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_WHEEL3_ENCODER");

   return (s);
} /* initialize_packets */

/*************************************************************************
*
*  initialize_values()
*
*  Args: none
*
*  Action:
*    Initializes meaningful values for RCP packets.  This was taken
*    directly from the RATLER source code.
*
*  Returns:
*    nothing
*
*  Created:  WAA, 12-02-93
*
**************************************************************************/
void initialize_values (void)
{
   vehstat.pan = RCP_CENTER_PAN;
   vehstat.xpos = 0;
   vehstat.ypos = 0;
   vehstat.heading = 0; /* North */
   vehstat.altimeter = 0;
   vehstat.vehmode = RCP_TELEOP;
   vehstat.kill = RCP_KILL;

   vehstat.video_power = RCP_OFF;
   vehstat.parking_brake = RCP_OFF;

   vehstat.abort_code = 0;
   vehstat.left_pitch = RCP_LEVEL_PITCH;
   vehstat.right_pitch = RCP_LEVEL_PITCH;
   vehstat.roll = RCP_LEVEL_ROLL;
   vehstat.speed = 0;

   vehstat.battery_voltage = 0;

   vehstat.odometer = 0;  /* incremental */
   vehstat.accum_odom = 0;
   vehstat.eng_temp = 0;

   vehstat.package = 0;
   vehstat.scale_speed = 0;  /* .01m/s/tick.  Default RCP is 1 == .1m/s/t */
   vehstat.turn_rate = 0;

   vehcmd.vehmode = RCP_TELEOP;
   vehcmd.kill = RCP_NO_KILL;
   vehcmd.parking_brake = RCP_OFF;  /* Parking brake OFF */
   vehcmd.pan = RCP_CENTER_PAN;  /* centered */
   vehcmd.xpos = 0;    /* arbitrary */
   vehcmd.ypos = 0;    /*     "     */
   vehcmd.tilt = RCP_CENTER_TILT; /* level */
   vehcmd.scale_speed = 0; /* 0.01 m/s/tick, not 0.1 m/s/tick */
   vehcmd.left_throttle = 0;
   vehcmd.right_throttle = 0;
   vehcmd.hi_res_left_throttle = 1000;
   vehcmd.hi_res_right_throttle = 1000;
   vehcmd.left_gear = RCP_FORWARD;
   vehcmd.right_gear = RCP_FORWARD;
   vehcmd.package = RCP_OFF;     /* Package (LIBS) off */
   vehcmd.video_power = RCP_OFF; /* Video power off */
   vehcmd.logging = RCP_OFF;
   vehcmd.send_triple = RCP_ON;

   return;
} /* initialize_values */

/*************************************************************************
*
*  display_commands()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*     vcp is a pointer to a VEHSTAT structure containing the vehicle
*        commands to be displayed.
*
*  Action:
*     Displays the status contained in the structure pointed to by
*     the input argument.  The location of the text is determined
*     by the x,y input arguments which define the upper-left corner
*     of a text window.  The data is displayed in the upper-left
*     corner of that text window.  The coordinates of the upper-left
*     corner in a full size screen are 1,1.  If the coords are
*     invalid, the window call is ignored.
*
*  Created:  WAA, 09-06-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*     WAA, 06-08-94 -- Added additional code from Roger S. to differentiate
*                      better between the modes.
*
**************************************************************************/
void display_commands(int x, int y, VEHCMD *vcp)
{
   window (x, y, 80, 25);

   gotoxy (VEHMODE_CMD_TEXT_POS);
   switch (vcp->vehmode)
	  {
		case RCP_SAFE :    cputs ("SAFE  ");
					          break;
		case RCP_SURV :    cputs ("SURV  ");
					          break;
		case RCP_PAUSE :   cputs ("PAUSE ");
					          break;
		case RCP_TELEOP :  cputs ("TELEOP");
					          break;
		case RCP_AUTON :   cputs ("AUTON ");
					          break;
		case RCP_QUIET :   cputs ("QUIET ");
					          break;
		default: cprintf ("%d",vcp->vehmode);
					break;
   }/* end switch */

   gotoxy (KILL_CMD_TEXT_POS);
   cprintf ("%-6s", vcp->kill ? "KILLED" : "ALIVE ");

   gotoxy (PBRAKE_CMD_TEXT_POS);
   cprintf ("%-s", vcp->parking_brake ? "PBRAKE ON" : "OFF      ");

   gotoxy (VIDEO_POWER_CMD_TEXT_POS);
   cprintf ("%-s", vcp->video_power ? "VIDEO ON" : "OFF     ");

   gotoxy (THROTTLE_CMD_TEXT_POS);
   cprintf ("%-3d    %-3d", vcp->left_throttle, vcp->right_throttle);

   gotoxy (GEAR_CMD_TEXT_POS);
   cprintf ("%-3d    %-3d", vcp->left_gear, vcp->right_gear);

   gotoxy (PANTILT_CMD_TEXT_POS);
   cprintf ("%-5lx, %-4x", vcp->pan, vcp->tilt);

   gotoxy (VEHPOS_CMD_TEXT_POS);
   cprintf ("%-5ld, %-4ld", vcp->xpos, vcp->ypos);

   gotoxy (SCALE_SPEED_CMD_TEXT_POS);
   cprintf ("%-3d", vcp->scale_speed);

   gotoxy (LIBS_CMD_TEXT_POS);
   cprintf ("%-2d", vcp->package);

   gotoxy (HI_RES_THROTTLE_CMD_TEXT_POS);
   cprintf ("%-5d  %-5d", vcp->hi_res_left_throttle,
                          vcp->hi_res_right_throttle);

   gotoxy (LOGGING_CMD_TEXT_POS);
   cprintf ("%-3d", vcp->logging);

   gotoxy (IMAGE_TRIPLE_CMD_TEXT_POS);
   cprintf ("%-3d", vcp->send_triple);

   window (FULL_WINDOW);

   return;
} /* display_commands */


/*************************************************************************
*
*  display_status()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*     vsp is a pointer to a VEHSTAT structure containing the vehicle
*        status to be displayed.
*
*  Action:
*     Displays the status contained in the structure pointed to by
*     the input argument.  The location of the text is determined
*     by the x,y input arguments which define the upper-left corner
*     of a text window.  The data is displayed in the upper-left
*     corner of that text window.  The coordinates of the upper-left
*     corner in a full size screen are 1,1.  If the coords are
*     invalid, then the window call is ignored.
*
*  Created:  WAA, 09-06-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*     WAA, 03-30-94 -- Removed altitude display to make more room for
*                      x,y values.
*     WAA, 06-08-94 -- Added additional code from Roger S. to differentiate
*                      better between the modes.
*
**************************************************************************/
void display_status(int x, int y, VEHSTAT *vsp)
{
   window (x, y, 80, 25);

   gotoxy (VEHMODE_STAT_TEXT_POS);
   switch (vsp->vehmode)
	  {
		case RCP_SAFE :    cputs ("SAFE  ");
					          break;
		case RCP_SURV :    cputs ("SURV  ");
					          break;
		case RCP_PAUSE :   cputs ("PAUSE ");
					          break;
		case RCP_TELEOP :  cputs ("TELEOP");
					          break;
		case RCP_AUTON :   cputs ("AUTON ");
					          break;
		case RCP_QUIET :   cputs ("QUIET ");
					          break;
		default: cprintf ("%d",vsp->vehmode);
		         break;
	  }/* end switch */


   gotoxy (KILL_STAT_TEXT_POS);
   cprintf ("%-6s", vsp->kill ? "KILLED" : "ALIVE ");

   gotoxy (PBRAKE_STAT_TEXT_POS);
   cprintf ("%-s", vsp->parking_brake ? "PBRAKE ON" : "OFF      ");

   gotoxy (VIDEO_POWER_STAT_TEXT_POS);
   cprintf ("%-s", vsp->video_power ? "VIDEO ON" : "OFF     ");

   gotoxy (PAN_STAT_TEXT_POS);
   cprintf ("%-5lx", vsp->pan);

   gotoxy (HEADING_STAT_TEXT_POS);
   cprintf ("%-3d", vsp->heading);

   gotoxy (VEHPOS_STAT_TEXT_POS);
   cprintf ("%-6ld, %-6ld", vsp->xpos, vsp->ypos);

   gotoxy (ABORT_CODE_TEXT_POS);
   cprintf ("%-3d", vsp->abort_code);

   gotoxy (LEFT_PITCH_TEXT_POS);
   cprintf ("%-3d", vsp->left_pitch);

   gotoxy (RIGHT_PITCH_TEXT_POS);
   cprintf ("%-3d", vsp->right_pitch);

   gotoxy (ROLL_TEXT_POS);
   cprintf ("%-3d", vsp->roll);

   gotoxy (SPEED_TEXT_POS);
   cprintf ("%-3d", vsp->speed);

   gotoxy (VOLTAGE_TEXT_POS);
   cprintf ("%-2d", vsp->battery_voltage);

   gotoxy (ODOMETER_TEXT_POS);
   cprintf ("%-7d  %-7d", vsp->odometer, vsp->accum_odom);

   gotoxy (ENGINE_TEMP_TEXT_POS);
   cprintf ("%-3d", vsp->eng_temp);

   gotoxy (LIBS_STATUS_TEXT_POS);
   cprintf ("%-3d", vsp->package);

   gotoxy (SCALE_SPEED_STAT_TEXT_POS);
   cprintf ("%-3d", vsp->scale_speed);

   gotoxy (TURN_RATE_TEXT_POS);
   cprintf ("%-4d", vsp->turn_rate);

   gotoxy (LOGGING_STAT_TEXT_POS);
   cprintf ("%-3d", vsp->logging);

   gotoxy (WHEEL0_TEXT_POS);
   cprintf ("%-6ld", vsp->wheel0);

   gotoxy (WHEEL1_TEXT_POS);
   cprintf ("%-6ld", vsp->wheel1);

   gotoxy (WHEEL2_TEXT_POS);
   cprintf ("%-6ld", vsp->wheel2);

   gotoxy (WHEEL3_TEXT_POS);
   cprintf ("%-6ld", vsp->wheel3);

   window (FULL_WINDOW);

   return;
} /* display_status */

/*************************************************************************
*
*  display_ids()
*
*  Args:
*     dest_id and src_id are RCP ID's, ranging from 0 through 127
*
*  Action:
*     Displays the given values of destination RCP ID and source
*     RCP ID.  This routine does not employ any text windowing.
*
*  Created:  WAA, 03-30-94
*
**************************************************************************/
void display_ids(int dest_id, int src_id)
{
   gotoxy (SRC_ID_POS);
   cprintf ("%d", src_id);

   gotoxy (DEST_ID_POS);
   cprintf ("%d", dest_id);

   return;
} /* display_ids */

/*************************************************************************
*
*  display_timers()
*
*  Args: none
*
*  Action:
*     Displays the timer numbers and whether or not they are enabled.
*     Enabled timers have their numbers prefixed by a '+'.
*
*  Created:  WAA, 04-05-94
*
**************************************************************************/
void display_timers(void)
{
   int i;

   gotoxy (TIMERS_POS);
   cputs ("Timers:");
   for (i = 0; i < NUM_TIMERS; ++i)
      cprintf (" %c%d", timer[i].enabled ? '+' : ' ', i + 1);

   return;
} /* display_timers */

/*************************************************************************
*
*  display_help()
*
*  Args: none
*
*  Action:
*     Displays a help screen that tells what keystrokes are available
*     and what they do.  Since blocks can still be sent and received,
*     the blocks are displayed at the bottom as usual.
*
*  Created:  WAA, 04-05-94
*
**************************************************************************/
void display_help()
{
   window (10, 1, 80,25); /* Scoot margin over */

   cputs ("                   HELP SCREEN\n\r\n\r");
   cputs ("In general:\n\r");
   cputs ("Plain keys -- STATUS\n\r");
   cputs ("Shift keys -- COMMANDS\n\r");
   cputs ("Alt keys   -- internal\r\n");

   cputs ("\n\r");

   cputs ("Lower-case letters -- Send corresponding STATUS block\n\r");
   cputs ("Upper-case letters -- Send corresponding COMMAND block\n\r");

   cputs ("Function keys -- Send corresponding query for STATUS block\n\r");
   cputs ("Shift fn keys -- Send corresponding query for COMMAND block\n\r");

   cputs ("\n\r");

   cputs ("Alt-numeric keys -- Toggle corresponding TIMER enable\n\r");
   cputs ("                    '+' means enabled\n\r");
   cputs ("Alt-0 (zero)     -- Disable all timers\n\r");
   cputs ("Alt-k, alt-u     -- Send KILL/UNKILL command\n\r");
   cputs ("Alt-p  -- Toggle and send the parking brake command\n\r");
   cputs ("Alt-r  -- REFRESH screen, return to regular screen\n\r");
   cputs ("Alt-h  -- toggle this HELP screen\n\r");
   cputs ("Alt-q or Esc EXITS the program.\n\r");
   cputs ("Blinking block in upper right -- good RCP blocks received\n\r");

   window (FULL_WINDOW);

   gotoxy (OUTBLOCK_HEADING_POS);
   cputs ("Out:");
   gotoxy (INBLOCK_HEADING_POS);
   cputs ("In:");

} /* display_help */

/*************************************************************************
*
*  write_command_headings()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*
*  Action:
*    Writes the text for each of the command values.
*
*  Created:  WAA, 09-17-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*
**************************************************************************/
void write_command_headings(int x, int y)
{
   window (x, y, 80,25);

   gotoxy (VEHMODE_CMD_TEXT_POS);
   cputs ("Commanded mode");

   gotoxy (KILL_CMD_TEXT_POS);
   cputs ("Commanded kill");

   gotoxy (PBRAKE_CMD_TEXT_POS);
   cputs ("Parking brake");

   gotoxy (VIDEO_POWER_CMD_TEXT_POS);
   cputs ("Video power");

   gotoxy (THROTTLE_CMD_TEXT_POS);
   cputs ("Throttles");

   gotoxy (GEAR_CMD_TEXT_POS);
   cputs ("Gears");

   gotoxy (PANTILT_CMD_TEXT_POS);
   cputs ("Pan/tilt");

   gotoxy (VEHPOS_CMD_TEXT_POS);
   cputs ("Destination");

   gotoxy (SCALE_SPEED_CMD_TEXT_POS);
   cputs ("Speed scale");

   gotoxy (LIBS_CMD_TEXT_POS);
   cputs ("LIBS command");

   gotoxy (HI_RES_THROTTLE_CMD_TEXT_POS);
   cputs ("Hi-res throttles");

   gotoxy (LOGGING_CMD_TEXT_POS);
   cputs ("Log cmd");

   gotoxy (IMAGE_TRIPLE_CMD_TEXT_POS);
   cputs ("Image triple cmd");

   window (FULL_WINDOW);

   return;
} /* write_command_headings */

/*************************************************************************
*
*  write_status_headings()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*
*  Action:
*    Writes the text for each of the status values.  Uses same gotoxy()
*    values as in tt_show_status(), but with the display window
*    offset to the left.
*
*  Created:  WAA, 09-17-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*
**************************************************************************/
void write_status_headings(int x, int y)
{
   window (x, y, 80,25);

   gotoxy(VEHMODE_STAT_TEXT_POS);
   cputs ("Mode");

   gotoxy(KILL_STAT_TEXT_POS);
   cputs ("Kill");

   gotoxy(PBRAKE_STAT_TEXT_POS);
   cputs ("Parking brake");

   gotoxy(VIDEO_POWER_STAT_TEXT_POS);
   cputs ("Video power");

   gotoxy(PAN_STAT_TEXT_POS);
   cputs ("Pan");

   gotoxy(HEADING_STAT_TEXT_POS);
   cputs ("Heading");

   gotoxy(VEHPOS_STAT_TEXT_POS);
   cputs ("Location (x,y,z)");

   gotoxy(ABORT_CODE_TEXT_POS);
   cputs ("Abort code");

   gotoxy(LEFT_PITCH_TEXT_POS);
   cputs ("Left body");

   gotoxy(ROLL_TEXT_POS);
   cputs ("Roll");

   gotoxy(RIGHT_PITCH_TEXT_POS);
   cputs ("Right body");

   gotoxy(SPEED_TEXT_POS);
   cputs ("Speed");

   gotoxy(VOLTAGE_TEXT_POS);
   cputs ("Voltage");

   gotoxy(ODOMETER_TEXT_POS);
   cputs ("Odometers");

   gotoxy(ENGINE_TEMP_TEXT_POS);
   cputs ("Engine temp");

   gotoxy(LIBS_STATUS_TEXT_POS);
   cputs ("LIBS status");

   gotoxy(SCALE_SPEED_STAT_TEXT_POS);
   cputs ("Scale speed");

   gotoxy (TURN_RATE_TEXT_POS);
   cputs ("Turn rate");

   gotoxy (LOGGING_STAT_TEXT_POS);
   cputs ("Log status");

   gotoxy (WHEEL0_TEXT_POS);
   cputs ("Wheel0");

   gotoxy (WHEEL1_TEXT_POS);
   cputs ("Wheel1");

   gotoxy (WHEEL2_TEXT_POS);
   cputs ("Wheel2");

   gotoxy (WHEEL3_TEXT_POS);
   cputs ("Wheel3");

   window (FULL_WINDOW);

   return;
} /* write_status_headings */

/*************************************************************************
*
*  send_block()
*
*  Args:
*     port is the COM port to use, either 1 or 2.
*     block_num is the block to build and send
*
*  Action:
*    Builds and sends the given block.
*
*  Returns: nothing
*
*  Created: WAA, 03-30-94
*
**************************************************************************/
void send_block(int port, int block_num, int dest_id)
{
   byte outblock[80]; /* Storage for block to send */
   int i, count;      /* Counter and number of bytes in block to send */

   gotoxy (OUTBLOCK_POS);
   clreol();

   if (build_block(FIRST_PORT, block_num, outblock,
                       &count, dest_id) == RCP_SUCCESS)
   {
      if (send_stream (port, outblock, count) == OKAY)
         for (i=0; i<count; ++i)
            printf ("%02x ", outblock[i]);
         else
            printf ("Error sending block %d", block_num);
   }
   else
      printf ("Block %d data storage not defined", block_num);
   return;
} /* send_block */

/*************************************************************************
*
*  send_query()
*
*  Args:
*     port is the COM port to use, either 1 or 2.
*     block_num is the block to build a query for and send
*
*  Action:
*    Builds (byte-by-byte) and sends a query for the given block.
*
*  Returns: nothing
*
*  Created: WAA, 03-30-94
*
**************************************************************************/
void send_query(int port, int block_num, int dest_id, int src_id)
{
   byte outblock[QUERY_LENGTH]; /* Storage for query to send */
   int i, count;      /* Counter and number of bytes in block to send */

   outblock[0] = src_id | HIGH_BIT;
   outblock[1] = dest_id | HIGH_BIT;
   outblock[2] = block_num | HIGH_BIT;
   outblock[3] = QUERY_CHAR;
   count = 4;
   insert_crc (outblock, &count, TRUE);

   gotoxy (OUTBLOCK_POS);
   clreol();

   if (count != QUERY_LENGTH)
      printf ("Wrong number of chars (%d) in built query.  Should be %d",
              QUERY_LENGTH);
   else
   {
      if (send_stream (port, outblock, count) == OKAY)
         for (i=0; i<count; ++i)
            printf ("%02x ", outblock[i]);
      else
         printf ("Error sending block %d", block_num);
   }
   return;
} /* send_query */

/*************************************************************************
*
*  setup_display ()
*
*  Args: none
*
*  Action:
*    Clears the screen and writes the headings for the RCP values
*
*  Returns: nothing
*
*  Created: WAA, 12-06-93
*
**************************************************************************/
void setup_display (void)
{
   clrscr ();

   gotoxy (PROGRAM_NAME_POS);
   cputs (PROGRAM_NAME);
   cputs (" (ESC exits)");

   write_command_headings (COMMAND_HEADING_POS);
   write_status_headings (STATUS_HEADING_POS);

   gotoxy (SRC_ID_HEADING_POS);
   cputs ("My ID");

   gotoxy (DEST_ID_HEADING_POS);
   cputs ("Destination ID");

   gotoxy (OUTBLOCK_HEADING_POS);
   cputs ("Out:");
   gotoxy (INBLOCK_HEADING_POS);
   cputs ("In:");
   return;
} /* setup_display */

/*************************************************************************
*
*  get_ids ()
*
*  Args:
*     &dest_id is set to be the RCP ID of the repient of blocks from
*        this program
*     &src_id is set to be the RCP ID of this program
*
*  Action:
*     The operator inputs the destination and source RCP ID's to be
*     be used by this program.  The ID's must be in the range 0-127.
*     Note that 127 is the broadcast ID.
*
*  Returns: nothing
*
*  Created: WAA, 03-30-94
*
**************************************************************************/
void get_ids(int *dest_id, int *src_id)
{
   char str[80]; /* String from the keyboard for determining com port */

   /* Initialize with temporary, invalid values */
   *dest_id = -1;
   *src_id = -1;

   while (*src_id == -1)
   {
      printf ("Source (my) RCP ID? [0..127, 0=default] ");
      gets (str);

      if (sscanf(str, "%d", src_id) != 1)
         *src_id = 0;

      /* ID must be in range, otherwise it asks you again */
      if (*src_id < 0 || *src_id > 127)
         *src_id = -1;
   }

   while (*dest_id == -1)
   {
      printf ("Destination RCP ID? [0..127, 0=default] ");
      gets (str);

      if (sscanf(str, "%d", dest_id) != 1)
         *dest_id = 0;

      /* ID must be in range, otherwise it asks you again */
      if (*dest_id < 0 || *dest_id > 127)
         *dest_id = -1;
   }

   return;
} /* get_ids */

/*************************************************************************
*
*  get_event()
*
*  Args: none
*
*  Action:
*     This routine returns a non-zero value if there is an "event"
*     to be acted upon.  An event may be a key press or a timer
*     expiration.  Keyboard events take precedence over timed events
*     (to make sure you can quit the program, for instance).  Events
*     with a lower index in the event array have precedence over
*     those events with a higher index.
*
*  Returns:
*     non-zero event code if a keyboard or other event has occurred
*     0 if no events seen
*
*  Created:  WAA, 04-04-94
*
**************************************************************************/
int get_event()
{
   int c;
   int i;

   c = 0;
   if (kbhit())
   {
      c = getch() & LOW_BYTE;  /* Make sure high byte is clear */
      if (c == '\0')  /* extended char sequence */
         c = (getch() << 8) & HIGH_BYTE;
   }
   else
   {
      /* Search through last_reset times to find timer that has expired */
      for (i = 0; i < NUM_TIMERS; ++i)
      {
         if (timer[i].enabled == FALSE)
            continue;

         if (clock() - timer[i].last_reset > timer[i].interval)
         {
            c = timer[i].keystroke;
            timer[i].last_reset = clock();
            break; /* out of for loop */
         }
      }
   }

   return (c);
} /* get_event */

/*************************************************************************
*
*  toggle_timer(int tnum)
*
*  Args: tnum is the index of the timer to
*
*  Action:
*     Toggles the state of the given timer.  If enabled, disables,
*     if disabled, enables.
*
*  Created:  WAA, 04-05-94
*
**************************************************************************/
void toggle_timer (int tnum)
{
   /* make sure timer index is valid */
   if (tnum < 0 || tnum >= NUM_TIMERS)
      return;

   timer[tnum].enabled = !timer[tnum].enabled;

   return;
} /* toggle_timer */

/*************************************************************************
*
*  clear_timers()
*
*  Args: none
*
*  Action:
*     Disables all timers.
*
*  Created:  WAA, 04-05-94
*
**************************************************************************/
void clear_timers(void)
{
   int i;

   for (i = 0; i < NUM_TIMERS; ++i)
      timer[i].enabled = FALSE;

   return;
} /* clear_timers */

/*************************************************************************
*
*  update_blinker()
*
*  Args: none
*
*  Action:
*     After this has been called a number of times (BLINK_COUNT),
*     the blink character is toggled.  To be used to indicate
*     receipt of valid blocks.
*
*  Created:  WAA, 04-01-94
*
**************************************************************************/
void update_blinker()
{
   static int  blinker = 0;
   static char blink_char = BLOCK;

   /* Update comm-good blinker */
   if (++blinker == BLINK_COUNT)
   {
      blinker = 0;
      gotoxy (BLINKER_POS);
      blink_char = (blink_char == SPACE) ? BLOCK : SPACE;
      putch (blink_char);
   }
   return;
} /* update_blinker */

/*************************************************************************
*
*  main()
*
*  Args: none
*
*  Action:
*    Attempts to set up one RCP structure.  If this fails, all bets are
*    off anyway and the program exits.  The user inputs which comm port
*    to use (either 1 or 2).  The user is then asked to input the RCP
*    ID's to use for this node and for the destination.  Comm interrupts
*    are initialized next, and then the screen display is initialized.
*
*    Since this is a single thread program (aack!), it has a main loop
*    for polling for events.  Events can be receipt of an RCP block,
*    a keystroke, or a timer expiration (which looks like a keystroke).
*
*    Received blocks are RCP validated and if they are a block
*    recognized by this system, they are partitioned.  The parsed
*    values are displayed, if the regular display screen (not help)
*    is up.
*
*    Keystrokes are used to send blocks.  Blocks that are sent include
*    status, command, and query blocks.
*
*    Timers can be set up to simulate periodic keystrokes, say, to
*    send a keepalive query.
*
*    The regular display shows the values of all RCP data.  At the
*    bottom of the screen are displayed the state of all timers,
*    as well as the received and outgoing blocks.  If received
*    blocks are RCP invalid or unrecognized by this system, a
*    message is printed following the block bytes.  If a keystroke
*    is unrecognized by the program, a message giving the keycode
*    is output instead of an outgoing block.  This helps in adding
*    keycodes to the program at a later time.  In the upper right
*    corner of the screen is a character which "blinks" as valid
*    blocks are received by this program; an aid in determining
*    if good communication is occurring.
*
*    More information on the usage of this program can be found in
*    the program header.
*
*  Created:  WAA, 12-02-93
*
**************************************************************************/
main ()
{
   int  port;       /* Which com port is active, 1 or 2.  0 is undefined */
   char str[80];    /* String from the keyboard for determining com port */
   int  block_num;  /* Number of block to build */
   int  i, count;   /* Counter and number of bytes in block to send */
   byte inblock[80];  /* Storage for received block */
   byte command_byte; /* Command byte of received block */
   unsigned int c;    /* Character from keyboard */
   int  dest_id, src_id;  /* RCP ID's for dest and source ID */
   Screen_type screen;    /* Which screen currently displayed */

   if (setup_rcp() != OKAY)
   {
      printf ("Unable to initialize RCP.\n");
      exit (0);
   }

   clrscr();
   port = 0;
   while (port == 0)
   {
      printf ("COM1 or COM2? [1|2, default=1] ");
      gets (str);

      if (sscanf(str, "%d", &port) != 1)
         port = 1;
   }

   /* Initialize the RCP ID's */
   get_ids (&dest_id, &src_id);

   save_imr(); /* Interrupt Mask Register needs to be saved before int init */

   if (port == 2)
   {
      init_COM2();
      init_COM2_int();
   }
   else
   {
      init_COM1();
      init_COM1_int();
   }

   /* Initialize the screen */
   _setcursortype (_NOCURSOR);
   screen = REGULAR_SCREEN;
   setup_display ();
   display_ids (dest_id, src_id);
   display_timers ();
   display_commands (COMMAND_VALUE_POS, &vehcmd);
   display_status (STATUS_VALUE_POS, &vehstat);

   c = NUL;
   while (c != ESC && c != ALT_Q)
   {
      while (get_stream (port, inblock, &count))
      {
         gotoxy (INBLOCK_POS);
         clreol();
         for (i=0; i<count; ++i)
            printf ("%02x ", inblock[i]);
         if (validate_block (count, inblock) == RCP_SUCCESS)
         {
            command_byte = inblock[count-1];
            if (is_fixed_command(command_byte))
            {
               if (command_byte == QUERY_CHAR)
               {
                  if (count < QUERY_LENGTH)
                     cputs ("input query too short");
                  else
                  {
                     block_num = inblock[count - 5] & ~HIGH_BIT;
                     if (is_vehicle_command(block_num) ||
                         is_vehicle_status(block_num))
                        send_block (port, block_num, dest_id);
                     else
                        cprintf ("Don't know how to build block %d",
                                 block_num);
                  }
               }
            }
            else
            {
               /* Check to see if block is known */
               if (is_vehicle_command(command_byte) ||  /* warning okay */
                   is_vehicle_status(command_byte))
               {
                  if (partition_block (FIRST_PORT, command_byte, inblock)
                      == RCP_SUCCESS)
                  {
                     if (screen != HELP_SCREEN)
                     {
                        /* Command block */          /* Warning below okay */
                        if (is_vehicle_command(command_byte))
                           display_commands (COMMAND_VALUE_POS, &vehcmd);
                        else  /* Status block */
                        if (is_vehicle_status(command_byte))
                           display_status (STATUS_VALUE_POS, &vehstat);
                     }
                  }
                  else /* couldn't partition, RCP init not complete */
                     cprintf ("Block %d data storage not defined",
                              command_byte);
               }
               else  /* Not a command or a status... */
                  cputs ("unknown block");
            } /* not a fixed command */
            update_blinker();
         }
         else /* validate failed */
            cputs ("invalid");

      } /* get stream from port */

      if (c = get_event()) /* warning okay */
      {
         if ((c & LOW_BYTE) != 0) /* Not extended key sequence */
         {
            c &= LOW_BYTE;  /* make sure we got just the low byte */
            block_num = -1;

            if (c >= 'A' && c <= ('A'+ MAX_CMD_BLOCK - MIN_CMD_BLOCK))
               block_num = c - 'A' + MIN_CMD_BLOCK;
            else
            if (c >= 'a' && c <= ('a' + MAX_STATUS_BLOCK - MIN_STATUS_BLOCK))
               block_num = c - 'a' + MIN_STATUS_BLOCK;
            else
            {
               switch (c)
               {
                  case ESC :
                  case ALT_Q :
                          break;  /* recognized but "handled" elsewhere */
                  default :
                          gotoxy (OUTBLOCK_POS);
                          clreol();
                          cprintf ("Unused key code %#06x", c);
                          block_num = -1;
                          break;
               } /* switch */
            } /* not an alphabet key */

            if (block_num >= 0)
               send_block (port, block_num, dest_id);
            /* else fall through */
         }
         else /* extended key sequence */
         {
            c &= HIGH_BYTE;  /* make sure we got just the high byte */
            block_num = -1;

               /* Remember thine precedences:  +- >> >= & &&  hi to lo */

            /* Shift Fn keys */
            if (c >= SHIFT_F1 &&
                c <= SHIFT_F1 + ((MAX_CMD_BLOCK - MIN_CMD_BLOCK)<<8))
               block_num = (((c - SHIFT_F1) >> 8) & LOW_BYTE) + MIN_CMD_BLOCK;
            else /* Just Fn keys */
            if (c >= F1 &&
                c <= F1 + ((MAX_STATUS_BLOCK - MIN_STATUS_BLOCK)<<8))
               block_num = (((c - F1) >> 8) & LOW_BYTE) + MIN_STATUS_BLOCK;
            else /* check timer stuff */
            if (c >= ALT_1 && c < ALT_1 + (NUM_TIMERS<<8))
            {
               toggle_timer ((c - ALT_1>>8) & LOW_BYTE);
               display_timers ();
            }
            else
            if (c == ALT_0)
            {
               clear_timers();
               display_timers();
            }
            else
            {
               switch (c)
               {
                  case ALT_R:  /* Refresh screen */
                          screen = REGULAR_SCREEN;
                          setup_display();
                          display_ids(dest_id, src_id);
                          display_timers();
                          display_commands (COMMAND_VALUE_POS, &vehcmd);
                          display_status (STATUS_VALUE_POS, &vehstat);
                          break;
                  case ALT_K: /* Kill */
                          vehcmd.kill = RCP_KILL;
                          send_block (port, MODE_CHANGE_CMD_BLOCK, dest_id);
                          display_commands (COMMAND_VALUE_POS, &vehcmd);
                          send_block (port, MODE_CHANGE_CMD_BLOCK, dest_id);
                          break;
                  case ALT_U: /* Unkill */
                          vehcmd.kill = RCP_NO_KILL;
                          send_block (port, MODE_CHANGE_CMD_BLOCK, dest_id);
                          display_commands (COMMAND_VALUE_POS, &vehcmd);
                          send_block (port, MODE_CHANGE_CMD_BLOCK, dest_id);
                          break;
                  case ALT_P: /* Toggle parking brake */
                          if (vehcmd.parking_brake == RCP_ON)
                             vehcmd.parking_brake = RCP_OFF;
                          else
                             vehcmd.parking_brake = RCP_ON;
                          send_block (port, TRANSMITTER_CMD_BLOCK, dest_id);
                          display_commands (COMMAND_VALUE_POS, &vehcmd);
                          break;
                  case ALT_H: /* Toggle help screen */
                          if (screen == HELP_SCREEN)
                          {
                             screen = REGULAR_SCREEN;
                             setup_display();
                             display_ids(dest_id, src_id);
                             display_timers();
                          }
                          else
                          {
                             screen = HELP_SCREEN;
                             clrscr();
                             display_help();
                          }
                          break;
                  default:
                          gotoxy (OUTBLOCK_POS);
                          clreol();
                          cprintf ("Extended char sequence %#06x", c);
                          block_num = -1;
                          break;
               }
            } /* not an alphabet key */

            if (block_num >= 0)
               send_query (port, block_num, dest_id, src_id);
         } /* extended key sequence */

      } /* if event */

   } /* main loop */

   restore_imr();

   if (port == 2)
      restore_COM2_int();
   else
      restore_COM1_int();

   _setcursortype(_NORMALCURSOR);

   return (0);
} /* main */
