/*------------------------------------------------------------------------
|       RTD_LCD.C   : Liquid Crystal Display Handling Routines 1.1
|       AUTHOR      : original souce code (c) RTD Scandinavia Oy
|		MODIFIED BY	: Real Time Devices, USA
|       DATE        : 1992, 1993, 1994
|
|       For use with LCD character displays that use the Hitachi HD44780
|       display controller.  Works with 1 or 2 line displays.
|
| Note: Ensure the parallel port is a bidirectional port
|-------------------------------------------------------------------------*/

#include        "rtd_lcd.h"
#include        <conio.h>
#include        <time.h>

/* Global Variables */
static char LCD_columns;                  /* # columns in display */
static char LCD_rows;                     /* # rows in display   */
static char Cursor_Column;				  /* Current cursor column */
static char Cursor_Row;				  	  /* Current cursor row */
static int LCD_control_port;              /* LCD control port   */
static int LCD_status_port;               /* LCD status port   */
static int LCD_data_port;                 /* LCD data port    */

/*
   LCD_control defines
   Control bits are:
		AUTOFD = RS		(control bit is inverted)
		INIT   = R/W
		SLCTIN = Enable  (control bit is inverted)
*/
/* defines for display Enable signal using SLCTIN */
#define LCD_ENABLE  0
#define LCD_DISABLE 8

/* defines for display R/W (Read/Write) signal using INIT */
#define LCD_WRITE   0
#define LCD_READ    4

/* defines for display RS (Register Select) signal */
#define LCD_DATA    0
#define LCD_INST    2

/* LCD data port direction control bit */
#define LCD_DATA_OUT    0
#define LCD_DATA_IN     0x20

/*---------------------------------------------------------------
| Function:     millisecond_delay( clock_t wait )
| Input:        wait - time in milliseconds
| Returns:      Nothing
| Purpose:      Pauses for a specified number of milliseconds.
|----------------------------------------------------------------*/
void millisecond_delay( clock_t wait )
{
clock_t t1, t2;

	if( !wait )
		return;

	t1 = wait + clock();
	do
	{
		t2 = clock();
	} while( t2 < t1 );
} /* end millisecond_delay */


/*---------------------------------------------------------------
| Function:     int LCD_Read_Status(void)
| Input:        Nothing
| Returns:      Status byte as integer from LCD
|               Bit 7 is busy flag
|               bits 6-0 are address counter contents
| Purpose:      Sends an instruction byte to the LCD by shaking
|               the LPT port control lines
|----------------------------------------------------------------*/
int LCD_Read_Status(void)
{
int status;

	/* Set control line twice, needs 140 ns delay */
	outp(LCD_control_port, LCD_DISABLE | LCD_READ | LCD_INST | LCD_DATA_IN);
	outp(LCD_control_port, LCD_DISABLE | LCD_READ | LCD_INST | LCD_DATA_IN);

	/* Enable LCD */
	outp(LCD_control_port, LCD_ENABLE | LCD_READ | LCD_INST | LCD_DATA_IN);

	/* Read Data, needs 320 ns delay */
	status = inp(LCD_data_port);
	status = inp(LCD_data_port);
	status = inp(LCD_data_port);

	/* Disable LCD */
	outp(LCD_control_port, LCD_DISABLE | LCD_READ | LCD_INST | LCD_DATA_OUT);

	return (status);
}/* end LCD_Read_Status */


/*---------------------------------------------------------------
| Function:     LCD_Send_Instruction(unsigned char instr, char wait)
| Input:        unsigned char instruction
|               char wait, 0 = without busy wait, 1 = with busy wait
| Returns:      Nothing
| Purpose:      Sends an instruction byte to the LCD by shaking
|               the LPT port control lines
|----------------------------------------------------------------*/
void LCD_Send_Instruction(unsigned char instr, char wait)
{
	if (wait) {
		/* Wait until LCD is ready */
		while ( LCD_Read_Status() & 0x80 );
	} /* end if */

	/* Set control line twice, needs 140 ns delay */
	outp(LCD_control_port, LCD_DISABLE | LCD_WRITE | LCD_INST | LCD_DATA_OUT);
	outp(LCD_control_port, LCD_DISABLE | LCD_WRITE | LCD_INST | LCD_DATA_OUT);

	/* Enable LCD */
	outp(LCD_control_port, LCD_ENABLE | LCD_WRITE | LCD_INST | LCD_DATA_OUT);

	/* Write Data, needs 195 ns delay */
	outp(LCD_data_port, instr);
	outp(LCD_data_port, instr);

	/* Disable LCD */
	outp(LCD_control_port, LCD_DISABLE | LCD_WRITE | LCD_INST | LCD_DATA_OUT);
} /* end LCD_Send_Instruction */


/*---------------------------------------------------------------
| Function:     void LCD_Move_Cursor(char row, char col)
| Input:        char row, col
| Returns:      Nothing
| Purpose:      Places the LCD cursor at row & col
|----------------------------------------------------------------*/
void LCD_Move_Cursor(char row, char col)
{
unsigned int status;
int column;

	if ( (row < 1) || (row > LCD_rows) )
		row = 1;

	column = col - 1;
	if (column < 0 || column > (LCD_columns - 1))
		column = 0;

	if (row  == 1)
		status = (0x00 + column);
	else
		status = (0x40 + column);

	 /* Position the LCD cursor. */
	 LCD_Send_Instruction((unsigned char)(0x80|(unsigned char)status), 1);

	 Cursor_Row = row;
	 Cursor_Column = col;
} /* end LCD_Move_Cursor */


/*---------------------------------------------------------------
| Function:     void LCD_Char_Out(unsigned char data)
| Input:        unsigned char mki
| Returns:      Nothing
| Purpose:      Writes one character at current position
|----------------------------------------------------------------*/
void LCD_Char_Out(unsigned char data)
{
	/* Wait until LCD is ready */
	while ( LCD_Read_Status() & 0x80 );

	/* Set control line twice, needs 140 ns delay */
	outp(LCD_control_port, LCD_DISABLE | LCD_WRITE | LCD_DATA | LCD_DATA_OUT);
	outp(LCD_control_port, LCD_DISABLE | LCD_WRITE | LCD_DATA | LCD_DATA_OUT);

	/* Enable LCD */
	outp(LCD_control_port, LCD_ENABLE | LCD_WRITE | LCD_DATA | LCD_DATA_OUT);

	/* Write Data, needs 195 ns delay */
	outp(LCD_data_port, data);     /* Out with the character. */
	outp(LCD_data_port, data);     /* Out with the character. */

	/* Disable LCD */
	outp(LCD_control_port, LCD_DISABLE | LCD_WRITE | LCD_DATA | LCD_DATA_OUT);

	Cursor_Column++;
	if (Cursor_Column > LCD_columns) {
		Cursor_Column = 1;
    	Cursor_Row++;
		if (Cursor_Row > LCD_rows) {
			Cursor_Row = 1;
		} /* end if Cursor_Row */
	 	LCD_Move_Cursor(Cursor_Row, Cursor_Column);
	} /* end if Cursor_Column */

} /* end LCD_Char_Out */


/*---------------------------------------------------------------
| Function:     LCD_Text_Out(char *), PUBLIC
| Input:        A pointer to a null terminated character string
| Returns:      Nothing
| Purpose:      Prints a string starting from the current cursor
|               position
|----------------------------------------------------------------*/
void LCD_Text_Out(char *s)
{
int i = 0;

	while (*(s+i) != '\0') {
		LCD_Char_Out(*(s+i));
		i++;
	}
} /* end LCD_Text_Out */


/*---------------------------------------------------------------
| Function:     LCD_Clear()
| Input:        Nothing
| Returns:      Nothing
| Purpose:      Clears the entire LCD display
|----------------------------------------------------------------*/
void LCD_Clear(void)
{
	LCD_Send_Instruction(0x01, 1);       /* Clears the LCD  */
	Cursor_Column = 1;
	Cursor_Row = 1;
} /* end LCD_Clear */



/*---------------------------------------------------------------
| Function:     void LCD_Setup(char rows, char columns, int base_address)
| Input:        rows - # of rows on display (1 - 4)
|               columns - # of columns on display
|               base address - base address of parallel port
| Returns:      Nothing
| Purpose:      Initializes the LCD operating modes
|
|----------------------------------------------------------------*/
void LCD_Setup(char rows, char columns, int base_address)
{
unsigned char row_command;

	/* set global variables */
	LCD_columns = columns;                  /* # columns in display */
	LCD_rows = rows;                        /* # rows in display */
	LCD_data_port = base_address;           /* LCD data port */
	LCD_status_port = base_address + 1;     /* LCD status port */
	LCD_control_port = base_address + 2;    /* LCD control port */
	Cursor_Column = 1;					    /* Current cursor column */
	Cursor_Row = 1;				  	  		/* Current cursor row */

	if (rows == 1)
		row_command = 0;
	else
		row_command = 8;

	/* Reset display */
	LCD_Send_Instruction(0x30, 0);		/* Selects: 8 bit I/O for LCD   */
    millisecond_delay(5);
	LCD_Send_Instruction(0x30, 0);		/* Selects: 8 bit I/O for LCD   */
    millisecond_delay(1);

	/* initialize display */

	/* Selects: 8 bit I/O for LCD & # rows. */
	LCD_Send_Instruction((unsigned char)(0x30 | row_command), 0);
    millisecond_delay(1);

	/* Selects: Display off */
	LCD_Send_Instruction(0x08, 0);
    millisecond_delay(1);

	/* clear display */
	LCD_Send_Instruction(0x01, 0);
    millisecond_delay(2);

	/* Set entry mode */
	LCD_Send_Instruction(0x06, 0);
    millisecond_delay(1);

	/* Set display on */
	LCD_Send_Instruction(0x0e, 0);
    millisecond_delay(1);

} /* end LCD_Setup */

