/***************************************************************************
*
*  pendant.c -- Program for operating the RATLER vehicle via pendant
*
*  Contains: setup_rcp(), initialize_packets(), initialize_values(),
*            display_commands(), display_status(), write_command_headings(),
*            write_status_headings(), setup_display(), display_block(),
*            get_port(), build_keepalive(), send_keepalive(),
*            get_drive_commands(), main()
*
*  Created:  waa, 12-21-93
*  Modified:
*     waa, 03-xx-94 -- Re-compiled to work with new RTLRBLKS.H, etc.
*     waa, 08-09-94 -- Added calibration call, which is called as long
*                      as F1 is held down.  New pendant hardware with
*                      keypad and LCD display.
*     waa, 08-26-94 -- Changed vehicle number from 1 to 0 for Sandia.
*                      Added program_alive() func to let user know progam
*                      is running.  Added LCD code.
*
*
*  Required PC104 hardware:
*    CPU (CMF8680 CPU from RTD, Inc., was used for development)
*    A/D + digital I/O board, DM5406 from Real-Time Devices, Inc.
*
*  Inputs:
*    3-axis joystick is wired into the A/D
*       channel 0 = Y = fwd/back.  Values increase back to fwd.
*       channel 1 = X = left/right.  Values increase right to left.
*       channel 2 = Z = twist.  Values increase going clockwise.
*    deadman switch is connected to PA0
*       PA0 = 0 with deadman in
*       PA0 = 1 with deadman released
*    Serial information from the vehicle via COM1 at 4800 baud
*       Status from the vehicle is encoded in the Robotic Communication
*       Protocol (RCP).  The keepalive message to the vehicle is a
*       request for mode status, and that status is returned on COM1.
*       This is the only information returned to the pendant.  The
*       vehicle responds only to queries, and the pendant does not
*       send out queries for any other information, e.g., odometer,
*       location, fuel, etc.
*
*  Outputs:
*    Text translations of RCP commands and status are displayed to a screen
*       Although this is an embedded application, it is useful to display
*       the commands and status to the screen for current and future
*       development purposes.
*    Serial information to vehicle via COM1 at 4800 baud, RCP-encoded
*       1.  In order to keep the vehicle alive, a keepalive RCP command
*           block must be sent at least every 1.6 seconds or the vehicle
*           will safe itself.  The keepalive block for RATLER consists of
*           a request for mode status.
*       2.  When the deadman is pulled in, the pendant continually commands
*           the vehicle to be unkilled.  This ensures the motor relays
*           are energized.  Conversely, when the deadman is released,
*           the pendant continually commands a kill, to make sure the
*           motor relays are not energized.
*       3.  When the deadman is pulled in, joystick values are translated
*           into throttle and gear commands which are sent to the vehicle.
*           When the deadman is released, throttle values of zero (no
*           throttle) are sent to the vehicle.
*    Communication link status
*       A blinking block is displayed when the pendant is receiving valid
*       status information from the vehicle.  Like the text display, this
*       is primarily for current and future development, as a screen is
*       not likely to be attached to the pendant during normal operation.
*
***************************************************************************/

/* Standard library include files */
#include <assert.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

/* Local library include files */
#include "rcpvalue.h"
extern "C"  /* Otherwise linker can't find RCP functions */
{
   #include <rcpdef.h>
   #include <rcperrs.h>
   #include <rcpproto.h>
}

/* RATLER-specific include files */
#include "rtlrcp.h"
#include "rtlrblks.h"
#include "cdstruc.h"
#include "textposn.h"

/* Include files for RATLER pendant code only */
#include "comm.h"
#include "dm5406.h"
#include "joystick.h"
extern "C"
{
#include "rtd_lcd.h"  /* LCD function protos */
}

/* Miscellaneous defines */
#define ESC             '\x1b'     /* Escape char to for quitting pgm */
#define PROGRAM_NAME    "PENDANT"  /* Displayed at top of screen */

/* Timer values */
#define QUERY_INTERVAL  3          /* Tics betw queries, 18.2 tics/sec */
#define IO_INTERVAL     3          /* Tics betw io polling, 0.05 sec/tic */

/* RCP defines */
#define FIRST_PORT      0          /* First RCP port */
#define VEHICLE_ID      0          /* Sandia RATLER vehicle ID */
#define CDS_ID          101        /* Traditional RCP CDS ID */
#define BROADCAST_ID    127        /* RCP broadcast ID */

/* Hardware defines */
#define AD_ADDRESS      0x320      /* I/O address of A/D and digital board */
#define ALL_OUTPUT      0x90       /* Digital I/O port mode */
#define DEADMAN_SWITCH  0x01       /* Mask for digital I/O for deadman sw */

/* LCD defines */
#define LCD_ROWS    2
#define LCD_COLUMNS 16
#define LCD_PORT    0x378

/* For communication status display */
#define BLINK_COUNT     2          /* Frequency of updating comm good blk */
#define BLOCK          ''
#define SPACE          ' '

/* Text positioning values: */
#define FULL_SCREEN          1,1,80,25
#define PROGRAM_NAME_POS     30,1
#define STATUS_HEADING_POS   1,4
#define STATUS_VALUE_POS     20,4
#define COMMAND_HEADING_POS  40,4
#define COMMAND_VALUE_POS    60,4
#define OUTBLOCK_HEADING_POS 1,22
#define OUTBLOCK_POS         6,22
#define INBLOCK_HEADING_POS  1,24
#define INBLOCK_POS          6,24

#define BLINKER_TEXT_POS     1,2
#define BLINKER_POS          13,2

/* Locations for RCP data storage: */
VEHCMD  vehcmd;
VEHSTAT vehstat;

/* Prototypes, this file: */
int  initialize_packets (void);
void initialize_values (void);
void display_commands (int x, int y, VEHCMD *vcp);
void display_status (int x, int y, VEHSTAT *vsp);
void setup_display (void);
void display_block (int x, int y, byte *blk, int count);
void write_command_headings (int x, int y);
void write_status_headings (int x, int y);
void build_keepalive(byte *kpv_block);
void get_port(int *port);
void send_keepalive(int port, byte *kpv);
void program_alive(void);

/*************************************************************************
*
*  int setup_rcp(void)
*
*  Args: none
*
*  Action:
*    Allocates and sets up all the RCP structures and pointers.
*
*  Returns:
*    OKAY if all RCP components were successfully initialized
*    !OKAY otherwise.  NOTE that a failure in RCP initialization is fatal!
*
**************************************************************************/
int setup_rcp(void)
{
   char *fname = "setup_rcp";
   int  dummyint;  /* temporary storage */
   int  status;

   /* Set up one set of structures */
   if ((status = rcp_initialize(1)) != RCP_SUCCESS)  /* warning okay */
   {
      printf ("%s: %s with %d\n", fname, "rcp_initialize failed", status);
      return (!OKAY);
   }

   /* Set up the RCP blocks */
   if ((status = add_module (FIRST_PORT, rtlrblks, &dummyint))
        != RCP_SUCCESS)  /* warning okay */
   {
      printf ("%s: %s with %d\n", fname, "add_module failed", status);
      return (!OKAY);
   }

   /* Point packet pointers to data storage locations (vehcmd, vehstat) */
   if (initialize_packets())
   {
      printf ("%s: %s\n", fname, "initialize_cmd failed");
      return (!OKAY);
   }

   /* Load meaningful values into RCP data locations */
   initialize_values();

   return (OKAY);
} /* setup_rcp */

/*************************************************************************
*
*  int initialize_packets(void)
*
*  Args: none
*
*  Action:
*    Points RCP packet pointers to storage locations in the vehcmd and
*    vehstat structures.  set_user_pkt_addr() returns 0 if successful,
*    so a running count of errors is kept and returned.
*
*  Returns:
*    0 if there were no errors encountered
*    or a positive integer which is the number of errors encountered.
*
**************************************************************************/
int initialize_packets(void)
{
   int s; /* Running sum of returned status */
   char *fname = "initialize_packets";

   s = 0;
   if (s += set_user_pkt_addr(0, PKT_CMD_SCALE_SPEED, &vehcmd.scale_speed))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_SCALE_SPEED");
   if (s += set_user_pkt_addr(0, PKT_CMD_VEH_MODE, &vehcmd.vehmode))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_VEH_MODE");
   if (s += set_user_pkt_addr(0, PKT_CMD_KILL_STATE, &vehcmd.kill))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_KILL_STATE");
   if (s += set_user_pkt_addr(0, PKT_CMD_PARKING_BRAKE, &vehcmd.parking_brake))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_PARKING_BRAKE");
   if (s += set_user_pkt_addr(0, PKT_CMD_THROTTLE, &vehcmd.left_throttle))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_THROTTLE");
   if (s += set_user_pkt_addr(0, PKT_CMD_THROTTLE1, &vehcmd.right_throttle))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_THROTTLE1");
   if (s += set_user_pkt_addr(0, PKT_CMD_GEAR, &vehcmd.left_gear))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_GEAR");
   if (s += set_user_pkt_addr(0, PKT_CMD_GEAR1, &vehcmd.right_gear))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_GEAR1");
   if (s += set_user_pkt_addr(0, PKT_CMD_PAN0, &vehcmd.pan))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_PAN0");
   if (s += set_user_pkt_addr(0, PKT_CMD_TILT0, &vehcmd.tilt))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_TILT0");
   if (s += set_user_pkt_addr(0, PKT_CMD_PACKAGE0, &vehcmd.package))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_PACKAGE0");
   if (s += set_user_pkt_addr(0, PKT_CMD_VIDEO0_XMITER, &vehcmd.video_power))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_VIDEO0_XMITER");
   if (s += set_user_pkt_addr(0, PKT_CMD_XPOS, &vehcmd.xpos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_XPOS");
   if (s += set_user_pkt_addr(0, PKT_CMD_YPOS, &vehcmd.ypos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on CMD_YPOS");

   if (s += set_user_pkt_addr(0, PKT_STAT_VEH_MODE, &vehstat.vehmode))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_VEH_MODE");
   if (s += set_user_pkt_addr(0, PKT_STAT_KILL_STATE, &vehstat.kill))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_KILL_STATE");
   if (s += set_user_pkt_addr(0, PKT_STAT_VIDEO0_XMITER, &vehstat.video_power))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_VIDEO0_XMITER");
   if (s += set_user_pkt_addr(0, PKT_STAT_PARKING_BRAKE, &vehstat.parking_brake))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PARKING_BRAKE");
   if (s += set_user_pkt_addr(0, PKT_STAT_ERROR_CODE, &vehstat.abort_code))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ERROR_CODE");
   if (s += set_user_pkt_addr(0, PKT_STAT_PITCH, &vehstat.left_pitch))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PITCH");
   if (s += set_user_pkt_addr(0, PKT_STAT_ROLL, &vehstat.roll))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ROLL");
   if (s += set_user_pkt_addr(0, PKT_STAT_PAN0, &vehstat.pan))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PAN0");
   if (s += set_user_pkt_addr(0, PKT_STAT_HEADING0, &vehstat.heading))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_HEADING0");
   if (s += set_user_pkt_addr(0, PKT_STAT_SPEED, &vehstat.speed))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_SPEED");
   if (s += set_user_pkt_addr(0, PKT_STAT_FUEL, &vehstat.fuel))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_FUEL");
   if (s += set_user_pkt_addr(0, PKT_STAT_INCLINE2, &vehstat.right_pitch))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_INCLINE2");
   if (s += set_user_pkt_addr(0, PKT_STAT_XPOS, &vehstat.xpos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_XPOS");
   if (s += set_user_pkt_addr(0, PKT_STAT_YPOS, &vehstat.ypos))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_YPOS");
   if (s += set_user_pkt_addr(0, PKT_STAT_ALTIMETER, &vehstat.altimeter))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ALTIMETER");
   if (s += set_user_pkt_addr(0, PKT_STAT_INC_OD, &vehstat.odometer))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_INC_OD");
   if (s += set_user_pkt_addr(0, PKT_STAT_ENGINE_TEMP, &vehstat.eng_temp))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_ENGINE_TEMP");
   if (s += set_user_pkt_addr(0, PKT_STAT_PACKAGE0, &vehstat.package))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_PACKAGE0");
   if (s += set_user_pkt_addr(0, PKT_STAT_SCALE_SPEED, &vehstat.scale_speed))
      printf ("%s:%s\n", fname,
             "set_user_pkt_addr failed on STAT_SCALE_SPEED");

   return (s);
} /* initialize_packets */

/*************************************************************************
*
*  void initialize_values(void)
*
*  Args: none
*
*  Action:
*    Initializes meaningful values for RCP packets.
*
*  Returns:
*    nothing
*
**************************************************************************/
void initialize_values (void)
{
   vehstat.pan = RCP_CENTER_PAN;
   vehstat.xpos = 0;
   vehstat.ypos = 0;
   vehstat.heading = 0; /* North */
   vehstat.altimeter = 0;
   vehstat.vehmode = RCP_TELEOP;
   vehstat.kill = RCP_NO_KILL;

   vehstat.video_power = RCP_OFF;
   vehstat.parking_brake = RCP_OFF;

   vehstat.abort_code = 0;
   vehstat.left_pitch = 64;
   vehstat.roll = 64;
   vehstat.speed = 0;

   vehstat.fuel = 15;
   vehstat.right_pitch = 64;

   vehstat.odometer = 0;  /* incremental */
   vehstat.eng_temp = 0;

   vehstat.package = 0;
   vehstat.scale_speed = 0;  /* .01m/s/tick.  Default RCP is 1 == .1m/s/t */

   vehcmd.vehmode = RCP_TELEOP;
   vehcmd.kill = RCP_NO_KILL;
   vehcmd.parking_brake = RCP_OFF;  /* Parking brake OFF */
   vehcmd.pan = RCP_CENTER_PAN;  /* centered */
   vehcmd.xpos = 0;    /* arbitrary */
   vehcmd.ypos = 0;    /*     "     */
   vehcmd.tilt = RCP_CENTER_TILT; /* level */
   vehcmd.scale_speed = 0; /* 0.01 m/s/tick, not 0.1 m/s/tick */
   vehcmd.left_throttle = 0;
   vehcmd.right_throttle = 0;
   vehcmd.left_gear = RCP_NEUTRAL;
   vehcmd.right_gear = RCP_NEUTRAL;
   vehcmd.package = RCP_OFF;     /* Package (LIBS) off */
   vehcmd.video_power = RCP_OFF; /* Video power off */

   return;
} /* initialize_values */

/*************************************************************************
*
*  display_commands()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*     vcp is a pointer to a VEHSTAT structure containing the vehicle
*        commands to be displayed.
*
*  Action:
*     Displays the status contained in the structure pointed to by
*     the input argument.  The location of the text is determined
*     by the x,y input arguments which define the upper-left corner
*     of a text window.  The data is displayed in the upper-left
*     corner of that text window.  The coordinates of the upper-left
*     corner in a full size screen are 1,1.  If the coords are
*     invalid, the window call is ignored.
*
*  Created:  WAA, 09-06-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*
**************************************************************************/
void display_commands(int x, int y, VEHCMD *vcp)
{
   window (x, y, 80, 25);

   gotoxy (VEHMODE_CMD_TEXT_POS);
   cprintf ("%-6s", vcp->vehmode == RCP_AUTON ? "AUTON" : "TELEOP");

   gotoxy (KILL_CMD_TEXT_POS);
   cprintf ("%-6s", vcp->kill ? "KILLED" : "ALIVE ");

   gotoxy (PBRAKE_CMD_TEXT_POS);
   cprintf ("%-s", vcp->parking_brake ? "PBRAKE ON" : "OFF      ");

   gotoxy (VIDEO_POWER_CMD_TEXT_POS);
   cprintf ("%-s", vcp->video_power ? "VIDEO ON" : "OFF     ");

   gotoxy (THROTTLE_CMD_TEXT_POS);
   cprintf ("%-3d    %-3d", vcp->left_throttle, vcp->right_throttle);

   gotoxy (GEAR_CMD_TEXT_POS);
   cprintf ("%-3d    %-3d", vcp->left_gear, vcp->right_gear);

#if 0
   /* Enable these printouts when the commands are to be sent */
   gotoxy (PANTILT_CMD_TEXT_POS);
   cprintf ("%-5lx, %-4x", vcp->pan, vcp->tilt);

   gotoxy (VEHPOS_CMD_TEXT_POS);
   cprintf ("%-4ld, %-4ld", vcp->xpos, vcp->ypos);

   gotoxy (SCALE_SPEED_CMD_TEXT_POS);
   cprintf ("%-3d", vcp->scale_speed);

   gotoxy (LIBS_CMD_TEXT_POS);
   cprintf ("%-2d", vcp->package);

#endif

   window (FULL_SCREEN);

   return;
} /* display_commands */


/*************************************************************************
*
*  display_status()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*     vsp is a pointer to a VEHSTAT structure containing the vehicle
*        status to be displayed.
*
*  Action:
*     Displays the status contained in the structure pointed to by
*     the input argument.  The location of the text is determined
*     by the x,y input arguments which define the upper-left corner
*     of a text window.  The data is displayed in the upper-left
*     corner of that text window.  The coordinates of the upper-left
*     corner in a full size screen are 1,1.  If the coords are
*     invalid, then the window call is ignored.
*
*  Created:  WAA, 09-06-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*
**************************************************************************/
void display_status(int x, int y, VEHSTAT *vsp)
{
   window (x, y, 80, 25);

   gotoxy (VEHMODE_STAT_TEXT_POS);
   cprintf ("%-6s", vsp->vehmode == RCP_AUTON ? "AUTON" : "TELEOP");

   gotoxy (KILL_STAT_TEXT_POS);
   cprintf ("%-6s", vsp->kill ? "KILLED" : "ALIVE ");

   gotoxy (VIDEO_POWER_STAT_TEXT_POS);
   cprintf ("%-s", vsp->video_power ? "VIDEO ON" : "OFF     ");

   gotoxy (PBRAKE_STAT_TEXT_POS);
   cprintf ("%-s", vsp->parking_brake ? "PBRAKE ON" : "OFF      ");

   gotoxy (ABORT_CODE_TEXT_POS);
   cprintf ("%-3d", vsp->abort_code);

#if 0
   /* Enable these printouts when appropriate blocsk are queried for */
   gotoxy (PAN_STAT_TEXT_POS);
   cprintf ("%-5lx", vsp->pan);

   gotoxy (HEADING_STAT_TEXT_POS);
   cprintf ("%-3d", vsp->heading);
   gotoxy (VEHPOS_STAT_TEXT_POS);
   cprintf ("%-4ld, %-4ld, %-4d", vsp->xpos, vsp->ypos, vsp->altimeter);

   gotoxy (LEFT_PITCH_TEXT_POS);
   cprintf ("%-3d", vsp->pitch);

   gotoxy (ROLL_TEXT_POS);
   cprintf ("%-3d", vsp->roll);

   gotoxy (SPEED_TEXT_POS);
   cprintf ("%-3d", vsp->speed);

   gotoxy (RIGHT_PITCH_TEXT_POS);
   cprintf ("%-3d", vsp->joint_angle);

   gotoxy (FUEL_TEXT_POS);
   cprintf ("%-2d", vsp->fuel);

   gotoxy (ODOMETER_TEXT_POS);
   cprintf ("%-7d", vsp->odometer);  /* incremental */

   gotoxy (ENGINE_TEMP_TEXT_POS);
   cprintf ("%-3d", vsp->eng_temp);

   gotoxy (LIBS_STATUS_TEXT_POS);
   cprintf ("%-3d", vsp->package);

   gotoxy (SCALE_SPEED_STATUS_TEXT_POS);
   cprintf ("%-3d", vsp->scale_speed);
#endif

   window (FULL_SCREEN);

   return;
} /* display_status */


/*************************************************************************
*
*  write_command_headings()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*
*  Action:
*    Writes the text for each of the command values.
*
*  Created:  WAA, 09-17-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*
**************************************************************************/
void write_command_headings(int x, int y)
{
   window (x, y, 80,25);

   gotoxy (VEHMODE_CMD_TEXT_POS);
   cprintf ("Commanded mode");

   gotoxy (KILL_CMD_TEXT_POS);
   cprintf ("Commanded kill");

   gotoxy (PBRAKE_CMD_TEXT_POS);
   cprintf ("Parking brake");

   gotoxy (VIDEO_POWER_CMD_TEXT_POS);
   cprintf ("Video power");

   gotoxy (THROTTLE_CMD_TEXT_POS);
   cprintf ("Throttles");

   gotoxy (GEAR_CMD_TEXT_POS);
   cprintf ("Gears");

#if 0
   /* Enable the following writes when these commands are sent out */
   gotoxy (PANTILT_CMD_TEXT_POS);
   cprintf ("Pan/tilt");

   gotoxy (VEHPOS_CMD_TEXT_POS);
   cprintf ("Destination");

   gotoxy (SCALE_SPEED_CMD_TEXT_POS);
   cprintf ("Speed scale");

   gotoxy (LIBS_CMD_TEXT_POS);
   cprintf ("LIBS command");
#endif

   window (FULL_SCREEN);

   return;
} /* write_command_headings */

/*************************************************************************
*
*  write_status_headings()
*
*  Args:
*     x,y are the upper-left coordinates of the text window where
*        the data are displayed.  Norm coords are 1,1 for upper-left.
*
*  Action:
*    Writes the text for each of the status values.  Uses same gotoxy()
*    values as in tt_show_status(), but with the display window
*    offset to the left.
*
*  Created:  WAA, 09-17-93
*  Modified:
*     WAA, 12-06-93 -- Took out AMX references.  Removed save/restore
*                      window code that would put the cursor back
*                      exactly where it was before.  Not needed in a
*                      single threaded system.
*
**************************************************************************/
void write_status_headings(int x, int y)
{
   window (x, y, 80,25);

   gotoxy(VEHMODE_STAT_TEXT_POS);
   cprintf ("Mode");

   gotoxy(KILL_STAT_TEXT_POS);
   cprintf ("Kill");

   gotoxy(PBRAKE_STAT_TEXT_POS);
   cprintf ("Parking brake");

   gotoxy(VIDEO_POWER_STAT_TEXT_POS);
   cprintf ("Video power");

   gotoxy(ABORT_CODE_TEXT_POS);
   cprintf ("Error code");

#if 0
   /* Enable the following writes when the status is queried for */
   gotoxy(PAN_STAT_TEXT_POS);
   cprintf ("Pan");

   gotoxy(HEADING_STAT_TEXT_POS);
   cprintf ("Heading");

   gotoxy(VEHPOS_STAT_TEXT_POS);
   cprintf ("Location (x,y,z)");

   gotoxy(LEFT_PITCH_TEXT_POS);
   cprintf ("LPitch");

   gotoxy(ROLL_TEXT_POS);
   cprintf ("Roll");

   gotoxy(SPEED_TEXT_POS);
   cprintf ("Speed");

   gotoxy(RIGHT_PITCH_TEXT_POS);
   cprintf ("RPitch");

   gotoxy(FUEL_TEXT_POS);
   cprintf ("Fuel");

   gotoxy(ODOMETER_TEXT_POS);
   cprintf ("Odometer");

   gotoxy(ENGINE_TEMP_TEXT_POS);
   cprintf ("Engine temp");

   gotoxy(LIBS_STATUS_TEXT_POS);
   cprintf ("LIBS status");

   gotoxy(SCALE_SPEED_STATUS_TEXT_POS);
   cprintf ("Scale speed");
#endif

   window (FULL_SCREEN);

   return;
} /* write_status_headings */

/*************************************************************************
*
*  setup_display ()
*
*  Args: none
*
*  Action:
*    Clears the screen and writes the headings for the RCP values
*
*  Returns: nothing
*
*  Created: WAA, 12-06-93
*
**************************************************************************/
void setup_display (void)
{
   clrscr ();
   _setcursortype(_NOCURSOR);  /* Turn off the cursor */

   gotoxy (PROGRAM_NAME_POS);
   cputs (PROGRAM_NAME);
#ifndef NOSTOP
   cputs (" (ESC exits)");
#endif

#ifndef NDEBUG
   gotoxy (OUTBLOCK_HEADING_POS);
   cputs ("Out:");
   gotoxy (INBLOCK_HEADING_POS);
   cputs ("In:");
#endif

   write_command_headings (COMMAND_HEADING_POS);
   write_status_headings (STATUS_HEADING_POS);

   /* For pendant */
   gotoxy (BLINKER_TEXT_POS);
   cputs ("Comm status");

   return;
} /* setup_display */

/*************************************************************************
*
*  void display_block(int x, int y, byte *blk, int count)
*
*  Args: x,y are the screen coords of where to start the display,
*          upper left is 1,1
*        *blk is the array of values
*        count is the number of values in the array to display
*
*  Action:
*    Goes to the given location and clears the line.  Then displays
*    the array values in right-justified, zero-filled hexadecimal.
*
*  Created:  waa, 01-03-94
*
**************************************************************************/
void display_block (int x, int y, byte *blk, int count)
{
   int i;

   gotoxy (x, y);
   clreol();
   for (i=0; i<count; ++i)
      printf ("%02x ", blk[i]);
   return;

} /* display_block */

/*************************************************************************
*
*  void get_port(int *port)
*
*  Args: *port is set to port the user chooses, 1 or 2, or is set to
*          the pre-determined port
*
*  Action:
*    If compiled with the development symbol 'NDEBUG' defined, the
*    port number is hardwired to port 1.  If NDEBUG is not defined,
*    the user is asked which serial port to use.
*
*  Returns: new port number in *port
*
*  Created:  waa, 12-22-93
*
**************************************************************************/
void get_port(int *port)
{
#ifdef NDEBUG
   *port = 2;
#else
   char str[80]; /* String from the keyboard for determining com port */

   /* For development purposes only, ask which COM port to use: */
   while (*port == 0)
   {
      printf ("COM1 or COM2? [1|2, default=1] ");
      gets (str);

      if (sscanf(str, "%d", port) == 0)
         *port = 1;
   }
#endif

   save_imr(); /* Interrupt Mask Register needs to be saved before int init */

   if (*port == 2)
   {
      init_COM2();
      init_COM2_int();
   }
   else /* default to COM1 */
   {
      init_COM1();
      init_COM1_int();
   }
} /* get_port */

/*************************************************************************
*
*  void buid_keepalive(byte *kpv)
*
*  Args: kpv points to already allocated storage and is used to
*          return the newly built keepalive block
*
*  Action:
*    Builds the fixed command query which serves as the keepalive block
*    for the RATLER.
*
*  Returns:  new keepalive block in kpv
*
**************************************************************************/
void build_keepalive(byte *kpv)
{
   int count;

   /* Build the keep alive query block */
   kpv[0] = CDS_ID | HIGH_BIT;
   kpv[1] = VEHICLE_ID | HIGH_BIT;
   kpv[2] = (byte) (KEEP_ALIVE_BLOCK | HIGH_BIT);
   kpv[3] = QUERY_CHAR;
   count = 4;
   insert_crc (kpv, &count, TRUE);
   assert(count == QUERY_LENGTH);

   return;
} /* build_keepalive */

/*************************************************************************
*
*  void send_keepalive(int port, byte *kpv)
*
*  Args:  port is the port to send the keepalive to
*         kpv is the block to send.  Must be QUERY_LENGTH chars long.
*
*  Action:
*     Attempts to send the keepalive block.  If send_stream() allows
*     the block to be written to the output buffer, then the keepalive
*     block is displayed on the screen.  Otherwise, an error message
*     is printed.
*
*  Returns:  new keepalive block in kpv
*
**************************************************************************/
void send_keepalive(int port, byte *kpv)
{
   int status;

   status = send_stream (port, kpv, QUERY_LENGTH);

   #ifndef NDEBUG
   if (status == OKAY)
      display_block (OUTBLOCK_POS, kpv, QUERY_LENGTH);
   else
      printf ("Error sending keepalive query");
   #endif

   return;
} /* send_keepalive */ /* status warning okay */


/*************************************************************************
*
*  void program_alive(void)
*
*  Action:
*     Notifies the user in some way that the program is running.
*
*  Created:  WAA, 08-26-94
*
**************************************************************************/
void program_alive(void)
{
   LCD_Text_Out ("Program running");
   sound(600);
   delay(500);
   nosound();
   sound(700);
   delay(200);
   nosound();
   LCD_Move_Cursor(2, 1);
   LCD_Text_Out ("F1: calib");
} /* program_alive */

/*************************************************************************
*
*  int get_drive_commands(VEHCMD *cmd)
*
*  Args: *cmd is where the RCP throttle and gear commands are stored
*
*  Action:
*    Reads the joystick and translates the raw values into RCP throttle
*    and gear commands which are stored in the cmd structure.
*
*  Returns: 1 if A/D values read okay
*           0 if A/D values could not be read
*
*  Created:  WAA, 01-03-94
*
**************************************************************************/
int get_drive_commands (VEHCMD *cmd)
{
   JOYSTRUCT j;

   /* Get the raw joystick values */
   if (get_joystick_values(DRIVE_BITWINDOW, &j) != OKAY)
      return (!OKAY);

   /* Put the joystick values into normalized form 0..THROTTLE_MAX * 2 */
//   normalize_drive_values (&j);
   raw_to_rcp(&j);

   /* Produce RCP throttle and gear values from the joystick values */
   rcp_combined_drive (&cmd->left_throttle, &cmd->right_throttle,
                       &cmd->left_gear, &cmd->right_gear,
                       &j);

   return (OKAY);
} /* get_drive_commands */

/*************************************************************************
*
*  main()
*
*  Action:
*    Initializes the comm interrupt and one RCP structure.  Two time
*    variables are used to keep track of the last time a query/keepalive
*    was sent and the last time the joystick and deadman were read.
*    Status values are updated everytime a valid status block is
*    received.  Command values are updated everytime the joystick
*    and deadman are read.
*
*    Compiling this with NDEBUG defined will suppress auxiliary display
*    to the screen.  With NDEBUG undefined, incoming and outgoing blocks
*    are displayed in hex at the bottom of the screen.  Error messages
*    and other information (e.g., type of incoming block) are also
*    displayed.  Other files such as COMM.CPP also use NDEBUG, so make
*    sure to do a build if you define or undefine it.
*
*    Compiling this with NOSTOP defined will create an executable that
*    cannot be exited.  Otherwise, this program can be exited via ESC.
*    This may better be implemented as a command-line option which is
*    used to set a flag to be checked at the top of the while loop,
*    since this check wouldn't take much time.
*
*    Hitting 'r' will refresh the screen.
*
**************************************************************************/
main ()
{
   int port = 0;      /* Active comm port, 1 or 2.  0 is undefined */
   char c='\0';       /* Character from keyboard */
   byte keepalive[QUERY_LENGTH];       /* Holds keepalive block */
   int count;         /* Number of bytes in block */
   byte outblock[80]; /* Storage for block to send */
   byte inblock[80];  /* Storage for received block */
   byte command_byte; /* Command byte of received block */
   clock_t tics;      /* Tics since program start */
   clock_t query_tics;/* Tics since last query sent */
   clock_t io_tics;   /* Tics since joystick and switch last read */
   byte blinker;      /* Counter to determine frequency of blinking */
   byte blink_char;   /* Char printed to update comm blinker */
   int  status;       /* Return status from send_stream() */
   byte calib = 0;    /* Whether or not calib was pressed last loop */


   /* Initialize RCP structures */
   if (setup_rcp() != OKAY)
   {
      printf ("Fatal error:  Unable to initialize RCP.\n");
      exit (0);
   }

   /* Initialize the serial communication port */
   get_port (&port);

   /* Initialize the A/D and digital I/O board */
   init_dm5406 (AD_ADDRESS, ALL_OUTPUT, TRUE);

   /* Initialize the display */
   clrscr();
   setup_display ();

   /* Build the keepalive string which is reused throughout execution */
   build_keepalive (keepalive);

   /* Initialize the tics variables for sending queries and polling I/O */
   query_tics = io_tics = clock();

   /* Initialize LCD display */
   LCD_Setup(LCD_ROWS, LCD_COLUMNS, LCD_PORT);

   /* Notification that program is running */
   program_alive();

   c = '\0';
   blinker = 0;
   blink_char = BLOCK;
#ifdef NOSTOP
   while(1)
#else
   while (c != ESC)
#endif
   {
      /* tics will take more than 3 and a half years to roll over,
         so no need to check for it, eh? */
      tics = clock();

      /* Send keepalive if sufficient time has passed since last one */
      if (tics - query_tics >= QUERY_INTERVAL)
      {
         send_keepalive (port, keepalive);
         query_tics = clock();  /* Re-arm timer for next pass */
      }

      /* Read joystick and switch if it is time to do so */
      if (tics - io_tics >= IO_INTERVAL)
      {
         /* If deadman in, read joystick to get drive cmds, unkill vehicle */
         if ((get_dio (DIO_PORT_A) & DEADMAN_SWITCH) == 0)
         {
            if (get_drive_commands (&vehcmd) != OKAY)
            {
               #ifndef NDEBUG
               gotoxy (OUTBLOCK_POS);
               clreol();
               printf ("Couldn't get A/D readings from joystick");
               #endif
            }
            vehcmd.kill = RCP_NO_KILL;
            calib = 0;
         }
         else  /* Deadman released, make vehicle stop */
         {
            /* Calibrate as long as F1 held down */
            if ((get_dio(DIO_PORT_A) & 0x20) == 0) /* F1 */
            {
               /* If not pressed before, zero values */
               if (calib == 0)
                  zero_joystick_calib();

               calibrate_joystick();
               calib = 1;
            }
            else
            {
               calib = 0;
            }
            #if 0  /* doesn't work because of too quick digital reads */
            else
            if ((get_dio(DIO_PORT_A) & 0xc0) == 0) /* F2 and F3 */
               zero_joystick_calib();
            #endif

            /* Put the throttles at zero; gear is irrelevant */
            vehcmd.left_throttle = 0;
            vehcmd.right_throttle = 0;

            /* Make sure vehicle is killed */
            vehcmd.kill = RCP_KILL;
         }

         /* Send out throttle and gear commands */
         build_block (FIRST_PORT, DRIVE_CMD_BLOCK, outblock, &count,
                      VEHICLE_ID);
         status = send_stream (port, outblock, count);

         #ifndef NDEBUG
         if (status == OKAY)
            display_block (OUTBLOCK_POS, outblock, count);
         else
            cputs ("Error sending driving block");
         #endif

         /* Send out kill command */
         build_block (FIRST_PORT, MODE_CHANGE_CMD_BLOCK, outblock, &count,
                      VEHICLE_ID);
         status = send_stream (port, outblock, count);

         #ifndef NDEBUG
         if (status == OKAY)
            display_block (OUTBLOCK_POS, outblock, count);
         else
            cputs ("Error sending mode block");
         #endif

         display_commands (COMMAND_VALUE_POS, &vehcmd);

         io_tics = clock();
      }

      /*
       *  Get complete RCP blocks
       *  This section is actually irrelevant for the pendant, as the
       *  pendant will have no means of displaying the information,
       *  but this has little performance impact and is useful for
       *  development purposes.
       */
      while (get_stream (port, inblock, &count))
      {
         #ifndef NDEBUG
         display_block (INBLOCK_POS, inblock, count);
         #endif

         /* Check checksum of received block */
         if (validate_block (count, inblock) == RCP_SUCCESS)
         {
            /*
             * Put check against vehicle ID here if desired.
             * Will NOT be including it for RATLER pendant so that
             *   this pendant will be guaranteed to work for all
             *   RATLERs.
             * Don't forget to mask off high bit if you do the check.
             */

            command_byte = inblock[count-1];

            /* Load RCP data structure from received block */
            if (partition_block (FIRST_PORT, command_byte, inblock)
                == RCP_SUCCESS)
            {
               /* Update display */
               if (is_vehicle_status(command_byte))
                  display_status (STATUS_VALUE_POS, &vehstat);
               #ifndef NDEBUG
               else
               if (is_fixed_command(command_byte))
                  cputs ("fixed command");
               else
                  cputs ("unknown command");
               #endif

               /* Update comm-good blinker */
               if (++blinker == BLINK_COUNT)
               {
                  blinker = 0;
                  gotoxy (BLINKER_POS);
                  blink_char = (blink_char == SPACE) ? BLOCK : SPACE;
                  putch (blink_char);
               }
            }
         }
         #ifndef NDEBUG
         else
            cputs ("invalid");
         #endif

      }

      /* Read keyboard.  Can later expand this to a switch statement */
      if (kbhit())
      {
         c = getch();
         if (c == 'r')
         {
            clrscr();
            setup_display();
         }
      }

   } /* while not ESC pressed */

   /* Restore comm registers and interrupt */
   restore_imr();

   if (port == 2)
      restore_COM2_int();
   else /* default is to restore COM1 */
      restore_COM1_int();

   /* Restore the cursor */
   _setcursortype(_NORMALCURSOR);

   return (0);
} /* main */ /* warning okay */
