/*************************************************************************
*
*  joystick.h -- Defines and prototypes for public functions found in
*                joystick.cpp.  The raw observed joystick limits are
*                coded into this file, which automatically generates
*                the appropriate values used in the conversion routines.
*
**************************************************************************/

#ifndef JOYSTICKDOTH
#define JOYSTICKDOTH

#include "cdstruc.h"  /* For byte */

/* Actual A/D resolution seen */
#define  AD_BITS         11  /* Using only 0-5V of 0-10V */

/* Mask for extracting part of the total A/D bits */
#define DRIVE_BITWINDOW   8  /* 7 bits plus sign */

/* Tolerances, in raw joystick counts after bitshifting and normalization */
#define DRIVE_TOL       10
#define Z_DRIVE_TOL     10

/* Observed joystick limits for the x and y axes */
#define X_DRIVE_MIN      0x02b0                                  /* 0x00 */
#define X_DRIVE_MAX      (0x0495 >> (AD_BITS - DRIVE_BITWINDOW)) /* 0xf8 */
#define X_DRIVE_CENTER   (0x03a0 >> (AD_BITS - DRIVE_BITWINDOW)) /* 0x78 */

#define Y_DRIVE_MIN      0x020b                                  /* 0x00 */
#define Y_DRIVE_MAX      (0x0495 >> (AD_BITS - DRIVE_BITWINDOW)) /* 0xf8 */
#define Y_DRIVE_CENTER   (0x03a0 >> (AD_BITS - DRIVE_BITWINDOW)) /* 0x77 */

/* Observed joystick limits for the z, or twist, axis */
#define Z_DRIVE_MIN      (0x0278 >> (AD_BITS - DRIVE_BITWINDOW-1)) /* 0x9e */
#define Z_DRIVE_MAX      (0x0544 >> (AD_BITS - DRIVE_BITWINDOW-1)) /* 0x151 */
#define Z_DRIVE_CENTER   (0x03dc >> (AD_BITS - DRIVE_BITWINDOW-1)) /* 0xf7 */

/* RCP values */
#define THROTTLE_BITS    7  /* Number of bits in RCP throttle */
#define THROTTLE_MASK 0x7f  /* Mask of THROTTLE_BITS length */
#define THROTTLE_MAX   127  /* Max RCP value for throttle */


/* Multipliers for normalizing raw observed values */
#define X_HIGH ((float)THROTTLE_MAX / (float)(X_DRIVE_MAX - X_DRIVE_CENTER))
#define X_LOW  ((float)THROTTLE_MAX / (float)(X_DRIVE_CENTER - X_DRIVE_MIN))

#define Y_HIGH ((float)THROTTLE_MAX / (float)(Y_DRIVE_MAX - Y_DRIVE_CENTER))
#define Y_LOW  ((float)THROTTLE_MAX / (float)(Y_DRIVE_CENTER - Y_DRIVE_MIN))

#define Z_HIGH ((float)THROTTLE_MAX / (float)(Z_DRIVE_MAX - Z_DRIVE_CENTER))
#define Z_LOW  ((float)THROTTLE_MAX / (float)(Z_DRIVE_CENTER - Z_DRIVE_MIN))

typedef struct
{
   int x;
   int y;
   int z;
} JOYSTRUCT;

/* Function prototype: */
int  get_joystick_values(int bitwindow, JOYSTRUCT *joyptr);
void raw_to_rcp(JOYSTRUCT *joy);
void calibrate_joystick(void);
void normalize_drive_values (JOYSTRUCT *joy);
void rcp_combined_drive (byte *left_throttle, byte *right_throttle,
                         byte *left_gear, byte *right_gear,
                         JOYSTRUCT *joy);
void zero_joystick_calib(void);

#endif
