/***************************************************************************
*
*  joystick.cpp -- Routines that convert from raw A/D values to RCP
*                  throttle and gear values, for the RATLER pendant
*
*  Public: get_joystick_values(), normalize_drive_values(),
*          rcp_combined_drive(), raw_to_rcp(), calibrate_joystick()
*
*  Usage:
*     get_joystick_values() is called first to get values from the
*        joystick.  Raw values from the joystick are 12 bits, so
*        these values are shifted so that only 8 are returned.
*
*     normalize_drive_values() takes the raw 8-bit values and uses
*        the hardcoded limit values to normalize the values into
*        the range 0..(THROTTLE_MAX * 2).  This is so the values
*        will be symmetric around THROTTLE_MAX and will be easy
*        to convert into RCP values.
*
*     rcp_combined_drive() takes the normalized 8-bit values and
*        converts them into 7-bit RCP throttle and gear commands.
*
*  Created:  WAA, 01-03-94
*  Modified:
*     WAA, 08-09-94 -- Cleaned up by using some floating point math
*                      instead of doing everything in integer.  New
*                      functions raw_to_rcp() and calibrate_joystick()
*                      make life much easier.
*     WAA, 08-26-94 -- Added default calibration values.  And took them
*                      back out.
*
***************************************************************************/


/* Global includes: */
#include <math.h>  /* abs */
#include <conio.h>

/* Sandia RVR includes: */
#include "rcpvalue.h"

/* Global to RATLER CDS software: */
#include "cdstruc.h"   /* For gear defines */

/* Local to RATLER pendant software only: */
#include "dm5406.h"
#include "joystick.h"

typedef  unsigned char  byte;

/* Analog channels */
#define X_CHAN   2
#define Y_CHAN   1
#define Z_CHAN   3

/* Calibration values */
#ifdef RUNNER
JOYSTRUCT min = { 0x02b0, 0x02b0, 0x0000}; /* determined on dev stn */
JOYSTRUCT max = { 0x0495, 0x0495, 0x0495};
#else
/* Use the following to make sure the bounds are tight */
JOYSTRUCT min = { 0x7fff, 0x7fff, 0x7fff}; /* A/D max values */
JOYSTRUCT max = { 0x0000, 0x0000, 0x0000};
#endif

/*************************************************************************
*
*  int get_joystick_values(int bitwindow, JOYSTRUCT *joyptr)
*
*  Args:
*    bitwindow is number of bits to return, from 1 through 12
*    *joyptr is a pointer to a JOYSTRUCT into which to load the new values
*
*  Action:
*    Gets the raw joystick values from the appropriate joystick and
*    shifts the 12-bit raw values into the given bit window.  E.g.,
*    if only 7 bits are needed, then all raw values are shifted
*    right by 5 bits.
*
*  Returns:  New raw x,y,z values in the output structure
*
*
*  Created:  WAA, 09-14-93
*  Modified:
*     WAA, 09-21-93 -- Added kludge to allow return of z-axis values in
*                      similar range as x and y.  Since the z-axis has
*                      a range less than half that of the other axes,
*                      the return value is shifted by one bit less
*                      in order to preserve the range without exceeding
*                      the range of values being returned for the other
*                      axes.
*                      There are about 1415 z counts returned, out of
*                      a possible 4096.
*                      Commented out kludge as callers were getting
*                      confused by values.
*     WAA, 01-03-94 -- Stripped down for single joystick mode as
*                      required for the pendant.
*
**************************************************************************/
int get_joystick_values(int bitwindow, JOYSTRUCT *joyptr)/*!end!*/
{
   int offset, z_offset;

   if (bitwindow < 1 || bitwindow > AD_BITS)
      return (!OKAY);

   offset = AD_BITS - bitwindow;

#if 1
/* Comment this out if and when the Z axis returns values comparable */
/*   to X and Y values.  Currently (as of 01-94) goes from 0x270 to */
/*   0x544 lock-to-lock, rather than 0 to 0xfff */
/* Also modify the #defines for Z_DRIVE_MIN, _MAX, and _CENTER */

   /* Z, or twist, axis needs extra resolution because it has fewer
      counts lock-to-lock than the X or Y axes */
   z_offset = AD_BITS - bitwindow - 1;
   if (z_offset < 0)
      z_offset = 0;
#else
   z_offset = offset;
#endif

   joyptr->x = get_ad (X_CHAN) >> offset;
   if (_dm5406_error != DM5406_OKAY)
      return(!OKAY);
   joyptr->y = get_ad (Y_CHAN) >> offset;
   if (_dm5406_error != DM5406_OKAY)
      return(!OKAY);
/* Commented out 08-09-94
   New joystick does not have Z axis hooked up. */

/*   joyptr->z = get_ad (Z_CHAN) >> z_offset; */
   joyptr->z = 0;

   if (_dm5406_error != DM5406_OKAY)
      return(!OKAY);

   return (OKAY);
} /* get_joystick_values */

/*************************************************************************
*
*  void normalize_drive_values (JOYSTRUCT *joy)
*
*  Args:
*    *joy is input with raw values, output with normalized values
*
*  Action:
*    Converts raw drive values from the joystick into normalized
*    8-bit drive values 0..254, where 127 is the center.  The range
*    of the normalized symmetric about 127, so that deflection values
*    of 0..127 and a direction can be determined.  This routine uses
*    empirically-determined limits (see #defines for _CENTER, _HIGH,
*    and _LOW in joystick.h) to make sure that 1) the entire range
*    of deflection values 0 to 127 correspond to the physical joystick
*    travel, 2) 127 is always returned when the joystick is centered,
*    and 3) the values increase left to right, back to front, and
*    clockwise.
*
*  Returns:
*    Normalized x, y, and z drive values in *joy
*
*  Created:  WAA, 01-04-93
*
**************************************************************************/
void normalize_drive_values (JOYSTRUCT *joy)
{
   int x, y, z;

   x = joy->x;
   y = joy->y;
   z = joy->z;

   /* Left/right axis */
   if (x >= X_DRIVE_CENTER)
   {
      x -= X_DRIVE_CENTER;
      x = (int) ((float) x * X_HIGH);
      x += THROTTLE_MAX;
   }
   else
      x = (int) ((float) x * X_LOW);

   if (x > (THROTTLE_MAX * 2))
      x = (THROTTLE_MAX * 2);
   else
   if (x < 0)
      x = 0;

   /* Fwd/back axis */
   if (y >= Y_DRIVE_CENTER)
   {
      y -= Y_DRIVE_CENTER;
      y = (int) ((float) y * Y_HIGH);
      y += THROTTLE_MAX;
   }
   else
      y = (int) ((float) y * Y_LOW);

   if (y > (THROTTLE_MAX * 2))
      y = (THROTTLE_MAX * 2);
   else
   if (y < 0)
      y = 0;

   /* Twist axis */
   if (z >= Z_DRIVE_CENTER)
   {
      z -= Z_DRIVE_CENTER;
      z = (int) ((float) z * Z_HIGH);
      z += THROTTLE_MAX;
   }
   else
   {
      z -= Z_DRIVE_MIN;
      z = (int) ((float) z * Z_LOW);
   }

   if (z > (THROTTLE_MAX * 2))
      z = (THROTTLE_MAX * 2);
   else
   if (z < 0)
      z = 0;

   /* Customize here so that X axis values increases left to right,
    * Y axis values increase back to front, and Z axis values increase
    * going clockwise.  This is so the RCP conversion routine will
    * work without modification. */
   joy->x = (THROTTLE_MAX * 2) - (byte) x;
   joy->y = (THROTTLE_MAX * 2) - (byte) y;

/*   joy->z = (byte) z;  */  /* Commented out, 08-09-94 by WAA
                                Z axis not hooked up on new pendant */
   joy->z = 0x7f; /* 127 centered */


   return;
} /* normalize_drive_values */

/*************************************************************************
*
*  void rcp_combined_drive (byte *left_throttle, byte *right_throttle,
*                           byte *left_gear, byte *right_gear,
*                           JOYSTRUCT *joy)
*
*  Args:
*    *left_throttle, *right_throttle are output vars containing the
*       RCP throttle value 0 to 127
*    *left_gear, *right_gear are output vars containing the RCP gear value
*    *joy is the input var containing the raw joystick values which
*       range from 0 through (THROTTLE_MAX * 2)
*
*  Action:
*    Converts the values from a single joystick into combined left and
*    right side drive values for the vehicle.  The input joystick values
*    range from 0..(THROTTLE_MAX * 2).
*
*    The values 0..THROTTLE_MAX represent the RCP values THROTTLE_MAX
*    through 0 in one direction (say, forward), while values
*    THROTTLE_MAX+1 through (THROTTLE_MAX * 2) represent the RCP
*    values 0 through THROTTLE_MAX in the other direction (reverse).
*
*  Returns:
*    left_throttle, right_throttle, left_gear, and right_gear are output
*      with RCP values
*
*  Created:  WAA, 09-20-93
*  Modified:
*    WAA, 01-03-94 -- Took out AMX references to floating pt semaphore
*    WAA, 01-04-94 -- Now using normalized values 0..(THROTTLE_MAX * 2)
*                     which come out of normalize_drive_values().
*    WAA, 01-08-94 -- Added DRIVE_TOL back into x, y, and z values so
*                     that actual observed range will be about
*                     1 through THROTTLE_MAX - DRIVE_TOL.  Basically
*                     sacrificing total range for smooth ramp up.
*
**************************************************************************/
void rcp_combined_drive (byte *left_throttle, byte *right_throttle,
                         byte *left_gear, byte *right_gear,
                         JOYSTRUCT *joy)
{
   byte x, y, z;
   unsigned int offset; /* 2 axes combined into percentage of full travel */
   float float_offset;  /* Ditto */

   x = joy->x;
   y = joy->y;
   z = joy->z;

   /* If the joystick knob is twisted, give a little more leeway in
      determining if the joystick is deflected */
   if ((abs (z - THROTTLE_MAX) > Z_DRIVE_TOL) &&
       (abs (x - THROTTLE_MAX) < (DRIVE_TOL * 6)) &&
       (abs (y - THROTTLE_MAX) < (DRIVE_TOL * 6)))
   {
      if (z > THROTTLE_MAX + Z_DRIVE_TOL)
      {
         z -= (THROTTLE_MAX + Z_DRIVE_TOL);

         *left_throttle = z;
         *right_throttle = z;
         *left_gear = RCP_FORWARD;
         *right_gear = RCP_REVERSE;
      }
      else /* z < THROTTLE_MAX - Z_DRIVE_TOL */
      {
         z = THROTTLE_MAX - Z_DRIVE_TOL - z;

         *left_throttle = z;
         *right_throttle = z;
         *left_gear = RCP_REVERSE;
         *right_gear = RCP_FORWARD;
      }

      return;
   }


   if (x > THROTTLE_MAX + DRIVE_TOL)  /* right */
   {
      /* Determine forward or reverse direction */
      if (y > THROTTLE_MAX + DRIVE_TOL)  /* forward */
      {
         *left_gear = RCP_FORWARD;
         *right_gear = RCP_FORWARD;
      }
      else
      if (y < THROTTLE_MAX - DRIVE_TOL)  /* reverse */
      {
         *left_gear = RCP_REVERSE;
         *right_gear = RCP_REVERSE;
      }
      else /* going nowhere */
      {
         *left_throttle = 0;
         *right_throttle = 0;
         *left_gear = RCP_NEUTRAL;
         *right_gear = RCP_NEUTRAL;

         return;
      }

      /* Bring X back into 0..127 range */
      x -= (THROTTLE_MAX + DRIVE_TOL);


      /* Calculate fwd/back joystick deflection */
      y = (y > THROTTLE_MAX) ?
            (y - THROTTLE_MAX - DRIVE_TOL) : (THROTTLE_MAX - DRIVE_TOL - y);


#if 0
      offset = x * (y+1);   /* integer multiplication, 14-bit result */
      offset >>= 7;         /* put back into 7-bit range 0..127 */
#endif
      float_offset = x * y;
      float_offset /= (THROTTLE_MAX - DRIVE_TOL);
      if (float_offset > THROTTLE_MAX - DRIVE_TOL)
         float_offset = THROTTLE_MAX - DRIVE_TOL;

      *left_throttle = y;
      *right_throttle = y - (byte) float_offset;
   }
   else
   if (x < THROTTLE_MAX - DRIVE_TOL)  /* left */
   {
      if (y > THROTTLE_MAX + DRIVE_TOL)  /* forward */
      {
         *left_gear = RCP_FORWARD;
         *right_gear = RCP_FORWARD;
      }
      else
      if (y < THROTTLE_MAX - DRIVE_TOL)  /* reverse */
      {
         *left_gear = RCP_REVERSE;
         *right_gear = RCP_REVERSE;
      }
      else /* going nowhere */
      {
         *left_throttle = 0;
         *right_throttle = 0;
         *left_gear = RCP_NEUTRAL;
         *right_gear = RCP_NEUTRAL;

         return;
      }

      x = THROTTLE_MAX - DRIVE_TOL - x;

      /* Calculate fwd/back joystick deflection */
      y = (y > THROTTLE_MAX) ?
            (y - THROTTLE_MAX - DRIVE_TOL) : (THROTTLE_MAX - DRIVE_TOL - y);

#if 0
      offset = x * (y+1); /* integer multiplication, 14-bit result */
      offset >>= 7;       /* put back into 7-bit range 0..127 */
#endif

      float_offset = x * y;
      float_offset /= (THROTTLE_MAX - DRIVE_TOL);
      if (float_offset > THROTTLE_MAX - DRIVE_TOL)
         float_offset = THROTTLE_MAX - DRIVE_TOL;

      *left_throttle = y - (byte) float_offset;
      *right_throttle = y;
   }
   else  /* No x-axis movement, so check for y-axis movement */
   if (y > THROTTLE_MAX + DRIVE_TOL) /* both forward */
   {
      y -= DRIVE_TOL;
      *left_throttle = y - THROTTLE_MAX;
      *right_throttle = y - THROTTLE_MAX;
      *left_gear = RCP_FORWARD;
      *right_gear = RCP_FORWARD;
   }
   else
   if (y < THROTTLE_MAX - DRIVE_TOL) /* both in reverse */
   {
      y += DRIVE_TOL;
      *left_throttle = THROTTLE_MAX - y;
      *right_throttle = THROTTLE_MAX - y;
      *left_gear = RCP_REVERSE;
      *right_gear = RCP_REVERSE;
   }
   else /* going nowhere */
   {
      *left_throttle = 0;
      *right_throttle = 0;
      *left_gear = RCP_NEUTRAL;
      *right_gear = RCP_NEUTRAL;
   }

   return;
} /* rcp_combined_drive */


/*************************************************************************
*
*  raw_to_rcp(JOYSTRUCT *joy)
*
*  Action:
*    Converts directly from raw joystick values to RCP, 0..0xfe, where
*    0x7f is centered.
*
*  Returns:  OKAY if A/D conversion to get raw values worked,
*            !OKAY otherwise.
*
*  Created:  WAA, 08-09-94
*  Modified:
*     WAA, 09-21-94 -- Added return parameter to let caller know if
*                      A/D conversion worked.
*
**************************************************************************/
int raw_to_rcp(JOYSTRUCT *joy)
{
  JOYSTRUCT raw;

  /* Get raw values */
  if (get_joystick_values (AD_BITS, &raw) != OKAY)
     return (!OKAY);

  joy->x = (raw.x - min.x);
  if (joy->x < 0)
     joy->x = 0;
  joy->x = (int) ((joy->x * 254.0) / (max.x - min.x));
  if (joy->x > 0xfe)
     joy->x = 0xfe;

  joy->y = (raw.y - min.y);
  if (joy->y < 0)
     joy->y = 0;
  joy->y = (int) ((joy->y * 254.0) / (max.y - min.y));
  if (joy->y > 0xfe)
     joy->y = 0xfe;

  joy->z = 0x7f;  /* centered */

  return (OKAY);
} /* raw_to_rcp */

void calibrate_joystick(void)
{
   JOYSTRUCT raw;

   get_joystick_values (AD_BITS, &raw);
   if (raw.x > max.x)
      max.x = raw.x;
   if (raw.x < min.x)
      min.x = raw.x;

   if (raw.y > max.y)
      max.y = raw.y;
   if (raw.y < min.y)
      min.y = raw.y;

   gotoxy (40, 15);
   cprintf ("max %02x %02x", max.x, max.y);
   gotoxy (40, 16);
   cprintf ("min %02x %02x", min.x, min.y);
   return;
} /* calibrate_joystick */

void zero_joystick_calib(void)
{
   max.x = 0;
   min.y = 0;
   min.x = 0x7fff;
   min.y = 0x7fff;
} /* zero_joystick_calib(void) */

/**************************************************************************
---------------------------------------------------------------------------
***************************************************************************
*                                                                         *
*  This section sets up the routines in this file for standalone          *
*  operation.  In the Borland programming environment, in                 *
*  Options/Compiler/Code Generation.../Defines add TESTMAIN in order      *
*  to enable this section.                                                *
*                                                                         *
***************************************************************************
---------------------------------------------------------------------------
***************************************************************************/

#ifdef TESTMAIN

#include <conio.h>
#include <stdio.h>

#define AD_ADDRESS      0x320      /* I/O address of A/D and digital board */
#define ALL_OUTPUT      0x90       /* Digital I/O port mode */

void main()
{
   JOYSTRUCT j;
   byte lt, rt, lg, rg;

   init_dm5406 (AD_ADDRESS, ALL_OUTPUT, TRUE);

   while (!kbhit())
   {
      get_joystick_values(DRIVE_BITWINDOW, &j);
      printf ("%04x %04x %04x\t", j.x, j.y, j.z);

      normalize_drive_values (&j);
      printf ("%04x %04x %04x", j.x, j.y, j.z);


      rcp_combined_drive (&lt, &rt, &lg, &rg, &j);
      printf ("\t%02x %02x\t%02x %02x\n", lt, lg, rt, rg);

   }
   getch();

} /* main */

#endif
