/***************************************************************************
*
*  dm5406.cpp -- Module for digital I/O and AD conversion routines for the
*                RTD DM5406 board
*
*  Public: init_dm5406(), get_dio(), select_ad_channel(),
*            get_ad() overloaded two times
*
*  Private:  main() when TESTMAIN defined
*
*  Usage, A/D routines:
*     First, init_dm5406_base() must be called with the address of the
*       DM5406 if the default value of 0x300 is incorrect.  The ad_enable
*       flag must be set to TRUE if the A/D routines are to be used.
*
*     If a single A/D channel is going to be sampled all the time, then
*       call select_ad_channel() once, and then get_ad(void) can be
*       called repeatedly.
*
*     If more than one A/D channel are to be sampled, then just use
*       get_ad(channel).  This is a little slower than get_ad(void) but
*       does not require the user explicitly to call select_ad_channel().
*
*  Usage, digital I/O routine:
*     First, init_dm5406_base() must be called with the address of the
*       DM5406 if the default value of 0x300 is incorrect.
*
*     Call get_dio() with DIO_PORT_A or DIO_PORT_C as the argument, to
*       get the byte values from those ports.
*
*     The corresponding output function put_dio() has not been
*     implemented yet.
*
*
*  All routines except init_dm5406_base() return an error value of type
*     DM5406_ERROR_TYPE.  select_ad_channel() returns the value, while
*     all other functions return the error value in _dm5406_error.
*
*  Created:  waa, 12-22-93
*
***************************************************************************/

#include <conio.h>
#include <dos.h>
#include <stdio.h>

#include "dm5406.h"


static int dm5406_base = DEFAULT_BASE;   /* I/O base address for DM5406 board */
DM5406_ERROR_TYPE _dm5406_error = DM5406_OKAY;

/***************************************************************************
*
*  void init_dm5406(int base_addr, int ppi_mode, int ad_enable)
*
*  Args:  base_addr is an I/O address from 0x200 through 0x3f0 in increments
*           of 0x010.
*         ppi_mode is the 8255 PPI control byte that controls the input/
*           output direction of the digital ports.
*         ad_enable is TRUE (1) or FALSE (0) to state whether the A/D
*           registers should be initialized
*
*  Action:
*    Sets the internal variable dm5406_base to the given I/O base address.
*    It is the caller's responsibility to make sure this is a valid
*    address.  dm5406_base defaults to the factory setting of 0x300 if
*    not explicitly set by this routine.
*
*    If A/D conversions are NOT to be done, the ppi_mode control byte
*    is written with the high bit set (must be set so that control
*    byte is used).
*
*    If A/D conversions ARE enabled, the ppi_mode control byte is
*    written with the high bit set and Port B set to output in Mode 0.
*    This is so the analog routines will work.  The the DM5406 is
*    reset to be ready for A/D conversions and makes an initial
*    reading to clear out garbage analog data.  Not sure if all
*    channels need to be cleared.
*
*  Created:  waa, 12-22-93
*
***************************************************************************/
void init_dm5406 (int base_addr, int ppi_mode, int ad_enable)
{
   /* Iniitialize base address */
   dm5406_base = base_addr;

   /* Port B must always be to set to output, Mode 0 */
   /* High bit must be set to enable the control byte write */
   outportb (PPI_CONTROL_ADDR, (ppi_mode & 0xf9) | 0x80);

   /* If A/D enabled, then reset registers and read first data */
   if (ad_enable == TRUE)
   {
      /* Reset the A/D registers */
      outportb (RESET_ADDR, 0);     /* Value written is irrelevant */

      /* Do an initial read to clear out A/D garbage */
      get_ad(0);
   }

   return;
} /* init_dm5406 */

/***************************************************************************
*
*  int get_dio (int port)
*
*  Arg:  port is the digital I/O port, DIO_PORT_A or DIO_PORT_C
*
*  Action:
*    Reads the given port if the port is valid (must be Port A or C),
*    otherwise returns zero and _dm5406_error is set.  Port B is
*    used for internal board functions, and is unavailable for
*    digital I/O.
*
*  Returns:  The value from the given I/O port, or 0 if invalid port
*
*  Created:  waa, 12-22-93
*
***************************************************************************/
int get_dio (int port)
{
   int result;

   switch (port)
   {
      case DIO_PORT_A :
              result = inportb (PPI_PORT_A);
              _dm5406_error = DM5406_OKAY;
              break;
      case DIO_PORT_C :
              result = inportb (PPI_PORT_C);
              _dm5406_error = DM5406_OKAY;
              break;
      default :
              result = 0;
              _dm5406_error = DM5406_INVALID_DIO_PORT;
              break;
   }

   return (result);
} /* get_dio */

/***************************************************************************
*
*  DM5406_ERROR_TYPE select_ad_channel(int channel)
*
*  Arg:  channel is an A/D channel from 1 through 16
*
*  Action:
*    Selects a conversion channel.  Subsequent conversions will use this
*    channel.  The IRQ is NOT enabled.
*
*  Returns: DM5406_OKAY if channel selected
*           DM5406_INVALID_CHANNEL if channel number is invalid
*
*  Created:  waa, 12-22-93
*
***************************************************************************/
DM5406_ERROR_TYPE select_ad_channel(int channel)
{
   if (is_valid_ad_channel(channel))
   {
      outportb (CHANNEL_ADDR, channel-1); /* AD uses chan nos. 0..15 */
      return (DM5406_OKAY);
   }
   else
      return (DM5406_INVALID_CHANNEL);
} /* select_ad_channel */

/***************************************************************************
*
*  int get_ad(void)
*
*  Args:  none (see overloaded version of this fn)
*
*  Action:
*    Starts A/D conversion and then reads the result.  Since the
*    conversion time is at least 20us, we delay for one millisecond
*    which should be enough and which is the minimum resolution of
*    the delay() routine.
*    The data is obtained from the same 8-bit port in two reads;
*    the first read gets the 4 right-justified high bits (MSB), while
*    the second read gets the 8 low bits (LSB).  The complete 12-bit
*    value is constructed by shifting the MSB left by 8 bits
*    then OR-ing in the LSB 8 bits.
*
*
*  Returns: 12-bit A/D value
*
*  Created:  waa, 12-22-93
*
***************************************************************************/
int get_ad(void)
{
   int result;

   /* Start the conversion and pause while it completes */
   outportb (CONVERT_ADDR, 0);
   delay (1);

   /* The End of Conversion bit is set to 1 when the conversion is done */
   if (inportb (STATUS_ADDR) & 0x01)
   {
      result = inportb (DATA_ADDR);  /* MSB */
      result <<= 8;                  /* Data is right justified */
      result |= inportb (DATA_ADDR); /* LSB */
      _dm5406_error = DM5406_OKAY;
   }
   else
   {
      result = 0;
      _dm5406_error = DM5406_CONVERSION_INCOMPLETE;
   }

   return (result);

} /* get_ad */

/***************************************************************************
*
*  int get_ad(int channel)
*
*  Args:  channel is the A/D channel which ranges from 1 through 8
*         (see also overloaded version of this fn)
*
*
*  Action:
*    First selects the given channel for data.  If the channel number is
*    invalid, returns 0 with _dm5406_error set to DM5406_INVALID_CHANNEL.
*    Starts A/D conversion and then reads the result.  Since the conversion
*    time is at least 20us, we delay for one millisecond which is enough,
*    and which is the minimum resolution of the delay() routine.
*    The result must be shifted to the right by 4 bits, because the
*    read value is left justified.
*
*  Returns: 12-bit A/D value
*
*  Created:  waa, 12-22-93
*
***************************************************************************/
int get_ad(int channel)
{
   int result;

   /* Select given channel */
   if (select_ad_channel(channel) == DM5406_OKAY)
   {
      /* Start conversion and pause while it completes */
      outportb (CONVERT_ADDR, 0);
      delay (1);

      /* The End of Conversion bit is set to 1 when the conversion is done */
      if (inportb (STATUS_ADDR) & 0x01)
      {
         result = inportb (DATA_ADDR);  /* MSB */
         result <<= 8;                  /* Data is right justified */
         result |= inportb (DATA_ADDR); /* LSB */
         _dm5406_error = DM5406_OKAY;
      }
      else
      {
         result = 0;
         _dm5406_error = DM5406_CONVERSION_INCOMPLETE;
      }
   }
   else /* Invalid channel number */
   {
      result = 0;
      _dm5406_error = DM5406_INVALID_CHANNEL;
   }

   return (result);

} /* get_ad */


#ifdef TESTMAIN
/***************************************************************************
*
*  This is a test main which is compiled only when TESTMAIN is defined
*  during compilation.
*
*  Created:  waa, 12-22-93
*
***************************************************************************/
main ()
{
   char str[20];
   int channel;
   int val;

   /* Enable this section if you want to sample one particular channel */
   /* Then hack the insides of the kbhit loop to read and display that */
   /*   one channel */
/*
   do
   {
      printf ("A/D channel? [1..16, default=1] ");
      gets (str);

      if ((*str == '\0') || sscanf(str, "%d", &channel) == 0)
         channel = 1;
   } while (!is_valid_ad_channel(channel));
*/

   /* Initialize base address, port modes, and enable A/D system */
   init_dm5406 (0x320, 0x90, TRUE);

   /* Flush kbd buffer */
   if (kbhit())
      getch();

   /* Run until any key hit */
   while (!kbhit())
   {
      /* Customized for reading a 3-axis joystick */

      /* X */
      val = get_ad(2);
      if (_dm5406_error != DM5406_OKAY)
         printf ("? %02x ", _dm5406_error);
      else
         printf ("%04x ", val);

      /* Y */
      val = get_ad(1);
      if (_dm5406_error != DM5406_OKAY)
         printf ("? %02x ", _dm5406_error);
      else
         printf ("%04x ", val);

      /* Z */
      val = get_ad(3);
      if (_dm5406_error != DM5406_OKAY)
         printf ("? %02x\n", _dm5406_error);
      else
         printf ("%04x\n", val);

   }

   /* Flush that last char */
   getch();


   return (0);

} /* main */
#endif