/*---------------------------------------------------------------------
*
*  comm.cpp -- interrupt driven serial communication routines
*
*  Public:  init_COM1(), init_COM2(), init_COM1_int(), init_COM2_int(),
*           restore_COM1_int(), restore_COM2_int(), save_imr(),
*           restore_imr(),
*
*   !!!!
*
*  NOTE: This code is from _The Waite Group's MS-DOS Developer's Guide,
*        Second Edition_, Howard W. Sams & Co., 1989
*
*   !!!!
*
*
*  This program interfaces with COM1 and COM2 on IBM PCs.
*
*  Two data queues are set up for each com port, the code will access
*  data coming in and out of the ports via these queues.  Use the routines
*  q_puton() and q_getfrom() to pass data in and out of the queues.
*
* 	byte rcv1_status_flag = FALSE;
*	byte rcv2_status_flag = FALSE;
*
*  The rcvX_status_flag variables indicate that a receive
*  line status error has occurred.  These are overrun, parity, framing
*  or break interrupt errors, and are an indication of garbage data.
*  The user must reset these flags when action is taken.
*  e.g.- flush the buffer, request a repeat message, ...
*
*  Created: 1/23/92 jbp modified tcom.c to create this program
*  Modified:
*     11-02-93, WAA -- Using this as a generic library of routines,
*                      with the main available only if compiled with
*                      TESTMAIN defined.  Added ellipses to interrupt
*                      prototypes and declarations, as this is now a
*                      .CPP file.
*
*------------------------------------------------------------------------*/


#include <stdio.h>
#include <conio.h>
#include <dos.h>

#include "comm.h" /* for #define OKAY */

/* COM1 UART register #defines */

#define 	COM1		0x3f8   /* address of com1 tx rx buf */
#define 	IER1		0x3f9   /* com1 int enable reg */
#define	IIR1		0x3fa   /* com1 int ID reg */
#define 	LCR1		0x3fb   /* com1 line control register */
#define 	MCR1		0x3fc   /* com1 modem control reg */
#define	LSR1		0x3fd   /* com1 line status reg */
#define 	MSR1		0x3fe   /* com1 modem status reg */
#define 	COM1_EOI	0x64    /* specific EOI for com1 int */


/* COM2 UART register #defines */

#define 	COM2		0x2f8   /* address of com2 tx/rx buf */
#define 	IER2		0x2f9   /* com2 int enable reg */
#define	IIR2		0x2fa   /* com2 int ID reg */
#define 	LCR2		0x2fb	  /* com2 line control register */
#define 	MCR2		0x2fc   /* com2 modem control reg */
#define	LSR2		0x2fd   /* com2 line status reg */
#define 	MSR2		0x2fe   /* com2 modem status reg */
#define 	COM2_EOI	0x63    /* specific EOI for com2 int */

/*interrupt enable/disable masks */
#define RDINT	0x01    /* enables rcv data interrupts at UART */
#define THREINT 0x02   /* enables xmit holding reg empty ints at UART */
#define RXSINT	0x04    /* enables receive status interrupts */
							  /* (overrun, parity, framing & break interrupts) */
#define THREOFF	0xfd /* turns off xmit interrupts */
#define RXSOFF	0xfb    /* turns off rcv status ints */

#define		TRUE	1
#define		FALSE	0

#define		BIT0	1       /* bit masks */
#define		BIT1	2
#define		BIT2	4
#define		BIT3	8
#define		BIT4	16
#define		BIT5	32
#define		BIT6	64
#define		BIT7	128

#define Port_A	0x20       /* 8259 PIC register 1 */
#define Port_B	0x21       /* 8259 PIC register 2 */

#define RXQSIZE 256
#define TXQSIZE 256

typedef unsigned char byte;

/*setup of data queues */
typedef struct QTYPE             /* data stucture for a queue */
{
	int count;
	int front;
	int rear;
	int maxsize;
	byte *data;
} QTYPE;

byte txbuf1[TXQSIZE],rxbuf1[RXQSIZE],txbuf2[TXQSIZE],rxbuf2[RXQSIZE];

QTYPE trmq1 = {0,-1,-1,TXQSIZE,txbuf1},
		rcvq1 = {0,-1,-1,RXQSIZE,rxbuf1},
		trmq2 = {0,-1,-1,TXQSIZE,txbuf2},
		rcvq2 = {0,-1,-1,RXQSIZE,rxbuf2};


QTYPE *txq1 = &trmq1, *rxq1 = &rcvq1, *txq2 = &trmq2, *rxq2 = &rcvq2;

byte rcv1_status_flag = FALSE; /* flag indicating a rcvr line status intr */
byte rcv2_status_flag = FALSE; /* flag indicating a rcvr line status intr */

/* Counting flags for telling when complete RCP blocks have arrived */
int block_arrived1 = 0;
int block_arrived2 = 0;

byte old_imr;  /* saved Interrupt Mask Register */

void interrupt (*old_12)(...);	/* old vector for IR4, COM1 */

void interrupt (*old_11)(...); /* old vector for IR3, COM2 */

/* Prototypes, this file: */

void interrupt COM1_handler(...);		/* ISR for COM1 */

void interrupt COM2_handler(...);      /* ISR for COM2 */

byte *q_getfrom( QTYPE *, byte *);

int q_puton(QTYPE *, byte *);

void display(void);


/*-----------------------------------------------------------------
*
*  void init_COM1()
*
*  Args:  None
*
*  Action:	Initializes COM1 port as 4800 baud, 8 bits data, 1 stop bit,
*  			no parity, DTR and RTS disabled.  Interrupts are enabled
*  			for data ready in the receive buffer, and receive line status.
*
*
*             Common
*           Baud Rates   Divisors
*               300        384
*              1200         96
*              2400         48
*              4800         24
*              9600         12
*              19.2k         6
*
*  Result:  None
*
*---------------------------------------------------------------------*/

void init_COM1()
{
	outportb(LCR1,0x80); /* set DLAB=1 in LCR to access 3f8/9 as baud div */

	outportb(0x3f8,24);	/* 4800 baud divisor LSB */
	outportb(0x3f9,0);  	/* 4800 baud divisor MSB */

	outportb(LCR1,0x03);	/* 8 bits, 1 stop, no parity DLAB = 0 */
	outportb(MCR1,0x0b);	/* enable interrupts, DTR & RTS active */
	outportb(IER1,(RDINT | RXSINT)); /* enable RX data avail and */
												/* RX status interrupts */

}  /* end init_COM1() */


/*-----------------------------------------------------------------
*
*  void init_COM2()
*
*  Args:  None
*
*  Action:	Initializes COM2 port as 1200 baud, 8 bits data, 1 stop bit,
*  			no parity, DTR and RTS disabled.  Interrupts are enabled
*  			for data ready in the receive buffer, and receive line status.
*
*
*             Common
*           Baud Rates   Divisors
*               300        384
*              1200         96
*              2400         48
*              4800         24
*              9600         12
*              19.2k         6
*
*  Result:  None
*
*---------------------------------------------------------------------*/

void init_COM2()
{
	outportb (LCR2, 0x80);  /* set DLAB=1 in LCR to access 2f8/9 as baud div */

	outportb(0x2f8,24);	/* 4800 baud divisor LSB */
	outportb(0x2f9,0);  	/* 4800 baud divisor MSB */

	outportb (LCR2,0x03);	/* 8 bits, 1 stop, no parity DLAB = 0 */
	outportb (MCR2,0x0b);	/* enable interrupts, DTR & RTS active */
	outportb (IER2, (RDINT | RXSINT)); /* enable RX data avail and */
												  /* RX status interrupts */

} /* end init_COM2() */


/*-------------------------------------------------------------------
*
*  void interrupt COM1_handler()
*
*  Args:  None
*
*  Action:  Handles interrupts for COM1 generated on IRQ4.  First the
*  			interrupt ID is read to determine the type of the highest
* 			priority interrupt.  Interrupt priorities are from highest
*           to lowest - receiver line status, received data available,
*			tranmitter holding register empty, and modem status.  This
*			routine does not process modem status interrupts.  Note that a
*			specific EOI is issued before exiting the interrupt handler
*			only after all COM1 related interrupts are serviced.
*
*  Returns:  Nothing
*
*------------------------------------------------------------------------*/

void interrupt COM1_handler (...)
{
	byte c, iir, msr, lsr, ier;
	byte invalue1;

	while (TRUE)  /* service all IRQ before returning */
	{
		iir = inportb(IIR1);  /* read in interrupt ID */
		iir = (iir & 0x07);  /* mask out upper bits */

		if((iir & BIT0) == BIT0)  /* if bit 0 is set no interrupts pending */
		{
			outportb (Port_A, COM1_EOI); /* issue EOI and return */
			return;
		}
		switch(iir) /* determine interrupt ID */
		{
			case 0:	/* modem status intr, read MSR to clear int */
					msr = inportb(MSR1); /* compiler warning MSR not used okay */
					break;

			case 2:	/* tx empty interrupt, int was cleared when IIR read */
					lsr = inportb(LSR1); /* read LSR to see if tx empty */

					if ((lsr & BIT5) == BIT5)	/*Tx Buffer really Empty ? */
					{

						if(q_getfrom(txq1,&c) != NULL) /* if q not empty */
						{
							outportb(COM1,c);  /* send out the char */
						}  /* end if */
						else   /* queue is empty, turn off tx interrupts */
						{
							ier = inportb(IER1);
							outportb (IER1, (ier & THREOFF));  /*end TX int*/
						}  /* end else */

					}    	/* end if(TX really empty)  */


					break;   /* case 2 */

			case 4:	/* rx data ready int, read data from rx buf to clear  */

					invalue1 = inportb (COM1);   /* get the byte */
					if (q_puton(rxq1,&invalue1)) /* and put it on the queue */
                  if (invalue1 < 128)  /* command byte */
                     ++block_arrived1; /* complete block arrived */

					break;

			case 6: /* line status int, read LSR to clear int */
					lsr = inportb(LSR1);
					rcv1_status_flag = TRUE;   /* set rcv status flag */
					break;

			default: break;

		} /* end switch(iir) */

	} /* end endless while loop */

}  	/* end of COM1_handler() */   /* "msr never used" warning okay */



/*-------------------------------------------------------------------
*
*  void interrupt COM2_handler()
*
*  Args:  None
*
*  Action:  Handles interrupts for COM2 generated on IRQ3.  First the
*  			interrupt ID is read to determine the type of the highest
* 			priority interrupt.  Interrupt priorities are from highest
*           to lowest - receiver line status, received data available,
*			tranmitter holding register empty, and modem status.  This
*			routine does not process modem status interrupts.  Note that a
*			specific EOI is issued before exiting the interrupt handler
*			only after all COM2 related interrupts are serviced.
*
*  Returns:  Nothing
*
*------------------------------------------------------------------------*/

void interrupt COM2_handler (...)
{
	byte c, iir, msr, lsr, ier;
	byte invalue2;

	while (TRUE)  /* service all IRQ before returning */
	{
		iir = inportb(IIR2);  /* read in interrupt ID */
		iir = (iir & 0x07);  /* mask out upper bits */

		if((iir & BIT0) == BIT0)  /* if bit 0 is set no interrupts pending */
		{
			outportb (Port_A, COM2_EOI); /* issue EOI and return */
			return;
		}
		switch(iir)
		{
			case 0:	/* modem status intr, read MSR to clear int */
					msr = inportb(MSR2);
					break;

			case 2:	/* tx empty interrupt, int was cleared when IIR read */
					lsr = inportb(LSR2); /* read LSR to see if tx empty */

					if((lsr & BIT5) == BIT5) /*Tx Buffer really empty ? */
					{


						if(q_getfrom(txq2,&c) != NULL) /* if q not empty */
						{
							outportb(COM2,c);  /* send out the char */
						}   /* end if */
						else
						{
							ier = inportb(IER2); /* q empty, turn off int */
							outportb (IER2, (ier & THREOFF));
						} /* end else */

					}    	/* end if(TXreally empty) */

					break;   /* case 2 */

			case 4:	/* rx data ready int, read data from rx buf to clear  */
					invalue2 = inportb (COM2);   /* get byte */
					if (q_puton(rxq2,&invalue2)) /* and put on queue */
                  if (invalue2 < 128)       /* command byte */
                     ++block_arrived2;      /* complete block has arrived */

					break;

			case 6: /* line status int, read LSR to clear int */
					lsr = inportb(LSR2);
					rcv2_status_flag = TRUE; /* and set rcv status flag */
					break;

			default: break;

		} /* end switch(iir) */

	} /* end endless while loop */

}  	/* end of COM2_handler() */  /* "msr never used" warning okay */

/*--------------------------------------------------------------------
*
*   void display()
*
*  Args: None
*
*  Action:  Displays dat from the COM ports
*
*  Results: None
*
*-----------------------------------------------------------------------*/
void display()
{
	byte inc1, inc2;
	int i = 0;

	/* display COM 1 chars */
	while(q_getfrom(rxq1, &inc1) != NULL)
	{
		gotoxy(1,(5+i));
		printf("COM1 inc%01d hex value = %02x",i,inc1);
		printf("  char = %c",(char)inc1);
		i++;
	} /* end while(q_getfrom) */

	i = 0;

	/* display COM2 chars */
	while(q_getfrom(rxq2, &inc2) != NULL)
	{
		gotoxy(1,(15+i));
		printf("COM2 inc%01d hex value = %02x",i,inc2);
		printf("  char = %c",(char)inc2);
		i++;
	} /* end while(q_getfrom) */

	i = 0;


} /* end display */

/*----------------------------------------------------------------------
*
*  byte *q_getfrom( QTYPE *queue, byte *data)
*
*  Args:	*queue is the pointer to the structure QTYPE which contains
*			 the communication ring buffer.  This arg is an input.
*			*data is a pointer to the next data byte in the queue, note that
*			 this is used as an output from the routine and is how the
*			 caller gets the next byte from the queue.
*
*  Action:	copies the next data byte in the queue to the variable data
*			which is accessed by the pointer *data by the caller.  The
*			front queue index is incremented and the queue count is
*			decremented.
*
*  Returns:	a pointer to the next location in the queue.  Note that this
*			is only checked to see if it is equal to NULL, an indication
*			that there is no data in the queue.
*
*------------------------------------------------------------------------*/

byte *q_getfrom(QTYPE *queue, byte *data)
{
	byte *current;
	current = NULL;
	if(queue->front == -1)
		return(current); /* front index = -1 only if queue empty */

	/* else retrieve data from queue */
	current = &(queue->data[queue->front]);
	*data = *current;   /* and store in data */
	queue->count--;     /* decriment queue count */

	if(queue->count == 0) /* the queue is now empty, reset front and rear */
	{
		queue->front = queue->rear = -1;
		return(current);                 /* and return */
	}

	/* else check for wraparound */
	if(queue->front == queue->maxsize-1)
		queue->front = 0;      /* and wraparound */
	else
		queue->front++;        /* or increment front index */
	return(current);          /* return pointer to current element */

} /* end q_getfrom() */

/*---------------------------------------------------------------------
*
*  int q_puton( QTYPE *queue, byte *data)
*
*  Args:	*queue is the pointer to the structure QTYPE which contains
*			 the communication ring buffer.  This arg is an input.
*			*data is a pointer to the data byte that you want to put in
*			 the queue.  This arg is an input.
*  Action:	Puts the byte *data on the specified *queue if there is room.
*			If the element is sucessfully inserted, the rear index is
*			incremented and wrapped around if it is at the end of the
*			buffer.  The front index is also checked for a -1, which would
*			indicate that the buffer is empty and the indexes must be
*			initialized.
*
*  Returns:	Returns a 0 if the queue is full and the byte cannot be inserted.
*			Returns a 1 if the byte was sucessfully inserted.
*
*------------------------------------------------------------------------*/

int q_puton(QTYPE *queue, byte *data)
{
/* first check to see if queue is full, return 0 if full */
	if (queue->count == queue->maxsize)
		return(0);

/* else check for wraparound and increment rear index */
	if(queue->rear == queue->maxsize-1)
		queue->rear = 0;
	else
		queue->rear++;

/* save the byte at the location of the rear index */
	queue->data[queue->rear] = *data;
	queue->count++;                    /* and increment queue count */
	if(queue->front == -1)
		queue->front = 0;  /* initialize the front index */
	return(1);             /* sucessfully inserted the byte */

} /* end q_puton() */


/*---------------------------------------------------------------------
*
*  void init_COM1_int ()
*
*  Args: none
*
*  Action:
*    Initializes the com1 interrupt vector (saves the old one), and
*    enables the comm interrupts.  The old interrupt mask must be
*    saved outside of this routine!
*
*  Returns: nothing
*
*------------------------------------------------------------------------*/
void init_COM1_int()
{
   byte imr;

	old_12 = getvect(12); /* old com1 interrupt vector */

	setvect(12, *COM1_handler); /* set the new com1 IRQ handler */

   imr = inportb (Port_B);

	outportb (Port_B, imr & ~BIT4); /*turn off mask bits for IRQ4=COM1*/

	inportb(COM1); /* clear any extra chars in the rcv buffer */

} /* init_COM1_int */


/*---------------------------------------------------------------------
*
*  void init_COM2_int ()
*
*  Args: none
*
*  Action:
*    Initializes the com2 interrupt vector (saves the old one), and
*    enables the comm interrupts.  The old interrupt mask must be
*    saved outside of this routine!
*
*  Returns: nothing
*
*------------------------------------------------------------------------*/
void init_COM2_int()
{
   byte imr;

	old_11 = getvect(11); /* old com2 interrupt vector */

	setvect(11, *COM2_handler); /* set the new com2 IRQ handler */

   imr = inportb (Port_B);

	outportb (Port_B, imr & ~BIT3); /*turn off mask bits for IRQ3=COM2*/

	inportb(COM2); /* clear any extra chars in the rcv buffer */

} /* init_COM2_int */

/*---------------------------------------------------------------------
*
*  void restore_COM1_int ()
*
*  Args: old_imr contains the contents of the Interrupt Mask Register
*          before the new comm interrupts were installed.
*
*  Action:
*    Restores the com1 interrupt.
*
*  Returns: nothing
*
*------------------------------------------------------------------------*/
void restore_COM1_int ()
{
	setvect(12, old_12);  /* replace old interrupt vectors */

} /* restore_COM1_int */


/*---------------------------------------------------------------------
*
*  void restore_COM2_int ()
*
*  Args: none
*
*  Action:
*    Restores the com2 interrupt.
*
*  Returns: nothing
*
*------------------------------------------------------------------------*/
void restore_COM2_int ()
{
	setvect(11, old_11);  /* replace old interrupt vectors */

} /* restore_COM2_int */

/*---------------------------------------------------------------------
*
*  void save_imr()
*  void restore_imr()
*
*  Args: none
*
*  Action:
*    This pair of routines save and restore the contents of the
*    Interrupt Mask Register, respectively.  This register is used
*    enable and disable interrupts by setting the corresponding
*    bit position to 0 to enable, 1 to disable.
*
*  Returns: nothing
*
*------------------------------------------------------------------------*/
void save_imr (void)
{
	old_imr = inportb (Port_B); /* Save old IMR */
} /* save_imr */

void restore_imr (void)
{
	outportb (Port_B, old_imr);	/* restore original interrupt mask */
} /* restore_imr */


/*************************************************************************
*
*  send_stream()
*
*  Args:
*    port is the comm port to send the bytes, 1 or 2
*    data is a byte stream to send out
*    count is the number of bytes to send out
*
*  Action:
*    Sends byte stream out designated port.
*    Will stuff as many of the characters onto the transmit list as
*    possible.
*
*  Returns:  OKAY if data was successfully added to output list
*            !OKAY otherwise
*
*  Created:  WAA, 8-93
*  Modified:
*     WAA, 09-01-93 -- Increased semaphore timeout from 2 to 5 ticks in
*                      preparation for encapsulating the ajatl() calls.
*                      It'll take longer.
*     WAA, 12-06-93 -- Deleted AMX semaphore code, since it is not
*                      needed under a single threaded system.
*
**************************************************************************/
int send_stream (int port, byte *data, int count)
{
   unsigned char ier;
   int status, i;

   if (port == 1)
   {
      for (i=0; i<count; ++i)
      {
         if ((status = q_puton (txq1, &data[i])) < 0)
           break;
      }
	   if(status >= 0)               /* enable xmit ints if data went on ok */
   	{
	   	ier = inportb(IER1);
		   ier = (ier | THREINT);
		   outportb(IER1, ier);        /* enable com1 tx interrupts */
         return (OKAY);
      }
      else
	      return(!OKAY);
   }
   else
   if (port == 2)
   {
      for (i=0; i<count; ++i)
      {
         if ((status = q_puton (txq2, &data[i])) < 0)
           break;
      }
	   if(status >= 0)               /* enable xmit ints if data went on ok */
   	{
	   	ier = inportb(IER2);
		   ier = (ier | THREINT);
		   outportb(IER2, ier);        /* enable com1 tx interrupts */
         return (OKAY);
      }
      else
	      return(!OKAY);
   }
   else
      return (!OKAY); /* unknown port */
} /* send_stream */

/*************************************************************************
*
*  get_stream()
*
*  Args:
*    port is the comm port to get the bytes from, 1 or 2
*    data is where to write the received data
*    count is the number of bytes read
*
*  Action:
*    Checks the appropriate block_arrived flag, and if a complete
*    block has been read, takes bytes off the queue up to and including
*    the command byte, and puts them into the data array.  The count
*    parameter is returned with the number of bytes read in.
*
*  Returns:  TRUE if a block was successfully transferred to the array
*            FALSE w/ count if a complete block could not be read
*            FALSE w/ zero count if there was not block to be read, or
*              an invalid port number was passed in.
*
*  Created:  WAA, 12-06-93
*
**************************************************************************/
int get_stream (int port, byte *data, int *count)
{
   byte c;

   *count = 0;

   if (port == 1)
   {
      /* Check to see if a block has arrived */
      if (block_arrived1 > 0)
      {
         --block_arrived1;  /* Decrement counting flag */

         /* Get the chars out one by one until cmd byte found or q empty */
         while (q_getfrom (rxq1, &c))
         {
            data[*count] = c;
            ++*count; /* Okay to increment count after char read */
            if (c < 128)  /* Cmd byte found! */
               return (TRUE);
         }

         /* Cmd byte not found before queue emptied */
         return (FALSE);
      }
      else
         return (FALSE); /* No block available */
   }
   else
   if (port == 2)
   {
      /* Check to see if a block has arrived */
      if (block_arrived2 > 0)
      {
         --block_arrived2;  /* Decrement counting flag */

         /* Get the chars out one by one until cmd byte found or q empty */
         while (q_getfrom (rxq2, &c))
         {
            data[*count] = c;
            ++*count; /* Okay to increment count after char read */
            if (c < 128)  /* Cmd byte found! */
               return (TRUE);
         }

         /* Cmd byte not found before queue emptied */
         return (FALSE);
      }
      else
         return (FALSE); /* No block available */
   }
   else
      return (FALSE);   /* Unknown port */
} /* get_stream */


/**************************************************************************
---------------------------------------------------------------------------
***************************************************************************
*                                                                         *
*  This section sets up the routines in this file for standalone          *
*  operation.  In the Borland programming environment, in                 *
*  Options/Compiler/Code Generation.../Defines add TESTMAIN in order      *
*  to enable this section.                                                *
*                                                                         *
***************************************************************************
---------------------------------------------------------------------------
***************************************************************************/

#ifdef TESTMAIN

#include "comm.h"  /* For function prototypes */

/*------------------------------------------------------------------
*
* void main()
*
*---------------------------------------------------------------------*/

void main()
{
	char inchar = 55;
	byte old_imr, ier;

	/* initialize the COM ports */
	init_COM1();
	init_COM2();

   /* Enable the interrupts */
   save_imr ();
   init_COM1_int();
   init_COM2_int();

	clrscr();


	printf("Communications Started, hit ESC to get out.\n");

	while (inchar != 27)        /* ESC to get out */
	{
		if (kbhit())
		{
		inchar = getch();


		if(q_puton(txq1,&inchar) == 1); /*put key on com1 tx buffer*/
			{
			ier = inportb(IER1);         /* if sucessfull */
			ier = (ier | THREINT);
			outportb(IER1, ier);         /* enable com1 tx interrupts */
			} /* end if */

		if(q_puton(txq2,&inchar) == 1); /*put key on com2 tx buffer*/
			{
			ier = inportb(IER2);        /* if sucessfull */
			ier = (ier | THREINT);
			outportb(IER2, ier);         /* enable com2 tx interrupts */
			 }


		} /* end if(kbhit) */

		display();

		if(rcv1_status_flag == TRUE)
			{
			gotoxy(30,20);
			printf("com1 receive line status error! ");
			rcv1_status_flag = FALSE;
			}

		if(rcv2_status_flag == TRUE)
			{
			gotoxy(30,21);
			printf("com2 receive line status error! ");
			rcv1_status_flag = FALSE;
			}

	} /* end main loop */

   restore_imr();  /* turns off comm interrupts */

   restore_COM1_int();
   restore_COM2_int();

	clrscr();

} /* end main() */

#endif
