/*****************************************************************************
 * PROJECT: TCA
 *
 * (c) Copyright Richard Goodwin, 1995. All rights reserved.
 *
 * FILE: tcaDev.c
 *
 * ABSTRACT:
 * 
 * This file provides routines for connecting TCA to the set of device
 * interfaces (see devUtils.{c,h}).
 *
 * ADAPTED FROM XAVIER SOFTWARE TO FACILITATE ADDING I/O TO MODULES.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/utils/tcaDev.c,v $ 
 * $Revision: 1.25 $
 * $Date: 1996/07/27 21:19:24 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: tcaDev.c,v $
 * Revision 1.25  1996/07/27  21:19:24  rich
 * Try closing TCA on a null input.
 *
 * Revision 1.24  1996/06/29  00:41:46  rich
 * Don't disconnect on zero now that we check for accept fds.
 *
 * Revision 1.23  1996/06/27  15:41:53  rich
 * Added fixes for accepting fds in tca.
 *
 * Revision 1.22  1996/06/27  03:07:28  rich
 * Fixed problem where it would not listen for tca.
 *
 * Revision 1.21  1996/06/25  20:53:35  rich
 * Fixed memory and other problems found with purify.
 * Also fixed TCA_isConnected and TCA_isDisconnected.  Added TCA_disconnect.
 *
 * Revision 1.20  1996/06/07  21:37:38  reids
 * devIsDisconnected was still broken (it returned TRUE if trying to reconnect).
 *   Fixed TCA_connect: would segv if centralhost not set.
 *
 * Revision 1.19  1996/05/14  22:41:05  rich
 * Need to save the current context.
 *
 * Revision 1.18  1996/02/21  18:36:00  rich
 * Use new routine for handling input.
 *
 * Revision 1.17  1996/02/10  16:54:20  rich
 * Made private functions static and fixed some forward declarations.
 *
 * Revision 1.16  1996/01/12  00:55:55  rich
 * Simplified GNUmakefiles. Fixed header include problem with release 8.3.
 *
 * Revision 1.15  1995/12/15  01:32:07  rich
 * Added routines to free data, devFreeDev and devFreeLineBuffer.
 * Added a parameter to set the behaviour when there is a signal, but no
 * characters to read on a socket.  Added routines to connect fds for
 * reading or sending only.  This is useful when spawning sub-processes and
 * you want to manage stdin, stdout and stderr.  This is used by nanny.
 * Added EZX_DestroyPopupPrompt and EZX_CenterPopup.
 *
 * Revision 1.14  1995/11/03  03:05:53  rich
 * Reid's fixes for handling multiple centrals.
 *
 * Revision 1.13  1995/11/01  21:27:38  rich
 * Allow registerAll function to be NULL.
 *
 * Revision 1.12  1995/10/29  18:30:14  rich
 * Fixes for context switching.  Propagated from 8.2 branch.
 *
 * Revision 1.11  1995/10/25  22:52:42  rich
 * The tca-device can now handle multiple central servers.
 * devUpdateConnections now takes the maximum number of connections.
 * The default stdin output handler now prints a prompt.
 *
 * Revision 1.10  1995/10/10  00:44:26  rich
 * Fixed a bug that would cause a crash if tca was not started before a
 * module using tcaDev tried to connect.
 *
 * Revision 1.9  1995/10/07  19:11:43  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Changed devIsConnected to devHasFd.
 *
 * Revision 1.8  1995/08/14  21:35:04  rich
 * Changed devReadN and devWriteN to take the same parameters as read and
 * write.  This allows macro substitution for the system calls.
 * Added "dev" prefix to everything.  Some minor fixes.
 *
 * Revision 1.7  1995/08/05  23:23:36  rich
 * Added functional interface to devUtils.  See the README for details.
 *
 * Revision 1.6  1995/07/19  14:32:57  rich
 * Error checking on select and accept calls.
 *
 * Revision 1.5  1995/06/14  03:26:15  rich
 * Added DBMALLOC_DIR.
 * Fixed problems with multiple connections.
 *
 * Revision 1.4  1995/06/01  00:28:28  rich
 * Use TIME_MAX rather than LONG_MAX for timeouts.
 * Fix some problems with devUtils processOutput.
 *
 * Revision 1.3  1995/04/07  05:11:21  rich
 * Fixed GNUmakefiles to find the release directory.
 * Moved all system includes into libc.h
 * Moved global variables into the c files and got rid of #define DECLARE_...
 * Now works with xavier stuff.
 *
 * Revision 1.2  1995/04/04  19:48:25  rich
 * Added sgi support.
 * Changed setAlarm to setTimer.
 * Numerous improvements and bug fixes in devUtils.
 *
 * Revision 1.1  1995/03/30  15:53:34  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added ezx library and the devutils library.
 *
 *
 * 30-Jan-1993 Richard Goodwin Created.
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "tca.h"

#include "handlers.h"
#include "timeUtils.h"
#include "devUtils.h"
#include "tcaDev.h"

/*****************************************************************************
 * Global constants
 *****************************************************************************/

/*****************************************************************************
 * Global Types
 *****************************************************************************/

typedef struct _TCA_DEV_TYPE {
  DEV_PTR dev;
  char *moduleName;
  char *centralHost;
  void (* registerAll)(void);
  void (* disconnectHnd)(void);
  void (* reconnectHnd)(void);
  const char **provides;
  const char **requires;
  TCA_CONTEXT_PTR context;
} TCA_DEV_TYPE;

/*****************************************************************************
 * Global variables
 *****************************************************************************/

TCA_DEV_PTR    TCA_device = NULL;
/*
{
  { 
    NOT_CONNECTED,
    FALSE,
    { "", DEFAULT_PORT},
    { "TCA central", DEFAULT_BAUD},
    "TCA central",
    NO_FDS,
    NO_FDS,
    LISTENING | TALKING | ACCEPTING,
    FALSE,
    (FILE *) NULL,
    &devConnections,
    (DEVICE_OUTPUT_HND) TCA_outputHnd,
    Null_Handler,  
    {TIME_MAX, 0},
    Null_Handler,
    {0, 0},
    {TIME_MAX, 0},
    (void (*)(DEV_PTR)) NULL,  
    (void (*)(DEV_PTR)) NULL,  
    (void (*)(DEV_PTR)) NULL,  
  },
  (char *) NULL,
  (void (*)(void)) NULL,
  (void (*)(void)) NULL,
  (void (*)(void)) NULL,
  (const char **)NULL,
  (const char **)NULL
};
*/

static TCA_DEV_PTR tcaDevices[FD_SETSIZE];
static char *centralHost = NULL;

/*****************************************************************************
 * Forward Declarations
 *****************************************************************************/

static void tcaExitHnd(void);
static void tcaUpdateConnections(TCA_DEV_PTR tcaDev);

/*****************************************************************************
 *
 * FUNCTION: void TCA_outputHnd(int fd, long chars_available)
 *
 * DESCRIPTION: Handles TCA events.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * HISTORY:
 *
 *****************************************************************************/

void TCA_outputHnd(int fd, long chars_available)
{
  TCA_DEV_PTR tcaDev = tcaDevices[fd];

  if (chars_available > 0) {
    /* Switch in the correct context, if needed. */
    if (!(FD_ISSET(fd,tcaGetConnections()))) {
      if (tcaDev != NULL) {
	tcaSetContext(tcaDev->context);
	TCA_device = tcaDev;
      }
    }
  }
  if (tcaHandleFdInput(fd) != Success) {
    if (fd_isZero(tcaGetConnections()))
      devDisconnectDev(tcaDev->dev,FALSE,TRUE);
  }
  
  tcaUpdateConnections(tcaDev);  
}

/*****************************************************************************
 *
 * FUNCTION: void tcaUpdateConnections(void)
 *
 * DESCRIPTION: Fixes the information on connections to tca.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * HISTORY:
 *
 *****************************************************************************/

static void tcaUpdateConnections(TCA_DEV_PTR tcaDev)
{
  int i, maxFD;
  fd_set *connections;
  
  /* Now have to make sure everything is up to date. */
  connections = tcaGetConnections();
  maxFD = tcaGetMaxConnection();

  for (i=0; i<=maxFD; i++) {
    if (FD_ISSET(i,connections))
      tcaDevices[i] = tcaDev;
    else if (tcaDevices[i] == tcaDev)
      tcaDevices[i] = NULL;
  }
  devUpdateConnections(tcaDev->dev, tcaGetConnections(), maxFD);
  fd_copy(tcaGetAcceptFds(),&(tcaDev->dev->acceptFds));
}

static void try_connect(void)
{
  tcaRegisterExitProc(tcaExitHnd);
  tcaConnectModule(TCA_device->moduleName, TCA_device->centralHost);
  
  if (tcaGetServer() > 0) {
    devConnectDev(TCA_device->dev,tcaGetServer());
    /* Do this here, so that user can override within the registration fn */
    tcaEnableDistributedResponses();
    tcaWillListen(TRUE);
    if (TCA_device->registerAll != NULL)
      (* TCA_device->registerAll)();
    tcaModuleProvidesArgv(TCA_device->provides);
    tcaModuleRequiresArgv(TCA_device->requires);
    tcaWaitUntilReady();
    tcaUpdateConnections(TCA_device);  
    
  } else {
    devConnectDev(TCA_device->dev,NO_FD);
  }
}

static void TCA_disconnectHnd(TCA_DEV_PTR tcaDev)
{
  if (TCA_device->disconnectHnd != NULL)
    (* TCA_device->disconnectHnd)();
  tcaClose();
  tcaRegisterExitProc(tcaExitHnd);
  devDisconnectDev(TCA_device->dev,TRUE,TRUE);
  /* Grab the current context */
  TCA_device->context = tcaGetContext();
}

void TCA_disconnect(TCA_DEV_PTR tcaDev)
{
  if (TCA_device->disconnectHnd != NULL)
    (* TCA_device->disconnectHnd)();
  tcaClose();
  devDisconnectDev(TCA_device->dev,FALSE,FALSE);
}

static void TCA_reconnect(DEV_PTR tcaDev)
{
  TCA_device = devGetClientData(tcaDev);
  
  if (TCA_device == NULL) return;
  
  tcaSetContext(TCA_device->context);
  try_connect();
  if (TCA_isConnected(TCA_device))
    if (TCA_device->reconnectHnd != NULL)
      (* TCA_device->reconnectHnd)();
}


void TCA_setCentralHost(const char *machineName)
{
  centralHost = malloc(strlen(machineName)+1);
  strcpy(centralHost,machineName);
}

TCA_DEV_PTR TCA_connect(char *moduleName,
			void registerAll(void),
			void disconnectFn(void),
			void reconnectFn(void),
			const char **provides, 
			const char **requires)
{
  /* Initialize: This is the first tca connection.   */
  if (TCA_device == NULL) {
    bzero(&tcaDevices,sizeof(tcaDevices));
  }

  /* Create a new device if first connection or connecting to
     a different centralHost */
  if ((TCA_device == NULL) || (TCA_device->centralHost == NULL) ||
      (centralHost && strcmp(TCA_device->centralHost,centralHost) != 0)) {
    TCA_device = (TCA_DEV_PTR)malloc(sizeof(TCA_DEV_TYPE));
    bzero(TCA_device,sizeof(TCA_DEV_TYPE));
    TCA_device->dev = 
      devCreateDev("TCA device",
		   DEV_OUTPUTHND, TCA_outputHnd,
		   DEV_LISTENING, LISTENING | TALKING | ACCEPTING,
		   DEV_DISCONNECTHND, TCA_disconnectHnd,
		   DEV_RECONNECTHND, TCA_reconnect,
		   DEV_CLIENT_DATA, (void *)TCA_device,
		   DEV_CLOSE_ON_ZERO, (void *)TRUE,
		   NULL);
  }
  
  TCA_device->moduleName = moduleName;
  if (centralHost != NULL) {
    TCA_device->centralHost = malloc(strlen(centralHost)+1);
    strcpy(TCA_device->centralHost,centralHost);
  } else {
    TCA_device->centralHost = (char *)tcaServerMachine();
  }
  
  TCA_device->registerAll = registerAll;
  TCA_device->disconnectHnd = disconnectFn;
  TCA_device->reconnectHnd = reconnectFn;
  TCA_device->provides = provides;
  TCA_device->requires = requires;
  
  tcaRegisterExitProc(tcaExitHnd);
  TCA_device->moduleName = moduleName;

  try_connect();
  
  /* Grab the current context */
  TCA_device->context = tcaGetContext();

  return TCA_device;
}

void tcaExitHnd(void)
{
  /* prevent the disconnectin handler from being reentered. */
  static BOOLEAN inTCAExitHnd=FALSE;

  if (inTCAExitHnd)
    return;
  inTCAExitHnd = TRUE;
  fprintf(stderr,"DevUtils clearing TCA connection and retrying later.\n");
  if (tcaGetServer() <= 0) {
    devDisconnectDev(TCA_device->dev,TRUE,TRUE);
  }
  inTCAExitHnd = FALSE;
}

BOOLEAN TCA_isConnected(TCA_DEV_PTR tcaDev)
{
  return devHasFd(tcaDev->dev);
}


BOOLEAN TCA_isDisconnected(TCA_DEV_PTR tcaDev)
{
  return devIsDisconnected(tcaDev->dev);
}


DEV_PTR TCA_getDev(TCA_DEV_PTR tcaDev)
{
  return tcaDev->dev;
}

void TCA_setContext(TCA_DEV_PTR tcaDev)
{
  tcaSetContext(tcaDev->context);
}
