/*****************************************************************************
 * PROJECT: APEX
 *
 * (c) Copyright 1995 Richard Goodwin. All rights reserved.
 *
 * FILE: simDev.c
 *
 * ABSTRACT:
 * 
 * This file provides routines for connecting Sim to the set of device
 * interfaces (see devUtils.{c,h}).
 *
 * ADAPTED FROM XAVIER SOFTWARE TO FACILITATE ADDING I/O TO MODULES.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/utils/simDev.c,v $ 
 * $Revision: 1.15 $
 * $Date: 1996/02/10 16:54:18 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: simDev.c,v $
 * Revision 1.15  1996/02/10  16:54:18  rich
 * Made private functions static and fixed some forward declarations.
 *
 * Revision 1.14  1996/01/12  00:55:49  rich
 * Simplified GNUmakefiles. Fixed header include problem with release 8.3.
 *
 * Revision 1.13  1995/12/17  20:27:56  rich
 * Fixed some problems with close on zero.
 *
 * Revision 1.12  1995/10/07  19:11:40  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Changed devIsConnected to devHasFd.
 *
 * Revision 1.11  1995/08/05  23:23:29  rich
 * Added functional interface to devUtils.  See the README for details.
 *
 * Revision 1.10  1995/07/30  17:13:07  rich
 * Changes to avoid deadlock on reconnecting to the simulator device.
 * Check for other events if a timeout occures when waiting for the
 * connection to the simulator.  Allow one device to disconnect another in
 * its disconnect handler.
 *
 * Revision 1.9  1995/07/30  03:54:44  rich
 * Wrong version.
 *
 * Revision 1.8  1995/07/30  03:50:09  rich
 * Fix for output parsing routine.  Compile utils with -g.
 *
 * Revision 1.7  1995/07/19  14:32:52  rich
 * Error checking on select and accept calls.
 *
 * Revision 1.6  1995/07/12  04:57:22  rich
 * Release of 8.0.
 *
 * Revision 1.5  1995/07/10  16:20:44  rich
 * Interm save.
 *
 * Revision 1.4  1995/06/14  03:26:10  rich
 * Added DBMALLOC_DIR.
 * Fixed problems with multiple connections.
 *
 * Revision 1.3  1995/06/06  00:00:45  rich
 * Fixed problem where sockets where closed when there was no error.
 * Improved the handling of SIGPIPE errors.
 *
 * Revision 1.2  1995/06/01  00:28:26  rich
 * Use TIME_MAX rather than LONG_MAX for timeouts.
 * Fix some problems with devUtils processOutput.
 *
 * Revision 1.1  1995/04/07  05:16:35  rich
 * Added x11Utils to connect tca to the X11 main loop.
 * Added sim device to use in creating simulators.
 *
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "tca/basics.h"
#include "handlers.h"
#include "timeUtils.h"
#include "devUtils.h"
#include "simDev.h"

/*****************************************************************************
 * Global constants
 *****************************************************************************/

/*****************************************************************************
 * Global variables
 *****************************************************************************/

SIM_PTR  sim_device = NULL;

/*
{
  {
    NOT_CONNECTED,
    FALSE,
    { "", DEFAULT_PORT},
    { "Simulator server", DEFAULT_BAUD},
    "Simulator server",
    NO_FDS,
    NO_FDS,
    LISTENING | TALKING | ACCEPTING,
    FALSE,
    (FILE *) NULL,
    &devConnections,
    (DEVICE_OUTPUT_HND) Sim_outputHnd,
    Null_Handler,  
    {TIME_MAX, 0},
    Null_Handler,  
    {0, 0},
    {TIME_MAX, 0},
    (void (*)(DEV_PTR)) NULL,
    (void (*)(DEV_PTR)) NULL,
    (void (*)(DEV_PTR)) simServerInitialize,
  },
  0,
  { 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
  }
};
*/

SIM_TYPE defaultSimDev = 
{
  NULL,
  0,
  { 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
  }
};

/******************************************************************************
 *
 * FUNCTION: BOOLEAN simServerInitialize(const char *name)
 *
 * DESCRIPTION:
 * Initialize the server. Creates a socket, looks up the server port
 * number and binds the socket to that port number, and listens to the
 * socket. The machine name for the simulation server is assumed to be the
 * machine on which it is running and so it needs not be specified.
 *
 * OUTPUTS:
 * Returns FALSE if there was an error detected, TRUE otherwise.
 *
 *****************************************************************************/

BOOLEAN simServerInitialize(char *name)
{
  if (sim_device == NULL) {
    sim_device = (SIM_PTR)malloc(sizeof(SIM_TYPE));
    *sim_device = defaultSimDev;
    
    sim_device->dev = 
      devCreateDev(name,
		   DEV_LISTENING, LISTENING | TALKING | ACCEPTING,
		   DEV_OUTPUTHND, Sim_outputHnd,
		   DEV_RECONNECTHND, Sim_outputHnd,
		   NULL);
  }
  return devServerInitialize(sim_device->dev);
}

/******************************************************************************
 *
 * FUNCTION: int readNChars(sd, buf, nchars)
 *
 * DESCRIPTION:
 * Read nchars of data from sd into buf. Continue to read until
 * nchars have been read.  Return 0 if end of file reached and -1 if
 * there was an error.
 *
 * INPUTS:
 * int sd;
 * char *buf;
 * int nchars;
 *
 * OUTPUTS: int numRead.
 *
 * NOTES:
 *
 * buf is a preallocated pointer to storage equal to nchars.
 * Propagation of low level errors - failures on read/writes still an issue.
 *
 *****************************************************************************/

static int readNChars(int sd, char *buf, int nchars)
{
  int amountRead, amountToRead;
  int i;
  
  amountToRead = nchars;
  for(;;){
    amountRead = read(sd, buf, amountToRead);
    if (amountRead == 0)
      { /* just got an end of file indication.
	 * close the socket and remove it from the list of active
	 * connections
	 */
	for(i=1; ((i<sim_device->numDevs) && 
		  !FD_ISSET(sd,(devGetFds(sim_device->devs[i]->dev))));
	    i++);
	if(FD_ISSET(sd,(devGetFds(sim_device->devs[i]->dev)))) {
	  printf("Connection closed %s\n",
		 devGetName(sim_device->devs[i]->dev));
	  devDisconnectDev(sim_device->devs[i]->dev,TRUE,TRUE);
	}
	return 0;
    } else if (amountRead < 0)
      /*
       * some other problem.  Try to continue?
       */
      return amountRead;
    amountToRead -= amountRead;
    if (amountToRead == 0)
      return amountRead;
    buf += amountRead;
  }
}

/*****************************************************************************
 *
 * FUNCTION: void Sim_outputHnd(int fd, long chars_available)
 *
 * DESCRIPTION: Handles Sim events.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * HISTORY:
 *
 *****************************************************************************/

void Sim_outputHnd(int fd, long ignore)
{
  int new_sd=NO_FD;
  long chars_available;
  char inbuf[DEFAULT_LINE_LENGTH+1];
  int i;
  fd_set readSet;
  int stat;
  struct timeval zeroTimeout = {0,0};
  
  fprintf(stderr, "New Connection Requested \n");
  
  FD_ZERO(&readSet);
  FD_SET(fd,&readSet);
  if (select(FD_SETSIZE, &readSet, NULL, NULL, &zeroTimeout) == 1) {
    if (FD_ISSET(fd,&readSet)) {
      new_sd = accept(fd, NULL, NULL);
    }
  }

  if (new_sd < 0) {
    fprintf(stderr, "Accept Failed\n");
    return;
  }
  
  /* Write a string to the new socket and wait for
   * a reply to indicate which device to connect to
   */
  
  write(new_sd, devGetName(sim_device->dev),
	strlen(devGetName(sim_device->dev)));
  
  do {
    /* wait for the name of the simulator */
    FD_ZERO(&readSet);
    FD_SET(new_sd,&readSet);
    do {
      stat = select(FD_SETSIZE, &readSet, NULL, NULL, NULL);
      if (stat == 0) devProcessDevices(NULL);
    } while (stat < 0 && errno == EINTR);

    if (stat < 0 ) {
      fprintf(stderr,"Sim_outputHnd: Select failed %d\n",errno);
      return;
    }
    
    chars_available = devCharsAvailable(new_sd);
  } while (chars_available <= 0);
  
  /* read in the device name. */
  bzero(inbuf,DEFAULT_LINE_LENGTH+1);
  if (readNChars(new_sd,inbuf,MIN(chars_available,DEFAULT_LINE_LENGTH)) <= 0)
    /* got a bad read */
    return;
  
  fprintf(stderr, " Requested from %s \n", inbuf);
  
  /* find the device in the device_info array and set up the connection. */
  
  for (i=0;
       ((i < sim_device->numDevs) &&
	(strncmp(inbuf,devGetName(sim_device->devs[i]->dev),
		 strlen(devGetName(sim_device->devs[i]->dev))) != 0));
       i++);
  
  if (i == sim_device->numDevs) {
    fprintf(stderr, "Request to connect to unknown device %s", inbuf);
    close(new_sd);
    return;
  };
  
  fprintf(stderr, "Connecting to device %s\n", inbuf);
  
  /* connect to devUtils loop */
  devConnectDev((sim_device->devs[i]->dev),new_sd);
  if (sim_device->devs[i]->initFN != NULL) {
    (* sim_device->devs[i]->initFN)();
  }
  
  /* echo back the name of the connecting device */
  write(new_sd, inbuf, strlen(inbuf));
}

/*****************************************************************************
 *
 * FUNCTION: void Sim_addDevice(SIM_DEV_PTR dev)
 *
 * DESCRIPTION: Adds new simulated devices.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * HISTORY:
 *
 *****************************************************************************/

void Sim_addDevice(SIM_DEV_PTR dev)
{
  if (sim_device->numDevs >= MAX_SIM_DEVS) {
    fprintf(stderr, "Too many devices, increase MAX_SIM_DEVS\n");
    return;
  }
  sim_device->devs[sim_device->numDevs++] = dev;
}

/*****************************************************************************
 *
 * FUNCTION: void Sim_closeDevices(void)
 *
 * DESCRIPTION: closes all open simulated devices.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * HISTORY:
 *
 *****************************************************************************/

void Sim_closeDevices(void)
{
  int i;

  for (i=0; i < sim_device->numDevs; i++) {
    if (devHasFd((sim_device->devs[i]->dev)))
      devDisconnectDev((sim_device->devs[i]->dev), FALSE, FALSE);
    else
      devDisconnectDev((sim_device->devs[i]->dev), TRUE, FALSE);
  }
}
