/*
 * FILE:  devConfig.h
 * DESC:  
 *
 * $Id: devConfig.h,v 1.1 1996/02/10 23:38:43 rich Exp $
 *
 * Modification history: $Log: devConfig.h,v $
 * Revision 1.1  1996/02/10  23:38:43  rich
 * Added RWI's paramter parsing routines.
 *
 * Revision 1.2  1996/01/23  07:27:25  jal
 * Added config iterators, IsDefined operator, AddDef, AddValue, etc.
 * Changed to use realloc() to create a variable sized array for
 * definitions.
 *
 * Revision 1.1  1996/01/14  04:31:20  jal
 * Initial checkin.  Still testing.
 *
 * Revision 1.1.1.1  1996/01/13  14:20:48  jal
 * First checkin of RAI-2.
 *
 *
 */

#ifndef _DEVCONFIG_H
#define _DEVCONFIG_H

#ifdef __cplusplus
extern "C" {
#endif

#include "tca/basics.h"

/*
 *   FUNCTION:  devConfigLoadFile(file)
 *
 *   ABSTRACT:  Uses this config file for setting information.
 *
 *   DESCRIPTION:
 *
 *     Adds the definitions in "file" to the runtime configuration
 *     information.  This function can be called multiple times with
 *     different filenames.
 *
 *     Each succeeding call reads and merges the definitions of the
 *     given file.  Definitions from files read later have precedence in
 *     the case of duplicate names.
 *
 *   RETURN VALUES:
 *
 *     True return value on success, false otherwise.
 *
 *   NOTES:
 *
 *     See also devConfigSetSearchPath() for information on how files
 *     given to devConfigLoadFile() are located.
 *
 *   FILE FORMAT:
 *
 *     Lines which begin with '#' are treated as comments.
 *     Lines of the form "<name>=<value>" are treated as definitions
 *       (whitespace is important, as in the Bourne shell).
 *
 *     The <name> can be any alphanumeric string, not including '='.
 *     The <value> continues until the end of line (no quoting necessary).
 *
 */

BOOLEAN devConfigLoadFile(const char *file);

/*
 *   FUNCTION:  devConfigSetSearchPath(path)
 *
 *   ABSTRACT:  Sets the search path for config files.
 *
 *   DESCRIPTION:
 *
 *     Sets the configuration file search path to "path", a
 *     colon seperated set of directory paths.
 *
 *     When a file cannot be opened by devConfigLoadFile(), the search
 *     path will be used as a set of prefixes to the filename.  The
 *     path is searched in left-to-right order.
 *
 *     Set to NULL to get rid of the search path.
 *
 *   RETURN VALUES:
 *
 *
 *   NOTES:
 *
 *
 */

void devConfigSetSearchPath(const char *path);

/*
 * paramParseArgs : Given argc and argv, look for -D<param name>=<value> 
 * in the parameter strings.  This is the same format the compiler accepts for
 * defining parameters.
 */

void devParseParamArgs(int argc, char **argv);

/*
 *   FUNCTION:  devConfigFindString(name)
 *
 *   ABSTRACT:  Finds a string definition.
 *
 *   DESCRIPTION:
 *
 *     Looks through the runtime configuration information for the
 *     string value associated with "name".
 *
 *   RETURN VALUES:
 *
 *     Character string, or NULL if not found.
 *
 *   NOTES:
 *
 *
 */

const char *devConfigFindString(const char *name);

/*
 *   FUNCTION:  devConfigFind(name, converter, value, convdata)
 *
 *   ABSTRACT:  Finds a definition (applying conversion as necessary).
 *
 *   DESCRIPTION:
 *
 *     Finds the value associated with "name" in the runtime configuration
 *     definitions and sets the contents of "value" to the results of the
 *     conversion by "converter".
 *
 *     The "convdata" pointer is passed to the converter function.  Its
 *     use varies between converters.
 *
 *   RETURN VALUES:
 *
 *     Returns true if the definition was found and conversion succeeded.
 *     False otherwise.
 *
 *   NOTES:
 *
 *     See the typedef for DEV_CONFIG_CONV_HND for the definition
 *     of a converter.
 *
 *     Converters are provided for some default types.  See function
 *     prototypes below of the form devConfigConvertXXX().
 *
 *   EXAMPLES:
 *
 *     float f;
 *     devConfigFind("avoidRadius", devConfigConvertFloat, &f, NULL);
 *     [gets the value of "avoidRadius" as a floating point number]
 */

typedef BOOLEAN (*DEV_CONFIG_CONV_HND)(const char *strValue, void *value,
				  void *convdata);

BOOLEAN devConfigGet(const char *name, DEV_CONFIG_CONV_HND converter,
		     void *value, void *data);

/* Value converters */
BOOLEAN devConfigConvertInt(const char*, void*, void*);
BOOLEAN devConfigConvertShort(const char*, void*, void*);
BOOLEAN devConfigConvertLong(const char*, void*, void*);
BOOLEAN devConfigConvertFloat(const char*, void*, void*);
BOOLEAN devConfigConvertDouble(const char*, void*, void*);
BOOLEAN devConfigConvertString(const char*, void*, void*);


/*
 * devGetParam<type> : Given a parameter name, set the value, if one is 
 * available. Otherwise, leave the value unchanged and return FALSE.
 */

BOOLEAN devGetParamInt(char *paramName, int *value);
BOOLEAN devGetParamShort(char *paramName, short *value);
BOOLEAN devGetParamLong(char *paramName, long *value);
BOOLEAN devGetParamFloat(char *paramName, float *value);
BOOLEAN devGetParamDouble(char *paramName, double *value);
BOOLEAN devGetParamString(char *paramName, char *value);

/*
 *   FUNCTION:  devConfigSetValue(name, value)
 *
 *   ABSTRACT:  Adds a definition into the runtime environment.
 *
 *   DESCRIPTION:
 *
 *     Associates "value" to "name" in the runtime configuration
 *     environment.  If a value is already bound to "name", it is
 *     replaced.
 *
 *   RETURN VALUES:
 *
 *   NOTES:
 *
 *   EXAMPLES:
 *
 */

void devConfigSetValue(const char *name, const char *value);
void devSetParamInt(const char *name, int value);
void devSetParamShort(const char *name, short value);
void devSetParamLong(const char *name, long value);
void devSetParamFloat(const char *name, float value);
void devSetParamDouble(const char *name, double value);
void devSetParamString(const char *name, const char *value);

/*
 *   FUNCTION:  devConfigSetDef(name, value)
 *
 *   ABSTRACT:  Adds a definition into the runtime environment.
 *
 *   DESCRIPTION:
 *
 *     Takes a definition of the form "<name>=<value>", where
 *     whitespace counts in the definition.
 *
 *     Associates "value" to "name" in the runtime configuration
 *     environment.  If a value is already bound to "name", it is
 *     replaced.
 *
 *   RETURN VALUES:
 *
 *   NOTES:
 *
 *   EXAMPLES:
 *
 */

void devConfigSetDef(const char *def);

/*
 *   FUNCTION:  devConfigIsDefined(name)
 *
 *   ABSTRACT:  Checks whether "name" is a defined config variable.
 *
 *   DESCRIPTION:
 *
 *     Checks whether "name" is defined in the runtime configuration
 *     environment.
 *
 *   RETURN VALUES:
 *
 *     Returns a true value if defined, false otherwise.
 *
 *   NOTES:
 *
 *     It is safer to use this function rather than
 *        devConfigFindString(name) != NULL
 *     because a NULL value might be associated with "name".
 *
 *   EXAMPLES:
 *
 */

BOOLEAN devConfigIsDefined(const char *name);

/*
 *   FUNCTION:  devConfigCreateIterator()
 *
 *   ABSTRACT:  Creates a parameter iterator.
 *
 *   DESCRIPTION:
 *
 *     Creates an iterator which can be used to examine the bindings
 *     in the runtime configuration environment.
 *
 *     Iterator must be destroyed using devConfigDestroyIterator().
 *
 *     Iterator operations are functions of the form devConfigIter*().
 *
 *   RETURN VALUES:
 *
 *     DEV_CONFIG_ITERATOR_PTR (opaque type)
 *
 *   NOTES:
 *
 *   EXAMPLES:
 *
 */

struct DEV_CONFIG_ITERATOR_TYPE;
typedef struct DEV_CONFIG_ITERATOR_TYPE *DEV_CONFIG_ITERATOR_PTR;

typedef struct DEV_CONFIG_DEF_TYPE {
  const char *name;
  const char *value;
} DEV_CONFIG_DEF_TYPE, *DEV_CONFIG_DEF_PTR;
typedef const DEV_CONFIG_DEF_TYPE *DEV_CONFIG_BINDING_PTR;

DEV_CONFIG_ITERATOR_PTR devConfigCreateIterator(void);

/*
 *   FUNCTION:  devConfigDestroyIterator(iter)
 *
 *   ABSTRACT:  Destroys a configuration iterator.
 *
 *   DESCRIPTION:
 *
 *     Destroys and deallocates a runtime configuration iterator.
 *
 *   RETURN VALUES:
 *
 *   NOTES:
 *
 *   EXAMPLES:
 *
 */

void devConfigDestroyIterator(DEV_CONFIG_ITERATOR_PTR iter);

/*
 *   FUNCTION:  devConfigIterNext(iter)
 *
 *   ABSTRACT:  Iterates over the next binding in the environment.
 *
 *   DESCRIPTION:
 *
 *     Returns a pointer to the next definition in the runtime 
 *     configuration environment.
 *
 *   RETURN VALUES:
 *
 *     DEV_CONFIG_BINDING_PTR containing a binding in the environment.
 *     The returned structure should not by modified or freed.
 *
 *     Returns NULL if there are no further entries.
 *
 *   NOTES:
 *
 *   EXAMPLES:
 *
 */

DEV_CONFIG_BINDING_PTR devConfigIterNext(DEV_CONFIG_ITERATOR_PTR iter);

/*
 *   FUNCTION:  devConfigIterPrev(iter)
 *
 *   ABSTRACT:  Iterates over the previous binding in the environment.
 *
 *   DESCRIPTION:
 *
 *     Returns a pointer to the previous definition in the runtime 
 *     configuration environment.
 *
 *   RETURN VALUES:
 *
 *     DEV_CONFIG_BINDING_PTR containing a binding in the environment.
 *     The returned structure should not by modified or freed.
 *
 *   NOTES:
 *
 *   EXAMPLES:
 *
 */

DEV_CONFIG_BINDING_PTR devConfigIterPrev(DEV_CONFIG_ITERATOR_PTR iter);

#ifdef __cplusplus
}
#endif

#endif /* _DEVCONFIG_H */
