/*
 * FILE:  devConfig.c
 * DESC:  
 *
 * $Id: devConfig.c,v 1.2 1996/02/11 04:15:00 rich Exp $
 *
 * Modification history: $Log: devConfig.c,v $
 * Revision 1.2  1996/02/11  04:15:00  rich
 * Fixed line numbers in warnings.
 *
 * Revision 1.1  1996/02/10  23:38:42  rich
 * Added RWI's paramter parsing routines.
 *
 * Revision 1.2  1996/01/23  07:27:25  jal
 * Added config iterators, IsDefined operator, AddDef, AddValue, etc.
 * Changed to use realloc() to create a variable sized array for
 * definitions.
 *
 * Revision 1.1  1996/01/14  04:31:19  jal
 * Initial checkin.  Still testing.
 *
 * Revision 1.1.1.1  1996/01/13  14:20:48  jal
 * First checkin of RAI-2.
 *
 *
 */

#include "tca/libc.h"
#include "devConfig.h"

/*
 *   Static data structures
 */

#define CONF_MAX_LINE 1000
#define CONF_DEF_BLOCK 1000
#define CONF_COMMENT_CHAR '#'
#define CONF_NAME_VAL_SEPS "="
#define CONF_PATH_SEPS ":"

static int confNumDefs = 0;
static int confMaxDefs = 0;
static DEV_CONFIG_DEF_PTR *confDefs = NULL;

static char *searchPath = NULL;
static char *searchPathScratch = NULL;

/**********************************************************/

static DEV_CONFIG_DEF_PTR confNewDef(const char *name,
				     const char *value)
{
  DEV_CONFIG_DEF_PTR newNode;
  char *newName, *newValue;
  
  newNode = (DEV_CONFIG_DEF_PTR) malloc(sizeof(DEV_CONFIG_DEF_TYPE));
  
  newName = (char*) malloc(strlen(name) + 1);
  strcpy(newName, name);
  newNode->name = newName;
  
  if (value != NULL) {
    newValue = (char*) malloc(strlen(value) + 1);
    strcpy(newValue, value);
    newNode->value = newValue;
  }
  else {
    newNode->value = NULL;
  }
  
  return newNode;
}

static BOOLEAN defNameCmp(DEV_CONFIG_DEF_PTR *a, DEV_CONFIG_DEF_PTR *b)
{
  return strcmp((*a)->name, (*b)->name);
}

static BOOLEAN defNameFindCmp(const char *name, DEV_CONFIG_DEF_PTR *b)
{
  return strcmp(name, (*b)->name);
}

static void configSortDefs(void)
{
  qsort(confDefs, confNumDefs, sizeof(DEV_CONFIG_DEF_PTR), (void*) defNameCmp);
}

static DEV_CONFIG_DEF_PTR configDefFind(const char *name)
{
  void *ret;
  
  if (confDefs == NULL) return NULL;
  ret = bsearch(name, confDefs, confNumDefs, sizeof(DEV_CONFIG_DEF_PTR),
		(void*) defNameFindCmp);
  
  if (ret == NULL) return NULL;
  else return *((DEV_CONFIG_DEF_PTR*) ret);
}

/* returns: boolean indicating a sort of the definition list is required */
static BOOLEAN confAddDef_NoSort(const char *name,
				 const char *value)
{
  DEV_CONFIG_DEF_PTR newDef;
  DEV_CONFIG_DEF_PTR found;
  BOOLEAN sortReq;
  
  if (confNumDefs == confMaxDefs) {
    /* need to expand the definition table */
    confMaxDefs += CONF_DEF_BLOCK;
    confDefs = (DEV_CONFIG_DEF_PTR*) realloc(confDefs,
					     (confMaxDefs *
					      sizeof(DEV_CONFIG_DEF_PTR)));
  }
  
  if ((found = configDefFind(name)) == NULL) {
    /* definition not present; add at end and trigger a sort */
    newDef = confNewDef(name, value);
    confDefs[confNumDefs++] = newDef;
    sortReq = 1;
  }
  else {
    /* definition present; replace it, no sort required */
    if (found->value != NULL) free((void*) found->value);
    if (value != NULL) {
      char *newValue;
      newValue = (char*) malloc(strlen(value) + 1);
      strcpy(newValue, value);
      found->value = newValue;
    }
    else {
      found->value = NULL;
    }
    sortReq = 0;
  }
  
  return sortReq;
}

static void addLine(char *line, const char *filename, int lineno)
{
  char *name, *value;
  
  if (*line == CONF_COMMENT_CHAR) return; /* skip comments */
  if (*line == 0) return;	/* skip blank lines */
  
  /* eliminate trailing <cr> */
  {
    int lineLen;
    lineLen = strlen(line);
    if (isspace(line[lineLen-1])) {
      line[lineLen-1] = 0;
    }
  }
  
  /* parse definition */
  name = line;
  value = strpbrk(line, CONF_NAME_VAL_SEPS);
  if (value != NULL) {
    *value++ = 0;
    if (*value == 0) value = NULL;
  }
  
  if (name == NULL) return;	/* skip blank lines */
  
  if (value == NULL) {
    /* check for blank line before writing error message */
    while (*name && isspace(*name)) name++;
    if (*name != 0) {
      /* line wasn't blank */
      fprintf(stderr, "Warning: Parse error in config file %s, line %d.\n",
	      filename, lineno);
    }
    return;
  }
  
  confAddDef_NoSort(name, value);
}

static void addDefsFromFile(const char *filename, FILE *f)
{
  char line[CONF_MAX_LINE];
  int lineno;
  
  for (lineno = 1; fgets(line, CONF_MAX_LINE, f) != NULL; lineno++) {
    addLine(line,filename,lineno);
  }
  
  configSortDefs();
}

BOOLEAN devConfigLoadFile(const char *file)
{
  FILE *fp;
  const char *mode = "r";
  BOOLEAN fileFound = 0;
  
  if ((fp = fopen(file, mode)) != NULL) {
    fileFound = 1;
    addDefsFromFile(file, fp);
    fclose(fp);
  }
  else {
    /* use the search path */
    char tmppath[255];
    char *prefix;
    
    if (searchPath != NULL) {
      strcpy(searchPathScratch, searchPath);
      
      for (prefix = strtok(searchPathScratch, CONF_PATH_SEPS);
	   prefix != NULL;
	   prefix = strtok(NULL, CONF_PATH_SEPS)) {
	/* build the path */
	strncpy(tmppath, prefix, sizeof(tmppath)-1);
	strcat(tmppath, "/");
	strcat(tmppath, file);
	
	fprintf(stderr, "devConfigLoadFile(): Trying %s...\n",
		tmppath);
	
	if ((fp = fopen(tmppath, mode)) != NULL) {
	  fileFound = 1;
	  fprintf(stderr, "   ... Opened.\n");
	  addDefsFromFile(tmppath, fp);
	  fclose(fp);
	  break;
	}
      }
    }
  }
  
  return fileFound;
}

void devConfigSetSearchPath(const char *path)
{
  int pathLen;
  
  if (searchPath != NULL) free(searchPath);
  if (searchPathScratch != NULL) free(searchPathScratch);
  
  if (path != NULL) {
    pathLen = strlen(path);
    
    searchPath = (char*) malloc(pathLen + 1);
    strcpy(searchPath, path);
    
    searchPathScratch = (char*) malloc(pathLen + 1);
  }
  else {
    searchPath = NULL;
    searchPathScratch = NULL;
  }
}

const char *devConfigFindString(const char *name)
{
  DEV_CONFIG_DEF_PTR found;
  
  found = configDefFind(name);
  if (found == NULL) return NULL;
  
  return found->value;
}

BOOLEAN devConfigGet(const char *name, DEV_CONFIG_CONV_HND converter,
		     void *value, void *data)
{
  const char *str;
  
  str = devConfigFindString(name);
  if (str == NULL) return 0;
  
  return (*converter)(str, value, data);
}

BOOLEAN devConfigConvertInt(const char *str, void *value, void *convdata)
{
  *((int*) value) = (int) strtod(str, NULL);
  return 1;
}

BOOLEAN devConfigConvertShort(const char *str, void *value, void *convdata)
{
  *((short*) value) = (short) strtod(str, NULL);
  return 1;
}

BOOLEAN devConfigConvertLong(const char *str, void *value, void *convdata)
{
  *((long*) value) = (long) strtod(str, NULL);
  return 1;
}

BOOLEAN devConfigConvertFloat(const char *str, void *value, void *convdata)
{
  *((float*) value) = (float) strtod(str, NULL);
  return 1;
}

BOOLEAN devConfigConvertDouble(const char *str, void *value, void *convdata)
{
  *((double*) value) = (double) strtod(str, NULL);
  return 1;
}

BOOLEAN devConfigConvertString(const char *str, void *value, void *convdata)
{
  value = (void*) str;
  
  if (value != NULL) return 1;
  else return 0;
}

BOOLEAN devGetParamInt(char *paramName, int *value)
{
  return devConfigGet(paramName, devConfigConvertInt, value, NULL);
}

BOOLEAN devGetParamShort(char *paramName, short *value)
{
  return devConfigGet(paramName, devConfigConvertShort, value, NULL);
}

BOOLEAN devGetParamLong(char *paramName, long *value)
{
  return devConfigGet(paramName, devConfigConvertLong, value, NULL);
}

BOOLEAN devGetParamFloat(char *paramName, float *value)
{
  return devConfigGet(paramName, devConfigConvertFloat, value, NULL);
}

BOOLEAN devGetParamDouble(char *paramName, double *value)
{
  return devConfigGet(paramName, devConfigConvertDouble, value, NULL);
}

BOOLEAN devGetParamString(char *paramName, char *value)
{
  return devConfigGet(paramName, devConfigConvertString, value, NULL);
}


void devConfigSetValue(const char *name, const char *value)
{
  if (confAddDef_NoSort(name, value))
    configSortDefs();
}

void devSetParamInt(const char *name, const int value)
{
  char paramStr[80];
  sprintf(paramStr,"%d",value);
  devConfigSetValue(name,paramStr);
}

void devSetParamShort(const char *name, short value)
{
  char paramStr[80];
  sprintf(paramStr,"%d",value);
  devConfigSetValue(name,paramStr);
}

void devSetParamLong(const char *name, long value)
{
  char paramStr[80];
  sprintf(paramStr,"%ld",value);
  devConfigSetValue(name,paramStr);
}

void devSetParamFloat(const char *name, float value)
{
  char paramStr[80];
  sprintf(paramStr,"%f",value);
  devConfigSetValue(name,paramStr);
}

void devSetParamDouble(const char *name, double value)
{
  char paramStr[80];
  sprintf(paramStr,"%f",value);
  devConfigSetValue(name,paramStr);
}

void devSetParamString(const char *name, const char *value)
{
  devConfigSetValue(name,value);
}


void devConfigSetDef(const char *def)
{
  char buf[CONF_MAX_LINE];
  char *name, *value;
  
  if (def == NULL || !*def) return;
  
  strncpy(buf, def, CONF_MAX_LINE);
  
  name = buf;
  value = strpbrk(buf, CONF_NAME_VAL_SEPS);
  if (value != NULL) {
    *value++ = 0;
    if (*value == 0) value = NULL;
  }
  
  if (confAddDef_NoSort(name, value))
    configSortDefs();
}

typedef struct DEV_CONFIG_ITERATOR_TYPE {
  int loc;
} DEV_CONFIG_ITERATOR_TYPE;

DEV_CONFIG_ITERATOR_PTR devConfigCreateIterator(void)
{
  DEV_CONFIG_ITERATOR_PTR newIter;
  
  newIter = (DEV_CONFIG_ITERATOR_PTR)
    malloc(sizeof(DEV_CONFIG_ITERATOR_TYPE));
  
  newIter->loc = -1;
  
  return newIter;
}

void devConfigDestroyIterator(DEV_CONFIG_ITERATOR_PTR iter)
{
  if (iter != NULL) free(iter);
}

DEV_CONFIG_BINDING_PTR devConfigIterNext(DEV_CONFIG_ITERATOR_PTR iter)
{
  if (iter == NULL) return NULL;
  
  ++iter->loc;
  
  if (iter->loc < 0 ||
      iter->loc >= confNumDefs) return NULL;
  else {
    return confDefs[iter->loc];
  }
}

DEV_CONFIG_BINDING_PTR devConfigIterPrev(DEV_CONFIG_ITERATOR_PTR iter)
{
  if (iter == NULL) return NULL;
  
  --iter->loc;
  
  if (iter->loc < 0 ||
      iter->loc >= confNumDefs) return NULL;
  else {
    return confDefs[iter->loc];
  }
}

BOOLEAN devConfigIsDefined(const char *name)
{
  return (configDefFind(name) != NULL);
}


/*
 * paramParseArgs : Given argc and argv, look for -D<param name>=<value> 
 * in the parameter strings.  This is the same format the compiler accepts for
 * defining parameters.
 */

void devParseParamArgs(int argc, char **argv)
{
  int i;
  char paramStr[80];
  
  for(i=1; i<argc; i++) {
    if (strlen(argv[i]) >1) {
      if ((argv[i][0] == '-') && (argv[i][1] == 'D')) {
	strcpy(paramStr, &(argv[i][2]));
	addLine(paramStr,"Command Line Args",0);
      }
    }
  }
  configSortDefs();
}
