/*-----------------------------------------------------------------------------

File:		wiretaps.c

Programmer:	Reid Simmons

Description:	TCA module that illustrates use of wiretaps.

		Implement a simple dead-reckoning module using wiretaps
		of the move and turn commands.

		Message handlers for the following (wiretap) messages
		are implemented:
			DEAD_RECK_TURN_CONSTRAINT
			DEAD_RECK_MOVE_GOAL
			DEAD_RECK_POSITION_QUERY

		This is a stand-alone module.  
		The command line options are: wiretaps <x> <y> <angle>,
		where (x,y) is the initial position of the robot, and
		"angle" is the initial orientation, in degrees.
		The defaults used are x=150, y=100, angle=0.0 (this is where
		the simulator starts the robot)

		To run this example:
		  1. central 3 -lmdsi
		  2. simulator/simulator sim.param
		  3. wiretaps
		  4. taskTrees

-----------------------------------------------------------------------------*/

#include "tca/libc.h"
#include "tca.h"
#include "simMessages.h"
#include "wiretaps.h"

static LOCATION_TYPE locationGlobal;

static void printLocation (LOCATION_PTR locationPtr)
{
  fprintf(stderr, "  Robot position is (%3.1f, %3.1f), orientation is %3.1f\n",
	  locationPtr->position.x, locationPtr->position.y,
	  locationPtr->orientation);
}

/*-------------------------- deadReckTurnHandler ---------------------------

	Handler for the DEAD_RECK_TURN_CONSTRAINT message.

   Increment the global position by the commanded turn angle. 

----------------------------------------------------------------------------*/

static void deadReckTurnHandler(TCA_REF_PTR ref, void *data)
{
  DEAD_RECK_TURN_PTR anglePtr;

  anglePtr = (DEAD_RECK_TURN_PTR)data;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));

  locationGlobal.orientation += *anglePtr;
  
  if (locationGlobal.orientation >= 360.0) {
  	locationGlobal.orientation -= 360.0;
  } else if (locationGlobal.orientation < 0.0) {
  	locationGlobal.orientation +=360.0;
  }
  
  printLocation(&locationGlobal);

  tcaFreeData(tcaReferenceName(ref), data);
}


/*-------------------------- deadReckMoveHandler ---------------------------

	Handler for the DEAD_RECK_MOVE_GOAL message.

   Find the distance actually travelled during the last move, and
   update the global position accordingly.

   The data argument is ignored.

----------------------------------------------------------------------------*/

/* ARGSUSED */
static void deadReckMoveHandler(TCA_REF_PTR ref, void *dummy)
{
  DISTANCE_MOVED_REPLY_TYPE distanceMoved;
  BOUNDED_MOVE_COMMAND_PTR commandedMovePtr;
  GUARDED_MOVE_COMMAND_PTR guardedMovePtr;
  TCA_REF_PTR moveRef;
  RADIANS theta;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));

  /* Find actual distance moved */
  tcaQuery(DISTANCE_MOVED_QUERY, NULL, (void *)&distanceMoved);
  
  fprintf(stderr, "  Moved %3.1f cms", distanceMoved); fflush(stderr);

  /* Find commanded distance */ 
  moveRef = tcaFindTappedReference(ref);
  
  if (!strcmp(tcaReferenceName(moveRef), BOUNDED_MOVE_COMMAND)) {
    commandedMovePtr = (BOUNDED_MOVE_COMMAND_PTR)tcaReferenceData(moveRef);
    fprintf(stderr, " out of %3.1f cms commanded move\n", *commandedMovePtr);
    tcaFreeData(tcaReferenceName(moveRef), (void *)commandedMovePtr);
  } else if (!strcmp(tcaReferenceName(moveRef), GUARDED_MOVE_COMMAND)) {
    guardedMovePtr = (GUARDED_MOVE_COMMAND_PTR)tcaReferenceData(moveRef);
    fprintf(stderr,
	    " out of %3.1f cms commanded move\n", guardedMovePtr->distance);
    tcaFreeData(tcaReferenceName(moveRef), (char *)guardedMovePtr);
  } else {
    fprintf(stderr, "\n");
  }

  theta = SIM_DEG_TO_RAD(locationGlobal.orientation);
  locationGlobal.position.x += distanceMoved * cos(theta);
  locationGlobal.position.y += distanceMoved * sin(theta);

  printLocation(&locationGlobal);

  tcaReferenceRelease(moveRef);
  tcaSuccess(ref);
}


/*-------------------------- deadReckLocationHandler --------------------------

	Handler for the DEAD_RECK_LOCATION_QUERY message.

   Return the dead-reckoned position and orientation.

   The data argument is NULL, and hence ignored.

----------------------------------------------------------------------------*/

/* ARGSUSED */
static void deadReckLocationHandler(TCA_REF_PTR ref, void *dummy )
{
  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));

  tcaReply(ref, &locationGlobal);
}


/*----------------------- main ------------------------
  WIRETAPS module: Implement a dead-reckoning module using wiretaps.

  The command line options are: wiretaps <x> <y> <angle>,
  where (x,y) is the initial position of the robot, and "angle" is the
  initial orientation, in degrees. The defaults used are x=y=angle=0.0;

----------------------------------------------------------------------------*/
void main (int argc, char **argv)
{
  LOCATION_TYPE initialLocation;

  if (argc > 1 && !strcmp(argv[1],"-h")) {
    fprintf(stderr, "Usage: %s <x> <y> <angle>\n", argv[0]);
    exit(0);
  }

  locationGlobal.position.x =  ((argc > 1) ? (CMS)atof(argv[1]) : X_DEFAULT);
  locationGlobal.position.y =  ((argc > 2) ? (CMS)atof(argv[2]) : Y_DEFAULT);
  locationGlobal.orientation = ((argc > 3) ? (CMS)atof(argv[3]) :
				ANGLE_DEFAULT);

  tcaConnectModule("WIRETAPS", tcaServerMachine());

  tcaRegisterInform(DEAD_RECK_TURN_INFORM, DEAD_RECK_TURN_FORM,
		    deadReckTurnHandler);
  tcaTapMessage(AfterAchieved, TURN_COMMAND, DEAD_RECK_TURN_INFORM);

  tcaRegisterGoal(DEAD_RECK_MOVE_GOAL, DEAD_RECK_MOVE_FORM,
		  deadReckMoveHandler);
  tcaTapMessage(WhenAchieved, BOUNDED_MOVE_COMMAND, DEAD_RECK_MOVE_GOAL);
  tcaTapMessage(WhenAchieved, UNBOUNDED_MOVE_COMMAND, DEAD_RECK_MOVE_GOAL);
  tcaTapMessage(WhenAchieved, GUARDED_MOVE_COMMAND, DEAD_RECK_MOVE_GOAL);

  tcaRegisterNamedFormatter(POSITION_NAME, POSITION_FORM);
  tcaRegisterNamedFormatter(ORIENTATION_NAME, ORIENTATION_FORM);
  tcaRegisterQuery(DEAD_RECK_LOCATION_QUERY, 
		   DEAD_RECK_LOCATION_QUERY_FORM,
		   DEAD_RECK_LOCATION_REPLY_FORM,
		   deadReckLocationHandler);

  tcaWaitUntilReady();

  tcaQuery(DEAD_RECK_LOCATION_QUERY, NULL, (void *)&initialLocation);
  printLocation(&initialLocation);

  tcaModuleListen();
}
