/*-----------------------------------------------------------------------------

File:		taskTrees.c

Programmer:	Reid Simmons

Description:	TCA module that illustrates use of goals and task trees.
                1. Define a "move_turn" goal that moves a given distance,
		   then turns a given number of degrees.
		2. Define a "shape" goal that moves the robot in a given
		   shape, defined by the number of sides and the length of
		   each side.

		Message handlers for the following message are implemented:

		  MOVE_TURN_GOAL
		  SHAPE_GOAL

		This is a stand-alone module.  It takes the following
		command-line options: taskTrees <numSides> <length> <range>,
		where "numSides" is an integer greater than 2, and "length"
		is a positive real.  The "stopRange" argument is passed
		to the guarded move command, and indicates the range at 
		which to stop before objects.
		If the command line options are not given, the defaults
		(numSides=3, length=100.0 cms, stopRange = 25.0 cms) are used.

		To run this example:
		  1. central 2 -lmdsi
		  2. simulator/simulator sim.param
		  3. taskTrees

-----------------------------------------------------------------------------*/

#include "tca/libc.h"
#include "tca.h"
#include "simMessages.h"
#include "taskTrees.h"

static CMS stopRangeGlobal = RANGE_DEFAULT;


/*-------------------------- moveTurnHandler ---------------------------

	Handler for the MOVE_TURN_GOAL message.

----------------------------------------------------------------------------*/

static void moveTurnHandler(TCA_REF_PTR ref, void *data)
{
  MOVE_TURN_PTR moveTurnData;
  GUARDED_MOVE_COMMAND_TYPE guardedMove;

  moveTurnData = (MOVE_TURN_PTR)data;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));
  fprintf(stderr, "  Planning ... please Wait\n");
  sleep(3); /* Simulate planning time */
  guardedMove.distance = moveTurnData->distance;
  guardedMove.stopRange = stopRangeGlobal;
  
  tcaExecuteCommand(GUARDED_MOVE_COMMAND, (void *)&guardedMove);
  tcaExecuteCommand(TURN_COMMAND, (void *)&(moveTurnData->angle));

  tcaSuccess(ref);
  
  tcaFreeData(tcaReferenceName(ref), data);
}


/*----------------------- shapeGoalHandler ------------------------

	Handler for the SHAPE_GOAL message.

	The angle to turn for the polygonal shape is determined as follows:
   A polygon of N sides has 180*(N-2) total interior angles, and thus
   180-360/N for each interior angle.  The exterior angle (which is the
   angle needed) is 180-interior angle, therefore, the angle to turn
   is simply 360/N.

----------------------------------------------------------------------------*/

static void shapeGoalHandler(TCA_REF_PTR ref, void *data)
{
  SHAPE_PTR shapeData;
  MOVE_TURN_TYPE moveTurn;
  int i;

  shapeData = (SHAPE_PTR)data;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));
  if ((shapeData->numSides < 3) || (shapeData->length <= 0.0)) {
    fprintf(stderr, "  Failure: Malformed Shape (%d %f)\n", 
	    shapeData->numSides, shapeData->length);
    tcaFailure(ref, "Malformed Shape", NULL);
  } else {
    moveTurn.distance = shapeData->length;
    moveTurn.angle = (int)(0.5 + 360/(shapeData->numSides));
    fprintf(stderr,
	    "  Each side is %3.2f cms; With interior angles of %3.2f\n",
	    moveTurn.distance, 180-moveTurn.angle);
    
    for (i=0; i<shapeData->numSides; i++) {
      tcaExpandGoal(MOVE_TURN_GOAL, (void *)&moveTurn);
    }
    tcaSuccess(ref);
  }
  tcaFreeData(tcaReferenceName(ref), data);
}


/*----------------------- main ------------------------
  TASK_TREES module: create and execute a task tree for moving the robot
  in a given shape.

  Command line options are: taskTrees <numSides> <length> <stopRange>,
  where "numSides" is an integer greater than 2, "length" is a positive real,
  and "stopRange" is the nominal stopping distance for guarded move.
  If the command line options are not given, the defaults (numSides=3, 
  length=100.0, stopRange=25.0) are used.  Units are in cms.
  The command line option "-h" provides a help message.
----------------------------------------------------------------------------*/
void main (int argc, char **argv)
{
  SHAPE_TYPE shape;

  if (argc >= 2 && !strcmp(argv[1], "-h")) {
    fprintf(stderr, "Usage: %s <numSides> <length> <stopRange>\n", argv[0]);
    exit(0);
  }

  if (argc >= 4) stopRangeGlobal = (CMS)atof(argv[3]);
  else stopRangeGlobal = RANGE_DEFAULT;

  if (argc >= 3) shape.length = (CMS)atof(argv[2]);
  else shape.length = LENGTH_DEFAULT;

  if (argc >= 2) shape.numSides = atoi(argv[1]);
  else shape.numSides = NUMSIDES_DEFAULT;

  tcaConnectModule("TASK TREES", tcaServerMachine());

  tcaRegisterGoal(MOVE_TURN_GOAL, MOVE_TURN_FORM,
		  moveTurnHandler);

  tcaRegisterGoal(SHAPE_GOAL, SHAPE_FORM, shapeGoalHandler);

  tcaWaitUntilReady();

  tcaExpandGoal(SHAPE_GOAL, (void *)&shape);

  tcaModuleListen();
}
