/*****************************************************************************
 * PROJECT: TCA Tutorial
 *
 * (c) Copyright 1994 Reid Simmons & Richard Goodwin. All rights reserved.
 *
 * FILE: simInterface.c
 *
 * ABSTRACT:
 *
 * Interface to simulator for use with TCA tutorial
 *
 * The functions which control movement are currently implemented as
 * blocking functions.  Any movement command will not return until it
 * has completed.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tutorial/simulator/simInterface.c,v $
 * $Revision: 1.3 $
 * $Date: 1995/04/07 05:10:08 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: simInterface.c,v $
 * Revision 1.3  1995/04/07  05:10:08  rich
 * Fixed GNUmakefiles to find the release directory.
 * Moved all system includes into libc.h
 * Moved global variables into the c files and got rid of #define DECLARE_...
 *
 * Revision 1.2  1995/01/25  00:07:27  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Formatting changes.
 *
 * Revision 1.1  1994/05/24  21:24:09  rich
 * Checking in Reid's tutorial code, with some fixes to the simulator.
 * Mostly problems with X11.
 *
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "common.h"
#include "tca.h"
#include "tca/ezx.h"
#include "files.h"
#include "action.h"
#include "sensors.h"
#include "updates.h"
#include "simInterface.h"

extern void handleSimEvents (void); /* Defined in sim.c */

/* local state variables */
static double currentTransVelocity = DEF_TRANS_VELOCITY;
static double currentRotVelocity = DEF_ROT_VELOCITY;
static int motionStatus = FALSE;
static CMS simLastDistance = 0.0;

/*---------------------------- simBoundedMove ---------------------------

This function tells the robot to move a given distance.  The velocity
will be the current velocity set by the simSetVelocity() function or
the default value set on startup.  The direction of movement
is forward only, in a straight line.  This is a bounded move only
so no check is done to see whether obstacles must be avoided.

The calling syntax is:

	simBoundedMove( distance );

where the parameters are as follows:

	distance  - the distance in centimeters for the robot to travel

Returns: nothing

------------------------------------------------------------------------*/

void simBoundedMove (CMS distance)
{
    motionStatus = TRUE;
    simLastDistance = ABS(distance);

    translation_velocity = currentTransVelocity;
    translation_distance = distance;
    resetTime();
    while ( translation_velocity != 0 ) {
      handleSimEvents();
    }
    simLastDistance = ABS(distance - translation_distance);
    simStop();
}


/*-------------------------- simUnboundedMove ---------------------------

This function tells the robot to move until a stop command is issued.
The velocity will be the current velocity set by the simSetVelocity()
function or the default value set on startup.  The direction to
move will be forward in the direction the robot is facing.  No checks
are done to see whether obstacles must be avoided.

The calling syntax is:

	simUnboundedMove();

Returns: nothing

Note: For simplicity, an unbounded move is really a bounded move with
      a very large distance.

------------------------------------------------------------------------*/

void simUnboundedMove()

{
    simBoundedMove( 1000000.0 );
}


/*---------------------------- simGuardedMove ---------------------------

This function tells the robot to move a given distance.  The velocity
will be the current velocity set by the simSetVelocity() function or
the default value set on startup.  This is a guarded move so
the sonar is checked to see if obstacles are within a given radius of
the robot.  The radius (in centimeters) is passed in as a parameter.

The calling syntax is:

	rtn = heroGuardedMove( distance, stopRange );

where the parameters are as follows:

	distance  - the distance in centimeters for the robot to travel
	stopRange - the minimum allowable distance from an obstacle
		    before the robot stops moving, the accuracy for
		    the stop range is about 5 centimeters

Returns: the distance travelled

Note: There is no unbounded guarded move.  This would be trivial to add
      if it is needed.

------------------------------------------------------------------------*/

CMS simGuardedMove (CMS distance, CMS stopRange)
{
    motionStatus = TRUE;
    simLastDistance = ABS(distance);

    translation_velocity = currentTransVelocity;
    translation_distance = distance;
    sensors();
    stopRange += robot_r;
    resetTime();
    while ( translation_velocity != 0 && (sensor[0] > stopRange)
    		&& (sensor[23] > stopRange) && (sensor[1] > stopRange)) {
      handleSimEvents();
      /* handleSimEvents updates sonar sensors */
    }
    simLastDistance = ABS(distance - translation_distance);
    simStop();
    return simLastDistance;
}


/*---------------------------- simTurn ----------------------------------

This function tells the robot to rotate in place.  The rotation is
given in degrees, relative to the current orientation of the robot.
Positive means clockswise, and negative means counterclockwise

The calling syntax is:

	simTurn( turn );

where the parameters are as follows:

	turn - the angle to rotate, positive means clockswise,
		  negative means counterclockwise

Returns: nothing

------------------------------------------------------------------------*/

void simTurn (DEGREES turn)
{
    motionStatus = TRUE;

    rotation_velocity = DEG_TO_RAD(currentRotVelocity);
    rotation_distance = DEG_TO_RAD(turn);
    if (turn < 0) {
      rotation_velocity = -rotation_velocity;
    }
    resetTime();
    while ( rotation_velocity != 0 ) {
      handleSimEvents();
    }
    simStop();
}



/*---------------------------- simStop ---------------------------------

This function tells the robot to stop all movement.

The calling syntax is:

	simStop();

Returns: nothing

------------------------------------------------------------------------*/

void simStop(void)

{
    motionStatus = FALSE;
    translation_velocity = rotation_velocity = 0.0;
}


/*---------------------------- simSetVelocity ---------------------------

This function sets the velocity at which the robot will move when it
receives a movement command.  The velocity is given in centimeters per second.

The calling syntax is:

	simSetVelocity(transVelocity, rotVelocity);

where the parameters are as follows:

	transVelocity - velocity in centimeters per second
	rotVelocity - velocity in radians per second

Returns: nothing

------------------------------------------------------------------------*/

void simSetVelocity(double transVelocity, double rotVelocity)
{
  currentTransVelocity = transVelocity;
  currentTransVelocity = rotVelocity;
}


/*---------------------------- simVelocity ---------------------------

This function will return the current velocity used for movement.

The calling syntax is:

	simVelocity(&transVelocity, &rotVelocity);

Where: transVelocity - is set to the current translational velocity (cm/sec)
       rotVelocity - is set to the current rotational velocity (rad/sec)

------------------------------------------------------------------------*/

void simVelocity(double *transVelocity, double *rotVelocity)
{
  *transVelocity = currentTransVelocity;
  *rotVelocity = currentTransVelocity;
}


/*---------------------------- simTalk ---------------------------

This function makes the robot "talk".

The calling syntax is:

	simTalk( char *message );

where the parameters are as follows:

	message - a string to "speak"

Returns: nothing

------------------------------------------------------------------------*/

void simTalk(char *message)
{
    printf("%s\n", message);
}


/*---------------------------- sonarScan ---------------------------

This function returns a set of 24 sonar readings, with sonar 0 facing forward,
and numbers increasing clockwise.  The sonar readings are in centimeters and
are calculated from the center of the robot.

The calling syntax is:

	sonarScan( sonarData );

where the parameters are as follows:

	sonarData - an array of 24 sonar readings (in centimeters)

Returns: an array containing the distance to any obstacle detected
	 in the sonarData parameter.  A value greater or equal to
	 MAX_SONAR_RANGE indicates no obstacle was detected

------------------------------------------------------------------------*/

void sonarScan(CMS sonarData[24])
{
  int i;
  
  sensors();
  for (i=0; i<24; i++) sonarData[i] = sensor[i];
}


/*---------------------------- simInMotion  ---------------------------

  This function will return whether the robot is currently moving.
  
  The calling syntax is:
  
  inMotion = simInMotion();
  
  Returns: if the robot is currently moving (TRUE/1 == moving;
  FALSE/0 == stopped)
  
  ---------------------------------------------------------------------------*/

int simInMotion (void)
{
  return motionStatus;
}


/*---------------------------- simLastMoveDistance  ---------------------------
  
  This function will return the distance of the last move made by the
  robot (in centimeters).
  
  The calling syntax is:
  
  distance = simLastMoveDistance ();
  
  Returns: the distance of the last move
  
  ---------------------------------------------------------------------------*/

CMS simLastMoveDistance(void)
{
  return simLastDistance;
}

