/*****************************************************************************
 * PROJECT: TCA Tutorial
 *
 * (c) Copyright 1994 Richard Goodwin & Reid Simmons. All rights reserved.
 *
 * FILE: sim.c
 *
 * ABSTRACT:
 *
 * Simulator main program.
 * Includes all the TCA message handlers and registration code
 * The following messages and queries are supported:
 * 
 * 		  BOUNDED_MOVE_COMMAND
 * 		  UNBOUNDED_MOVE_COMMAND
 * 		  GUARDED_MOVE_COMMAND
 * 		  TURN_COMMAND
 * 		  STOP_COMMAND
 * 		  SET_VELOCITY_COMMAND
 * 		  TALK_COMMAND
 * 		  SONAR_PING_QUERY
 * 		  SONAR_SCAN_QUERY
 * 		  SONAR_SCAN_LIST_QUERY
 * 		  SONAR_SCAN_VAR_QUERY
 * 		  VELOCITY_QUERY
 * 		  DISTANCE_MOVED_QUERY
 * 
 * 		For more information on the behavior and function of
 * 		each of these messages, see the corresponding functions
 * 		in simInterface.c (see also simMessages.h).
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tutorial/simulator/sim.c,v $
 * $Revision: 1.12 $
 * $Date: 1996/02/14 22:16:09 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: sim.c,v $
 * Revision 1.12  1996/02/14  22:16:09  rich
 * Fixed static problems.
 *
 * Revision 1.11  1995/07/26  20:43:47  rich
 * Return had wronge value.
 *
 * Revision 1.10  1995/07/19  14:31:18  rich
 * Use new tca_connect.
 *
 * Revision 1.9  1995/07/08  18:26:22  rich
 * Change all /afs/cs to /afs/cs.cmu.edu to get ride of conflict problems.
 *
 * Revision 1.8  1995/06/14  17:51:53  rich
 * Added support for 3 types of event loop.
 *
 * Revision 1.7  1995/04/07  05:10:03  rich
 * Fixed GNUmakefiles to find the release directory.
 * Moved all system includes into libc.h
 * Moved global variables into the c files and got rid of #define DECLARE_...
 *
 * Revision 1.6  1995/03/16  18:06:21  rich
 * Merged in changes to the 7.9 branch.
 * Changed the VERSION_ to TCA_VERSION_
 *
 * Revision 1.5  1995/01/25  00:07:25  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Formatting changes.
 *
 * Revision 1.4  1994/11/02  21:41:31  rich
 * Now works for linux machines (i486).
 * Uses tca/libc.h.
 *
 * Revision 1.3  1994/05/31  03:28:46  rich
 * Modified the examples to use the new msg/hnd macros.
 *
 * Revision 1.2  1994/05/25  23:55:40  rich
 * *** empty log message ***
 *
 * Revision 1.1  1994/05/24  21:24:01  rich
 * Checking in Reid's tutorial code, with some fixes to the simulator.
 * Mostly problems with X11.
 *
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "tca/ezx.h"
#include "tca/devUtils.h"
#include "tca/tcaDev.h"
#include "tca/stdinDev.h"
#include "tca/x11Dev.h"
#include "tca/x11Utils.h"

#include "sim-xwindows.h"

#include "updates.h"
#include "draw_env.h"
#include "environment.h"
#include "sensors.h"
#include "files.h"
#include "action.h"
#include "tca.h"
#include "simMessages.h"
#include "simInterface.h"
#include "sim.h"

#define HOME_GROWN_LOOP 1
#define X11_LOOP 2
#define DEVUTILS_LOOP 3
/*#define MAIN_LOOP DEVUTILS_LOOP*/

/*****************************************************************************
 * Global variables
 *****************************************************************************/

int useResources, registerGuardedMove;

void handleSimEvents(void)
{
  fd_set listenList;
  int xSocket = ConnectionNumber(theDisplay);
  int tcaSocket = tcaGetServer();
  XtInputMask mask;
  struct timeval time;
  int stat;
  
  FD_ZERO(&listenList);
  FD_SET(xSocket, &listenList);
  FD_SET(tcaSocket, &listenList);
  time.tv_sec = time.tv_usec = 0;
  do {
    stat = select(FD_SETSIZE, &listenList, 0, 0, &time);
  } while (stat < 0 && errno == EINTR);
  
  if (stat < 0 ) {
    fprintf(stderr,"handleSimEvents: Select failed %d\n",errno);
    return;
  }
  
  if (FD_ISSET(xSocket, &listenList)) {
    while ((mask = (XtAppPending(app_context) & (XtIMXEvent | XtIMTimer)))!= 0)
      {
	XtAppProcessEvent(app_context,mask);
      }
  } else if (FD_ISSET(tcaSocket, &listenList)) {
    tcaHandleMessage(0);
  }
  simUpdate();
}

/*--------------------------------------------------
  
  Handler for the BOUNDED_MOVE_COMMAND message.
  
  ---------------------------------------------------------------------------*/

static void boundedMoveCommandHandler(TCA_REF_PTR ref, void *data)
{
  BOUNDED_MOVE_COMMAND_PTR boundedMoveData;
  
  boundedMoveData = (BOUNDED_MOVE_COMMAND_PTR)data;
  
  simBoundedMove(*boundedMoveData);
  tcaSuccess(ref);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the UNBOUNDED_MOVE_COMMAND message.
  
  ---------------------------------------------------------------------------*/

/* ARGSUSED */
static void unboundedMoveCommandHandler(TCA_REF_PTR ref, void *data)
{
  simUnboundedMove();
  tcaSuccess(ref);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the GUARDED_MOVE_COMMAND message.
  
  ---------------------------------------------------------------------------*/

static void guardedMoveCommandHandler (TCA_REF_PTR ref, void *data)
{
  GUARDED_MOVE_COMMAND_PTR guardedMoveData;
  CMS distTravelled;		/* the distance travelled so far */
  
  guardedMoveData = (GUARDED_MOVE_COMMAND_PTR)data;
  
  distTravelled = simGuardedMove(guardedMoveData->distance,
				 guardedMoveData->stopRange );
  if (ABS(distTravelled - guardedMoveData->distance) <= 0.05) {
    tcaSuccess(ref);
  } else {
    tcaFailure(ref, "GuardedMoveFailure", &distTravelled);
  }
  
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the TURN_COMMAND message.
  
  ---------------------------------------------------------------------------*/

static void turnCommandHandler(TCA_REF_PTR ref, void *data)
{
  TURN_COMMAND_PTR turnData;
  
  turnData = (TURN_COMMAND_PTR)data;
  
  simTurn(*turnData);
  tcaSuccess(ref);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*--------------------------------------------------------------
  
  Handler for the STOP_COMMAND message.
  
  ---------------------------------------------------------------------------*/

/* ARGSUSED */
static void stopCommandHandler(TCA_REF_PTR ref, void *data)
{
  simStop();
  
  tcaSuccess(ref);
  tcaFreeData( tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the SET_VELOCITY_INFORM message.
  
  ---------------------------------------------------------------------------*/

static void setVelocityInformHandler(TCA_REF_PTR ref, void *data)
{
  SET_VELOCITY_INFORM_PTR setVelocityData;
  
  setVelocityData = (SET_VELOCITY_INFORM_PTR)data;
  
  simSetVelocity(setVelocityData->transVelocity, setVelocityData->rotVelocity);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the VELOCITY_QUERY message.
  
  ---------------------------------------------------------------------------*/

/* ARGSUSED */
static void velocityQueryHandler(TCA_REF_PTR ref, void *data)
{
  VELOCITY_REPLY_TYPE velocity;
  
  simVelocity(&velocity.transVelocity, &velocity.rotVelocity);
  tcaReply(ref, &velocity);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the TALK_COMMAND message.
  
  ---------------------------------------------------------------------------*/

static void talkCommandHandler(TCA_REF_PTR ref, void *data)
{
  TALK_COMMAND_PTR talkData;
  
  talkData = (TALK_COMMAND_PTR)data;
  
  simTalk(*talkData);
  tcaSuccess(ref);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------------
  
  Handler for the SONAR_PING_QUERY message.
  
  -------------------------------------------------------------------------*/

static void sonarPingQueryHandler(TCA_REF_PTR ref, void *data)
{
  CMS sonarData[24];
  SONAR_PING_QUERY_TYPE sonarPingData;
  SONAR_PING_REPLY_TYPE sonarPingReply;
  
  sonarPingData = *(SONAR_PING_QUERY_PTR)data;
  
  if (sonarPingData < 0 || sonarPingData > 23) {
    fprintf(stderr, "Sonar Ping request out of range: %d\n", sonarPingData);
    tcaNullReply(ref);
  } else {
    sonarScan(sonarData);
    sonarPingReply = sonarData[sonarPingData];
    tcaReply(ref, &sonarPingReply);
  }
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the SONAR_SCAN_QUERY message.
  
  ---------------------------------------------------------------------------*/

static void sonarScanQueryHandler(TCA_REF_PTR ref, void *data)
{
  CMS sonarData[24];
  SONAR_SCAN_QUERY_PTR sonarScanData;
  SONAR_SCAN_REPLY_TYPE sonarScanReply;
  int i, j, num;
  
  sonarScanData = (SONAR_SCAN_QUERY_PTR)data;
  
  if (sonarScanData->startSonar < 0 || sonarScanData->startSonar > 23 ||
      sonarScanData->endSonar < 0 || sonarScanData->endSonar > 23) {
    fprintf(stderr, "Sonar Scan request out of range: (%d..%d)\n",
	    sonarScanData->startSonar, sonarScanData->endSonar);
    tcaNullReply(ref);
  } else {
    sonarScan(sonarData);
    for (j=0;j<24;j++) sonarScanReply.sonarData[j] = 0.0;
    num = (sonarScanData->endSonar - sonarScanData->startSonar + 1 + 24)%24; 
    for (i=sonarScanData->startSonar, j=0; j<num; i++, j++) {
      sonarScanReply.sonarData[j] = sonarData[i%24];
    }
    tcaReply(ref, &sonarScanReply);
  }
  tcaFreeData(tcaReferenceName(ref), data);
}


/*----------------------- hero_tcaSonarScanVarArray ------------------------
  
  Handler for the HERO_SONAR_SCAN_VAR message.
  
  ---------------------------------------------------------------------------*/

static void sonarScanVarQueryHandler(TCA_REF_PTR ref, void *data)
{
  CMS sonarData[24], sonarReplyData[24];
  SONAR_SCAN_VAR_QUERY_PTR sonarScanVarData;
  SONAR_SCAN_VAR_REPLY_TYPE sonarScanVarReply;
  int i, j, num;
  
  sonarScanVarData = (SONAR_SCAN_VAR_QUERY_PTR)data;
  
  if (sonarScanVarData->startSonar < 0 || sonarScanVarData->startSonar > 23 ||
      sonarScanVarData->endSonar < 0 || sonarScanVarData->endSonar > 23) {
    fprintf(stderr, "Sonar Scan Var request out of range: (%d..%d)\n",
	    sonarScanVarData->startSonar, sonarScanVarData->endSonar);
    tcaNullReply(ref);
  } else {
    sonarScan(sonarData);
    num = (sonarScanVarData->endSonar-sonarScanVarData->startSonar+1 + 24)%24; 
    for (i=sonarScanVarData->startSonar, j=0; j<num; i++, j++) {
      sonarReplyData[j] = sonarData[i%24];
    }
    sonarScanVarReply.numItems = num;
    sonarScanVarReply.sonarData = sonarReplyData;
    tcaReply(ref, &sonarScanVarReply);
  }
  tcaFreeData(tcaReferenceName(ref), data);
}


/*-----------------------------------------------
  
  Handler for the SONAR_SCAN_LIST_QUERY message.
  
  Note: we are not freeing any list space here - bad move
  
  ---------------------------------------------------------------------------*/

static void sonarScanListQueryHandler(TCA_REF_PTR ref, void *data)
{
  CMS sonarData[24];
  SONAR_SCAN_LIST_QUERY_PTR sonarScanListData;
  SONAR_SCAN_LIST_REPLY_TYPE sonarScanListReply;
  SONAR_LIST_PTR nextItem;
  int i, j, num;
  
  sonarScanListData = (SONAR_SCAN_LIST_QUERY_PTR)data;
  
  if (sonarScanListData->startSonar < 0 || 
      sonarScanListData->startSonar > 23 ||
      sonarScanListData->endSonar < 0 || sonarScanListData->endSonar > 23) {
    fprintf(stderr, "Sonar Scan List request out of range: (%d..%d)\n",
	    sonarScanListData->startSonar, sonarScanListData->endSonar);
    tcaNullReply(ref);
  } else {
    sonarScan(sonarData);
    sonarScanListReply = NULL;
    num = (sonarScanListData->endSonar-sonarScanListData->startSonar+1 +24)%24;
    for (i=sonarScanListData->startSonar, j=0; j<num; i++, j++) {
      nextItem = (SONAR_LIST_PTR)calloc(1,sizeof(SONAR_LIST_TYPE));
      nextItem->next = sonarScanListReply;
      nextItem->sonarData = sonarData[i%24];
      sonarScanListReply = nextItem;
    }
    tcaReply(ref, &sonarScanListReply);
  }
  
  tcaFreeData(tcaReferenceName(ref), data);
  tcaFreeReply(tcaReferenceName(ref), &sonarScanListReply);
}


/*-----------------------------------------------
  
  Handler for the MOTION_QUERY message.
  
  ---------------------------------------------------------------------------*/

/* ARGSUSED */
static void motionQueryHandler(TCA_REF_PTR ref, void *data)
{
  MOTION_REPLY_TYPE moving;
  
  moving = simInMotion();
  tcaReply(ref, &moving);
  tcaFreeData(tcaReferenceName(ref), data);
}

/*-----------------------------------------------
  
  Handler for the DISTANCE_MOVED_QUERY message.
  
  ---------------------------------------------------------------------------*/

/* ARGSUSED */
static void distanceMovedQueryHandler(TCA_REF_PTR ref, void *data)
{
  DISTANCE_MOVED_REPLY_TYPE	distance;
  
  distance = simLastMoveDistance();
  tcaReply(ref, &distance);
  tcaFreeData(tcaReferenceName(ref), data);
}

static void registerAll(void)
{
  tcaRegisterNamedFormatter(CMS_NAME, CMS_FORMAT);
  tcaRegisterNamedFormatter(DEGREES_NAME, DEGREES_FORMAT);
  
  tcaRegisterCommand(BOUNDED_MOVE_COMMAND, BOUNDED_MOVE_FORM,
		     boundedMoveCommandHandler);
  
  tcaRegisterCommand(UNBOUNDED_MOVE_COMMAND, UNBOUNDED_MOVE_FORM,
		     unboundedMoveCommandHandler);
  
  if (registerGuardedMove) {
    tcaRegisterCommand(GUARDED_MOVE_COMMAND, GUARDED_MOVE_FORM,
		       guardedMoveCommandHandler);
  }
  
  tcaRegisterCommand(TURN_COMMAND, TURN_FORM, turnCommandHandler);
  
  tcaRegisterCommand(STOP_COMMAND, STOP_FORM, stopCommandHandler);
  
  tcaRegisterInform(SET_VELOCITY_INFORM, SET_VELOCITY_FORM,
		    setVelocityInformHandler);
  
  tcaRegisterCommand(TALK_COMMAND, TALK_FORM, talkCommandHandler);
  
  tcaRegisterQuery(SONAR_PING_QUERY, SONAR_PING_QUERY_FORM,
		   SONAR_PING_REPLY_FORM,
		   sonarPingQueryHandler);
  
  tcaRegisterQuery(SONAR_SCAN_QUERY, SONAR_SCAN_QUERY_FORM,
		   SONAR_SCAN_REPLY_FORM, sonarScanQueryHandler);
  
  tcaRegisterQuery(SONAR_SCAN_LIST_QUERY, SONAR_SCAN_LIST_QUERY_FORM,
		   SONAR_SCAN_LIST_REPLY_FORM,
		   sonarScanListQueryHandler);
  
  tcaRegisterQuery(SONAR_SCAN_VAR_QUERY, SONAR_SCAN_VAR_QUERY_FORM,
		   SONAR_SCAN_VAR_REPLY_FORM,
		   sonarScanVarQueryHandler);
  
  tcaRegisterQuery(VELOCITY_QUERY, VELOCITY_QUERY_FORM,
		   VELOCITY_REPLY_FORM, velocityQueryHandler);
  
  tcaRegisterQuery(MOTION_QUERY, MOTION_QUERY_FORM, MOTION_REPLY_FORM,
		   motionQueryHandler);
  
  tcaRegisterQuery(DISTANCE_MOVED_QUERY, DISTANCE_MOVED_QUERY_FORM,
		   DISTANCE_MOVED_REPLY_FORM, distanceMovedQueryHandler);
  
  if (useResources) {
    tcaRegisterResource(TALK_RESOURCE, 1);
    tcaAddHndToResource(TALK_COMMAND, TALK_RESOURCE);
    
    tcaRegisterResource(STOP_RESOURCE, 1);
    tcaAddHndToResource(STOP_COMMAND, STOP_RESOURCE);
    
    tcaRegisterResource(QUERY_RESOURCE, 1);
    tcaAddHndToResource(SONAR_PING_QUERY, QUERY_RESOURCE);
    tcaAddHndToResource(SONAR_SCAN_QUERY, QUERY_RESOURCE);
    tcaAddHndToResource(SONAR_SCAN_VAR_QUERY, QUERY_RESOURCE);
    tcaAddHndToResource(SONAR_SCAN_LIST_QUERY, QUERY_RESOURCE);
    tcaAddHndToResource(MOTION_QUERY, QUERY_RESOURCE);
    tcaAddHndToResource(DISTANCE_MOVED_QUERY, QUERY_RESOURCE);
  }
}

/*****************************************************************************
 *
 * FUNCTION: void disconnectTCA(void)
 *
 * DESCRIPTION: Stop the robot if tca goes down.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * HISTORY:
 *
 *****************************************************************************/

static void disconnectTCA(void)
{
  /* called when a central disconnect is detected. */
}

/*****************************************************************************
 *
 * FUNCTION: static void reconnectTCA(void)
 *
 * DESCRIPTION: Reconnect to tca is possible.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * HISTORY:
 *
 *****************************************************************************/

static void reconnectTCA(void)
{
  /* called when a central is reconnected. */
}

void main(int argc, char *argv[])
{
  
  if (argc < 2 || !strcmp(argv[1], "-h")) {
    fprintf(stderr, "Usage: %s parameter-file <-r> <-g>\n", argv[0]);
    fprintf(stderr, "  -r : Use multiple resources (optional)\n");
    fprintf(stderr, "  -g : Don't register guarded move command (optional)\n");
  }
  
  useResources = (argc > 2 && !strcmp(argv[2], "-r"));
  registerGuardedMove = !(argc > 3 && !strcmp(argv[3], "-g"));
  
  init_simx(argv[1]);
  
  RedrawWholeMap();
  
  sensors();  
  
  RedrawPosition(xposW,NULL,NULL,NULL);
  
  plot_robot(1);
  
#if (MAIN_LOOP == HOME_GROWN_LOOP)
  tcaConnectModule(CONTROLLER_MODULE, tcaServerMachine());
  
  registerAll();
  
  tcaWaitUntilReady();
  
  resetTime();
  
  while (TRUE) handleSimEvents();
#elif (MAIN_LOOP == X11_LOOP)
  tcaConnectModule(CONTROLLER_MODULE, tcaServerMachine());
  
  registerAll();
  
  tcaWaitUntilReady();
  
  resetTime();
  
  connectTCA_X11(app_context);

  XtAppMainLoop(app_context);
#else /* DEVUTILS_LOOP */

  x11_connect(ConnectionNumber(theDisplay));

  /* connect up the handlers for standard in. */
  stdin_connect(stdin_defaultInputHnd); 
  
  TCA_connect(CONTROLLER_MODULE,registerAll,disconnectTCA,reconnectTCA,
	      NULL, NULL);
  
  resetTime();

  /* loop forever in the main device loop */
  devMainLoop();

  /* should never reach here. */
  devShutdown();
  exit(0);
  
#endif
}
