/*-----------------------------------------------------------------------------

File:		resources2.c

Programmer:	Reid Simmons

Description:	TCA module that illustrates example 2 in the use of resources.
	        Define a "takePicture" command that locks the controller
		 (simulator) resource.  Show how that interacts with
		 the "shape" module.

		Message handlers for the following message are implemented:

		  TAKE_PICTURE_COMMAND

		This is a stand-alone module.  
		It takes no command line options.

		To run this example:
		  1. central 3 -lmdsi
		  2. simulator/simulator sim.param
		  3. taskTrees
		  4. resources2

-----------------------------------------------------------------------------*/

#include "tca/libc.h"
#include "tca.h"
#include "simMessages.h"
#include "resources.h"

int getbool(char *prompt, int ret_default)
{
  char input[80];
  printf("%s",prompt);
  fflush(stdout);
  gets(input);
  if ((input[0] == 'y') || (input[0] == 'Y')) 
    return TRUE;
  if ((input[0] == 'n') || (input[0] == 'N')) 
    return FALSE;
  return ret_default;
}

/*-------------------------- takePictureHandler ---------------------------

Handler for the TAKE_PICTURE_COMMAND message.

Lock the controller resource to ensure that no motion occurs during the
image acquisition.

----------------------------------------------------------------------------*/

static void takePictureHandler(TCA_REF_PTR ref, void *data)
{
  TAKE_PICTURE_PTR picData;
  TCA_REF_PTR lock;

  picData = (TAKE_PICTURE_PTR)data;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));
  fprintf(stderr, "  Waiting to lock the simulator resource\n");
  lock = tcaLockResource(CONTROLLER_MODULE);
  if (!lock || tcaReferenceName(lock)==NULL) {
    fprintf(stderr, "  Could not acquire a lock\n");
    tcaFailure(ref, "takePicture: no lock", NULL);
  } else {
    /* The code to actually acquire the image goes here */
    fprintf(stderr, "  Acquiring an image named '%s'...", *picData);
    fflush(stderr);
    sleep(5);
    fprintf(stderr, "Done\n");
    fprintf(stderr, "  Unlocking the resource\n");
    tcaUnlockResource(lock);
    tcaSuccess(ref);
  }
  tcaFreeData(tcaReferenceName(ref), data);
}


/*----------------------- main ------------------------
  RESOURCES module: Examples in using resource locking.

  No command line arguments.

  The "taskTrees" program should be running concurrently with this program.

----------------------------------------------------------------------------*/
void main (void)
{
  TAKE_PICTURE_TYPE takePicture;

  tcaConnectModule("Resources", tcaServerMachine());

  tcaRegisterCommand(TAKE_PICTURE_COMMAND, TAKE_PICTURE_FORM,
		     takePictureHandler);

  tcaWaitUntilReady();

  do {
    takePicture = "Color Camera";
    tcaExecuteCommand(TAKE_PICTURE_COMMAND, (void *)&takePicture);
    tcaHandleMessage(WAITFOREVER);
  } while (getbool("Take another picture?", FALSE));
}
