/*-----------------------------------------------------------------------------

File:		monitor.c

Programmer:	Reid Simmons

Description:	TCA module that illustrates use of monitors.

                Reimplement the guarded move command using a bounded
		move and a polling interval monitor.

		Message handlers for the following message are implemented:

		  RANGE_QUERY
		  STOP_ACTION_COMMAND
		  GUARDED_MOVE_COMMAND (Supercedes simulator's definition)

		A polling interval monitor is defined:
  		  GUARDED_MOVE_MONITOR,
		that moves the robot and checks for obstacles within range
		periodically. 

		This is a stand-alone module.  
		The command line options are: monitors <period>,
		where "period" is how often to poll the sonars (in seconds).
		If not specified, "period" defaults to 1 second.

		To run this example:
		  1. central 3 -lmdsi
		  2. simulator/simulator sim.param -r -g
		  3. monitors
		  4. taskTrees 4 100

-----------------------------------------------------------------------------*/

#include "tca/libc.h"
#include "tca.h"
#include "simMessages.h"
#include "monitors.h"

static int monitorPeriodGlobal = PERIOD_DEFAULT;

/*-------------------------- rangeQueryHandler ---------------------------

Handler for the RANGE_QUERY message.

   Take sonar readings from +/- 45 degrees: if readings are less than
   the "range" argument, return the closest reading; otherwise NullReply

----------------------------------------------------------------------------*/

static void rangeQueryHandler(TCA_REF_PTR ref, void *data)
{
  RANGE_QUERY_PTR rangePtr;
  SONAR_SCAN_VAR_QUERY_TYPE sonarQuery;
  SONAR_SCAN_VAR_REPLY_TYPE sonarReply;
  CMS min=1000.0;
  int i;

  rangePtr = (RANGE_QUERY_PTR)data;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));
  sonarQuery.startSonar = -45/15 + 24; /* -45 degree sonar */
  sonarQuery.endSonar   =  45/15;      /* +45 degree sonar */
  tcaQuery(SONAR_SCAN_VAR_QUERY, (void *)&sonarQuery, (void *)&sonarReply);
  for (i=0; i<sonarReply.numItems; i++) {
    if (sonarReply.sonarData[i] < min)
      min = sonarReply.sonarData[i];
  }
  if (min <= *rangePtr) {
    fprintf(stderr, "  Sonar reading %3.1f is within stopping range (%3.1f)\n",
	    min, *rangePtr);
    tcaReply(ref, (void *)&min);
  } else {
    fprintf(stderr, "  All sonar readings outside of stopping range\n");
    tcaNullReply(ref);
  }

  tcaFreeData(tcaReferenceName(ref), data);
  tcaFreeReply(SONAR_SCAN_VAR_QUERY, (void *)&sonarReply);
}


/*-------------------------- stopActionHandler ---------------------------

	Handler for the STOP_ACTION_COMMAND message.

   Stop all current motions and return a failure (the failure data is
   the distance travelled so far).

   The data argument, passed on from the RANGE_QUERY, is ignored.

----------------------------------------------------------------------------*/

/* ARGSUSED */
static void stopActionHandler(TCA_REF_PTR ref, void *data)
{
  DISTANCE_MOVED_REPLY_TYPE distanceMoved;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));
  fprintf(stderr, "  Stopping motion\n");
  tcaExecuteCommand(STOP_COMMAND, NULL);

  /* Get distance travelled from move and return as failure data */
  tcaQuery(DISTANCE_MOVED_QUERY, NULL, (void *)&distanceMoved);
  fprintf(stderr, "  Raising exception: Moved only %3.1f\n", distanceMoved);
  tcaFailure( ref, "GuardedMoveFailure", &distanceMoved );

  tcaFreeData(tcaReferenceName(ref), data);
}


/*-------------------------- guardedMoveHandler ---------------------------

	Handler for the GUARDED_MOVE_COMMAND message.

   Command a regular (bounded) move, and use the GUARDED_MOVE_MONITOR
   to check for objects coming within range.

   The data is the distance to move and the range at which to stop 
   for obstacles.  The global variable "guardedMoveMonitorGlobal" gives
   the polling frequency, in seconds.

----------------------------------------------------------------------------*/

static void guardedMoveHandler(TCA_REF_PTR ref, void *data)
{
  GUARDED_MOVE_COMMAND_PTR guardedMoveData;
  TCA_REF_PTR moveRef;
  INTERVAL_MONITOR_OPTIONS_PTR monitorOptions;
  BOUNDED_MOVE_COMMAND_TYPE boundedMoveData;
  
  guardedMoveData = (GUARDED_MOVE_COMMAND_PTR)data;

  fprintf(stderr, "Handling the %s message\n", tcaReferenceName(ref));
  fprintf(stderr, "  Creating reference to move command\n");
  moveRef = tcaCreateReference(BOUNDED_MOVE_COMMAND);

  fprintf(stderr,
	  "  Setting up guarded move monitor to poll every %d seconds\n",
	  monitorPeriodGlobal);
  monitorOptions = tcaCreateMonitorOptions();
  monitorOptions->maxFire = 1;
  monitorOptions->period = monitorPeriodGlobal;
  tcaIntervalMonitorWithConstraints(NULL, GUARDED_MOVE_MONITOR, 
				    &(guardedMoveData->stopRange),
				    tcaStartOf(tcaAchievementOf(moveRef)),
				    tcaEndOf(tcaAchievementOf(moveRef)),
				    monitorOptions);
  free(monitorOptions);

  boundedMoveData = guardedMoveData->distance;
  fprintf(stderr, "  Sending move command of %3.2f cms\n", boundedMoveData);
  tcaExecuteCommandWithConstraints(moveRef, BOUNDED_MOVE_COMMAND,
				   &boundedMoveData, SEQ_ACH);

  tcaSuccess(ref);
  tcaReferenceRelease(moveRef);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*----------------------- main ------------------------
  MONITORS module: Implement guarded move command using polling monitor.

  Command line arguments are: monitors <period>,
  where "period" is how often to poll the sonars (in seconds).
  If not specified, "period" defaults to 1 second.

  Note: The simulator must be started in "multiple resource mode"
        (command line option -r).
----------------------------------------------------------------------------*/
void main (int argc, char **argv)
{
  if (argc < 2) {
    monitorPeriodGlobal = PERIOD_DEFAULT;
  } else {
    monitorPeriodGlobal = atoi(argv[1]);
  }
  tcaConnectModule("MONITORS", tcaServerMachine());

  tcaRegisterQuery(RANGE_QUERY, RANGE_QUERY_FORM, 
		   RANGE_QUERY_REPLY_FORM, rangeQueryHandler);
  
  tcaRegisterCommand(STOP_ACTION_COMMAND, STOP_ACTION_FORM, 
		     stopActionHandler);

  tcaRegisterCommand(GUARDED_MOVE_COMMAND, GUARDED_MOVE_FORM,
		     guardedMoveHandler);

  tcaRegisterPollingMonitor(GUARDED_MOVE_MONITOR, 
			    RANGE_QUERY, STOP_ACTION_COMMAND);

  tcaWaitUntilReady();

  tcaModuleListen();
}
