/*-----------------------------------------------------------------------------

File:		exceptions.c

Programmer:	Reid Simmons

Description:	Header file containing type definitions and
                TCA message definitions for the TCA exceptions usage example.

		The example handles the "GuardedMoveFailure" exception
		issued by the HERO_GUARDED_MOVE command.  
		Very simple recovery strategy: turn left or right 90 degrees,
		move a distance to the side, continue the rest of the original
		commanded distance, and then turn back to the original goal.
		If one of the detour moves encounters an obstacle, the
		exception handler can be invoked recursively.
		VERY SIMPLE: ALMOST DEFINITELY WILL BREAK IN COMPLEX SITUATIONS,
		BUT SUITABLE TO DEMONSTRATE EXCEPTION HANDLING.

		The following messages are available:
			GUARDED_MOVE_EXCEPTION
		This handles the failure generated by the guarded move command.

		This is a stand-alone module.  
		There are no command line options.

		To run this example:
		  1. central 3 -lmdsi
		  2. simulator/simulator sim.param.exception
		  3. exceptions
		  4. taskTrees 5 120

-----------------------------------------------------------------------------*/

#include "tca/libc.h"
#include "tca.h"
#include "simMessages.h"
#include "taskTrees.h"
#include "exceptions.h"

/*-------------------------- guardedMoveExceptionHandler ----------------------

Handler for the GUARDED_MOVE_EXCEPTION failure message.

    Find which direction is more open (to the left or right side of the robot).
    Turn in that direction, move a small distance away, turn back in the 
    original direction, complete the move distance, and then return to the
    original goal location and orientation.

    If both sides are closed in (less than the stop range), bypass the
    exception.

----------------------------------------------------------------------------*/
/* failure data is the distance travelled so far */
static void guardedMoveExceptionHandler(TCA_REF_PTR ref, void *data)
{
  GUARDED_MOVE_EXCEPTION_PTR distTravelledPtr;
  TCA_REF_PTR gmoveRef;
  GUARDED_MOVE_COMMAND_PTR gmoveDataPtr;
  SONAR_SCAN_VAR_QUERY_TYPE range;
  SONAR_SCAN_VAR_REPLY_TYPE sonarLeft, sonarRight;
  TURN_COMMAND_TYPE turnDir;
  MOVE_TURN_TYPE moveTurn;
  CMS minLeft, minRight, remainingDist, detourDist = DETOUR_DISTANCE;
  int i;

  distTravelledPtr = (GUARDED_MOVE_EXCEPTION_PTR)data;

  /* Get commanded distance and stop range of active guarded move command */
  gmoveRef = tcaFindAnteReferenceByName(ref, GUARDED_MOVE_COMMAND);
  gmoveDataPtr = (GUARDED_MOVE_COMMAND_PTR)tcaReferenceData(gmoveRef);
  fprintf(stderr, "  Moved %3.1f centimeters out of %3.1f commanded move\n",
	  *distTravelledPtr, gmoveDataPtr->distance);

  /* Get sonar readings to the right and left side, and choose the one that has
     the most open space.  Cannot handle the exception if all readings are
     below
     the stop range */
  range.startSonar = (-90 - 45)/15 + 24;
  range.endSonar = (-90 + 45)/15 + 24;
  tcaQuery(SONAR_SCAN_VAR_QUERY, (void *)&range, (void *)&sonarLeft);
  minLeft = sonarLeft.sonarData[0];
  for (i=1; i<sonarLeft.numItems; i++) {
    if (sonarLeft.sonarData[i] < minLeft) minLeft = sonarLeft.sonarData[i];
  }
  fprintf(stderr, "  Minimum sonar reading on the left is %3.1f cms\n",
	  minLeft);

  range.startSonar = (90 - 45)/15;
  range.endSonar = (90 + 45)/15;
  tcaQuery(SONAR_SCAN_VAR_QUERY, (void *)&range, (void *)&sonarRight);
  minRight = sonarRight.sonarData[0];
  for (i=1; i<sonarRight.numItems; i++) {
    if (sonarRight.sonarData[i] < minRight) minRight = sonarRight.sonarData[i];
  }
  fprintf(stderr, "  Minimum sonar reading on the right is %3.1f cms\n",
	  minRight);

  if (minLeft <= gmoveDataPtr->stopRange &&minRight <= gmoveDataPtr->stopRange)
    {
      tcaByPassException(ref);
  } else {
    turnDir = ((minLeft > minRight) ? -90 : 90);
    remainingDist = gmoveDataPtr->distance - (*distTravelledPtr);
    /* Turn in the detour direction */
    fprintf(stderr, "  Turning to the %s\n", (turnDir < 0 ? "left" : "right"));
    tcaExecuteCommand(TURN_COMMAND, (void *)&turnDir);
    /* Move the detour distance and turn back to original direction */
    moveTurn.distance = detourDist; moveTurn.angle = -turnDir;
    tcaExpandGoal(MOVE_TURN_GOAL, (void *)&moveTurn);
    /* Move parallel to original path and turn toward goal */
    moveTurn.distance = remainingDist; moveTurn.angle = -turnDir;
    tcaExpandGoal(MOVE_TURN_GOAL, (void *)&moveTurn);
    /* Move to the original goal location and turn back to original
       direction */
    moveTurn.distance = detourDist; moveTurn.angle = turnDir;
    tcaExpandGoal(MOVE_TURN_GOAL, (void *)&moveTurn);
  }

  tcaSuccess(ref);
  tcaFreeData(tcaReferenceName(gmoveRef), (void *)gmoveDataPtr);
  tcaReferenceRelease(gmoveRef);
  tcaFreeData(tcaReferenceName(ref), data);
}


/*----------------------- main ------------------------
  EXCEPTIONS module: Implements the exception handler for the failure generated
                     by the guarded move command.

  No command line options.
----------------------------------------------------------------------------*/
void main (void)
{
  tcaConnectModule("EXCEPTIONS", tcaServerMachine());

  tcaRegisterExceptionMessage(GUARDED_MOVE_EXCEPTION,
			      GUARDED_MOVE_EXCEPTION_FORM);
  tcaRegisterHandler(GUARDED_MOVE_EXCEPTION, GUARDED_MOVE_EXCEPTION_HANDLER,
		     guardedMoveExceptionHandler);
  tcaAddExceptionHandlerToMessage(GUARDED_MOVE_COMMAND,
				  GUARDED_MOVE_EXCEPTION_HANDLER);

  tcaWaitUntilReady();

  tcaModuleListen();
}

