/*-----------------------------------------------------------------------------

File:		billBoard.c

Programmer:	Richard Goodwin

Description:	TCA module that provides a graphics server.  It is used to
                illustrate how to register messages and handlers.

		Very little error checking is done, X may crash if you use 
                unreasonable parameter values.

		Message handlers for the following message are implemented:
		BB_CLEAR_CMD
		BB_FLASH_CMD
		BB_FILL_CMD
		BB_DRAW_SQUARE_CMD
		BB_DRAW_CIRCLE_CMD
		BB_DRAW_LINE_CMD
		BB_DRAW_LINES_CMD
		BB_CONNECT_DOTS_CMD

		This is a stand-alone module.  It does not take any 
		command-line options.

		To run this example:
		  1. central 2 -lmdsi
		  2. billBoard
		  3. artist <your name>

-----------------------------------------------------------------------------*/

#include "tca/libc.h"
#include "tca.h"
#include "tca/ezx.h"
#include "tca/basics.h"
#include "billBoard.h"

/* Global Definitions */
#define BILLBOARD_MODULE_NAME "billBoard"

/* Global Variables */
Window theWindow;

/* Utility routines. */
static void setColour(char *colour)
{
  int theColour = C_RED;
  if (strstr(colour,"red")) {
    printf("Red Fill\n");
    theColour = C_RED;
  } else if (strstr(colour,"white")) {
    printf("White Fill\n");
    theColour = C_WHITE;
  } else if (strstr(colour,"blue")) {
    printf("Blue Fill\n");
    theColour = C_BLUE;
  } else if (strstr(colour,"green")) {
    printf("Green Fill\n");
    theColour = C_LAWNGREEN;
  } else if (strstr(colour,"yellow")) {
    printf("Yellow Fill\n");
    theColour = C_YELLOW;
  } else {
    printf("Unknown Colour\n");
  }
  fflush(stdout);
  EZX_SetColor(theColour);
}

/* Message Handlers */

static void bb_clearHnd(TCA_REF_PTR ref, void *empty)
{
  EZX_ClearWindow(theWindow); 
  tcaSuccess(ref);
}

static void bb_flashHnd(TCA_REF_PTR ref, void *empty)
{
  EZX_ClearWindow(theWindow); 
  EZX_Flush();
  sleep(1);
  EZX_SetColor(C_RED);
  EZX_FillRectangle(theWindow, 0, 0, BB_WIDTH, BB_HEIGHT);
  EZX_Flush();
  sleep(1);
  EZX_SetColor(C_YELLOW);
  EZX_FillRectangle(theWindow, 0, 0, BB_WIDTH, BB_HEIGHT);
  EZX_Flush();
  sleep(1);
  EZX_ClearWindow(theWindow); 
  EZX_Flush();
  tcaSuccess(ref);
}

static void bb_fillHnd(TCA_REF_PTR ref, char **colour)
{

  setColour(*colour);
  EZX_FillRectangle(theWindow, 0, 0, BB_WIDTH, BB_HEIGHT);
  tcaSuccess(ref);
  tcaFreeData(BB_FILL_CMD,colour);
  EZX_Flush();
}

static void bb_drawSquareHnd(TCA_REF_PTR ref, SQUARE_PTR sqr)
{
  setColour(sqr->colour);
  
  EZX_FillRectangle(theWindow, sqr->left, sqr->top, sqr->size, sqr->size);
  tcaSuccess(ref);
  tcaFreeData(BB_DRAW_SQUARE_CMD,sqr);
  EZX_Flush();
}

static void bb_drawCircleHnd(TCA_REF_PTR ref, CIRCLE_PTR crl)
{
  setColour(crl->colour);
  if (crl->filled)
    EZX_FillCircle(theWindow, crl->center.x, crl->center.y, crl->radius);
  else
    EZX_DrawCircle(theWindow, crl->center.x, crl->center.y, crl->radius);
  tcaSuccess(ref);
  tcaFreeData(BB_DRAW_CIRCLE_CMD,crl);
  EZX_Flush();
}

static void bb_drawLineHnd(TCA_REF_PTR ref, LINE_PTR line)
{
  EZX_DrawLine(theWindow, (*line)[0].x, (*line)[0].y, 
	       (*line)[1].x, (*line)[1].y);
  tcaSuccess(ref);
  tcaFreeData(BB_DRAW_LINE_CMD,line);
  EZX_Flush();
}

static void bb_drawLinesHnd(TCA_REF_PTR ref, LINES_PTR lines)
{
  int i;

  for ( i=0; i<lines->num_lines; i++) {
    tcaExecuteCommand(BB_DRAW_LINE_CMD,&(lines->lines[i]));
  }
  tcaSuccess(ref);
  tcaFreeData(BB_DRAW_LINES_CMD,lines);
  EZX_Flush();
}

static void bb_connectDotsHnd(TCA_REF_PTR ref, POINT_LIST *list)
{
  LINE_TYPE line;
  POINT_LIST next;
  
  if(*list != NULL) {
    line[1] = (*list)->point;
    next = (*list)->next;
    while (next != NULL) {
      line[0] = line[1];
      line[1] = next->point;
      tcaExecuteCommand(BB_DRAW_LINE_CMD,&(line));
      next = next->next;
    }
  }
  tcaSuccess(ref);
  tcaFreeData(BB_CONNECT_DOTS_CMD,list);
  EZX_Flush();
}

static void registerAll(void)
{
  /* Register any named formats. */
  tcaRegisterNamedFormatter(BB_POINT_NAME,BB_POINT_NAME_FORMAT);

  /* Register messages and handlers. */
  tcaRegisterCommand(BB_CLEAR_CMD, BB_CLEAR_CMD_FORMAT, bb_clearHnd);
  tcaRegisterCommand(BB_FLASH_CMD, BB_FLASH_CMD_FORMAT, bb_flashHnd);
  tcaRegisterCommand(BB_FILL_CMD, BB_FILL_CMD_FORMAT, bb_fillHnd);
  tcaRegisterCommand(BB_DRAW_SQUARE_CMD, BB_DRAW_SQUARE_CMD_FORMAT, 
		     bb_drawSquareHnd);
  tcaRegisterCommand(BB_DRAW_CIRCLE_CMD, BB_DRAW_CIRCLE_CMD_FORMAT, 
		     bb_drawCircleHnd);
  tcaRegisterCommand(BB_DRAW_LINE_CMD, BB_DRAW_LINE_CMD_FORMAT, 
		     bb_drawLineHnd);
  tcaRegisterCommand(BB_DRAW_LINES_CMD, BB_DRAW_LINES_CMD_FORMAT, 
		     bb_drawLinesHnd);
  tcaRegisterCommand(BB_CONNECT_DOTS_CMD, BB_CONNECT_DOTS_CMD_FORMAT, 
		     bb_connectDotsHnd);
}

void main(int argc, char *argv[])
{

  /* Initialize X11. */
  EZX_InitGraphics(getenv("DISPLAY"),BILLBOARD_MODULE_NAME);

  /* Create the window */
  theWindow = EZX_MakeWindow(BILLBOARD_MODULE_NAME,
			     BB_WIDTH, BB_HEIGHT,
			     "+20+20");
  
  /* Connect to TCA */
  tcaConnectModule(BILLBOARD_MODULE_NAME, tcaServerMachine());

  /* Register messages and handlers */
  registerAll();
  
  /* Wait until the system has started. */
  tcaWaitUntilReady();

  /* Enter the main loop handling messages. */
  tcaModuleListen();
}

