/*
 * File: TreeGUI.c
 * Author: Domingo Gallardo, CMU
 * Purpose: Graphical User Interface for the task tree
 *
 * REVISION HISTORY
 *
 * $Log: TreeGUI.c,v $
 * Revision 1.8  1996/02/10  16:53:55  rich
 * Made private functions static and fixed some forward declarations.
 *
 * Revision 1.7  1996/02/07  15:40:00  reids
 * Cleaned up a bit of the code -- removing extraneous arguments, adding some
 *   "define"s for string constants.  Fixed the initialization of menu items so
 *   that their labels depend on global variables, rather than being hard-coded
 *
 * Revision 1.6  1996/01/27  21:56:56  rich
 * Pre-release of 8.4.
 *
 * Revision 1.5  1995/07/10  16:20:24  rich
 * Interm save.
 *
 * Revision 1.4  1995/07/08  17:51:51  rich
 * Linux Changes.  Also added GNUmakefile.defs.
 *
 * Revision 1.3  1995/05/31  21:00:14  rich
 * Fixed conflict with tca declarations.
 *
 * Revision 1.2  1995/04/07  05:07:42  rich
 * Fixed GNUmakefiles to find the release directory.
 * Fixed problems found by sgi cc compiler.  It would not compile.
 *
 * Revision 1.1  1995/04/05  18:32:24  rich
 * Moved tview files to a subdirectory.
 *
 * Revision 1.4  1995/03/28  01:16:59  rich
 * Fixed problem where pointers to local variables were being returned.
 *
 * Revision 1.3  1995/01/25  00:04:37  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Formatting changes.
 *
 * Revision 1.2  1994/11/02  21:39:07  rich
 * Now works for linux machines (i486).
 * Got afs to work on alpha (and hopefully other vendor OS's)
 * Added generic Makefile and asynchronous sender/receiver.
 * Renamed some X11 files and modified routines so we don't get library
 * conflicts.
 *
 * Revision 1.1  1994/05/31  03:26:39  rich
 * Moved Domingo's tview tool into the main tca module.
 *
 * Revision 1.5  1994/05/27  05:35:00  rich
 * Can now read from file that is being written to.
 * Fixed Menu and button names.
 * Fixed Indentation.
 * Added menu item to change orientation of the tree.
 *
 * Revision 1.4  1993/09/13  04:12:32  domingo
 * Added two set of options to the tree menu:
 *
 * - Erase queries (on/off) : activate/deactivate the deletion of queries
 *      from the task tree. Erase queries by default.
 *
 * - Display constraints (on/off) : activate/deactivate the display of
 *      constraints messages. No display by default.
 *
 * Revision 1.3  1993/09/07  00:24:50  domingo
 * Fixed almost all the warnings
 *
 * Revision 1.2  1993/08/13  02:09:31  domingo
 * Updated function declarations for compilation under gcc in ANSI C (still
 * a lot of warnings to fix).
 * Added automatic logging.
 *
 * Jan 11 1993 - Domingo Gallardo at School of Computer Science, CMU
 * Created.
 *
 */

#include "TreeGUI.h"
#include "Widgets.h"
#include "tcaTree.h"

#include <X11/StringDefs.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Command.h>

Widget global_tree_widget;
BOOLEAN   _center_tree = False;
BOOLEAN   _tree_horizontal = True;
BOOLEAN global_display_tree = True;

/* Global variables */

static void PushTaskHandler(Widget w, XtPointer client_data, 
			    XtPointer call_data);
static void PushTaskHandlerOff(Widget w, XtPointer client_data, 
			       XtPointer call_data);
static BOOLEAN FlagTrue(taskType task_type, _DisplayFlag flag_table[]);
static Task FindTaskWithWidget(Task root_task, Widget widget);
static void ModifyTaskWidget(Task task);
static Widget CreateTaskWidget(Task task);
#ifdef COLOR_WORKSTATION
static Pixel TaskColor(Task task);
static int ConvertColor(Widget w, char *color_name);
#endif

static Array  global_unmanaged_children;
static Array  global_widgets_to_destroy;

_DisplayFlag  task_display_flag_table[MAX_TASK_TYPES] = {
  {task_QUERY,            True},
  {task_GOAL,             True},
  {task_COMMAND,          True},
  {task_EXCEPTION,        True},
  {task_INTERVAL_MONITOR, True},
  {task_POINT_MONITOR,    True},
  {task_INFORM,           True}
};

_DisplayFlag  task_display_when_finished[MAX_TASK_TYPES] = {
  {task_QUERY,            False},
  {task_GOAL,             True},
  {task_COMMAND,          True},
  {task_EXCEPTION,        True},
  {task_INTERVAL_MONITOR, True},
  {task_POINT_MONITOR,    True},
  {task_INFORM,           False}
};

static String StringTemporalConstraint(int constraint, char *text)
{
  strcpy(text,"");
  switch (constraint) {
  case MSG_SEQ_ACH: strcat(text, "SEQ_ACH "); break;

  case MSG_SEQ_PLAN: strcat(text, "SEQ_PLAN "); break;

  case MSG_PLAN_FIRST: strcat(text, "PLAN_FIRST "); break;

  case MSG_DELAY_PLAN: strcat(text, "DELAY_PLAN "); break;

  case MSG_INV_SEQ_ACH: strcat(text, "INV_SEQ_ACH "); break;

  case MSG_INV_SEQ_PLAN: strcat(text, "INV_SEQ_PLAN "); break;

#ifndef TEST_CASE_COVERAGE
  default: strcat(text, "???? "); break;
#endif
  }
  return(text);
}

static String StringIntervalOp(intervalOp interval_op)
{
  switch(interval_op) {
  case(START): return("start");

  case(END): return("end");
  }
  return("?");
}

static String StringInterval(interval interv)
{
  switch(interv) {
  case(HANDLING_INTERVAL): return("handling interval");

  case(PLANNING_INTERVAL): return("planning interval");

  case(ACHIVEMENT_INTERVAL): return("achievement interval");
  }
  return("?");
}

static String StringPointConstraints(PointTC p_constraint, char *text)
{
  
  strcpy(text,"");
  strcat(text, StringIntervalOp(p_constraint->my_op));
  strcat(text, " ");
  strcat(text, StringInterval(p_constraint->my_interval));
  strcat(text, "\n");
  if (p_constraint->point_TC == '\\')
    strcat (text, "<= ");
  else if (p_constraint->point_TC == '/')
    strcat (text, ">= ");
  else {
    text[0] = p_constraint->point_TC;
    text[1] = ' ';
  }
  strcat(text, StringIntervalOp(p_constraint->task_interval_op));
  strcat(text, " ");
  strcat(text, StringInterval(p_constraint->task_interval));
  strcat(text, " ");
  strcat(text, p_constraint->to_task->name);
  return(text);
}

static String TemporalConstraints(Task task,char *text)
{
  PointTC p_constraint;
  TemporalConstraint temporal_constraint;
  static char subText[100];
  
  text[0] = '\0';
  ArrayRewind(task->temporal_constraint);
  temporal_constraint = (TemporalConstraint) 
    ArrayNext(task->temporal_constraint);
  while(temporal_constraint) {
    strcat(text, 
	   StringTemporalConstraint(temporal_constraint->temporal_constraints,
				    &(subText[0])));
    if (temporal_constraint->to_task)
      strcat(text, temporal_constraint->to_task->name);
    temporal_constraint = (TemporalConstraint)
      ArrayNext(task->temporal_constraint);
    strcat(text, "\n");
  }
  strcat(text, "\n");
  ArrayRewind(task->point_constraints);
  p_constraint = (PointTC) ArrayNext(task->point_constraints);
  while(p_constraint) {
    strcat(text, StringPointConstraints(p_constraint,&(subText[0])));
    p_constraint = (PointTC) ArrayNext(task->point_constraints);
    strcat(text, "\n");
  }
  return(text);
}

static String TaskDescription(Task task, char *text)
{
  static char constraintsText[200];
  GUI_Task_Additional taskAddData = GUI_TASK_DATA(task);

  if (taskAddData->expanded) {
    sprintf(text, "%s    \n%s {%d}    \n",
	    TaskTypeString(task->type), task->name, (task->id & ~MSB));
    
    if (task->state2 != task_NORMAL) {
      strcat(text, TaskState2String(task->state2));
      strcat(text, "    \n");
    }
    strcat(text, TaskStateString(task->state));
    strcat(text, "    \n");
    strcat(text, task->module);
    strcat(text, "\n");
    strcat(text, TemporalConstraints(task,&(constraintsText[0])));
    return(text);
  }
  else {
    strcpy(text, task->name);
    return(text);
  }
}

void DisplayWhenFinished(taskType task_type, BOOLEAN flag)
{
  int i;
  
  for(i=0;i<MAX_TASK_TYPES;i++) {
    if (task_display_when_finished[i].type == task_type)
      task_display_when_finished[i].flag = flag;
  }
}

BOOLEAN isDisplayedWhenFinished(taskType task_type)
{
  int i;
  
  for(i=0;i<MAX_TASK_TYPES;i++) {
    if (task_display_when_finished[i].type == task_type)
      return task_display_when_finished[i].flag;
  }
  return False;
}

void DisplayInTaskTree(taskType task_type, BOOLEAN flag)
{
  int i;
  
  for(i=0;i<MAX_TASK_TYPES;i++) {
    if (task_display_flag_table[i].type == task_type)
      task_display_flag_table[i].flag = flag;
  }
}

BOOLEAN isDisplayedInTaskTree(taskType task_type)
{
  int i;
  
  for(i=0;i<MAX_TASK_TYPES;i++) {
    if (task_display_flag_table[i].type == task_type)
      return task_display_flag_table[i].flag;
  }
  return False;
}


void CreateTree(Widget widget_parent)
{
  Arg args[1];
  
  treeWidgetClass = (WidgetClass) &tcaTreeClassRec;

  XtSetArg(args[0], XtNgravity, NorthGravity);
  global_tree_widget = XtCreateManagedWidget("tree", treeWidgetClass, 
					     widget_parent, args, ONE);
  global_unmanaged_children = ArrayNew();
  global_widgets_to_destroy = ArrayNew();
}



/*	Function Name: InsertRootTask
 *      Description:
 *      Arguments: 
 *      Returns: Created tree widget
 */

static void InsertTaskAsRootUI(Task root_task)
{
  Widget  button;
  Arg args[2];
  int i;
  
  i = 0;
  XtSetArg(args[i], XtNshapeStyle, XmuShapeRoundedRectangle);     i++;
  button = XtCreateManagedWidget(root_task->name, commandWidgetClass,
				 global_tree_widget, args, i);
  i = 0;
  
  GUI_TASK_DATA(root_task)->widget = button;

#ifdef COLOR_WORKSTATION
  { int pixel = TaskColor(root_task);
    XtSetArg(args[i], XtNbackground, (Pixel)pixel);   i++;
    XtSetValues(button, args, i);
  }
#endif
}


static void InsertTaskAsChildUI(Task parent, Task task)
{
  Arg args[3];
  Widget button;
  int i;
  GUI_Task_Additional parentAddData = GUI_TASK_DATA(parent);
  
  if (!FlagTrue(task->type, task_display_flag_table) ||
      parentAddData->widget == NULL) 
    return;
  
  button = CreateTaskWidget(task);
  if (global_display_tree) {
    i = 0;
    XtSetArg(args[i], XtNtreeParent, parentAddData->widget);   i++;
    XtSetValues(button, args, i); 
    XUnmapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
    XtManageChild(button); 
    tcaXawTreeForceLayout (global_tree_widget);
    XMapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
    CenterTree(task);
  }
  else {
    ArrayInsert(global_unmanaged_children, (Pointer) task);
  }

  GUI_TASK_DATA(task)->widget = button;
}


static void ModifyTaskUI(Task task, taskState new_state, int time)
{
  Arg args[2];
  int state, i;
  GUI_Task_Additional taskAddData = GUI_TASK_DATA(task);

  if (taskAddData->widget == NULL)
    return;
  if (new_state == task_COMPLETED && 
      (!FlagTrue(task->type, task_display_when_finished))) {
    ArrayDeleteItem(global_unmanaged_children, (Pointer) task);
    
    if (global_display_tree) {
      
      i = 0;
      XtSetArg(args[i], XtNtreeParent, NULL);       i++;
      XtSetValues(taskAddData->widget, args, i);
      XtUnmanageChild(taskAddData->widget);
      XUnmapWindow (XtDisplay(global_tree_widget),
		    XtWindow(global_tree_widget)); 
      tcaXawTreeForceLayout (global_tree_widget);
      XMapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
    }
    else {
      ArrayInsert(global_widgets_to_destroy, (Pointer)taskAddData->widget);
    }
    return;
  }
  if (XtIsManaged(taskAddData->widget)) {
    state = task->state;
    task->state = new_state;
    ModifyTaskWidget(task);
    task->state = state;
    if (global_display_tree) {
      XUnmapWindow (XtDisplay(global_tree_widget),
		    XtWindow(global_tree_widget)); 
      tcaXawTreeForceLayout (global_tree_widget);
      XMapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
      CenterTree(task);
    }
  }
}


static void ModifyTask2UI(Task task, taskState2 new_state, int time)
{
  int state;
  GUI_Task_Additional taskAddData = GUI_TASK_DATA(task);
  
  if (taskAddData->widget == NULL) {
    return;
  }
  if (XtIsManaged(taskAddData->widget)) {
    state = task->state2;
    task->state2 = new_state;
    ModifyTaskWidget(task);
    task->state2 = state;
    if (global_display_tree) {
      XUnmapWindow (XtDisplay(global_tree_widget),
		    XtWindow(global_tree_widget)); 
      tcaXawTreeForceLayout (global_tree_widget);
      XMapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
      CenterTree(task);
    }
  }
}


void TreeRefresh(Widget w, XtPointer client_data, XtPointer call_data)
{
  Widget widget, parent;
  Task task;
  Arg args[3];
  
  XUnmapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget)); 
  ArrayRewind(global_widgets_to_destroy);
  widget = (Widget) ArrayNext(global_widgets_to_destroy);
  while (widget) {
    XtUnmanageChild(widget);
    XtSetArg(args[0], XtNtreeParent, NULL);
    XtSetValues(widget, args, 1);
    /* The widgets must be destroyed somewhere... */
    widget = (Widget)ArrayNext(global_widgets_to_destroy);
  }
  ArrayDeleteAll(global_widgets_to_destroy);
  ArrayRewind(global_unmanaged_children);
  task = (Task)ArrayNext(global_unmanaged_children);
  while (task) {
    widget = GUI_TASK_DATA(task)->widget;
    parent = GUI_TASK_DATA(task->creator)->widget;
    XtSetArg(args[0], XtNtreeParent, parent);
    XtSetValues(widget, args, ONE);
    ModifyTaskWidget(task);
    XtManageChild(widget);
    task = ArrayNext(global_unmanaged_children);
  }
  tcaXawTreeForceLayout (global_tree_widget);
  XMapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
  ArrayDeleteAll(global_unmanaged_children);
}

void CenterTreeName(char *task_name)
{
  Task task;
  BOOLEAN flag;
  
  flag = _center_tree;
  _center_tree = True;
  task = FindTask(task_name, -1, task_NULL);
  CenterTree(task);
  _center_tree = flag;
}

void CenterTree(Task task)
{
  Arg args[2];
  Widget task_widget;
  Position x_node, y_node;
  Position x_tree, y_tree;
  Dimension height, width;
  int i;
  GUI_Task_Additional taskAddData = GUI_TASK_DATA(task);
  
  if (!_center_tree) 
    return;
  task_widget = taskAddData->widget;
  if (task_widget == NULL)
    return;
  if (XtIsManaged(task_widget)) {
    i = 0;
    XtSetArg(args[i], XtNx, &x_node); i++;
    XtSetArg(args[i], XtNy, &y_node); i++;
    XtGetValues(task_widget, args, i);
    
    i = 0;
    XtSetArg(args[i], XtNwidth, &width); i++;
    XtSetArg(args[i], XtNheight, &height); i++;
    XtGetValues(global_porthole, args, i);
    
    i = 0;
    XtSetArg(args[i], XtNx, &x_tree); i++;
    XtSetArg(args[i], XtNy, &y_tree); i++;
    XtGetValues(global_tree_widget, args, i);
    
    /* return if node is visible */
    
    if (x_node + x_tree > 0     &&
	x_node + x_tree < width &&
	y_node + y_tree > 0     &&
	y_node + y_tree < height)
      
      return;
    
    x_tree = - (x_node - width / 2);
    y_tree = - (y_node - height / 2);
    i = 0;
    XtSetArg(args[i], XtNx, x_tree); i++;
    XtSetArg(args[i], XtNy, y_tree); i++;
    
    XtSetValues(global_tree_widget, args, i);
  }
}



static void PushTaskHandler(Widget w, XtPointer client_data,
			    XtPointer call_data)
{    
  Task task;
  
  task = FindTaskWithWidget(mission, w);
  GUI_TASK_DATA(task)->expanded = True;
  XUnmapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget)); 
  ModifyTaskWidget(task);
  tcaXawTreeForceLayout (global_tree_widget);
  XMapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
  /*
     ConnectNodes(task, mission);
     */
  XtRemoveCallback(w, XtNcallback, PushTaskHandler, NULL);
  XtAddCallback(w, XtNcallback, PushTaskHandlerOff, NULL);
}


static void PushTaskHandlerOff(Widget w, XtPointer client_data,
			       XtPointer call_data)
{    
  Task task;
  
  task = FindTaskWithWidget(mission, w);
  GUI_TASK_DATA(task)->expanded = False;
  XUnmapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget)); 
  ModifyTaskWidget(task);
  tcaXawTreeForceLayout (global_tree_widget);
  XMapWindow (XtDisplay(global_tree_widget), XtWindow(global_tree_widget));
  XtRemoveCallback(w, XtNcallback, PushTaskHandlerOff, NULL);
  XtAddCallback(w, XtNcallback, PushTaskHandler, NULL);
}


static BOOLEAN FlagTrue(taskType task_type, _DisplayFlag flag_table[])
{
  int i;
  
  for(i=0;i<MAX_TASK_TYPES;i++) {
    if (flag_table[i].type == task_type)
      return(flag_table[i].flag);
  }
  return(False);
}


/*
   static void ConnectNodes(Task task1, Task task2)
   {
   Arg args[2];
   Position x1,y1;
   Position x2,y2;
   int i;
   Widget task1_widget, task2_widget;
   GC gc;
   
   task1_widget = task1->widget;
   task2_widget = task2->widget;
   if (task1_widget && task2_widget) {
   if (XtIsManaged(task1_widget) &&
   XtIsManaged(task1_widget)) {
   
   XClearWindow( XtDisplay(top_level), XtWindow(top_level));
   
   i = 0;
   XtSetArg(args[i], XtNx, &x1); i++;
   XtSetArg(args[i], XtNy, &y1); i++;
   XtGetValues(task1_widget, args, i);
   
   i = 0;
   XtSetArg(args[i], XtNx, &x2); i++;
   XtSetArg(args[i], XtNy, &y2); i++;
   XtGetValues(task2_widget, args, i);
   
   gc = XtGetGC((Widget) task1_widget, NULL, NULL);
   XDrawLine(XtDisplay(task1_widget),  XtWindow(task1_widget),
   gc, x1, y1, x2, y2);
   }
   }
   }
   
   */

static Task FindTaskWithWidget(Task root_task, Widget widget)
{
  Task task;
  
  if (GUI_TASK_DATA(root_task)->widget == widget)
    return(root_task);
  if (root_task->sibiling) {
    task = FindTaskWithWidget(root_task->sibiling, widget);
    if (task)
      return(task);
  }
  if (root_task->first_child) {
    task = FindTaskWithWidget(root_task->first_child, widget);
    return(task);
  }
  return(NULL);
}


static void ModifyTaskWidget(Task task)
{    
  Arg args[3];
  int i, shape;
  char text[200];
  
  i = 0;
  XtSetArg(args[i], XtNlabel, TaskDescription(task,&(text[0])));         i++;
  if (GUI_TASK_DATA(task)->expanded) {
    XtSetArg(args[i], XtNshapeStyle, XmuShapeRectangle);    i++;
  }
  else {
    switch (task->type) {
    case task_GOAL:
      shape = XmuShapeOval;
      break;
    case task_QUERY:
      shape = XmuShapeEllipse;
      break;
    case task_COMMAND:
    default:
      shape = XmuShapeRectangle;
      break;
    }
    XtSetArg(args[i], XtNshapeStyle, shape);            i++;
  }
#ifdef COLOR_WORKSTATION
  { int pixel;

    pixel = TaskColor(task);
    XtSetArg(args[i], XtNbackground, (Pixel) pixel);        i++;
  }
#endif
  XtSetValues(GUI_TASK_DATA(task)->widget, args, i);
}


/* Creates an unmanaged widget */

static Widget CreateTaskWidget(Task task)
{
  Arg args[4];
  char text[100];
  int shape, i;
  GUI_Task_Additional taskAddData = GUI_TASK_DATA(task);

  switch (task->type) {
  case task_GOAL:
    shape = XmuShapeOval;
    break;
  case task_QUERY:
    shape = XmuShapeEllipse;
    break;
  case task_COMMAND:
  default:
    shape = XmuShapeRectangle;
    break;
  }
  i = 0;
  XtSetArg(args[i], XtNshapeStyle, shape);            i++;
  XtSetArg(args[i], XtNlabel, TaskDescription(task,&(text[0]))); i++;
  XtSetArg(args[i], XtNtreeParent, GUI_TASK_DATA(mission)->widget);
  i++;
  
  taskAddData->widget = XtCreateWidget(text, commandWidgetClass,
				       global_tree_widget, args, i);
  
#ifdef COLOR_WORKSTATION
  { int pixel;

    pixel = TaskColor(task);
    i = 0;
    XtSetArg(args[i], XtNbackground, (Pixel) pixel);        i++;
    XtSetValues(taskAddData->widget, args, i);
  }
#endif
  /* 
   * Amazing hack needed to be able to create the widget without inserting
   * it in the tree: you have to create it hanging of a parent and, after that,
   * declare its parent null and unmanage it. This really doesn't make sense
   * because the widget is never managed, it is not created with 
   * XtCrateManagedWidget ... but it works.
   * I promise that I've been checking a lot of ways of doing that
   * in other way, but ...
   * It's a hard life for Xwindow programmers!
   */
  
  i = 0;
  XtSetArg(args[i], XtNtreeParent, NULL);       i++;
  XtSetValues(taskAddData->widget, args, i);
  XtUnmanageChild(taskAddData->widget);
  XtAddCallback(taskAddData->widget, XtNcallback, PushTaskHandler, NULL);
  return (taskAddData->widget);
}

#ifdef COLOR_WORKSTATION
static Pixel TaskColor(Task task)
{
  Pixel pixel;
  GUI_Task_Additional taskAddData = GUI_TASK_DATA(task);  

  switch (task->state) {
  case task_ON_HOLD:
    pixel = ConvertColor(taskAddData->widget, ON_HOLD_TASK_COLOR);
    break;
  case task_KILLED:
    pixel = ConvertColor(taskAddData->widget, KILLED_TASK_COLOR);
    break;
  case task_PENDING:
    pixel = ConvertColor(taskAddData->widget, PENDING_TASK_COLOR);
    break;
  case task_ACTIVE:
    pixel = ConvertColor(taskAddData->widget, ACTIVE_TASK_COLOR);
    break;
  case task_WAITING:
    pixel = ConvertColor(taskAddData->widget, WAITING_TASK_COLOR);
    break;
  case task_COMPLETED:
    pixel = ConvertColor(taskAddData->widget, COMPLETED_TASK_COLOR);
    break;
  default:
    pixel = ConvertColor(taskAddData->widget, DEFAULT_TASK_COLOR);
    break;
  }
  if (task->state2 == task_WILL_BE_KILLED &&
      task->state != task_KILLED)
    pixel =  ConvertColor(taskAddData->widget, WILL_BE_KILLED_TASK_COLOR);
  return(pixel);
}

static int ConvertColor(Widget w, char *color_name)
{
  XrmValue from, to;
  
  from.size = strlen(color_name) + 1;  
  from.addr = color_name;
  
  /*
   * This conversion accepts a colorname from rgb.txt, or a #rrrgggbbb 
   * rgb color definition, or the special toolkit strings "XtDefaultForeground"
   * and "XtDefaultBackground".
   */
  
  XtConvert(w, XtRString, (XrmValuePtr) &from, XtRPixel, (XrmValuePtr) &to);
  if (to.addr == NULL) {
    return(-1);
  }
  
  return( (int) *((Pixel *) to.addr) );
}
#endif

static void *AddWidgetData (void)
{
  GUI_Task_Additional taskWidgetData;

  taskWidgetData = (GUI_Task_Additional)malloc(sizeof(_GUI_Task_Additional));
  taskWidgetData->expanded = False;
  taskWidgetData->widget = taskWidgetData->expanded_widget = NULL;

  return taskWidgetData;
}

void TreeGuiInitialize (void)
{
  setAdditionalTaskDataFn(AddWidgetData);
  setInsertTaskRootFn(InsertTaskAsRootUI);
  setInsertTaskChildFn(InsertTaskAsChildUI);
  setModifyTaskFn(ModifyTaskUI);
  setModifyTask2Fn(ModifyTask2UI);
}
