/*
 * File: MList.c
 * Author: Domingo Gallardo, CMU
 * Purpose: Implementation of module that manages xwindow lists
 * 
 *
 * REVISION HISTORY
 *
 * $Log: MList.c,v $
 * Revision 1.3  1996/02/10  23:38:02  rich
 * Ifdef out tests rather than comment them out.  Allows comments in the
 * test routines.
 *
 * Revision 1.2  1995/04/07  05:07:35  rich
 * Fixed GNUmakefiles to find the release directory.
 * Fixed problems found by sgi cc compiler.  It would not compile.
 *
 * Revision 1.1  1995/04/05  18:32:11  rich
 * Moved tview files to a subdirectory.
 *
 * Revision 1.3  1995/04/04  19:45:23  rich
 * Added sgi support.
 * Fixed some warnings.
 *
 * Revision 1.2  1995/01/25  00:04:15  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Formatting changes.
 *
 * Revision 1.1  1994/05/31  03:26:03  rich
 * Moved Domingo's tview tool into the main tca module.
 *
 * Revision 1.5  1994/05/27  05:34:40  rich
 * Can now read from file that is being written to.
 * Fixed Menu and button names.
 * Fixed Indentation.
 * Added menu item to change orientation of the tree.
 *
 * Revision 1.4  1993/09/13  04:12:26  domingo
 * Added two set of options to the tree menu:
 *
 * - Erase queries (on/off) : activate/deactivate the deletion of queries
 *      from the task tree. Erase queries by default.
 *
 * - Display constraints (on/off) : activate/deactivate the display of
 *      constraints messages. No display by default.
 *
 * Revision 1.3  1993/09/07  00:24:41  domingo
 * Fixed almost all the warnings
 *
 * Revision 1.2  1993/08/13  02:09:18  domingo
 * Updated function declarations for compilation under gcc in ANSI C (still
 * a lot of warnings to fix).
 * Added automatic logging.
 *
 *
 * Dec 23 1992 - Domingo Gallardo at School of Computer Science, CMU
 * Created.
 *
 */

#include "MList.h"

static void ListCallbackFunc(Widget w, XtPointer client_data,
			     XtPointer call_data);

extern MList MListCreate(Widget parent)
{   
  Arg args[4];
  int i;
  MList list;
  
  list = (MList)malloc((unsigned)sizeof(_MList));
  list->num_items = 0;
  list->allocated_items = 10;
  list->strings = (String *) XtMalloc(sizeof(String) * 10);
  list->funcs = (StrFunction *) XtMalloc(sizeof(VoidFunction) * 10);
  i = 0;
  XtSetArg(args[i], XtNallowVert, True); i++;
  XtSetArg(args[i], XtNforceBars, True); i++;
  list->viewport = XtCreateManagedWidget( "viewport", viewportWidgetClass,
					 parent, args, i);
  
  i = 0;
  XtSetArg(args[i], XtNdefaultColumns, 1); i++;
  XtSetArg(args[i], XtNverticalList, True); i++;
  XtSetArg(args[i], XtNrowSpacing, 1); i++;
  list->widget = XtCreateManagedWidget( "", listWidgetClass,
				       list->viewport, args, i);
  XtAddCallback(list->widget, XtNcallback, ListCallbackFunc, (XtPointer)list);
  return list;
}



/*	Function Name: MListAddItem
 *	Arguments:     list -- my_list
 *                     item -- new string to add to the list
 *                     func_callback -- function to execute when an item is 
 *                                      selected. Must have as parameter an
 *                                      string, that will be the selected item
 *                     update -- boolean. True to display.
 *	Description:   Add a new string to the list widget
 *	Returns:       Nothing
 */

int MListAddItem(MList list, 
		 String item, 
		 StrFunction func_callback,
		 BOOLEAN display)
{
  if (list->num_items == list->allocated_items) {
    list->allocated_items += 10;
    list->strings = (String *) 
      XtRealloc((char *)list->strings, 
		sizeof(String) * list->allocated_items);
    list->funcs = (StrFunction *) 
      XtRealloc((char *)list->funcs,
		sizeof(StrFunction) * list->allocated_items);
  }
  
  /*
   * The value returned by XawDialogGetValueString() does not remain good
   * forever so we must store is somewhere.  This creates a memory leak 
   * since I don't remember these values, and free them later.  I know about
   * it, but it doesn't seem worth fixing in this example.
   */
  
  list->strings[list->num_items] = strdup(item);
  list->funcs[list->num_items++] = func_callback;
  if (display)
    return(MListRefresh(list));
  else
    return 0;
}

static int pos_showing = 0;

extern int MListRefresh(MList list)
{
  int n_items;
  
  n_items = 10;
  pos_showing = list->num_items - n_items;
  if (pos_showing < 0) {
    pos_showing = 0;
    n_items = list->num_items;
  }
  XawListChange(list->widget, list->strings+pos_showing, n_items, 0, True);
  /*
   * The list is moved to show the item just inserted.
   * Undocumented function!! I've found it looking at the
   * source code, after more than three hours of 
   * trying to find a way of doing it.
   */
  XawViewportSetLocation(list->viewport, 0.0, 1.0); 
  return(list->num_items);
}


extern int MListJump(MList list, int num)
{
  int n_items;
  
  if( pos_showing + num >= list->num_items )
    return (list->num_items);
  
  pos_showing = pos_showing + num;
  
  if( pos_showing < 0 )
    pos_showing = 0;
  
  n_items = 10;
  
  if( pos_showing + n_items > list->num_items )
    n_items = list->num_items - pos_showing;
  
  XawListChange(list->widget, list->strings+pos_showing, n_items, 0, True);
  /*
   * The list is moved to show the item just inserted.
   * Undocumented function!! I've found it looking at the
   * source code, after more than three hours of 
   * trying to find a way of doing it.
   */
  XawViewportSetLocation(list->viewport, 0.0, 1.0); 
  return(pos_showing + n_items);
}





/*
 *  PRIVATE FUNCTIONS
 */

/* Client_data is the pointer to the structure MList */

static void ListCallbackFunc(Widget w, XtPointer client_data,
			     XtPointer call_data)
{
  XawListReturnStruct *item = (XawListReturnStruct*)call_data;
  MList my_list = (MList) client_data;
  StrFunction function;
  
  function = my_list->funcs[item->list_index];
  if (function)
    (*function)(item->string);
}



/*
 *  TEST
 */

#ifdef TEST   
#include <X11/Xaw/Command.h>
#include "Memory.h"

void p_function(item)
     String item;
{
  printf("%s\n", item);
}


String fallback_resources[] = {
  NULL,
};


void main(int argc, char **argv)
     
{
  XtAppContext app_con;
  Widget top_level;
  MList list;
  
  top_level =  XtAppInitialize(&app_con, "Xcommand",
			       NULL, ZERO,
			       &argc, argv, fallback_resources, 
			       NULL, ZERO);
  
  list = MListCreate(top_level);
  
  MListAddItem(list, "Pepe", p_function);
  MListAddItem(list, "Antonio", p_function);
  MListAddItem(list, "Juanita", p_function);
  MListAddItem(list, "Domingo", p_function);
  
  XtRealizeWidget(top_level);
  XtAppMainLoop(app_con);
}

#endif
