/*
 * File: MDialog.c
 * Author: Domingo Gallardo, CMU
 * Purpose: Implementation of module that manages dialogs
 * 
 *
 * REVISION HISTORY
 *
 * $Log: MDialog.c,v $
 * Revision 1.3  1996/02/10  23:38:00  rich
 * Ifdef out tests rather than comment them out.  Allows comments in the
 * test routines.
 *
 * Revision 1.2  1995/04/07  05:07:32  rich
 * Fixed GNUmakefiles to find the release directory.
 * Fixed problems found by sgi cc compiler.  It would not compile.
 *
 * Revision 1.1  1995/04/05  18:32:02  rich
 * Moved tview files to a subdirectory.
 *
 * Revision 1.4  1995/03/30  15:50:43  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Do not return pointers to local variables.
 *
 * Revision 1.3  1995/01/25  00:04:11  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Formatting changes.
 *
 * Revision 1.2  1994/11/02  21:38:57  rich
 * Now works for linux machines (i486).
 * Got afs to work on alpha (and hopefully other vendor OS's)
 * Added generic Makefile and asynchronous sender/receiver.
 * Renamed some X11 files and modified routines so we don't get library
 * conflicts.
 *
 * Revision 1.1  1994/05/31  03:25:59  rich
 * Moved Domingo's tview tool into the main tca module.
 *
 * Revision 1.4  1994/05/27  05:34:38  rich
 * Can now read from file that is being written to.
 * Fixed Menu and button names.
 * Fixed Indentation.
 * Added menu item to change orientation of the tree.
 *
 * Revision 1.3  1993/09/07  00:24:37  domingo
 * Fixed almost all the warnings
 *
 * Revision 1.2  1993/08/13  02:09:16  domingo
 * Updated function declarations for compilation under gcc in ANSI C (still
 * a lot of warnings to fix).
 * Added automatic logging.
 *
 *
 * Dec 23 1992 - Domingo Gallardo at School of Computer Science, CMU
 * Created.
 *
 */

#include "MDialog.h"

static void CancelFunction(Widget widget, XtPointer client_data,
			   XtPointer call_data);
static void OkFunction(Widget widget, XtPointer client_data,
		       XtPointer call_data);
static void PopupCentered(XEvent *event, Widget w, XtGrabKind mode);

#define MAX_DIALOGS 10

static MDialog array_dialogs[MAX_DIALOGS];
static int n_dialogs = 0;

void DialogInitStrings(MDialog dialog, String header, 
		       String error_msg, String default_value)
{
  dialog->header = header; 
  dialog->error_msg = error_msg;
  dialog->last_value = default_value;
}


void DialogInitFuncs(MDialog dialog, BOOLEAN (*test)(char *), 
		     DIALOG_FUNCTION_TYPE ok, DIALOG_FUNCTION_TYPE error,
		     DIALOG_FUNCTION_TYPE cancel)
{
  dialog->test_function = test;
  dialog->ok_function = ok;
  dialog->error_function = error;
  dialog->cancel_function = cancel;
  dialog->widget = NULL;
  dialog->popup = NULL;
}


void DialogCreate(MDialog dialog, Widget parent, int type, String name)
{
  Arg args[2];
  int i;
  String popup_name = (name ? name : (strdup("pop_up")));
  WidgetClass widget_class;
  
  switch (type) {
  case TOP_LEVEL:
    widget_class = topLevelShellWidgetClass;
    break;
  case TRANSIENT:
  default:
    widget_class = transientShellWidgetClass;
    break;
  }
  i = 0;
  XtSetArg(args[i], XtNallowShellResize, TRUE);               i++;
  dialog->popup = XtCreatePopupShell(popup_name, widget_class, parent,
				     args, i);
  i = 0;
  XtSetArg(args[i], XtNlabel, dialog->header);               i++;
  if (dialog->last_value != NULL)
    XtSetArg(args[i], XtNvalue, dialog->last_value);
  else
    XtSetArg(args[i], XtNvalue, "");                       
  i++;
  dialog->widget = XtCreateManagedWidget(dialog->header,
					 dialogWidgetClass, dialog->popup, 
					 args, i);
  XawDialogAddButton(dialog->widget, "ok", OkFunction, (XtPointer) dialog);
  XawDialogAddButton(dialog->widget, "cancel", CancelFunction,
		     (XtPointer) dialog);
  if (n_dialogs < MAX_DIALOGS) 
    array_dialogs[n_dialogs++] = dialog;
}

/* use_last_value says if the prompt must shows the value
   entered in the previous fired of the dialog */


void DialogFire(MDialog dialog, BOOLEAN use_last_value)
{
  Arg args[2];
  int i;
  
  if (dialog->widget) {
    i = 0;
    XtSetArg(args[i], XtNlabel, dialog->header);
    i++;
    if (use_last_value && dialog->last_value) {
      XtSetArg(args[i], XtNvalue, dialog->last_value);
      i++;
    }
    XtSetValues(dialog->widget, args, ONE);
    PopupCentered(NULL, dialog->popup, XtGrabNone);
  }
}

void OkDialog(Widget widget, XEvent event, String params, Cardinal num_params)
{
  int i;
  Widget dial_widget;
  
  dial_widget = XtParent(widget);
  for(i=0; i<n_dialogs; i++)
    if (array_dialogs[i]->widget == dial_widget)
      OkFunction(widget, (XtPointer)array_dialogs[i], (XtPointer)NULL);
}


/*
 * Private functions
 */

static void CancelFunction(Widget widget, XtPointer client_data,
			   XtPointer call_data)
{
  String name;
  MDialog dialog;
  Widget popup;
  
  dialog = (MDialog) client_data;
  popup = XtParent(dialog->widget);
  name =  XawDialogGetValueString(dialog->widget);
  dialog->last_value = strdup(name);
  XFlush(XtDisplay(popup));
  XtPopdown(popup);
  if (dialog->cancel_function)
    (*dialog->cancel_function)(name);
}


static void OkFunction(Widget widget, XtPointer client_data,
		       XtPointer call_data)
{
  Arg args[1];
  String name;
  MDialog dialog;
  Widget popup;
  
  dialog = (MDialog) client_data;
  popup = XtParent(dialog->widget);
  name =  XawDialogGetValueString(dialog->widget);
  dialog->last_value = strdup(name);
  if (dialog->test_function)
    if (!(*dialog->test_function)(name)) {
      XtSetArg(args[0], XtNlabel, dialog->error_msg);
      XtSetValues(dialog->widget, args, ONE);
      if (dialog->error_function)
	(*dialog->error_function)(name);
      return;
    }
  XtPopdown(popup);
  XFlush(XtDisplay(popup));
  if (dialog->ok_function)
    (*dialog->ok_function)(name);
}


/*	Function Name: PopupCentered
 *	Description: Pops up the window specified under the location passed
 *                   in the event, or under the cursor.
 *	Arguments: event - the event that we should use (NULL, ButtonPress,
 *                         KeyPress)
 *                 w - widget to popup.
 *                 mode - mode to pop it up in.
 *	Returns: none
 */

static void PopupCentered(XEvent *event, Widget w, XtGrabKind mode)
{
  BOOLEAN get_from_cursor = FALSE;
  Arg args[3];
  Cardinal num_args;
  Dimension width, height, b_width;
  int x, y, max_x, max_y;
  
  XtRealizeWidget(w);
  
  if (event == NULL)
    get_from_cursor = TRUE;
  else {
    switch (event->type) {
    case ButtonPress:
    case ButtonRelease:
      x = event->xbutton.x_root;
      y = event->xbutton.y_root;
      break;
    case KeyPress:
    case KeyRelease:
      x = event->xkey.x_root;
      y = event->xkey.y_root;
      break;
    default:
      get_from_cursor = TRUE;
      break;
    }
  }
  
  if (get_from_cursor) {
    Window root, child;
    int win_x, win_y;
    unsigned int mask;
    
    XQueryPointer(XtDisplay(w), XtWindow(w),
		  &root, &child, &x, &y, &win_x, &win_y, &mask);
  }
  
  num_args = 0;
  XtSetArg(args[num_args], XtNwidth, &width); num_args++;
  XtSetArg(args[num_args], XtNheight, &height); num_args++;
  XtSetArg(args[num_args], XtNborderWidth, &b_width); num_args++;
  XtGetValues(w, args, num_args);
  
  width += 2 * b_width;
  height += 2 * b_width;
  
  x -= ((int) width/2);
  if (x < 0) 
    x = 0;
  if ( x > (max_x = (int) (XtScreen(w)->width - width)) )
    x = max_x;
  
  y -= ( (Position) height/2 );
  if (y < 0) 
    y = 0;
  if ( y > (max_y = (int) (XtScreen(w)->height - height)) )
    y = max_y;
  
  num_args = 0;
  XtSetArg(args[num_args], XtNx, x); num_args++;
  XtSetArg(args[num_args], XtNy, y); num_args++;
  XtSetValues(w, args, num_args);
  
  XtPopup(w, mode);
}


/* TEST MODULE */

#ifdef TEST

#include <X11/Xaw/Command.h>
#include "Memory.h"


void Ok(name)
     String name;
{
  printf("Ok %s\n", name);
}

void Cancel(name)
     String name;
{
  printf("Cancel %s\n", name);
}

BOOLEAN Test(name)
     String name;
{
  return(StringEqual(name, "Domingo"));
}

String fallback_resources[] = {
  NULL,
};


static void StartDialog(w, client_data, call_data)
     Widget w;
     XtPointer call_data, client_data;
{
  static MDialog dialog = NULL;
  
  if (!dialog) {
    dialog = _new(_MDialog);
    DialogInitStrings(dialog, "Dialog test",
		      "Who's the boss?",
		      "Wrong. Try again");
    DialogInitFuncs(dialog, Test, Ok, NULL, Cancel);
    DialogCreate(dialog, w, TOP_LEVEL);
  }
  DialogFire(dialog, True);
}



void main(int argc, char **argv)
{
  XtAppContext app_con;
  Widget top_level, command;;
  
  top_level =  XtAppInitialize(&app_con, "Xcommand",
			       NULL, ZERO,
			       &argc, argv, fallback_resources, 
			       NULL, ZERO);
  
  command = XtCreateManagedWidget("command", commandWidgetClass, top_level,
				  NULL, ZERO);
  XtAddCallback(command, XtNcallback, StartDialog, NULL);
  
  XtRealizeWidget(top_level);
  XtAppMainLoop(app_con);
}


#endif
