/*****************************************************************************
 * PROJECT: Xavier
 *
 * (c) Copyright 1993 Richard Goodwin & Joseph O'Sullivan. All rights reserved.
 *
 * FILE: xfCallbacks.c
 *
 * ABSTRACT:
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/xfCallbacks.c,v $
 * $Revision: 1.5 $
 * $Date: 1996/07/10 17:19:50 $
 * $Author: reids $
 *
 * REVISION HISTORY:
 *
 * $Log: xfCallbacks.c,v $
 * Revision 1.5  1996/07/10  17:19:50  reids
 * Replaced the "Destroy->" menu (which didn't really work right) with a
 *   "Choose->" menu for picking which process to view (Can do "destroy" with
 *   a "choose" and a "kill").
 *
 * Revision 1.4  1996/06/28  14:07:42  reids
 * Fixed quite a few bugs -- with graphics, interaction with script, and
 *   killing processes
 *
 * Revision 1.3  1996/03/29  15:57:43  reids
 * Consolidated the common code between xCallbacks and xfCallbacks, and
 *   xRunConsole and xfRunConsole.
 * Added a way to add macro definitions to resource files ("define: <x> <y>").
 * Fixed a bug that was causing the xfRunConsole to crash when a task was
 *   restarted.
 *
 ****************************************************************************/
#include "tca/libc.h"
#include "tca/devUtils.h"
#include "tca/stdinDev.h"

#include "forms.h"
#include "xfConsole.h"

#include "xfRunConsole.h"
#include "callbacks.h"
#include "messageHandle.h"
#include "messages.h"
#include "script.h"

/* Forward references */
static void DoProcess(FL_OBJECT *obj, long index);

static void prevHistory(FL_OBJECT *obj, long foo)
{
  if (!consoleProc[currentProc].history.pos)
    return;

  fl_set_input(xfConsole->inputForm, 
	       historyPrev(&(consoleProc[currentProc].history)));
}

static void nextHistory(FL_OBJECT *obj, long foo)
{
  if (!consoleProc[currentProc].history.pos)
    return;

  fl_set_input(xfConsole->inputForm,
	       historyPrev(&(consoleProc[currentProc].history)));
}

void inputCallback(FL_OBJECT *obj, long foo)
{
  static char userString[1025];
  static char current[1025], new[1025];
  int line; 

  strcpy(userString, fl_get_input(xfConsole->inputForm));
  fl_set_input(xfConsole->inputForm,"");

  historyAddCommand(&(consoleProc[currentProc].history), userString);

  /* Send to nanny... */
  userString[strlen(userString)+1]='\0';
  userString[strlen(userString)]='\n';

  EditAppend(userString, strlen(userString));

  runCommandProcess(consoleProc[currentProc].name, userString);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void closeConsoleCallback(FL_OBJECT *obj, long foo)
{
  fl_hide_form(xfConsole->xfConsole);
  safeExit();
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void reloadCallback(FL_OBJECT *obj, long foo)
{
  const char *fname, *fname2;
  char *restName;
  static char scriptDirectory[100] = "\0";

  fname = fl_show_file_selector("Resource File To Load", scriptDirectory,
				"*.rc", "Simulator.rc");

  if (fname) {
    /* Update the directory name */
    restName = strrchr(fname, '/');
    if (restName) {
      restName[0] = '\0';
      strcpy(scriptDirectory, fname);
      restName[0] = '/';
    }

    reloadResource(fname);
  }
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void syncCallback(FL_OBJECT *obj, long foo)
{
  syncWithNanny();
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void helpCallback(FL_OBJECT *obj, long foo)
{
  fl_show_message("Help is not written as yet", 
		  "Please refer to the README instead",
		  "");
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void filterCallback(FL_OBJECT *obj, long foo)
{
  char *newFilter;

  newFilter = (char *)fl_show_input("Enter string to filter upon:",
				    (filterString != NULL ? filterString : ""));
  setFilterString(newFilter);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void messageClearBufferCallback(FL_OBJECT *obj, long foo)
{
  fl_clear_browser(xfConsole->messageBrowser);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void killCallback(FL_OBJECT *obj, long foo)
{
  if ((currentProc != -1) && 
      fl_show_question("The kill operation can't be undone for", 
		       consoleProc[currentProc].name,
		       "Do you wish to proceed?"))
    killCurrentProcess();
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void restartCallback(FL_OBJECT *obj, long foo)
{
  if ((currentProc != -1) && 
      fl_show_question("The restart operation can't be undone for", 
		       consoleProc[currentProc].name,
		       "Do you wish to proceed?")) 
    restartCurrentProcess();
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void breakCallback(FL_OBJECT *obj, long foo)
{
  sendBreak();
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void processClearBufferCallback(FL_OBJECT *obj, long foo)
{
  fl_clear_browser(xfConsole->processBrowser);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void runCallback(FL_OBJECT *obj, long foo)
{
  if (!fl_get_menu_maxitems(obj)) {
    getProcessList("Run");
  } else if (fl_get_menu(obj) != -1) {
    runNewProcess((char *)fl_get_menu_text(obj));
  }
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void destroyCallback(FL_OBJECT *obj, long foo)
{
  if (!fl_get_menu_maxitems(obj)) {
    getProcessList("Destroy");
  } else if (fl_get_menu(obj) != -1) {
    destroyProcess(fl_get_menu_text(obj));
  }
}

void chooseCallback(FL_OBJECT *obj, long foo)
{
  if (!fl_get_menu_maxitems(obj)) {
    getProcessList("Choose");
  } else if (fl_get_menu(obj) != -1) {
    printf("Choosing %s\n", fl_get_menu_text(obj));
    DoProcess(NULL, convertProcessNameToId(fl_get_menu_text(obj)));
  }
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void shutdownCallback(FL_OBJECT *obj, long foo)
{
  if( !nConsoleProcs ||
      fl_show_question("You have running processes.", 
		       "Shutdown will kill them", 
		       "Do you wish to proceed?")) {
    fl_hide_form(xfConsole->xfConsole);
    shutdownProcess();
  }
}

/* 
 * 
 */
void DoProcess(FL_OBJECT *obj, long index)
{
  int i;

  for (i=0; i<nConsoleProcs; i++)
    if (fl_get_button(consoleProc[i].graphics->button)) {
      fl_set_button(consoleProc[i].graphics->button, 0);
      fl_activate_object(consoleProc[i].graphics->button);
    }
     
  currentProc = index;
  fl_set_object_color(consoleProc[index].graphics->button,FL_COL1,FL_YELLOW);
  fl_set_button(consoleProc[index].graphics->button, 1);

  fl_activate_object(xfConsole->killButton);
  fl_activate_object(xfConsole->restartButton);
  fl_activate_object(xfConsole->breakButton);
  fl_activate_object(xfConsole->inputForm);

  fl_deactivate_object(consoleProc[index].graphics->button);

  historyFreeProcess(&(consoleProc[currentProc].history));
  fl_clear_browser(xfConsole->processBrowser);
  fl_freeze_form(xfConsole->xfConsole);

  startFlushing(consoleProc[currentProc].name);
}

void EditAppend(char *buffer, int n)
{
  const char *currentLine;
  char newLine[200];
  int lineNum;

  lineNum = fl_get_browser_maxline(xfConsole->processBrowser);
  currentLine = fl_get_browser_line(xfConsole->processBrowser, lineNum);

  if (!currentLine || currentLine[strlen(currentLine)-1] == '\n') {
    /* Need to do it this crufty way, because "addto" does not preserve <cr> */
    fl_addto_browser(xfConsole->processBrowser, " ");
    fl_replace_browser_line(xfConsole->processBrowser, lineNum+1, buffer);
  } else {
    strcpy(newLine, currentLine);
    strcat(newLine, buffer);
    fl_replace_browser_line(xfConsole->processBrowser, lineNum, newLine);
  }
}

static void addProcessButton1 (int index, const char *name)
{
  FL_OBJECT *obj;

  consoleProc[index].graphics->button = 
    obj = fl_add_lightbutton(FL_PUSH_BUTTON, 5+60*index, 410, 60, 30, name);
  fl_set_object_boxtype(obj, FL_SHADOW_BOX);
  fl_set_object_lsize(obj, FL_TINY_SIZE);
  fl_set_object_callback(obj, DoProcess, index);
}

void addProcessButton (const char *name)
{
  fl_addto_form(xfConsole->xfConsole);
  addProcessButton1(nConsoleProcs, name);
  fl_end_form();
}

void destroyProcessButton (int index)
{
  fl_delete_object(consoleProc[index].graphics->button);
  fl_free_object(consoleProc[index].graphics->button);
  consoleProc[index].graphics->button = NULL;
}

void resetProcessButtons (int index)
{
  int i;

  fl_addto_form(xfConsole->xfConsole);
  for (i=index; i<nConsoleProcs; i++) {
    destroyProcessButton(i);
    addProcessButton1(i, consoleProc[i].name);
  }
  fl_end_form();

  fl_clear_browser(xfConsole->processBrowser);

  fl_deactivate_object(xfConsole->killButton);
  fl_deactivate_object(xfConsole->restartButton);
  fl_deactivate_object(xfConsole->breakButton);
  fl_deactivate_object(xfConsole->inputForm);
}

BOOLEAN highlightProcess(char *name)
{
  int ntmp;

  /* Figure out which of consoleProcs we wish to reset */
  if ((ntmp = convertProcessNameToId(name)) == -1) return;

  fl_set_object_color(consoleProc[ntmp].graphics->button,FL_COL1,FL_RED);
  fl_set_button(consoleProc[ntmp].graphics->button, 0);
}

/* Lowlevel functions */
/*	Function Name: Feep
 *	Description: feeps the bell.
 *	Arguments: none.
 *	Returns: none.
 */

void Feep(void)
{
  /* XBell(CurDpy, 0); */
}

void WFlush(void)
{
  /* XFlush(CurDpy); */
}
/*
 * DESCRIPTION:
 *  prints string to message browser, which is forced to the bottom...
 * INPUTS:
 *
 * OUTPUTS:
 */
void MessagePrintf (const char * str)
{
  fl_addto_browser(xfConsole->messageBrowser, str);
}

static void setScrollPosition (void)
{
  fl_set_slider_value(xfConsole->statusSlider, getPosition());
}

static void updateStatus(Pointer call_back, Pointer client_data)
{
  updatePosition();
  setScrollPosition();
  ProcessXFHandler(call_back, client_data);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void activateStatus (const char *message)
{
  fl_set_object_label(xfConsole->statusLabel, message);

  initPosition();
  setScrollPosition();

  devStartPolling(stdin_device, &pollTime, (Handler) updateStatus, NULL);
}

/* DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void deactivateStatus (const char *message)
{
  fl_set_object_label(xfConsole->statusLabel, message);

  devStopPolling(stdin_device);
  devStartPolling(stdin_device, &pollTime, (Handler)ProcessXFHandler, NULL);

  initPosition();
  setScrollPosition();
}

/* Dynamic menu handling */
/* DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void attachProgramMenu(void)
{
  char *buf, program[128];
  int i;

  fl_clear_menu(xfConsole->runMenu);
#if 0
  fl_clear_menu(xfConsole->destroyMenu);
#endif
  fl_clear_menu(xfConsole->chooseMenu);
  buf = messageExtractKey(possiblePrograms, program);
  for (i=0; i<numPossiblePrograms; i++) 
    {
      fl_addto_menu(xfConsole->runMenu, program);
#if 0
      fl_addto_menu(xfConsole->destroyMenu, program);
#endif
      if (convertProcessNameToId(program) != -1) {
	fl_addto_menu(xfConsole->chooseMenu, program);
      }
      buf = messageExtractKey(buf, program);
    }
}

/*****************************************************************************
 ****************************************************************************/

void setSuspendResume(const char *title)
{
   /* reids script stuff. Xforms could do this nicer - but don't know
    * where reid is going with this all.
    */
  fl_clear_menu(xfConsole->scriptMenu);

  fl_addto_menu(xfConsole->scriptMenu, "Run...");
  fl_addto_menu(xfConsole->scriptMenu, "Rerun");
  fl_addto_menu(xfConsole->scriptMenu, title);
  fl_addto_menu(xfConsole->scriptMenu, "Stop");
}

static void rerunScriptCallback (FL_OBJECT *obj, long foo)
{
  rerunScript();
}

/* Used for both suspend and resume */
static void suspendScriptCallback (FL_OBJECT *obj, long index)
{
  suspendResume(fl_get_menu_text(obj));
}

static void runScriptCallback(FL_OBJECT *obj, long index)
{
  const char *fname;
  char *restName;
  static char scriptDirectory[100] = "\0";

  fname = fl_show_file_selector("Script File To Load", scriptDirectory,
				"*.script", "foo.script");
  if (fname) {
    /* Update the directory name */
    restName = strrchr(fname, '/');
    if (restName) {
      restName[0] = '\0';
      strcpy(scriptDirectory, fname);
      restName[0] = '/';
    }

    if (!scriptLoad((char *)fname)) {
      fl_show_alert("File not found or syntax error in parsing", "", "", 1);
    }
  }
}

static void stopScriptCallback (FL_OBJECT *obj, long index)
{
  stopScript();
}

void scriptCallback(FL_OBJECT *obj, long foo)
{
  static char   buf[128];

  if (!fl_get_menu_maxitems(obj))
    {
      /* If we have no items, that means we have no nanny connect yet...  */
      Feep();
      MessagePrintf("[CONSOLE] script menu weirdness...\n");
    }
  else
    {
      if (fl_get_menu(obj) != -1)
	{
	  if (!strcmp(fl_get_menu_text(obj), SUSPEND)) 
	    suspendScriptCallback(obj,foo);
	  else if (!strcmp(fl_get_menu_text(obj), RESUME)) 
	    suspendScriptCallback(obj,foo);
	  else if (!strcmp(fl_get_menu_text(obj), "Run...")) 
	    runScriptCallback(obj,foo);
	  else if (!strcmp(fl_get_menu_text(obj), "Rerun")) 
	    rerunScriptCallback(obj,foo);
	  else if (!strcmp(fl_get_menu_text(obj), "Stop")) 
	    stopScriptCallback(obj,foo);
	  else {
	    MessagePrintf("[CONSOLE] script menu weirdness (2)...\n");
	  }
	}
    }
}

void ProcessXFHandler(Pointer call_back, Pointer client_data)
{
  FL_OBJECT *obj;
  char   buf[DEFAULT_LINE_LENGTH];
  static int ping_timeout=0;

  /* Check that we are still talking... */
  if (ping_timeout++ > 20) {
    ping_timeout = 0;
    syncWithNanny();
  }

  if (flushing) {
    if (flushing_count++ > 20) {
      flushing=FALSE; flushing_count = 0;
      unfreeze();
    }
  }

  obj = fl_check_forms();
}


void unfreeze (void)
{
  fl_unfreeze_form(xfConsole->xfConsole);
}

struct _CONSOLE_GRAPHICS *createConsoleGraphics (void)
{
  return (CONSOLE_GRAPHICS_PTR)malloc(sizeof(CONSOLE_GRAPHICS_TYPE));
}

void clearConsoleGraphics (consoleProcPtr consoleProc)
{
  bzero(consoleProc->graphics, sizeof(CONSOLE_GRAPHICS_TYPE));
}

void copyConsoleGraphics (consoleProcPtr from, consoleProcPtr to)
{
  *to->graphics = *from->graphics;
}
