/*****************************************************************************
 * PROJECT: TCA
 *
 * (c) Copyright 1996 Reid Simmons. All rights reserved.
 *
 * FILE: scriptParse.c
 *
 * ABSTRACT: Parse a script file and create the appropriate data structures.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/scriptParse.c,v $
 * $Revision: 1.2 $
 * $Date: 1996/02/05 15:57:04 $
 * $Author: reids $
 *
 * REVISION HISTORY:
 *
 * $Log: scriptParse.c,v $
 * Revision 1.2  1996/02/05  15:57:04  reids
 * Added scripting capabilities to nanny -- using lex/bison to parse script
 *  files according to the script.bnf standard.
 * Integrated support for running/stopping/suspending scripts into the
 *  runConsole window.
 *
 * Revision 1.1  1996/01/22  21:31:32  reids
 * Fixed the way stdout is handled, using pseudo-terminals, so that it preserves
 *   the line-buffering mode of the real TTY.
 * Added support for running scripts (not really integrated, yet, but initial
 *   tests are working).
 *
 ****************************************************************/

#include "tca/libc.h"
#include "tca/basics.h"
#include "tca/timeUtils.h"

#include "nannyCommon.h"
#include "script.h"
#include "scriptParse.h"

/* For YACC/Bison */
extern FILE *yyin;
extern int yydebug;
extern int yyparse(void);

/*****************************************************************
 *  GLOBAL VARIABLES 
 ****************************************************************/
SCRIPT_LIST_PTR currentScript = NULL;

/* Forward References */
static void printScript (FILE *file, SCRIPT_LIST_PTR script, int indent);

static SCRIPT_STATEMENT_PTR scriptCreateStatement (SCRIPT_STATEMENT_ENUM type)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = (SCRIPT_STATEMENT_PTR)malloc(sizeof(SCRIPT_STATEMENT_TYPE));
  statement->type = type;
  statement->continual = FALSE;
  statement->process = statement->line = "";
  scriptInitTime(&statement->time);
  statement->subStatements = NULL;

  return statement;
}

SCRIPT_STATEMENT_PTR scriptCreateAction (SCRIPT_STATEMENT_ENUM type,
					 char *process)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = scriptCreateStatement(type);
  statement->process = process;

  return statement;
}

SCRIPT_STATEMENT_PTR scriptCreateSend (char *process, char *line)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = scriptCreateStatement(SendAction);
  statement->process = process;
  statement->line = line;

  return statement;
}

SCRIPT_STATEMENT_PTR scriptCreateWhenReady (char *process)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = scriptCreateStatement(WhenReadyConditional);
  statement->process = process;

  return statement;
}

SCRIPT_STATEMENT_PTR scriptCreateWhen (char *process, BOOLEAN continual,
				       char* line)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = scriptCreateStatement(WhenConditional);
  statement->process = process;
  statement->line = line;
  statement->continual = continual;

  return statement;
}

SCRIPT_STATEMENT_PTR scriptCreateAt (BOOLEAN continual, SCRIPT_TIME_TYPE time)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = scriptCreateStatement(AtConditional);
  statement->continual = continual;
  statement->time = time;

  return statement;
}

void scriptInitTime(SCRIPT_TIME_PTR time)
{
  time->useRelative = TRUE;
  time->relative = time->alarm = TIME_MAX;
  time->absolute.mth = time->absolute.day = -1;
  time->absolute.hr = time->absolute.min = time->absolute.sec = -1;
}

time_t relTimeFromAbs(ABS_TIME_PTR time)
{
  time_t relTime = 0;;

  if (TIME_IS_SET(time->sec)) relTime += time->sec;
  if (TIME_IS_SET(time->min)) relTime += time->min*60;
  if (TIME_IS_SET(time->hr))  relTime += time->hr*60*60;

  return relTime;
}
#if 0
static SCRIPT_STATEMENT_PTR scriptCreateRelAt (BOOLEAN continual, time_t delay)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = scriptCreateStatement(AtConditional);
  statement->continual = continual;
  statement->time.useRelative = TRUE;
  statement->time.relative = delay;

  return statement;
}

static SCRIPT_STATEMENT_PTR scriptCreateAbsAt (BOOLEAN continual, 
					       short mth, short day,
					       short hr, short min, short sec)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = scriptCreateStatement(AtConditional);
  statement->continual = continual;
  statement->time.useRelative = FALSE;
  statement->time.absolute.mth = mth;
  statement->time.absolute.day = day;
  statement->time.absolute.hr = hr;
  statement->time.absolute.min = min;
  statement->time.absolute.sec = sec;

  return statement;
}
#endif

SCRIPT_LIST_PTR scriptParse (FILE *scriptFile)
{
  yyin = scriptFile;

  if (yyparse() == 0) {
    return currentScript;
  } else {
    /* if (currentScript) freeScript(currentScript);*/
    return NULL;
  }
}

void setLocalTimeFromAbsTime(struct tm *localTime, ABS_TIME_PTR absTime)
{
  if (TIME_IS_SET(absTime->mth)) localTime->tm_mon = absTime->mth;
  if (TIME_IS_SET(absTime->day)) localTime->tm_mday = absTime->day;
  if (TIME_IS_SET(absTime->hr))  localTime->tm_hour = absTime->hr;
  if (TIME_IS_SET(absTime->min)) localTime->tm_min = absTime->min;
  if (TIME_IS_SET(absTime->sec)) localTime->tm_sec = absTime->sec;
}

static void printScriptTime (FILE *file, SCRIPT_TIME_PTR time, int indent)
{
  char buf[DEFAULT_LINE_LENGTH];
  struct tm *localTime;
  time_t zeroTime = 0;

  if (time->useRelative) {
    localTime = gmtime(&time->relative);
    strftime(buf, DEFAULT_LINE_LENGTH, "+%k:%M:%S", localTime);
  } else {
    localTime = gmtime(&zeroTime);
    setLocalTimeFromAbsTime(localTime, &time->absolute);
    if (TIME_IS_SET(time->absolute.mth) || TIME_IS_SET(time->absolute.day)) {
      strftime(buf, DEFAULT_LINE_LENGTH, "%h-%d %k:%M:%S", localTime);
    } else {
      strftime(buf, DEFAULT_LINE_LENGTH, "%k:%M:%S", localTime);
    }
  }
  fprintf(file, buf);
}

static void printIndent (FILE *file, int indent)
{
  int i;

  for (i=0; i<indent; i++) fprintf(file, " ");
}

static void printSubElements (FILE *file, SCRIPT_STATEMENT_PTR statement,
			      int indent)
{
  int num;

  num = listLength(statement->subStatements);
  if (num > 0) {
    if (num > 1) { printIndent(file, indent); fprintf(file, "BEGIN:\n"); }
    printScript(file, statement->subStatements,  indent+INDENT_INCREMENT);
    if (num > 1) { printIndent(file, indent); fprintf(file, "END:\n"); }
  }
}

static void printStatement (FILE *file, SCRIPT_STATEMENT_PTR statement,
			    int indent)
{
  printIndent(file, indent);
  switch (statement->type) {
  case SendAction: 
    fprintf(file, "SEND: %s \"%s\"\n", statement->process, statement->line);
    break;
  case RunAction:
    fprintf(file, "RUN: %s\n", statement->process); break;
  case KillAction:
    fprintf(file, "KILL: %s\n", statement->process); break;
  case SuspendAction:
    fprintf(file, "SUSPEND: %s\n", statement->process); break;
  case ContinueAction:
    fprintf(file, "CONTINUE: %s\n", statement->process); break;
  case WhenReadyConditional:
    fprintf(file, "WHENREADY: %s\n", statement->process); 
    printSubElements(file, statement, indent);
    break;
  case WhenConditional:
    fprintf(file, "WHEN%s: %s \"%s\"\n", (statement->continual ? "EVERY" : ""),
	    statement->process, statement->line);
    printSubElements(file, statement, indent);
    break;
  case AtConditional:
    fprintf(file, "AT%s: ", (statement->continual ? "EVERY" : ""));
    printScriptTime(file, &statement->time, indent);
    fprintf(file, "\n");
    printSubElements(file, statement, indent);
    break;
  default:;
  }
}

static void printScript (FILE *file, SCRIPT_LIST_PTR script, int indent)
{
  SCRIPT_STATEMENT_PTR statement;

  statement = (SCRIPT_STATEMENT_PTR)listFirst(script);
  while (statement) {
    printStatement(file, statement, indent);
    statement = (SCRIPT_STATEMENT_PTR)listNext(script);
  }
}

void scriptPrint (SCRIPT_LIST_PTR script)
{
  printf("\n");
  printScript(stdout, script, 0);
}

#ifdef TEST
void main (int argc, char **argv)
{
  SCRIPT_LIST_PTR script;

  /* Need (some of) this to initialize the list package */
  { extern void tcaModuleInitialize(void);
    tcaModuleInitialize();
  }

  yydebug = 0;

  script = scriptParse(fopen(argv[1], "r"));
  if (script) {
    scriptPrint(script);
  } else {
    fprintf(stderr, "Script parsing failed\n");
  }
}
#endif
