/*****************************************************************************
 * PROJECT: TCA
 *
 * (c) Copyright 1996 Reid Simmons. All rights reserved.
 *
 * FILE: script.c
 *
 * ABSTRACT: Run a script file.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/script.c,v $
 * $Revision: 1.7 $
 * $Date: 1996/07/29 05:03:18 $
 * $Author: josullvn $
 *
 * REVISION HISTORY:
 *
 * $Log: script.c,v $
 * Revision 1.7  1996/07/29  05:03:18  josullvn
 * Bloody hell. This commiting is a bit different than under Xavier. Short
 * story is cleaned up some purify bugs, and also made changes to nanny
 * which should make it a bit better - Improving performance over multiple
 * machines, explict quietening of nondisplayed processes, replacing of
 * runConsole with xfMiniConsole, which is multithreaded, vt102 compilant,
 * adds a uniform emacs-like command line editing feature, better on small
 * screens and otherwise fab.
 *
 * Revision 1.6  1996/06/28  14:07:37  reids
 * Fixed quite a few bugs -- with graphics, interaction with script, and
 *   killing processes
 *
 * Revision 1.5  1996/03/29  15:57:30  reids
 * Consolidated the common code between xCallbacks and xfCallbacks, and
 *   xRunConsole and xfRunConsole.
 * Added a way to add macro definitions to resource files ("define: <x> <y>").
 * Fixed a bug that was causing the xfRunConsole to crash when a task was
 *   restarted.
 *
 * Revision 1.4  1996/02/13  02:36:41  rich
 * Fixing corrupted script.c
 *
 * Revision 1.2  1996/02/05  15:56:58  reids
 * Added scripting capabilities to nanny -- using lex/bison to parse script
 *  files according to the script.bnf standard.
 * Integrated support for running/stopping/suspending scripts into the
 *  runConsole window.
 *
 * Revision 1.1  1996/01/22  21:31:27  reids
 * Fixed the way stdout is handled, using pseudo-terminals, so that it preserves
 *   the line-buffering mode of the real TTY.
 * Added support for running scripts (not really integrated, yet, but initial
 *   tests are working).
 *
 ****************************************************************/

#include "tca/libc.h"
#include "tca/basics.h"
#include "tca/timeUtils.h"
#include "tca/devUtils.h"

#include "nannyCommon.h"
#include "nannyDev.h"
#include "messages.h"
#include "runConsole.h"
#include "callbacks.h"
#include "script.h"
#include "scriptParse.h"

static SCRIPT_LIST_PTR activeListG = NULL;
static SCRIPT_LIST_PTR suspendedListG = NULL;
static SCRIPT_LIST_PTR loadedScriptG = NULL;

static struct timeval currentAlarmTime = {TIME_MAX, 0};

/* Forward References */
static void processConditional (SCRIPT_STATEMENT_PTR statement);
static void freeScript (SCRIPT_LIST_PTR *scriptHandle);

static time_t currentSeconds (void)
{
  struct timeval time;

  gettimeofday(&time, NULL);
  return time.tv_sec;
}

static time_t nextTime(time_t currentTime, SCRIPT_TIME_PTR scriptTime)
{
  struct tm *localTime;
  time_t nextTime;

  if (scriptTime->useRelative) {
    return scriptTime->relative + currentTime;
  } else {
    localTime = localtime(&currentTime);
    setLocalTimeFromAbsTime(localTime, &scriptTime->absolute);
#ifdef __linux__
    nextTime = mktime (localTime); 	/* no timelocal () on linux */
#else
    nextTime = timelocal(localTime);
#endif
    if (nextTime <= currentTime) {
      /* Increment time to the *next* matching absolute time (e.g., next
	 year if the month is set, next month if the day is set, etc.) */
      if (TIME_IS_SET(scriptTime->absolute.mth)) {
	localTime->tm_year++;
#ifdef __linux__
	nextTime = mktime (localTime); 	/* no timelocal () on linux */
#else
	nextTime = timelocal(localTime);
#endif
      } else if (TIME_IS_SET(scriptTime->absolute.day)) {
	localTime->tm_mon++;
	if (localTime->tm_mon > 11) {
	  localTime->tm_mon = 0;
	  localTime->tm_year++;
	}
#ifdef __linux__
	nextTime = mktime (localTime); 	/* no timelocal () on linux */
#else
	nextTime = timelocal(localTime);
#endif
	nextTime = timelocal(localTime);
      } else if (TIME_IS_SET(scriptTime->absolute.day)) {
	nextTime += 24*60*60;
      } else if (TIME_IS_SET(scriptTime->absolute.min)) {
	nextTime += 60*60;
      } else if (TIME_IS_SET(scriptTime->absolute.sec)) {
	nextTime += 60;
      }
    }
    return nextTime;
  }
}

static void updateStatementAndAlarmTimes(SCRIPT_TIME_PTR scriptTime, 
					 TIME_ITER_PTR timeData)
{
  scriptTime->alarm = nextTime(timeData->current, scriptTime);
  if (timeData->alarm > scriptTime->alarm) {
    timeData->alarm = scriptTime->alarm;
  }
}

static void clearAlarm (void)
{
  currentAlarmTime.tv_sec = TIME_MAX;
  devSetAlarm(theLocalDevice()->dev, &currentAlarmTime, (Handler)NULL, NULL);
}

static void resetAlarm (time_t alarmTime)
{
  if (alarmTime < currentAlarmTime.tv_sec) {
    currentAlarmTime.tv_sec = alarmTime;
    devSetAlarm(theLocalDevice()->dev, &currentAlarmTime,
		(Handler)processAtConditionals, NULL);
  }
}

static void initTimeData (TIME_ITER_PTR timeData)
{
  timeData->current = currentSeconds();
  timeData->alarm = TIME_MAX;
}

static BOOLEAN activateStatement(TIME_ITER_PTR timeData,
				 SCRIPT_STATEMENT_PTR statement)
{
  static char buf[DEFAULT_LINE_LENGTH];
  int ntmp;

  switch (statement->type) {
  case SendAction: 
    sprintf(buf, "%s\n", statement->line);
    runCommandProcess(statement->process, buf);
    if (convertProcessNameToId(statement->process) == currentProc) {
      EditAppend (buf, strlen(buf));
      WFlush();
    }
    break;

  case RunAction:      runNewProcess(statement->process); break;
  case KillAction:     runKillProcess(statement->process); break;
  case SuspendAction:  runSuspendProcess(statement->process); break;
  case ContinueAction: runContinueProcess(statement->process); break;

  case WhenReadyConditional: 
    /* If process is in process table, it has been announced -- is that
       always the same as being ready? */
    if ((ntmp = convertProcessNameToId(statement->process)) != -1 &&
	consoleProc[ntmp].ready) {
      processConditional(statement);
    } else {
      ACTIVATE(statement);
    }
    break;

  case WhenConditional: 
    ACTIVATE(statement);
    /* Make sure loudnes of process is VERBOSE */
    messageMakeVerbose(buf, statement->process);
    (void)safePassLocalMessage(buf);
    break;

  case AtConditional:
    ACTIVATE(statement);
    updateStatementAndAlarmTimes(&statement->time, timeData);
    break;

  default: WARNING2("Unknown statement type %d\n", statement->type);
  }
  return TRUE;
}

static void activateStatements (SCRIPT_LIST_PTR script)
{
  TIME_ITER_TYPE timeData;

  initTimeData(&timeData);
  ITERATE_SCRIPT_LIST(activateStatement, &timeData, script);
  resetAlarm(timeData.alarm);
}

static void processConditional (SCRIPT_STATEMENT_PTR statement)
{
  if (!statement->continual) DEACTIVATE(statement);
  if (statement->subStatements) activateStatements(statement->subStatements);
}

static BOOLEAN atHandler (TIME_ITER_PTR timeData, 
			  SCRIPT_STATEMENT_PTR statement)
{
  if (statement->type == AtConditional && 
      statement->time.alarm <= timeData->current) {
    clearAlarm();
    processConditional(statement);
    updateStatementAndAlarmTimes(&statement->time, timeData);
  }
  return TRUE;
}

void processAtConditionals (void)
{
  TIME_ITER_TYPE timeData;

  initTimeData(&timeData);
  ITERATE_ACTIVE_LIST(atHandler, &timeData);
  resetAlarm(timeData.alarm);
}

static BOOLEAN alarmTimeFromStatement (TIME_ITER_PTR timeData, 
				       SCRIPT_STATEMENT_PTR statement)
{
  if (statement->type == AtConditional) {
    if (timeData->alarm > statement->time.alarm) {
      timeData->alarm = statement->time.alarm;
    }
  }
  return TRUE;
}

void setAlarmTimeFromList (SCRIPT_LIST_PTR script)
{
  TIME_ITER_TYPE timeData;

  initTimeData(&timeData);
  ITERATE_SCRIPT_LIST(alarmTimeFromStatement, &timeData, script);
  resetAlarm(timeData.alarm);
}

static BOOLEAN whenHandler (WHEN_ITER_PTR whenData, 
			    SCRIPT_STATEMENT_PTR statement)
{
  if (statement->type == WhenConditional && 
      !strcmp(statement->process, whenData->process) &&
      strstr(whenData->line, statement->line)) {
    processConditional(statement);
  }
  return TRUE;
}

void processWhenConditionals (char *process, char *line)
{
  WHEN_ITER_TYPE whenData;

  whenData.process = process;   whenData.line = line; 
  ITERATE_ACTIVE_LIST(whenHandler, &whenData);
}

static BOOLEAN whenReadyHandler (WHEN_ITER_PTR whenData, 
				 SCRIPT_STATEMENT_PTR statement)
{
  if (statement->type == WhenReadyConditional && 
      !strcmp(statement->process, whenData->process)) {
    processConditional(statement);
  }
  return TRUE;
}

void processWhenReadyConditionals (char *process)
{
  WHEN_ITER_TYPE whenData;

  whenData.process = process;   whenData.line = NULL;
  ITERATE_ACTIVE_LIST(whenReadyHandler, &whenData);
}

static void runScript (SCRIPT_LIST_PTR script)
{
  scriptClear();
  activateStatements(script);
}

static int freeStatement (SCRIPT_STATEMENT_PTR statement)
{
  if (statement->process) free(statement->process);
  if (statement->line) free(statement->line);
  if (statement->subStatements) freeScript(&statement->subStatements);
  free(statement);
  return TRUE;
}

static void freeScript (SCRIPT_LIST_PTR *scriptHandle)
{
  if (*scriptHandle) {
    listFreeAllItems((LIST_FREE_FN)freeStatement, *scriptHandle);
    listFree(scriptHandle);
  }
}

void scriptClear (void)
{
  if (activeListG) listFree(&activeListG);
  if (suspendedListG) listFree(&suspendedListG);
  clearAlarm();
}

BOOLEAN scriptLoad (char *filename)
{
  FILE *scriptFile;

  scriptFile = fopen(filename, "r");
  if (!scriptFile) {
    return FALSE;
  } else {
    freeScript(&loadedScriptG);
    loadedScriptG = scriptParse(scriptFile);
    fclose(scriptFile);
    if (listLength(loadedScriptG) == 0) {
      return FALSE;
    } else {
      runScript(loadedScriptG);
      return TRUE;
    }
  }
}

BOOLEAN scriptRerun (void)
{
  if (listLength(loadedScriptG) == 0) {
    return FALSE;
  } else {
    runScript(loadedScriptG);
    return TRUE;
  }
}

BOOLEAN scriptSuspend (void)
{
  if (listLength(activeListG) == 0) {
    return FALSE;
  } else if (suspendedListG) {
    fprintf(stderr, "ERROR: Already have a suspended list\n");
    return FALSE;
  } else {
    suspendedListG = activeListG;
    activeListG = NULL;
    clearAlarm();
    return TRUE;
  }
}

BOOLEAN scriptResume (void)
{
  if (listLength(suspendedListG) == 0) {
    return FALSE;
  } else if (activeListG) {
    fprintf(stderr, "ERROR: Already have an active list\n");
    return FALSE;
  } else {
    activeListG = suspendedListG;
    suspendedListG = NULL;
    setAlarmTimeFromList(activeListG);
    return TRUE;
  }
}
