/*****************************************************************************
 * PROJECT: Xavier
 *
 * (c) Copyright 1993 Richard Goodwin & Joseph O'Sullivan. All rights reserved.
 *
 * FILE: runConsole.c
 *
 * ABSTRACT: This is the common code between xRunConsole and xfRunConsole
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/runConsole.c,v $
 * $Revision: 1.11 $
 * $Date: 1996/06/28 14:07:34 $
 * $Author: reids $
 *
 * REVISION HISTORY:
 *
 * $Log: runConsole.c,v $
 * Revision 1.11  1996/06/28  14:07:34  reids
 * Fixed quite a few bugs -- with graphics, interaction with script, and
 *   killing processes
 *
 * Revision 1.10  1996/03/29  15:57:23  reids
 * Consolidated the common code between xCallbacks and xfCallbacks, and
 *   xRunConsole and xfRunConsole.
 * Added a way to add macro definitions to resource files ("define: <x> <y>").
 * Fixed a bug that was causing the xfRunConsole to crash when a task was
 *   restarted.
 *
 *****************************************************************************/
#include "tca/libc.h"
#include "tca/devUtils.h"

#include "runConsole.h"
#include "callbacks.h"
#include "messages.h"
#include "nanny.h"
#include "nannyDev.h"
#include "script.h"

int nConsoleProcs = 0;
int currentProc   = -1;
consoleProcType consoleProc[MAX_PROCESSES];

static char *scriptFile = NULL;

static char buf[DEFAULT_LINE_LENGTH];

void runNewProcess (char *process)
{
  messageMakeExec(buf, process, getenv("DISPLAY"));
  if (safePassLocalMessage(buf)) {
    sprintf(buf,"Waiting for %s...", process);
    activateStatus(buf);
    sprintf(buf, "[CONSOLE] %s waiting for confirmation from nanny\n", 
	    process);
    MessagePrintf(buf);
  }
}

void runKillProcess (char *process) 
{ 
  messageMakeKill(buf, process);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] %s waiting for nanny kill confirmation\n",  
	    process);
    MessagePrintf(buf);
  }
}

void runSuspendProcess (char *process)
{
  messageMakePause(buf, process);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] %s suspended.\n", process);
    MessagePrintf(buf);
  }
}

void runContinueProcess (char *process)
{
  messageMakeBreak(buf, process);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] continuing %s\n", process);
    MessagePrintf(buf);
  }
}

void runCommandProcess (char *process, char *command)
{
  messageMakeCommand(buf, process, command);
  if (!safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] cannot send to %s\n", process);
    MessagePrintf(buf);
  }
}

static void displayVersion (void)
{
  fprintf(stderr, "runConsole %d.%d.%d \n",
	  NANNY_VERSION_MAJOR, NANNY_VERSION_MINOR, NANNY_VERSION_MICRO);
  fprintf(stderr, " Released : %s\n", NANNY_VERSION_DATE);
  fprintf(stderr, " Commited : %s\n", NANNY_COMMIT_DATE);
  fprintf(stderr, " Compiled : %s %s\n", __DATE__, __TIME__);
}

static void printHelp (void)
{
  printf("Usage: runConsole [options]\n");
  printf("Without options, displays a Xwindow allowing for GUI interaction.\n");
  printf("With options, allows for interaction with the Xavier nanny daemons:\n");
  printf(" -h | -help:   Print this message\n");
  printf(" -v        :   Display version information\n");
  printf(" -run <name>:  Runs program name (eg -run CTR)\n");
  printf(" -kill <name>: Kills program name, if it is running\n");
  printf(" -reload [current | <file>]: Forces nannys to reload either the\n");
  printf("                current resource file or from a new location\n");
  printf(" -script <file>: Load and run the given script file\n");
}

static void parseCommandLineOptions (int argc, char **argv)
{
  char *name = NULL, *file = NULL;
  char buffer[DEFAULT_LINE_LENGTH];
  int  i;

  for (i=1; i<argc && argv[i][0]=='-'; i++) {
    if (!strcmp(argv[i],"-run") && i+1<argc) {
      name = (argv[++i]);
      messageMakeExec(buffer, name, getenv("DISPLAY"));
      passLocalMessage(buffer);
      sleep(3);
      exit (0);
    } else if (!strcmp(argv[i],"-kill") && i+1<argc) {
      name = (argv[++i]);
      messageMakeKill(buffer, name);
      passLocalMessage(buffer);
      sleep(3);
      exit (0);
    } else if (!strcmp(argv[i], "-reload") && i+1<argc) {
      file = (argv[++i]);
      messageMakeReload(buffer);
      passLocalMessage(buffer);
      sleep(3);
      exit (0);
    } else if (!strcmp(argv[i], "-script") && i+1<argc) {
      scriptFile = (argv[++i]);
    } else if ((!strcmp(argv[i],"-help")) || (!strcmp(argv[i], "-h"))) {
      printHelp();
      exit(1);
    } else if (!strcmp(argv[i], "-v")) {
      displayVersion();
      exit(1);
    } else {
      WARNING2("Unknown option: %s\n", argv[i]);
      printHelp();
      exit(-1);
    }
  }
}

/* DESCRIPTION: Handles character typed from stdin.
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void stdin_inputHnd(int fd, long chars_available)
{
  static char buffer[DEFAULT_LINE_LENGTH+1];
  int numRead=0;
  char *buffPtr;
  
  bzero(buffer, DEFAULT_LINE_LENGTH+1);
  
  numRead = devReadN(fd, buffer,
		     MIN(chars_available,DEFAULT_LINE_LENGTH));
  for(buffPtr = buffer; ((*buffPtr == ' ') && (buffPtr != '\0')); buffPtr++);
}

static void processLocalMessage(int id, char *message)
{
  parseLine(TRUE, NULL, id, message);
}

void connectWithNanny (void)
{
  nanny_connect(processLocalMessage);
  
  /* Announce that we be here */
  syncWithNanny();
}

static void initConsoleProc (void)
{
  int i;

  for (i=0; i<MAX_PROCESSES; i++) {
    consoleProc[i].history.buffer = NULL;
    consoleProc[i].graphics = createConsoleGraphics();
  }
}

void main(int argc, char **argv)
{
  parseCommandLineOptions(argc, argv);

  displayVersion();

  /* Need (some of) this to initialize the list package */
  {
    extern void tcaModuleInitialize(void);
    tcaModuleInitialize();
  }

  fprintf(stdout, ">");
  fflush(stdout);

  initConsoleProc();

  xRunConsole(argc, argv);

  /* Preload a script file */
  if (scriptFile) {
    if (!scriptLoad(scriptFile)) {
      fprintf(stderr, "Cannot load script file %s\n", scriptFile);
    }
  }

  /* loop forever in the main device loop */
  devMainLoop();
  
  exit(0);
}
