/*****************************************************************************
 * PROJECT: Xavier
 *
 * (c) Copyright 1993 Richard Goodwin & Joseph O'Sullivan. All rights reserved.
 *
 * FILE: resource.c
 *
 * ABSTRACT:
 *
 * Implements a FSM for parsing resource files. Also handles creating and
 * managing the dependanciy/conflict structure stored in these files.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/resource.c,v $
 * $Revision: 1.13 $
 * $Date: 1996/07/29 05:03:16 $
 * $Author: josullvn $
 *
 * REVISION HISTORY:
 *
 * $Log: resource.c,v $
 * Revision 1.13  1996/07/29  05:03:16  josullvn
 * Bloody hell. This commiting is a bit different than under Xavier. Short
 * story is cleaned up some purify bugs, and also made changes to nanny
 * which should make it a bit better - Improving performance over multiple
 * machines, explict quietening of nondisplayed processes, replacing of
 * runConsole with xfMiniConsole, which is multithreaded, vt102 compilant,
 * adds a uniform emacs-like command line editing feature, better on small
 * screens and otherwise fab.
 *
 * Revision 1.12  1996/07/25  22:25:29  rich
 * Fixed uninitialized memory references.
 *
 * Revision 1.11  1996/07/18  02:23:10  rich
 * Fixed some memory problems found with purify.
 *
 * Revision 1.10  1996/07/10  02:11:01  reids
 * Enable "define"s to be multi-word (contain whitespace).
 *
 * Revision 1.9  1996/06/28  17:59:53  reids
 * Eliminate "uninitialized memory read" by null-terminating stdinBuffer
 *   and errorBuffer.
 *
 * Revision 1.8  1996/06/28  14:07:29  reids
 * Fixed quite a few bugs -- with graphics, interaction with script, and
 *   killing processes
 *
 * Revision 1.7  1996/03/29  15:57:17  reids
 * Consolidated the common code between xCallbacks and xfCallbacks, and
 *   xRunConsole and xfRunConsole.
 * Added a way to add macro definitions to resource files ("define: <x> <y>").
 * Fixed a bug that was causing the xfRunConsole to crash when a task was
 *   restarted.
 *
 * Revision 1.6  1996/02/10  16:52:22  rich
 * Made private functions static and fixed some forward declarations.
 *
 * Revision 1.5  1996/02/05  15:56:52  reids
 * Added scripting capabilities to nanny -- using lex/bison to parse script
 *  files according to the script.bnf standard.
 * Integrated support for running/stopping/suspending scripts into the
 *  runConsole window.
 *
 * Revision 1.4  1996/02/01  04:04:15  rich
 * Generalized updateVersion and added recursion.
 *
 * Revision 1.3  1996/01/27  21:56:30  rich
 * Pre-release of 8.4.
 *
 * Revision 1.2  1996/01/22  21:27:57  reids
 * Consolidated all the DEBUG and WARNING macro definitions into nannyCommon.h
 *
 * Revision 1.1  1995/12/17  20:25:54  rich
 * Moved Nanny to the tca release.
 *
 * Revision 1.17  1995/07/21  15:13:22  josullvn
 * Added a restarts option - see the README. updated some .rc files to give
 * examples. Also fixed conflicts to that runConsole is notified about dead
 * processes.
 *
 * Revision 1.16  1995/07/15  07:13:18  josullvn
 * ProcessDevices is _not_ reentrant - this was causing lots of problems as it
 * turns out. Restructing my blocking code fixed it.
 * Also, added a notion of dead - that is telling when something has
 * finally died.
 *
 * Revision 1.15  1995/07/11  11:21:39  josullvn
 * A lot more changes to message passing to handle multiple nannys better.
 * This basically involved removing fdclient, and being smarter about what
 * nannys are in the system
 *
 * Revision 1.14  1995/07/11  06:58:57  josullvn
 * Added capabilities for inter-nanny communication.
 * Mainly, there is now a list of connections, and each new executable is
 * announced to the whole list - may need to work on the kill part...
 *
 * Revision 1.13  1995/07/11  01:06:37  josullvn
 * Whesh. OK, fixed a bug in parsing lines, where by if messages get corrupted,
 * we can recover. Added a -clean option to nanny. Discovered that the old
 * double newProcess bug has returned to haunt me.
 *
 * Revision 1.12  1995/05/25  02:50:10  rich
 * Fixed lookup for nanny device and lost character problem.
 *
 * Revision 1.11  1995/05/24  15:50:48  josullvn
 * Had deleted two lines by accident in nannyUtils which prevented commands being
 * passed to processes. More support for intermachine communication. Now
 * can do rcsh to run a csh shell remotely on heart (according to new
 * Simulator.rc file)
 *
 * Revision 1.10  1995/05/23  23:58:42  josullvn
 * Yeah. Fixed the SIGCHLD problem (tured out that popen is internally
 * implemented with fork - when pclose was called, it generated a SIGCHLD
 * which intereupted the system call which lead to trouble).
 * Fixed environment variables, can now add env variables in .rc file, and
 * they are passed to appropriate child. Realized that need to also
 * provide an ability to pass the display variable from runConsole to the
 * nanny - being worked on.
 * Bug exists in devUtils (?) which large data streams. Run csh as a new process
 * and do ps auxww to cause it to occur.
 *
 * Revision 1.9  1995/05/20  02:07:35  josullvn
 * Some more bells and whistles to the console callback.
 * Added environment variables to the resource definition.
 * Extended the still buggy children catching for time outs.
 * Added diagnositic code to analyze startup failures.
 *
 * Revision 1.8  1995/05/19  11:32:30  josullvn
 * Updated resource files to include "ready strings"
 * Debugged some protocol problems with messages.c parsing of split lines.
 * (Feel that remaining problem is in devUtils).
 * Added Highlightening to buttons.
 * Removed SIGCHLD trapping - its too flakey at the moment.
 *
 * Revision 1.7  1995/05/15  17:07:24  rich
 * Updated interface to createLineBuffer so you can get partial lines and
 * the delimit character is not replaced.
 * Imporved layout of the X11 window of the console.
 *
 *
 *****************************************************************************/
#include "tca/libc.h"

#include "resource.h"
#include "tca/devUtils.h"

/*
 * Global variables
 */
int          rcNumAvailablePrograms = 0;
rcProgramPtr rcAvailableProgramList = NULL;

static int numDefns = 0;
static char **defines = NULL;

/*****************************************************************************/
/* Resource entry management */
/*****************************************************************************/
/*
 * Intended to be called each time a resource file is loaded, to clear
 * out old definations.
 */
static void resFreeAvailablePrograms(void)
{
  int i, j;
  rcProgramPtr r;
  
  for (i = 0; i< rcNumAvailablePrograms; i++)
    {
      r = &(rcAvailableProgramList[i]);
      if (r->conflicts) {
	free(r->conflicts);
	r->conflicts = NULL;
      }
      if (r->dependencies) {
	free(r->dependencies);
	r->dependencies = NULL;
      }
      if (r->ready_string!=NULL)
	free(r->ready_string);
      r->ready_string=NULL;
      for (j=0; j< r->noargs; j++)
	free(r->args[j]);
      if (r->args)
	free(r->args);
      r->args = NULL;
      r->timeouts = 0;
    }
  free(rcAvailableProgramList);
  rcNumAvailablePrograms = 0;
}

/*
 * when a new key/program is discovered, assign a unique id and create 
 * a resource table entry 
 */
static int resCreateLookup(char *key)
{
  rcProgramPtr r;
  
  if (rcNumAvailablePrograms == 0) {
    rcAvailableProgramList = (rcProgramPtr) malloc(sizeof(rcProgramType));
    bzero(rcAvailableProgramList,sizeof(rcProgramType));
  } else {
    rcAvailableProgramList = 
      (rcProgramPtr) realloc(rcAvailableProgramList, 
			     (rcNumAvailablePrograms+1) *
			     (sizeof(rcProgramType)));
    bzero(&rcAvailableProgramList[rcNumAvailablePrograms]
	  ,sizeof(rcProgramType));
  }
  r = &rcAvailableProgramList[rcNumAvailablePrograms];
  
  r->noConflicts = 0;
  r->conflicts = NULL;
  r->noDependencies = 0;
  r->dependencies = NULL;
  r->noargs = 0;
  r->args = NULL;
  r->noenvs = 0;
  r->env = NULL;
  r->ibufEnd = 0;
  r->ebufEnd = 0;
  r->executing = 0;
  r->wantToExecute = 0;
  r->ready = FALSE;
  r->loudness = QUIET;
  r->alwaysVerbose = FALSE;
  r->timeouts = 0;
  r->max_restarts = MAX_RESTARTS;
  r->ready_string = NULL;
  r->fdstdin = NO_FD;
  r->fdstdout = NO_FD;
  r->fdstderr = NO_FD;
  
  strcpy(r->name, key);
  r->id = rcNumAvailablePrograms; 
  
  return rcNumAvailablePrograms++;
}

/* 
 * Maps key into a previously created id, else returns error
 */
int rcLookupName(char *key)
{
  int i;
  
  for (i=0; i<rcNumAvailablePrograms; i++)
    if (!strcmp(rcAvailableProgramList[i].name, key))
      return rcAvailableProgramList[i].id;
  return ERROR; 
}

/*
 * Fills in an entry in the resource table entry with key.
 * legal entries of type string are
 *     rcEntMACHINE_NAME
 *     rcEntPATH
 *     rcEntEXECUTABLE
 *     rcEntARGS
 *     rcEntDISPLAY
 * Assumes that id is valid...
 */
void rcInsertStr (int id, int entry, char *key)
{
  static char *thisHost= NULL;
  int i;
  rcProgramPtr r = &rcAvailableProgramList[id];
  
  if (thisHost == NULL) {
    thisHost = (char *)malloc(80);
    gethostname(thisHost, 80);  
  }
  
  if ((key == NULL) || (key[0] == '\0')) {
    WARNING1("Corrupt string received by resource manager\n");
    return;
  }
  
  switch (entry) {
  case rcEntMACHINE_NAME:
    r->local = ((!strcmp("localhost", key)) || (strstr(thisHost,key) != NULL));
    strcpy(r->remoteMachine, key);
    break;
  case rcEntPATH:
    strcpy(r->pathToExecuteFrom, key);
    break;
  case rcEntEXECUTABLE:
    strcpy(r->executable, key);
    break;
  case rcEntDISPLAY:
    strcpy(r->display, key);
    break;
  case rcEntARGS:
    if (r->noargs == 0)
      r->args = (char **) malloc (sizeof(char *));
    else
      r->args = (char **) realloc (r->args, (r->noargs+1)*sizeof(char *));
    r->args[r->noargs] = (char *) malloc (strlen(key)+1);
    strcpy(r->args[r->noargs], key);
    r->noargs++;
    break;
  case rcEntENVS:
    if (r->noenvs == 0)
      r->env = (char **) malloc (2 * sizeof(char *));
    else
      r->env = (char **) realloc (r->env, (r->noenvs+2)*sizeof(char *));
    r->env[r->noenvs] = (char *) malloc (strlen(key)+1);
    strcpy(r->env[r->noenvs], key);
    r->noenvs++;
    r->env[r->noenvs] = NULL;
    break;
  case rcEntREADY:
    if (r->ready_string == NULL)
      r->ready_string = (char *) malloc (strlen(key)+1);
    strcpy(r->ready_string, key);
    break;
  case rcEntIBUF: 
    for (i=0; i<strlen(key); i++) 
      r->stdinBuffer[(r->ibufEnd + i)%MAXBUF] = key[i];
    r->ibufEnd += i;
    /* Make sure NULL-terminated */
    r->stdinBuffer[r->ibufEnd%MAXBUF] = '\0';
    break;
  case rcEntEBUF:
    for (i=0; i<strlen(key); i++)
      r->errorBuffer[(r->ebufEnd + i)%MAXBUF] = key[i];
    r->ebufEnd += i;
    /* Make sure NULL-terminated */
    r->errorBuffer[r->ebufEnd%MAXBUF] = '\0';
    break;
#ifndef TEST_CASE_COVERAGE
  default:
    fprintf(stderr, "Error in Insert Str\n"); fflush(stderr);
#endif
  }
}

/*
 * Fills in an entry in the resource table entry with key.
 * legal entries of type int are
 *       rcEntCONFLICTS
 *       rcEntDEPENDENCIES
 * Assumes that id is valid...
 */
void rcInsertId(int id, int entry, int id1)
{
  rcProgramPtr r = &rcAvailableProgramList[id];
  
  switch (entry)
    {
    case rcEntCONFLICTS:
      if (r->noConflicts == 0)
	r->conflicts = (int *) malloc (sizeof(int));
      else
	r->conflicts = (int *) realloc (r->conflicts, 
					(r->noConflicts+1)*sizeof(int));
      r->conflicts[r->noConflicts] = id1; 
      r->noConflicts++;
      break;
    case rcEntDEPENDENCIES:
      if (r->noDependencies == 0)
	r->dependencies = (int *) malloc (sizeof(int));
      else
	r->dependencies = (int *) realloc (r->dependencies, 
					   (r->noDependencies+1)*sizeof(int));
      r->dependencies[r->noDependencies] = id1; 
      r->noDependencies++;
      break;
#ifndef TEST_CASE_COVERAGE
    default:
      fprintf(stderr, "Error in Insert Id\n"); fflush(stderr);
#endif
    }
}

static void rcAddDefine (const char *define)
{
  if (numDefns == 0) {
    defines = (char **)malloc(2 * sizeof(char *));
  } else {
    defines = (char **) realloc (defines, (numDefns+2)*sizeof(char *));
  }
  defines[numDefns] = strdup(define);
  numDefns++;
  defines[numDefns] = NULL;
  numDefns++;
}

/*****************************************************************************/
/* parsing */
/*****************************************************************************/
/*
 * States in the FSM used for parsing the resource file. 
 */
typedef enum {  
  unknown,
  process,
  machine_name,
  path,
  executable,
  args,
  ready, 
  restartlist,
  restarts,
  envlist,
  envs,
  dependencies,
  dependent,
  conflicts,
  conflict,
  define,
  error
#ifdef FORCE_32BIT_ENUM
    , dummyParseState = 0x7FFFFFFF
#endif
} PARSE_STATE_TYPE;

/*
 * fills buff with chars from fp, up until max_char arrives, 
 * or we reach a newline/endofline
 */
static char *resGetLine(char *buff, int max_chars, FILE *fp)
{
  int i;
  int c;
  char *str;
  
  str = buff;
  for( i = 0; i < max_chars; i++){
    c = getc(fp);
    if (c == '\n' || c == EOF) {
      *buff++ = '\0';
      if (c == EOF)
	return NULL;
      else
	return str;
    }
    else 
      *buff++ = c;
  }
  return str;
}

static BOOLEAN resEndLinep(char *buf)
{
  return (*buf == '\0');
}

static BOOLEAN resCommentp(char *buf)
{
  return (*buf == '#');
}

static BOOLEAN resSpecialp(char *buf)
{
  return (*buf == '(' || *buf == ')' || *buf == '$');
}

static char *resSkipWhitespace(char *buf)
{
  while ((!resEndLinep(buf)) && (isspace(*buf)))
    buf++;
  
  return buf;
}

static char *resExtractKey(char *buf, char *key)
{
  int i=0;

  if (resSpecialp(buf)) {
    key[i++] = *(buf++);
  } else {
    while (!resEndLinep(buf) && !resCommentp(buf) && !isspace(*buf) &&
	   !resSpecialp(buf)) {
      key[i++] = *(buf++);
    }
  }
  key[i] = '\0';

  return buf;
}

/*
 * Implements a FSM which begins in unknown at the start of each line
 * This allows error handling etc to be quite instructive, although 
 * only a skeleton is provided.
 * The parsing also handles necessary actions for updating the resource
 * data.
 */ 
static BOOLEAN resParseLine(char *buf)
{
  PARSE_STATE_TYPE prev_state = unknown, state = unknown;
  int        id = -1, id1 = -1;
  char       key[MAXTOKEN];
  
  buf = resSkipWhitespace(buf);
  while (!resCommentp(buf) && !resEndLinep(buf) && (state!=error)) {
    prev_state = state;
    buf = resExtractKey(buf, key);
    
    switch (state)  {
    case unknown:
      if (!strcmp(key, "dependencies:"))
	state = dependencies;
      else if (!strcmp(key, "process:"))
	state = process;
      else if (!strcmp(key, "conflict:"))
	state = conflicts;
      else if (!strcmp(key, "ready:"))
	state = ready;
      else if (!strcmp(key, "env:"))
	state = envlist;
      else if (!strcmp(key, "restarts:"))
	state = restartlist;
      else if (!strcmp(key, "define:"))
	state = define;
      else
	state = error;
      break;
    case ready:
      if ((id = rcLookupName(key)) == ERROR)
	state = error;
      else { 
	buf = resSkipWhitespace(buf);
	rcInsertStr(id, rcEntREADY, buf);
	return TRUE;
      }
      break;
    case process:
      if (rcLookupName(key) != ERROR)
	state = error;
      else {
	id = resCreateLookup(key);
	state = machine_name;
      }
      break;
    case machine_name:
      rcInsertStr(id, rcEntMACHINE_NAME, key);
      state = path;
      break;
    case path:
      rcInsertStr(id, rcEntPATH, key);
      state = executable;
      break;
    case executable:
      rcInsertStr(id, rcEntEXECUTABLE, key);
      state = args;
      break;
    case args:
      rcInsertStr(id, rcEntARGS, key);
      break;
    case envlist:
      if ((id = rcLookupName(key)) == ERROR) {
	if (!strcmp(key,"*"))
	  state = envs;
	else
	  state = error;
      }
      else 
	state = envs;
      break;
    case restartlist:
      if ((id = rcLookupName(key)) == ERROR) {
	if (!strcmp(key,"*"))
	  state = restarts;
	else
	  state = error;
      }
      else 
	state = restarts;
      break;
    case restarts:
      {
	int i;
	
	if (id == ERROR)
	  for (i=0; i<rcNumAvailablePrograms; i++)
	    rcAvailableProgramList[i].max_restarts=atoi(key);
	else
	  rcAvailableProgramList[id].max_restarts=atoi(key);
      }
      break;
    case envs:
      {
	int i;
	
	if (id == ERROR)
	  for (i=0; i<rcNumAvailablePrograms; i++)
	    rcInsertStr(i, rcEntENVS, key);
	else
	  rcInsertStr(id, rcEntENVS, key);
      }
      break;
    case dependencies:
      if ((id = rcLookupName(key)) == ERROR)
	state = error;
      else
	state = dependent;
      break;
    case dependent:
      if ((id1 = rcLookupName(key)) == ERROR)
	{ if (strcmp(key, "none")) state = error;} 
      else
	rcInsertId(id, rcEntDEPENDENCIES, id1);
      break;
    case conflicts:
      if ((id = rcLookupName(key)) == ERROR)
	state = error;
      else
	state = conflict;
      break;
    case conflict:
      if ((id1 = rcLookupName(key)) == ERROR)
	{ if (strcmp(key, "none")) state = error; } 
      else
	rcInsertId(id, rcEntCONFLICTS, id1);
      break;
    case define:
      rcAddDefine(key);
      buf = resSkipWhitespace(buf);
      defines[numDefns-1] = strdup(buf);
      state = unknown;
      return TRUE;
      break;
    case error:
      break;
#ifndef TEST_CASE_COVERAGE
    default:
      fprintf(stderr, "Illegal state in resource parse\n");
      fflush(stderr);
#endif
    }
    buf = resSkipWhitespace(buf);
  }
  if (state == error) {
    switch (prev_state) {
    case process:     fprintf(stderr, "Failed in process name\n"); break;
    case ready:     fprintf(stderr, "Failed in ready name\n"); break;
    case machine_name:fprintf(stderr, "Failed in machine name\n"); break;
    case envlist:fprintf(stderr, "Failed in environment name\n"); break;
    case path:        fprintf(stderr, "Failed in executable path\n"); break;
    case executable:  fprintf(stderr, "Failed in executable name\n"); break;
    case dependencies:fprintf(stderr, "Failed in dependent name\n"); break;
    case conflicts:   fprintf(stderr, "Failed in conflict name\n"); break;
    case dependent:   fprintf(stderr, "Failed in dependency name\n"); break;
    case conflict:    fprintf(stderr, "Failed in conflicting name\n"); break;
    case args:        fprintf(stderr, "Failed in args\n"); break;
    case envs:        fprintf(stderr, "Failed in envs\n"); break;
    case restarts:    fprintf(stderr, "Failed in restarts\n"); break;
    case define:      fprintf(stderr, "Failed in define\n"); break;
    case unknown:     fprintf(stderr, "Unknown Keyword\n"); break;
#ifndef TEST_CASE_COVERAGE
    default:          fprintf(stderr, "Failed in illegal state in parse\n");
#endif
    }
    fflush(stderr);
    return FALSE; 
  }
  
  if ((state != conflict) && (state != dependent) && 
      (state != args) && (state != unknown) && 
      (state != envs) && (state !=restarts)) {
    switch (state) {
    case process:     fprintf(stderr, "Expecting process name\n"); break;
    case ready:     fprintf(stderr, "Expecting ready name\n"); break;
    case machine_name:fprintf(stderr, "Expecting machine name\n"); break;
    case path:        fprintf(stderr, "Expecting executable path\n"); break;
    case executable:  fprintf(stderr, "Expecting executable name\n"); break;
    case dependencies:fprintf(stderr, "Expecting dependent name\n"); break;
    case conflicts:   fprintf(stderr, "Expecting conflict name\n"); break;
#ifndef TEST_CASE_COVERAGE
    default:          fprintf(stderr, "Reached Illegal state in parse\n");
#endif
    }
    fflush(stderr);
    return FALSE; 
  }
   return TRUE;
} 

static void addToLine (char *line, char *addition, int maxChars)
{
  if (strlen(line) + strlen(addition) >= maxChars) {
    fprintf(stderr, "ERROR: addToLine: Line not long enough (%d)\n", maxChars);
  } else {
    strcat(line, addition);
  }
}

static char *insertDefinition (char *origLine, char *expandedLine,
			       int maxChars)
{
  BOOLEAN paren;
  int i;
  char key[MAXBUF], *env;

  paren = (*origLine == '(');
  origLine = resExtractKey((paren ? origLine+1 : origLine), key);
  if (paren != (*origLine == ')')) {
    fprintf(stderr, "Illegal syntax -- unmatched parentheses (%s)\n",
	    origLine); 
    exit(-1);
  } else {
    if (paren) origLine++;
    for (i=0; i<numDefns; i+=2) {
      if (!strcmp(defines[i], key)) {
	addToLine(expandedLine, defines[i+1], maxChars);
	return origLine-1;
      }
    }
    env = getenv(key);
    if (env) {
      addToLine(expandedLine, env, maxChars);
      return origLine-1;
    } else {
      fprintf(stderr, "ERROR: No definition found for %s\n", key);
      exit(-1);
    }
  }
  return origLine;
}

static void resExpandLine (char *origLine, int maxChars)
{
  static char buf[MAXBUF], single[2];
  char *line = origLine;

  bzero(buf, MAXBUF); bzero(single, 2);
  for (; *line != '\0'; line++) {
    if (*line == '#') {
      addToLine(buf, line, maxChars);
      break;
    } else if (*line == '\\' && *(line+1) == '$') {
      line++;
      addToLine(buf, "$", MAXBUF);
    } else if (*line == '$') {
      line = insertDefinition(++line, buf, MAXBUF);
    } else if (!strncmp("localhost", line, strlen("localhost"))) {
      line = insertDefinition(line, buf, MAXBUF);
    } else {
      single[0] = *line;
      addToLine(buf, single, MAXBUF);
    }
  }
  
  if (strlen(buf) > maxChars) {
    fprintf(stderr, "ERROR: resExpandLine: Line too long: %s\n", buf);
    exit(-1);
  } else {
    strcpy(origLine, buf);
  }
}

static void resFreeDefinitions (void)
{
  int i;

  if (numDefns > 0) {
    for (i=0; i<numDefns; i++) {
      free(defines[i]);
    }
    free(defines);
    defines = NULL;
    numDefns = 0;
  }
}

/*****************************************************************************/
/* Resource File/Data interaction */
/*****************************************************************************/
/*
 * Mainly useful for debug purposes. Allows the loaded resource file to be
 * saved in a format which can be reloaded.
 */
void rcDumpResources (FILE *stream)
{
  int i, j;
  rcProgramPtr r;
  
  for (i = 0; i< numDefns; i+=2) {
    fprintf(stream, "define: %s %s\n", defines[i], defines[i+1]);
  }

  for (i = 0; i< rcNumAvailablePrograms; i++) {
    r = &(rcAvailableProgramList[i]);
    
    fprintf(stream, "# %d \n", r->id);
    fprintf(stream, "process: %s %s %s %s", 
	    r->name, r->remoteMachine, r->pathToExecuteFrom, r->executable);
    for (j = 0; j < r->noargs; j++) 
      fprintf(stream, " %s", r->args[j]); 
    fprintf(stream, "\n"); 
    }
  
  fprintf(stream, "\n"); 
  for (i = 0; i< rcNumAvailablePrograms; i++) {
    r = &(rcAvailableProgramList[i]);
    
    if (r->noDependencies) {
      fprintf(stream, "dependencies: %s", r->name);
      for (j = 0; j < r->noDependencies; j++) 
	fprintf(stream, " %s", rcAvailableProgramList[r->dependencies[j]].name); 
      fprintf(stream, "\n"); 
    }
    }
  fprintf(stream, "\n"); 
  
  for (i = 0; i< rcNumAvailablePrograms; i++) {
    r = &(rcAvailableProgramList[i]);
    
    if (r->noConflicts) {
      fprintf(stream, "conflicts: %s", r->name);
      for (j = 0; j < r->noConflicts; j++) 
	fprintf(stream, " %s", rcAvailableProgramList[r->conflicts[j]].name); 
      fprintf(stream, "\n"); 
    }
  }
}

/*
 * Attempts to open filename. If we fail to open filename, or if filename
 * is null, failback to opening the default resource file defined in 
 * resource.h. 
 * If we fail to open a resource file return NULL, else return the fileid
 * of an opened resource file
 */
static FILE *resOpenFile(char* filename)
{
  FILE *fp;
  
  if (filename != NULL) {
    fp = fopen(filename, "r");
    
    if (!fp)  {
      fprintf(stderr, "Unable to open %s. ", filename);
    }
    else
      return fp;
  }
  
  fp = fopen(defaultResourceFileA, "r");
  if (fp)
    fprintf(stderr, "Using %s.\n", defaultResourceFileA);
  else {
    fp = fopen(defaultResourceFileB, "r");
    
    if (fp)
      fprintf(stderr, "Using %s.\n", defaultResourceFileB);
    else {
      fp = fopen(defaultResourceFileC, "r");
      
      if (fp)
	fprintf(stderr, "Using %s.\n", defaultResourceFileC);
    }
  } 
  return fp;
}

/*
 * Main exported function, which accesses either filename, or 
 * if that fails/is null, the default resource file. The resources
 * are then loaded into memory...
 */
BOOLEAN rcLoadResourceFile(char *filename)
{
  FILE *fp;
  int line=0;
  char buf[MAXBUF];
  
  if (!(fp=resOpenFile(filename))) return FALSE;
  
  if (rcNumAvailablePrograms) {
    resFreeAvailablePrograms();  /* we do be reloading ... */
    resFreeDefinitions();
  }
  { char hostname[80];
    rcAddDefine("localhost");
    gethostname(hostname, 80);
    defines[numDefns-1] = strdup(hostname);
  }

  while (!feof(fp))
    {
      resGetLine(buf, MAXBUF, fp);
      resExpandLine(buf, MAXBUF);
      if (!resParseLine(buf))
	{ fprintf(stderr, "Resource Error: Line %d\n", line); fflush(stderr); }
      line++;
    }
  fclose (fp);
  
  return TRUE;
}

/*****************************************************************************/
/* dependency/conflict management */
/*****************************************************************************/
/*
 * recursively generates subdependencies/conflicts.
 * conflicts are guaranteed to be at a leaf level, so no need to recurse.
 */
static BOOLEAN resGenerateSubDependencies(int id, 
					  int *dependent, 
					  int *conflict)
{
  int i;
  rcProgramPtr r = &rcAvailableProgramList[id];
  
  for (i=0; i < r->noDependencies; i++)
    if (!dependent[r->dependencies[i]]) {
      dependent[r->dependencies[i]] = TRUE;
      resGenerateSubDependencies(r->dependencies[i],dependent,conflict);
    }
  
  for (i=0; i < r->noConflicts; i++)
    conflict[r->conflicts[i]] = TRUE;
  
  return TRUE;
}

/* 
 * The exported version of the above, indexed by key (eg CTR etc). 
 * assumes that dependent and conflict are created with enough space
 * (rcNumAvailablePrograms)
 */
BOOLEAN rcGenerateDependencies(char *key, int *dependent, int *conflict)
{
  int id, i;
  rcProgramPtr r;
  
  if ((id = rcLookupName(key)) == ERROR) return FALSE;
  
  r = &rcAvailableProgramList[id];
  
  for (i=0; i < rcNumAvailablePrograms; i++)
    dependent[i] = conflict[i] = FALSE;
  
  return resGenerateSubDependencies(id, dependent, conflict);
}

/*
 * TO find dependent processes to execute, set executing to FALSE, 
 * to find conflicting processes to kill, set it to be TRUE 
 */
int rcExistDepOrCon(int *DepOrCon, BOOLEAN executing)
{
  int i;
  
  for (i=0; i<rcNumAvailablePrograms; i++)
    if ((rcAvailableProgramList[i].executing == executing) 
	&& DepOrCon[i])
      return i;
  
  return -1;
}

/*
 * Returns TRUE if some process that wants to execute (but is not currently
 * executing) depends on this process.
 */

BOOLEAN rcDependsOnProcess (rcProgramPtr process)
{
  int i, j;
  rcProgramPtr r;

  for (i=0; i<rcNumAvailablePrograms; i++) {
    r = &rcAvailableProgramList[i];
    if (r->wantToExecute && !r->executing) {
      for (j=0; j<r->noDependencies; j++) {
	if (r->dependencies[j] == process->id) {
	  return TRUE;
	}
      }
    }
  }
  return FALSE;
}
