/*****************************************************************************
 * PROJECT: APEX
 *
 * (c) Copyright 1993 Richard Goodwin & Joseph O'Sullivan. All rights reserved.
 *
 * FILE: nannyDev.c
 *
 * ABSTRACT:
 * 
 * This file provides routines for connecting Nanny to other nanny processes
 * or local processes.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/nannyDev.c,v $ 
 * $Revision: 1.10 $
 * $Date: 1996/07/18 02:23:08 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: nannyDev.c,v $
 * Revision 1.10  1996/07/18  02:23:08  rich
 * Fixed some memory problems found with purify.
 *
 * Revision 1.9  1996/06/28  20:07:53  reids
 * Fixed an insdiuous bug that arose in runConsole when nanny quit.
 * Purified some more (no memory leaks!)
 *
 * Revision 1.8  1996/06/28  18:29:14  reids
 * Removed some large (and small) memory leaks when killing devices.
 *
 * Revision 1.7  1996/02/20  11:40:34  josullvn
 * Made extensive changes:
 *    added xfRunConsole, an xforms based interface which was quick
 *    To change the interface, from the nanny dir run
 *    /afs/cs/user/josullvn/xforms/DESIGN/fdesign xfConsole
 *    to get going, and made debugging easier.
 *    added tons of comments.
 *    Debug spews a lot more info back, so just compile
 *    with DEBUG=DEBUG if you really need it.
 *    I've added some timeouts to the code, which will
 *    probably be tweaked later. There are in xfCallbacks.c
 *    Its become better at keeping processes going, and
 *    so its harder to quit. Using shutdown can be
 *    painful. Killing the processes individually is
 *    probably better. Need to do a reverse dependancy when quitting.
 *    Multiple machines are again poor. It doesn't wait sufficently
 *    for dependencies - so you need to run things explicitly...
 *
 * Revision 1.6  1996/02/10  16:52:18  rich
 * Made private functions static and fixed some forward declarations.
 *
 * Revision 1.5  1996/01/27  21:56:28  rich
 * Pre-release of 8.4.
 *
 * Revision 1.4  1996/01/22  21:27:53  reids
 * Consolidated all the DEBUG and WARNING macro definitions into nannyCommon.h
 *
 * Revision 1.3  1996/01/09  03:54:32  rich
 * Removed extra debugging.
 *
 * Revision 1.2  1996/01/05  16:33:32  rich
 * Nanny fixes.
 *
 * Revision 1.1  1995/12/17  20:25:46  rich
 * Moved Nanny to the tca release.
 *
 * Revision 1.29  1995/08/14  22:40:42  rich
 * Changes for the new functional devUtils (8.1.6).
 * "-clean" option on nanny now prompts before killing tasks, unless the
 * "-noprompt" flag is used.
 *
 * Revision 1.28  1995/08/06  00:05:04  rich
 * Changes for new devUtils in tca-8.1.
 *
 * Revision 1.27  1995/07/19  02:00:58  rich
 * Changes for sync target.
 *
 * Revision 1.26  1995/07/15  07:13:13  josullvn
 * ProcessDevices is _not_ reentrant - this was causing lots of problems as it
 * turns out. Restructing my blocking code fixed it.
 * Also, added a notion of dead - that is telling when something has
 * finally died.
 *
 * Revision 1.25  1995/07/13  11:09:47  josullvn
 * Two bugs were present. a) syscalls were being interrupted, and b) the
 * processes button management got screwy after a while. The first is due
 * to Devutils, second fixed by removing a reliance on xclient data. Also
 * added features to newProcess whereby it is now menu driven.
 * Added tca.rc, removing those processes from the Simulator resource files.
 *
 * Revision 1.24  1995/07/12  01:39:13  josullvn
 * Dooh. Had local and remote reversed.
 *
 * Revision 1.23  1995/07/12  01:29:11  josullvn
 * Added host information to my messages, as not happy with
 * what nannydev is doing.
 *
 * Revision 1.22  1995/07/11  01:06:32  josullvn
 * Whesh. OK, fixed a bug in parsing lines, where by if messages get corrupted,
 * we can recover. Added a -clean option to nanny. Discovered that the old
 * double newProcess bug has returned to haunt me.
 *
 * Revision 1.21  1995/07/04  05:46:28  josullvn
 * The latest update. Some stuff vanished (???), but recovered a bit.
 * Now have improved notification of whats happening between runConsole
 * and nanny, and better starting off of processes.
 *
 * Revision 1.20  1995/06/25  01:03:45  rich
 * Moved devUtils to tca.
 *
 * Revision 1.19  1995/06/14  01:02:18  rich
 * Fixed the INC_PATH and added TCA=[v7.8,v7.9,v8.0]
 *
 * Revision 1.18  1995/06/06  01:01:27  rich
 * Added return types to local message passing routines.
 *
 * Revision 1.17  1995/06/02  23:02:50  rich
 * Added support for OSF 2.
 *
 * Revision 1.16  1995/05/25  02:50:07  rich
 * Fixed lookup for nanny device and lost character problem.
 *
 * Revision 1.15  1995/05/24  15:50:44  josullvn
 * Had deleted two lines by accident in nannyUtils which prevented commands being
 * passed to processes. More support for intermachine communication. Now
 * can do rcsh to run a csh shell remotely on heart (according to new
 * Simulator.rc file)
 *
 * Revision 1.14  1995/05/23  23:58:38  josullvn
 * Yeah. Fixed the SIGCHLD problem (tured out that popen is internally
 * implemented with fork - when pclose was called, it generated a SIGCHLD
 * which intereupted the system call which lead to trouble).
 * Fixed environment variables, can now add env variables in .rc file, and
 * they are passed to appropriate child. Realized that need to also
 * provide an ability to pass the display variable from runConsole to the
 * nanny - being worked on.
 * Bug exists in devUtils (?) which large data streams. Run csh as a new process
 * and do ps auxww to cause it to occur.
 *
 * Revision 1.13  1995/05/19  17:56:16  rich
 * Include the null character in the strings sent.
 * Added back the stdin device.  Made runConsole a public bin.
 *
 * Revision 1.12  1995/05/15  17:07:17  rich
 * Updated interface to createLineBuffer so you can get partial lines and
 * the delimit character is not replaced.
 * Imporved layout of the X11 window of the console.
 *
 * Revision 1.11  1995/05/12  18:48:31  rich
 * Changed program output parser to break lines on \n
 *
 * Revision 1.10  1995/05/12  12:58:24  josullvn
 * Wheee. Running locally, its just about ready.
 * Catching and restarting children isn't fully implemented yet, reloading
 * leaves to be desired, and there is still those new widget and filter bugs.
 * The simulator.c is also updated. Since the -s optoin to the simulator can't
 * handle localhost, it is now geared up for stomach. starting, say, navigate
 * will cause a chain of dependencies to be fired.
 *
 * Revision 1.9  1995/05/11  23:35:06  rich
 * Nanny to Nanny comminications partially tested.
 *
 * Revision 1.8  1995/05/11  19:02:44  rich
 * Fixed some communications problems.
 *
 * Revision 1.7  1995/05/11  02:13:29  rich
 * Added routine for nanny to reply to local client.
 *
 * Revision 1.6  1995/05/11  01:55:54  rich
 * Communications from client to nanny partially tested.
 *
 * Revision 1.5  1995/05/11  01:22:52  rich
 * Both now compile.
 *
 * Revision 1.4  1995/05/10  18:33:08  rich
 * Added the routines to handle program output.
 *
 * Revision 1.3  1995/05/10  08:43:02  josullvn
 * "The I'm off to bed for the night, commit so rich can play" version.
 *
 * Revision 1.2  1995/05/10  06:38:07  josullvn
 * More fiddling - added some stubs to nannyDev.h...
 *
 * Revision 1.1  1995/05/10  02:39:31  rich
 * Forgot to add the Nanny files.
 *
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "tca/basics.h"
#include "tca/devUtils.h"
#include "nannyCommon.h"
#include "nannyDev.h"

/*****************************************************************************
 * Forward Declarations.
 *****************************************************************************/

void remoteNanny_outputHnd(int fd, long chars_available);
void Nanny_outputHnd(int fd, long ignore);
void local_outputHnd(int fd, long chars_available);
void Nanny_addNannyDevice(int sd, char *name);
void Nanny_addUserDevice(int sd, char *name);
void remoteNannyParse(char *line, void *clientData);
void localNannyParse(char *line, void *clientData);
void  program_outputHnd(int fd, long chars_available);
void programNannyParse(char *line, void *clientData);
BOOLEAN connectToLocalNanny(void);

/*****************************************************************************
 * Global constants 
 *****************************************************************************/

/*****************************************************************************
 * Global variables 
 *****************************************************************************/

static NANNY_TYPE  nanny_device = 
{
  NULL,
  0,
  { 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
  }
};

static NANNY_DEV_TYPE  remoteNanny_device = 
{
  NULL,
  NULL,
};

static NANNY_DEV_TYPE  local_device = 
{
  NULL,
  NULL
};

static NANNY_DEV_TYPE  program_device = 
{
  NULL,
  NULL
};

static BOOLEAN freeNeeded = FALSE;

/******************************************************************************
 *
 * FUNCTION: void insertNannyDev(NANNY_DEV_PTR dev)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void insertNannyDev(NANNY_DEV_PTR dev)
{
  nanny_device.devs[nanny_device.numDevs++] = dev;
  
}

/******************************************************************************
 *
 * FUNCTION: void removeNannyDev(NANNY_DEV_PTR dev)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void removeNannyDev(NANNY_DEV_PTR dev)
{
  int i;
  
  for(i=0;((i < nanny_device.numDevs) && 
	   (nanny_device.devs[i] != dev));
      i++);
  
  if (nanny_device.devs[i] == dev) {
    for(i++;(i < nanny_device.numDevs); i++)
      nanny_device.devs[i-1] = nanny_device.devs[i];
    nanny_device.numDevs -=1;
    nanny_device.devs[nanny_device.numDevs] = NULL;
  }

  devFreeDev(dev->dev);
  dev->dev = NULL;
  freeNeeded = TRUE;
  devFreeLineBuffer(dev->lineBuf);
  dev->lineBuf = NULL;
  if (dev != &local_device && 
      dev != &remoteNanny_device && dev != &program_device)
    free(dev);
}

/******************************************************************************
 *
 * FUNCTION: int getNannyDev(int fd)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static NANNY_DEV_PTR getNannyDev(int fd)
{
  int i;
  
  if ((fd < 0) || (fd > FD_SETSIZE)) return NULL;

  for(i=0;((i < nanny_device.numDevs) && 
	   (nanny_device.devs[i] != NULL) &&
	   !FD_ISSET(fd,devGetFds(nanny_device.devs[i]->dev)));
      i++);
  if ((nanny_device.devs[i] != NULL) &&
      FD_ISSET(fd,devGetFds(nanny_device.devs[i]->dev))) {
    return nanny_device.devs[i];
  }
  WARNING2("NANNYDEV: can't find device for fd %d\n", fd);
  return NULL;
}

/******************************************************************************
 *
 * FUNCTION: int findNannyDev(char *machine)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static NANNY_DEV_PTR findNannyDev(char *machine)
{
  int i;
  
  for(i=0;((i < nanny_device.numDevs) && 
	   ((devGetSimMachine(nanny_device.devs[i]->dev)[0] == '\0') ||
	    (strstr(devGetSimMachine(nanny_device.devs[i]->dev), machine) 
	     == NULL)));
      i++);
  if ((i <  nanny_device.numDevs) && 
      (strstr(devGetSimMachine(nanny_device.devs[i]->dev), machine) != NULL)) {
    return nanny_device.devs[i];
  }
  return NULL;
}

/******************************************************************************
 *
 * FUNCTION: BOOLEAN nannyServerInitialize(const char *name)
 *
 * DESCRIPTION: 
 * Initialize the server. Creates a socket, looks up the server port
 * number and binds the socket to that port number, and listens to the
 * socket. The machine name for the nanny server is assumed to be the
 * machine on which it is running and so it needs not be specified.
 *
 * OUTPUTS: 
 * Returns FALSE if there was an error detected, TRUE otherwise.
 *
 *****************************************************************************/

BOOLEAN nannyServerInitialize(PARSER_FN outputParser,
			      PARSER_FN localParser,
			      REMOTE_PARSER_FN remoteParser)
{
  static char* thisHost;
  
  thisHost = (char *)malloc(80);
  
#ifndef VXWORKS
  gethostname(thisHost, 80);  
#else
  thisHost[0]='\0';
#endif
  
  thisHost = strcat(thisHost,NANNY_SERVER_NAME);
  
  nanny_device.dev = devCreateDev(thisHost, 
				  DEV_LISTENING, 
				  (LISTENING | TALKING | ACCEPTING),
				  DEV_OUTPUTHND, Nanny_outputHnd,
				  DEV_PORT_NUMBER, NANNY_SERVER_PORT,
				  DEV_RECONNECTHND, devServerInitialize,
				  NULL
				  );
  
  remoteNanny_device.remoteParser = remoteParser;
  local_device.parser = localParser;
  program_device.parser = outputParser;
  
  return devServerInitialize(nanny_device.dev);
}

/******************************************************************************
 *
 * FUNCTION: void nanny_connect(PARSER_FN localParser)
 *
 * DESCRIPTION: 
 * Starts the nanny device.  Does not actually connect until a message is sent.
 *
 * OUTPUTS: 
 *
 *****************************************************************************/

void nanny_connect(PARSER_FN localParser)
{
  devInit();
  local_device.parser = localParser;
  local_device.dev = devCreateDev("Local Nanny server", 
				  DEV_LISTENING, 
				  (LISTENING | TALKING),
				  DEV_OUTPUTHND, local_outputHnd,
				  DEV_PORT_NUMBER, NANNY_SERVER_PORT,
				  NULL
				  );
}

NANNY_DEV_PTR theLocalDevice(void)
{ 
  return &local_device;
}

/* DESCRIPTION:
 * Read nchars of data from sd into buf. Continue to read until 
 * nchars have been read.  Return 0 if end of file reached and -1 if
 * there was an error.
 *
 * INPUTS: 
 * int sd;
 * char *buf;
 * int nchars;
 *
 * OUTPUTS: int numRead.
 *
 * NOTES:
 * 
 * buf is a preallocated pointer to storage equal to nchars.
 * Propagation of low level errors - failures on read/writes still an issue.
 */
static int readNChars(int sd, char *buf, int nchars)
{
  int amountRead, amountToRead;
  int i;
  
  amountToRead = nchars;
  for(;;){
    amountRead = read(sd, buf, amountToRead);
    if (amountRead == 0)
      { /* just got an end of file indication. 
	 * close the socket and remove it from the list of active
	 * connections 
	 */
	for(i=1; ((i<nanny_device.numDevs) && 
		  !FD_ISSET(sd,devGetFds(nanny_device.devs[i]->dev)));
	    i++);
	if(FD_ISSET(sd,devGetFds(nanny_device.devs[i]->dev))) {
	  DEBUG2("Connection closed %s\n",nanny_device.devs[i]->dev->devName);
	  devDisconnectDev(nanny_device.devs[i]->dev,TRUE,TRUE);
	}
	return 0;
    } else if (amountRead < 0)
      /* 
       * some other problem.  Try to continue?
       */
      return amountRead;
    amountToRead -= amountRead;
    if (amountToRead == 0)
      return amountRead;
    buf += amountRead;
  }
}

/* DESCRIPTION: Handles Nanny events. Only events are connection requests.
 *
 * INPUTS:
 *
 * HISTORY:
 */
void Nanny_outputHnd(int fd, long ignore)
{
  int new_sd=NO_FD;
  long chars_available;
  char inbuf[DEFAULT_LINE_LENGTH+1];
  fd_set readSet;
  /* int sockFlags = 0;*/
  struct timeval zeroTimeout = {0,0};
  
  DEBUG1("New Connection Requested \n");
  DEBUG2("fd: %d\n", fd);
  DEBUG2("Original fd settings %d\n",fcntl(fd, F_GETFL, FNDELAY));
  /* sockFlags = fcntl(fd, F_GETFL, FNDELAY);*/
  /* fcntl(new_sd, F_SETFL, sockFlags | FNDELAY | FNONBIO | FNBIO );*/
  /* DEBUG3("New fd settings %d, should be %d\n",fcntl(fd, F_GETFL, FNDELAY),
     sockFlags | FNDELAY | FNONBIO | FNBIO);*/
  FD_ZERO(&readSet);
  FD_SET(fd,&readSet);
  if (select(FD_SETSIZE, &readSet, NULL, NULL, &zeroTimeout) == 1) {
    if (FD_ISSET(fd,&readSet)) {
      new_sd = accept(fd, NULL, NULL);
    }
  }
  /* fcntl(fd, F_SETFL, sockFlags);*/
  /* DEBUG2("Reset fd settings %d\n",fcntl(fd, F_GETFL, FNDELAY));*/
  if (new_sd <= 0) {
    fprintf(stderr, "Interrupted accept system call...\n");fflush(stderr);
    return;
  }
  /* fcntl(new_sd, F_SETFL, sockFlags);*/
  /*
   * This test on accept was missing previously.
   */
  if (new_sd == -1) {
    fprintf(stderr, "%d\n", errno);
    perror("Nanny_outputFd:");
  }
  /* Write a string to the new socket and wait for 
   * a reply to indicate which device to connect to
   */
  
  write(new_sd, devGetName(nanny_device.dev), 
	strlen(devGetName(nanny_device.dev)));
  
  do {
    /* wait for the name of the nanny */
    FD_ZERO(&readSet);
    FD_SET(new_sd,&readSet);
    select(FD_SETSIZE, &readSet, NULL, NULL, NULL);
    chars_available = devCharsAvailable(new_sd);
  } while (chars_available <= 0);
  
  /* read in the device name. */
  bzero(inbuf,DEFAULT_LINE_LENGTH+1);
  if (readNChars(new_sd,inbuf,MIN(chars_available,DEFAULT_LINE_LENGTH)) <= 0)
    /* got a bad read */
    return;
  
  DEBUG2("Requested from %s \n", inbuf);
  
  /* Received a new request, check to see what kind */
  
  if (strstr(inbuf,NANNY_SERVER_NAME) != NULL) {
    Nanny_addNannyDevice(new_sd,inbuf);
  } else {
    Nanny_addUserDevice(new_sd,inbuf);
  }
  
  /* echo back the name of the connecting device */
  write(new_sd, inbuf, strlen(inbuf));
}

/* DESCRIPTION: Adds new nanny device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 *
 */
void Nanny_addNannyDevice(int sd, char *name)
{
  NANNY_DEV_TYPE *newDev = NULL;
  
  if (nanny_device.numDevs >= MAX_NANNY_DEVS) {
    WARNING1("Too many devices, increase MAX_NANNY_DEVS\n");
    return;
  }
  
  newDev = (NANNY_DEV_TYPE *)malloc(sizeof(NANNY_DEV_TYPE));
  *newDev = remoteNanny_device;
  newDev->dev = devCreateDev("Remote Nanny server", 
			     DEV_LISTENING, 
			     (LISTENING | TALKING),
			     DEV_OUTPUTHND, remoteNanny_outputHnd,
			     DEV_PORT_NUMBER, NANNY_SERVER_PORT,
			     NULL
			     );
  
  /*  newDev->dev.devName = name;*/
  
  insertNannyDev(newDev);
  
  DEBUG2("Connecting to nanny device %s\n", name);
  
  /* connect to devUtils loop */
  devConnectDev(newDev->dev,sd);
}

/* DESCRIPTION: Adds new user local device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
void Nanny_addUserDevice(int sd, char *name)
{
  NANNY_DEV_TYPE *newDev = NULL;
  
  if (nanny_device.numDevs >= MAX_NANNY_DEVS) {
    WARNING1("Too many devices, increase MAX_NANNY_DEVS\n");
    return;
  }
  
  newDev = (NANNY_DEV_TYPE *)malloc(sizeof(NANNY_DEV_TYPE));
  *newDev = remoteNanny_device;
  newDev->dev = devCreateDev("Local Nanny server", 
			     DEV_LISTENING, 
			     (LISTENING | TALKING),
			     DEV_OUTPUTHND, local_outputHnd,
			     DEV_PORT_NUMBER, NANNY_SERVER_PORT,
			     NULL
			     );
  
  /*  newDev->dev.devName = name;*/
  
  insertNannyDev(newDev);
  
  DEBUG2("Connecting to nanny device %s\n", name);
  
  /* connect to devUtils loop */
  devConnectDev(newDev->dev,sd);
}

/* DESCRIPTION: Handle characters from another nanny device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
void  remoteNanny_outputHnd(int fd, long chars_available)
{
  static char delimitChar = '\0';
  NANNY_DEV_PTR dev;
  
  dev = getNannyDev(fd);
  if (dev == NULL) 
    return;
  
  if (dev->lineBuf == NULL) {
    dev->lineBuf = devCreateLineBuffer(DEFAULT_LINE_LENGTH * 3,
				       &delimitChar, remoteNannyParse,
				       TRUE, FALSE,
				       (void *)getNannyDev(fd)->dev);
  }
  
  devProcessOutput(dev->dev, fd, dev->lineBuf, chars_available);
}


/* DESCRIPTION: Handle characters from another nanny device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
void remoteNannyParse(char *line, void *clientData)
{
  RDEBUG2("Calling remote parser with [%s]\n",line);
  (remoteNanny_device.remoteParser)(devGetName(((DEV_PTR)clientData)), line);
}

/* DESCRIPTION: Handle characters from another nanny device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
void  local_outputHnd(int fd, long chars_available)
{
  static char delimitChar = '\0';
  NANNY_DEV_PTR dev;
  
  dev = getNannyDev(fd);
  if (dev == NULL) 
    return;
  
  if (dev->lineBuf == NULL) {
    dev->lineBuf = devCreateLineBuffer(DEFAULT_LINE_LENGTH * 3,
				       &delimitChar, localNannyParse,
				       TRUE,FALSE,
				       (void *)(long)fd);
  }
  
  devProcessOutput(dev->dev, fd, dev->lineBuf, chars_available);
}


/* DESCRIPTION: Handle characters from a local interface.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
void localNannyParse(char *line, void *clientData)
{
  RDEBUG2("Calling local parser with [%s]\n",line);
  (local_device.parser)((int)clientData, line);
}

/* DESCRIPTION: Handle characters from another nanny device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
void  program_outputHnd(int fd, long chars_available)
{
  static char delimitChar = '\n';
  NANNY_DEV_PTR dev=NULL;
  
  dev = getNannyDev(fd);
  if (dev == NULL) 
    return;
  
  if (dev->lineBuf == NULL) {
    dev->lineBuf = devCreateLineBuffer(DEFAULT_LINE_LENGTH * 3,
				       &delimitChar, programNannyParse,
				       FALSE,TRUE,
				       (void *)(long)fd);
  }
  if (dev->dev != NULL) {
    devSetLineBufferData(dev->lineBuf, (void *) fd);
    devProcessOutput(dev->dev, fd, dev->lineBuf, chars_available);
  }
}


/* DESCRIPTION: Handle characters from a program interface.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
void programNannyParse(char *line, void *clientData)
{
  RDEBUG2("Calling program parser with [%s]\n",line);
  (program_device.parser)((int)clientData, line);
}

/* DESCRIPTION: Send output to the local device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
BOOLEAN passLocalMessage(char *message)
{
  RDEBUG2("Pass Local Message [%s]\n",message);
  if(!devIsConnected(local_device.dev)) {
    /* Have to connect to the local nanny.   */
    if (! connectToLocalNanny())
      return FALSE;
  }
  
  return devWriteN(devGetFd(local_device.dev),message,strlen(message)+1);
}

/* DESCRIPTION: Send output to the local device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
BOOLEAN passLocalReply(int id,char *message)
{
  NANNY_DEV_PTR dev;
  
  RDEBUG2("Pass Local Reply [%s]\n",message);
  dev = getNannyDev(id);
  if (dev == NULL) {
    WARNING1("Local Reply: Unknown device.\n");
    return FALSE;
  }
  
  if(!devIsConnected(dev->dev)) {
    WARNING1("Local Reply: Tried to write to disconnected device.\n");
    return FALSE;
  }
  
  return devWriteN(devGetFd(dev->dev),message,strlen(message)+1);
}

/* DESCRIPTION: Establish a connection to aanother nanny server.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
static NANNY_DEV_PTR connectToRemoteNanny(char *machine)
{
  static char thisHost[DEFAULT_LINE_LENGTH+1] = "";
  int sd;
  char inbuf[DEFAULT_LINE_LENGTH+1];
  long chars_available=0;
  fd_set readSet;
  int result=0;
  NANNY_DEV_PTR newDev = NULL;
  
  DEBUG2("Trying to open connection to nanny: %s\n",machine);
  
  if(strlen(thisHost) == 0)
#ifndef VXWORKS
    gethostname(thisHost, 80);  
#else
  thisHost[0]='\0';
#endif
  
  newDev = (NANNY_DEV_PTR)malloc(sizeof(NANNY_DEV_TYPE));
  *newDev = remoteNanny_device;
  newDev->dev = devCreateDev("Remote Nanny server", 
			     DEV_LISTENING, (LISTENING | TALKING),
			     DEV_OUTPUTHND, remoteNanny_outputHnd,
			     DEV_SIM_MACHINE, strdup(machine),
			     DEV_PORT_NUMBER, NANNY_SERVER_PORT,
			     NULL
			     );
  
  sd = devConnectToSocket(newDev->dev);
  if (sd != -1) {
    /* write the device name to the socket to allow the simulator
     * to know what device we wanted to open.
     */
    
    DEBUG2("Opened socket to nanny: %s\n",machine);
    
    do {
      /* wait for the name of the simulator */
      FD_ZERO(&readSet);
      FD_SET(sd,&readSet);
      select(FD_SETSIZE, &readSet, NULL, NULL, NULL);
      
      chars_available = devCharsAvailable(sd);
      if (chars_available == -1)
	WARNING1("POLL ERROR on socket\n");
    } while (chars_available <= 0);
    
    bzero(inbuf,DEFAULT_LINE_LENGTH+1);
    result = devReadN(sd, inbuf,
		      MIN(chars_available,DEFAULT_LINE_LENGTH));
    
    if (result <= 0)
      { /* cound not read the socket */
	free(newDev);
	return NULL;
      }
    
    DEBUG2("Connected to Nanny %s\n",inbuf);
    
    devWriteN(devGetFd(newDev->dev),devGetName(newDev->dev),
	      strlen(devGetName(newDev->dev)));
    bzero(inbuf,DEFAULT_LINE_LENGTH+1);
    devReadN(sd,inbuf,strlen(devGetName(newDev->dev)));
  }
  
  insertNannyDev(newDev);
  return newDev;
}

/* DESCRIPTION: Establish a connection to aanother nanny server.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
BOOLEAN connectToLocalNanny(void)
{
  int sd;
  char inbuf[DEFAULT_LINE_LENGTH+1];
  long chars_available=0;
  fd_set readSet;
  int result=0;
  
  DEBUG1("Trying to open connection to local nanny: \n");
  
  sd = devConnectToSocket(local_device.dev);
  if (sd != -1) {
    /* write the device name to the socket to allow the simulator
     * to know what device we wanted to open.
     */
    
    DEBUG1("Opened socket to local nanny.\n");
    
    do {
      /* wait for the name of the simulator */
      FD_ZERO(&readSet);
      FD_SET(sd,&readSet);
      select(FD_SETSIZE, &readSet, NULL, NULL, NULL);
      
      chars_available = devCharsAvailable(sd);
      if (chars_available == -1)
	WARNING1("POLL ERROR on socket\n");
    } while (chars_available <= 0);
    
    bzero(inbuf,DEFAULT_LINE_LENGTH+1);
    result = devReadN(sd, inbuf,
		      MIN(chars_available,DEFAULT_LINE_LENGTH));
    
    if (result <= 0)
      { /* cound not read the socket */
	return FALSE;
      }
    
    DEBUG2("Connected to Nanny %s\n",inbuf);
    
    devWriteN(devGetFd(local_device.dev),devGetName(local_device.dev),
	      strlen(devGetName(local_device.dev)));
    bzero(inbuf,DEFAULT_LINE_LENGTH+1);
    devReadN(sd,inbuf,strlen(devGetName(local_device.dev)));
    insertNannyDev(&local_device);
    return TRUE;
  } else {
    return FALSE;
  }
}

/* DESCRIPTION: Send output to the local device.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 * HISTORY:
 */
BOOLEAN passMessage(char *machine, char *message)
{
  NANNY_DEV_PTR dev;
  
  RDEBUG2("Pass Message [%s]",machine);
  RDEBUG2("[%s]\n",message);
  dev = findNannyDev(machine);
  
  if( dev == NULL) {
    /* Have to start a connection. */
    dev = connectToRemoteNanny(machine);
  }
  
  if ((dev != NULL) &&(devIsConnected(dev->dev))) {
    devWriteN(devGetFd(dev->dev),message,strlen(message)+1);
    return TRUE;
  } else {
    return FALSE;
  }
}


/* DESCRIPTION:
 * To handle output from processes created by nanny, we inform
 * the nanny monitoring loop that a fd exists, and thereafter information 
 * arriving on that fd will cause a function 
 *      processOutputMessage(char *name, char *message)
 * to be executed.
 *
 * INPUTS:
 * 
 * fd   = file descriptor to monitor.
 * name = a string to be use for identifying messages 
 *        (a typical string could be "CTR")
 * OUTPUTS: 
 *
 * HISTORY:
 */
NANNY_DEV_PTR Nanny_addFd(int stdinFd, int stdoutFd, int stderrFd, char *name)
{
  NANNY_DEV_TYPE *newDev = NULL;
  char *devName;
  
  if (nanny_device.numDevs >= MAX_NANNY_DEVS) {
    WARNING1("Too many devices, increase MAX_NANNY_DEVS\n");
    return NULL;
  }
  
#if 0
  devName = (char *)malloc(strlen(name)+1);
  strcpy((char *)devName,name);
#else
  devName = name;
#endif
  newDev = (NANNY_DEV_TYPE *)malloc(sizeof(NANNY_DEV_TYPE));
  *newDev = program_device;
  newDev->dev = devCreateDev(devName, 
			     DEV_LISTENING, 
			     (LISTENING | TALKING),
			     DEV_OUTPUTHND, program_outputHnd,
			     DEV_PORT_NUMBER, NANNY_SERVER_PORT,
			     /* DEV_DEBUG, TRUE,*/
			     NULL
			     );
  
  
  insertNannyDev(newDev);
  
  DEBUG2("Connecting to program %s\n", name);
  
  /* connect to devUtils loop */
  devConnectDevReceiveOnly(newDev->dev,stdoutFd);
  devConnectDevReceiveOnly(newDev->dev,stderrFd);
  devConnectDevSendOnly(newDev->dev,stdinFd);
  return newDev;
}


/* DESCRIPTION:
 * Remove a file descriptor from the set of valid file descriptors.
 *
 * INPUTS:
 * 
 * fd   = file descriptor to stop monitoring.
 * 
 * OUTPUTS: 
 *
 * HISTORY:
 */
BOOLEAN Nanny_removeFd(int fd)
{
  NANNY_DEV_PTR dev;
  
  dev = getNannyDev(fd);
  if (dev == NULL) {
    WARNING2("Nanny_removeFd: Tried to remove unkown device %d\n", fd);
    return FALSE;
  }
  
  DEBUG2("Disconnecting from program %s\n", dev->dev->devName);
  
  devDisconnectDev(dev->dev, FALSE, FALSE);
  
  removeNannyDev(dev);
  
  return TRUE;
}




/* DESCRIPTION:
 * Remove a file descriptor from the set of valid file descriptors.
 *
 * INPUTS:
 * 
 * fd   = file descriptor to stop monitoring.
 * 
 * OUTPUTS:
 *
 * HISTORY:
 */
static void Nanny_reallyFreeDev(void)
{
  int i;

  for(i=0;i < nanny_device.numDevs; i++)
    if ((nanny_device.devs[i] != NULL) && 
	(nanny_device.devs[i]->dev == NULL)) {
      if (nanny_device.devs[i]->lineBuf != NULL)
	devFreeLineBuffer(nanny_device.devs[i]->lineBuf);
      nanny_device.devs[i]->lineBuf = NULL;
      free(nanny_device.devs[i]);
      nanny_device.devs[i] = NULL;
    }

  freeNeeded = FALSE;
}


