/*****************************************************************************
 * PROJECT: Xavier
 *
 * (c) Copyright 1993 Richard Goodwin & Joseph O'Sullivan. All rights reserved.
 *
 * FILE: nanny.c
 *
 * ABSTRACT:
 *
 * Nanny main program.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/nanny.c,v $
 * $Revision: 1.10 $
 * $Date: 1996/06/28 14:07:22 $
 * $Author: reids $
 *
 * REVISION HISTORY:
 *
 * $Log: nanny.c,v $
 * Revision 1.10  1996/06/28  14:07:22  reids
 * Fixed quite a few bugs -- with graphics, interaction with script, and
 *   killing processes
 *
 * Revision 1.9  1996/02/20  11:40:30  josullvn
 * Made extensive changes:
 *    added xfRunConsole, an xforms based interface which was quick
 *    To change the interface, from the nanny dir run
 *    /afs/cs/user/josullvn/xforms/DESIGN/fdesign xfConsole
 *    to get going, and made debugging easier.
 *    added tons of comments.
 *    Debug spews a lot more info back, so just compile
 *    with DEBUG=DEBUG if you really need it.
 *    I've added some timeouts to the code, which will
 *    probably be tweaked later. There are in xfCallbacks.c
 *    Its become better at keeping processes going, and
 *    so its harder to quit. Using shutdown can be
 *    painful. Killing the processes individually is
 *    probably better. Need to do a reverse dependancy when quitting.
 *    Multiple machines are again poor. It doesn't wait sufficently
 *    for dependencies - so you need to run things explicitly...
 *
 * Revision 1.8  1996/02/10  16:52:16  rich
 * Made private functions static and fixed some forward declarations.
 *
 * Revision 1.7  1996/02/07  00:28:10  rich
 * Add prefix to VERSION_DATE and COMMIT_DATE.
 *
 * Revision 1.6  1996/02/05  15:56:48  reids
 * Added scripting capabilities to nanny -- using lex/bison to parse script
 *  files according to the script.bnf standard.
 * Integrated support for running/stopping/suspending scripts into the
 *  runConsole window.
 *
 * Revision 1.5  1996/02/01  02:03:14  reids
 * Cleaned up/standardized command line option parsing and help messages.
 *
 * Revision 1.4  1996/01/31  22:51:53  reids
 * Added automatic updating of (micro) version control numbers
 *
 * Revision 1.3  1996/01/22  21:27:47  reids
 * Consolidated all the DEBUG and WARNING macro definitions into nannyCommon.h
 *
 * Revision 1.2  1996/01/05  16:33:30  rich
 * Nanny fixes.
 *
 * Revision 1.1  1995/12/17  20:25:43  rich
 * Moved Nanny to the tca release.
 *
 * Revision 1.29  1995/08/14  22:40:38  rich
 * Changes for the new functional devUtils (8.1.6).
 * "-clean" option on nanny now prompts before killing tasks, unless the
 * "-noprompt" flag is used.
 *
 * Revision 1.28  1995/08/06  00:05:01  rich
 * Changes for new devUtils in tca-8.1.
 *
 * Revision 1.27  1995/07/30  23:22:45  rich
 * Seperated robotConfig from config.  Added more parameters.
 *
 * Revision 1.26  1995/07/30  02:01:16  rich
 * Moved common defines to etc/GNUmakefile.defs.
 * Added loadRobotConfig.
 *
 * Revision 1.25  1995/07/15  07:13:08  josullvn
 * ProcessDevices is _not_ reentrant - this was causing lots of problems as it
 * turns out. Restructing my blocking code fixed it.
 * Also, added a notion of dead - that is telling when something has
 * finally died.
 *
 * Revision 1.24  1995/07/13  17:04:29  josullvn
 * Added version info
 *
 * Revision 1.23  1995/07/13  11:09:46  josullvn
 * Two bugs were present. a) syscalls were being interrupted, and b) the
 * processes button management got screwy after a while. The first is due
 * to Devutils, second fixed by removing a reliance on xclient data. Also
 * added features to newProcess whereby it is now menu driven.
 * Added tca.rc, removing those processes from the Simulator resource files.
 *
 * Revision 1.22  1995/07/12  07:40:00  josullvn
 * Filled in README, updated resource file (Plus added Simulator2.rc)
 * Changed some messages on startup, and added a bugs file
 *
 * Revision 1.21  1995/07/11  22:59:14  josullvn
 * Needed to retain the concept of who sent a message, and so had to
 * restructure all the message passing. Also, made sure we don't
 * overwrite any of the parameters due to reentry problems.
 *
 * Revision 1.20  1995/07/11  11:21:33  josullvn
 * A lot more changes to message passing to handle multiple nannys better.
 * This basically involved removing fdclient, and being smarter about what
 * nannys are in the system
 *
 * Revision 1.19  1995/07/11  08:01:22  josullvn
 * Improved -clean option
 *
 * Revision 1.18  1995/07/11  01:06:31  josullvn
 * Whesh. OK, fixed a bug in parsing lines, where by if messages get corrupted,
 * we can recover. Added a -clean option to nanny. Discovered that the old
 * double newProcess bug has returned to haunt me.
 *
 * Revision 1.17  1995/06/25  01:03:40  rich
 * Moved devUtils to tca.
 *
 * Revision 1.16  1995/06/23  21:47:28  robocomp
 * Added TCA=linx to GNUmakefile
 * Removed some unLinux code
 *
 * Revision 1.15  1995/06/02  23:02:48  rich
 * Added support for OSF 2.
 *
 * Revision 1.14  1995/05/23  23:58:37  josullvn
 * Yeah. Fixed the SIGCHLD problem (tured out that popen is internally
 * implemented with fork - when pclose was called, it generated a SIGCHLD
 * which intereupted the system call which lead to trouble).
 * Fixed environment variables, can now add env variables in .rc file, and
 * they are passed to appropriate child. Realized that need to also
 * provide an ability to pass the display variable from runConsole to the
 * nanny - being worked on.
 * Bug exists in devUtils (?) which large data streams. Run csh as a new process
 * and do ps auxww to cause it to occur.
 *
 * Revision 1.13  1995/05/20  02:07:32  josullvn
 * Some more bells and whistles to the console callback.
 * Added environment variables to the resource definition.
 * Extended the still buggy children catching for time outs.
 * Added diagnositic code to analyze startup failures.
 *
 * Revision 1.12  1995/05/19  17:56:15  rich
 * Include the null character in the strings sent.
 * Added back the stdin device.  Made runConsole a public bin.
 *
 * Revision 1.11  1995/05/19  11:32:27  josullvn
 * Updated resource files to include "ready strings"
 * Debugged some protocol problems with messages.c parsing of split lines.
 * (Feel that remaining problem is in devUtils).
 * Added Highlightening to buttons.
 * Removed SIGCHLD trapping - its too flakey at the moment.
 *
 * Revision 1.10  1995/05/18  02:11:46  josullvn
 * hehe  - now error messages go to both windows.
 *
 * Revision 1.9  1995/05/17  23:39:32  josullvn
 * Now restarts dead processes - need to add timeouts.
 *
 * Revision 1.8  1995/05/15  17:16:46  rich
 * Added nanny.h and moved external declarations to the .h files.
 *
 * Revision 1.7  1995/05/12  12:58:22  josullvn
 * Wheee. Running locally, its just about ready.
 * Catching and restarting children isn't fully implemented yet, reloading
 * leaves to be desired, and there is still those new widget and filter bugs.
 * The simulator.c is also updated. Since the -s optoin to the simulator can't
 * handle localhost, it is now geared up for stomach. starting, say, navigate
 * will cause a chain of dependencies to be fired.
 *
 * Revision 1.6  1995/05/11  10:23:56  josullvn
 * Just about full handling of nanny-nanny and nanny-client messages in the
 * code. Bugs with the actual communication.
 * Also added a sample simulator resource file
 *
 * Revision 1.5  1995/05/11  01:22:51  rich
 * Both now compile.
 *
 * Revision 1.4  1995/05/10  18:33:04  rich
 * Added the routines to handle program output.
 *
 * Revision 1.3  1995/05/10  06:38:04  josullvn
 * More fiddling - added some stubs to nannyDev.h...
 *
 * Revision 1.2  1995/05/10  05:42:54  josullvn
 * More file shuffling... now have runConsole and nanny
 *
 * Revision 1.1  1995/05/10  02:39:28  rich
 * Forgot to add the nanny files.
 *
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "tca/devUtils.h"
#include "tca/stdinDev.h"

#include "nannyCommon.h"
#include "nanny.h"
#include "nannyDev.h"
#include "nannyUtils.h"
#include "resource.h"
#include "messages.h"

char *rcFileName = NULL;

char **environment;

static BOOLEAN clean = FALSE;
static BOOLEAN prompt = TRUE;

/*
 * DESCRIPTION:
 *  Catch all signals, 
 * INPUTS:
 *  
 */
static void sigint(int code)
{
  fprintf(stderr, "Interrupt signal received %d\n", code);
  signal(SIGINT, sigint);
}

/* DESCRIPTION: Handles character typed from stdin.
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void stdin_inputHnd(int fd, long chars_available)
{
  static char buffer[DEFAULT_LINE_LENGTH+1];
  int numRead=0;
  char *buffPtr;
  
  bzero(buffer, DEFAULT_LINE_LENGTH+1);
  
  numRead = devReadN(fd, buffer,
		     MIN(chars_available,DEFAULT_LINE_LENGTH));
  for(buffPtr = buffer; ((*buffPtr == ' ') && (buffPtr != '\0')); buffPtr++);
  if (strstr(buffer,"q")) {
    nannyShutdown();
    exit(0);
  } else if (strstr(buffer,"h")) {
    printf("h: Help, l: list connections, p: print processes, q: Quit");
  } else if (strstr(buffer,"l")) {
    nannyConnectList(stdout);
  } else if (strstr(buffer,"p")) {
    int i; 
    printf("Executing Processes are:");
    for (i=0; i<rcNumAvailablePrograms; i++) { 
      rcProgramPtr r = &(rcAvailableProgramList[i]);
      if (r->executing) printf(" %s",r->name);
    }
  }
  printf("\n> ");
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void processOutputMessage(int id, char *message)
{
  int i;
  rcProgramPtr r=NULL;
  char buf[MAXBUF];
  BOOLEAN found = FALSE;

  for (i=0; i<rcNumAvailablePrograms && !found; i++) {
    r = &(rcAvailableProgramList[i]);
    
    if (r->executing && r->local) {
      if (r->fdstdout == id) {
	rcInsertStr(i, rcEntIBUF, message);
	if (r->loudness & VERBOSE) {
	  messageMakeInfo(buf, r->name, FALSE, message);
	  nannyReportDualMessage(buf);
	}
	found = TRUE;
      } else if (r->fdstderr == id) {
	rcInsertStr(i, rcEntEBUF, message);
	rcInsertStr(i, rcEntIBUF, message);
	if  (r->loudness & QUIET) {
	  messageMakeInfo(buf, r->name, TRUE, message);
	  nannyReportDualMessage(buf);
	}
	found = TRUE;
      }
    }
  }
  if (found) {
    if (!r->ready && r->ready_string) {
      r->ready = (strstr(r->stdinBuffer, r->ready_string) != NULL);
      if (r->ready) {
	messageMakeAnnounce(buf, r->name, TRUE);
	nannyReportDualMessage(buf);
      }
    }
  } else {
    printf("processOutputMessage Stub called with unknown id, %d\n", id);
  }
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void processLocalMessage(int id, char *message)
{
  parseLine(TRUE, "", id, message);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void processNannyMessage(const char *machine, char *message)
{
  parseLine(FALSE, machine, -1, message);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void displayVersion(void)
{
  fprintf(stderr, "nanny Server %d.%d.%d \n",
	  NANNY_VERSION_MAJOR, NANNY_VERSION_MINOR, NANNY_VERSION_MICRO);
  fprintf(stderr, " Released : %s\n", NANNY_VERSION_DATE);
  fprintf(stderr, " Commited : %s\n", NANNY_COMMIT_DATE);
  fprintf(stderr, " Compiled : %s %s\n", __DATE__, __TIME__);
  fprintf(stderr, "Nanny process Id : %d\n", getpid());
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void printHelp (void)
{
  printf("Usage: nanny [options]:\n");
  printf(" -h | -help:   Print this message\n");
  printf(" -v        :   Display version information\n");
  printf(" -load <file>: Forces nannys to load resources from file.\n");
  printf(" -clean:       Will kill any competing processes at initialization time.\n");
  printf(" -noprompt:    Do not prompt the user before killing a process.\n");
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
static void parseCommandLineOptions (int argc, char **argv)
{
  int i;

  for (i=1; i<argc && argv[i][0]=='-'; i++) {
    if (!strcmp(argv[i], "-load") && i+1<argc) {
      rcFileName = (argv[++i]);
    } else if (!strcmp(argv[i], "-clean")) {
      clean = TRUE;
    } else if (!strcmp(argv[i], "-noprompt")) {
      prompt = FALSE;
    } else if (!strcmp(argv[i], "-help") || !strcmp(argv[i], "-h"))  {
      printHelp();
      exit(1);
    } else if (!strcmp(argv[i], "-v"))  {
      displayVersion();
      exit(1);
    } else {
      WARNING2("Unknown option: %s\n", argv[i]);
      printHelp();
      exit(-1);
    }
  }
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void main(int argc, char *argv[], char *env[])
{  
  parseCommandLineOptions(argc, argv);
  displayVersion();

  environment  = env;

  if (!rcLoadResourceFile(rcFileName))
    exit (-1);
  
  if (TRUE) {
    WARNING1("Checking existing processes - please wait...\n"); 
    nannyCheckEmptySystem(clean, prompt);
  }
  
  /* Do not enable the following yet... */
  /* nanny_daemon_start(); */
  
  /* Enable the following to restart children when they die */
  signal(SIGCHLD, nannyCatchChild);
  
  /* connect up the handlers for standard in. */
  stdin_connect(stdin_inputHnd); 
  
  WARNING1("h: Help, q: Quit\n>\n");
  WARNING1("Trying to open sockets, please wait for a further second or so...\n");
  WARNING1("(After that, if no further text appears, check for conflicting processes)\n");

  nannyServerInitialize(processOutputMessage,
			processLocalMessage,
			processNannyMessage);
  /* loop forever in the main device loop */
  devMainLoop();
  
  /* should never reach here. */
  devShutdown();
  exit(0);
}

#if 0
/*
 * DESCRIPTION:
 *   See Unix Network Programming, W. Richards Stevens
 *   Thought at one stage that nanny would be run as a kernel deamon.
 *   Realized that that was way too complex, but here are the relics.
 * 
 * INPUTS:
 *
 * OUTPUTS:
 */
void nanny_daemon_start(void)
{
  register int childpid, fd;
  
  /* If we are started by init (process 1) from the /etc/inittab file
   * there's no need to detach. The test is a bit unreliable due to 
   * unavoidable ambiguities
   */
  if (getppid() != 1) {
    /* Ignore various stop signals in BSD */
#ifdef SIGTTOU
    signal(SIGTTOU, SIG_IGN);
#endif
#ifdef SIGTTIN
    signal(SIGTTIN, SIG_IGN);
#endif
#ifdef SIGTTSTP
    signal(SIGTTSTP, SIG_IGN);
#endif
    
    /* 
     * if we are not started in the background, fork and let the 
     * parent exit. This guarantess the first child is not a 
     * process group leader.
     */
    if ( (childpid = fork() ) < 0) { 
      WARNING1("can't fork first child\n");
      exit(0);
    }
    else if (childpid > 0)
      exit(0);		/* parent */
    
    /* 
     * Definately a child process
     * Disassociate from the terminal and process group, 
     * Ensure that the process can't reacquire a new terminal
     */
    
#ifdef SIGTSTP
    if (setpgrp(0, getpid()) == -1) { 
      WARNING1("can't change process group\n");
      exit(0);
    }
    
    if ( (fd = open("/dev/tty", O_RDWR)) >= 0) {
      ioctl(fd,TIOCNOTTY,(char *)NULL); /* lose controlling tty */
      close(fd);
    }
#else
    if (setpgrp() == -1) { 
      WARNING1("can't change process group\n");
      exit(0);
    }
    
    signal(SIGHUP,SIG_IGN);	/* immune from pgrp leader death */
    
    if ( (childpid = fork() ) < 0) { 
      WARNING1("can't fork second child\n");
      exit(0);
    }
    else if (childpid > 0)
      exit(0);		/* first child */
#endif
  }
  
  /* 
   * close open descriptors.
   */
#ifdef PRODUCTION_TIME
  for (fd = 0; fd < NOFILE; fd++)
    close(fd);
  errno = 0; /* Surely some bad close's occurred */
#endif
  
  /*
   * move the current directory to root, to make sure we don't 
   * interfere with a mounted filesystem.
   */
  chdir("/");
  
  /*
   * Clear any inherited file mode creation masks.
   */
  umask(0); 
  
  /*
   * catch those darn kids in order to restart em
   */
  signal(SIGCHLD, nannyCatchChild);
}
#endif

