/*****************************************************************************
 * PROJECT: Xavier
 *
 * (c) Copyright 1993 Richard Goodwin & Joseph O'Sullivan. All rights reserved.
 *
 * FILE: messageConsoleHandle.c
 *
 * ABSTRACT:
 *    Deals with actions based on messages from the nanny. 
 *    three types of message in fact exist, 
 *      a communciation message eg stdin, stdout etc
 *      a command message eg Announce
 *      an ack/nack message which communicates from the nanny.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/messageConsoleHandle.c,v $
 * $Revision: 1.11 $
 * $Date: 1996/07/29 05:03:05 $
 * $Author: josullvn $
 *
 * REVISION HISTORY:
 *
 * $Log: messageConsoleHandle.c,v $
 * Revision 1.11  1996/07/29  05:03:05  josullvn
 * Bloody hell. This commiting is a bit different than under Xavier. Short
 * story is cleaned up some purify bugs, and also made changes to nanny
 * which should make it a bit better - Improving performance over multiple
 * machines, explict quietening of nondisplayed processes, replacing of
 * runConsole with xfMiniConsole, which is multithreaded, vt102 compilant,
 * adds a uniform emacs-like command line editing feature, better on small
 * screens and otherwise fab.
 *
 * Revision 1.10  1996/07/28  18:31:04  rich
 * Need the check for a currentProc.
 *
 * Revision 1.9  1996/07/18  02:23:04  rich
 * Fixed some memory problems found with purify.
 *
 * Revision 1.8  1996/06/28  14:07:15  reids
 * Fixed quite a few bugs -- with graphics, interaction with script, and
 *   killing processes
 *
 * Revision 1.7  1996/03/29  15:57:13  reids
 * Consolidated the common code between xCallbacks and xfCallbacks, and
 *   xRunConsole and xfRunConsole.
 * Added a way to add macro definitions to resource files ("define: <x> <y>").
 * Fixed a bug that was causing the xfRunConsole to crash when a task was
 *   restarted.
 *
 * Revision 1.6  1996/02/20  11:40:27  josullvn
 * Made extensive changes:
 *    added xfRunConsole, an xforms based interface which was quick
 *    To change the interface, from the nanny dir run
 *    /afs/cs/user/josullvn/xforms/DESIGN/fdesign xfConsole
 *    to get going, and made debugging easier.
 *    added tons of comments.
 *    Debug spews a lot more info back, so just compile
 *    with DEBUG=DEBUG if you really need it.
 *    I've added some timeouts to the code, which will
 *    probably be tweaked later. There are in xfCallbacks.c
 *    Its become better at keeping processes going, and
 *    so its harder to quit. Using shutdown can be
 *    painful. Killing the processes individually is
 *    probably better. Need to do a reverse dependancy when quitting.
 *    Multiple machines are again poor. It doesn't wait sufficently
 *    for dependencies - so you need to run things explicitly...
 *
 * Revision 1.5  1996/02/18  21:17:45  reids
 * Put script stuff back in;
 *   Try to fix pseudo-terminal handling for UNIX and LINUX;
 *   Resize dialog windows when text gets too long.
 *
 * Revision 1.4  1996/02/12  00:46:12  rich
 * Scripting does not compile, removed.
 *
 * Revision 1.3  1996/02/10  16:52:11  rich
 * Made private functions static and fixed some forward declarations.
 *
 * Revision 1.2  1996/01/22  21:30:58  reids
 * Fixed the way stdout is handled, using pseudo-terminals, so that it preserves
 *   the line-buffering mode of the real TTY.
 * Added support for running scripts (not really integrated, yet, but initial
 *   tests are working).
 *
 * Revision 1.1  1995/12/17  20:25:36  rich
 * Moved Nanny to the tca release.
 *
 * Revision 1.21  1995/09/28  19:31:51  josullvn
 * Lots o changes
 *
 * Revision 1.20  1995/09/26  20:35:30  rich
 * Fixed order of system specific libraries.
 * Removed extra "extern"s.
 *
 * Revision 1.19  1995/07/30  02:01:14  rich
 * Moved common defines to etc/GNUmakefile.defs.
 * Added loadRobotConfig.
 *
 * Revision 1.18  1995/07/21  15:13:18  josullvn
 * Added a restarts option - see the README. updated some .rc files to give
 * examples. Also fixed conflicts to that runConsole is notified about dead
 * processes.
 *
 * Revision 1.17  1995/07/15  07:12:57  josullvn
 * ProcessDevices is _not_ reentrant - this was causing lots of problems as it
 * turns out. Restructing my blocking code fixed it.
 * Also, added a notion of dead - that is telling when something has
 * finally died.
 *
 * Revision 1.16  1995/07/13  14:26:25  josullvn
 * Cosmetic mods (printfs etc)
 *
 * Revision 1.15  1995/07/13  11:09:37  josullvn
 * Two bugs were present. a) syscalls were being interrupted, and b) the
 * processes button management got screwy after a while. The first is due
 * to Devutils, second fixed by removing a reliance on xclient data. Also
 * added features to newProcess whereby it is now menu driven.
 * Added tca.rc, removing those processes from the Simulator resource files.
 *
 * Revision 1.14  1995/07/11  22:59:02  josullvn
 * Needed to retain the concept of who sent a message, and so had to
 * restructure all the message passing. Also, made sure we don't
 * overwrite any of the parameters due to reentry problems.
 *
 * Revision 1.13  1995/07/11  11:21:26  josullvn
 * A lot more changes to message passing to handle multiple nannys better.
 * This basically involved removing fdclient, and being smarter about what
 * nannys are in the system
 *
 * Revision 1.12  1995/07/11  01:06:19  josullvn
 * Whesh. OK, fixed a bug in parsing lines, where by if messages get corrupted,
 * we can recover. Added a -clean option to nanny. Discovered that the old
 * double newProcess bug has returned to haunt me.
 *
 * Revision 1.11  1995/07/09  05:25:04  josullvn
 * Added TCA test suite to Simulator.rc - helps with debugging
 * Fixed a display inheritance bug that interfere with dependancies starting up.
 * Fixed a problem with parsing data.
 * Improved communication feedbacks to runConsole from nanny.
 *
 * Revision 1.10  1995/07/04  05:46:21  josullvn
 * The latest update. Some stuff vanished (???), but recovered a bit.
 * Now have improved notification of whats happening between runConsole
 * and nanny, and better starting off of processes.
 *
 * Revision 1.9  1995/06/15  22:11:00  rich
 * Linux 1.2 changes. Still does not compile.
 *
 * Revision 1.7  1995/05/23  23:58:30  josullvn
 * Yeah. Fixed the SIGCHLD problem (tured out that popen is internally
 * implemented with fork - when pclose was called, it generated a SIGCHLD
 * which intereupted the system call which lead to trouble).
 * Fixed environment variables, can now add env variables in .rc file, and
 * they are passed to appropriate child. Realized that need to also
 * provide an ability to pass the display variable from runConsole to the
 * nanny - being worked on.
 * Bug exists in devUtils (?) which large data streams. Run csh as a new process
 * and do ps auxww to cause it to occur.
 *
 * Revision 1.6  1995/05/20  02:07:28  josullvn
 * Some more bells and whistles to the console callback.
 * Added environment variables to the resource definition.
 * Extended the still buggy children catching for time outs.
 * Added diagnositic code to analyze startup failures.
 *
 * Revision 1.5  1995/05/19  11:32:22  josullvn
 * Updated resource files to include "ready strings"
 * Debugged some protocol problems with messages.c parsing of split lines.
 * (Feel that remaining problem is in devUtils).
 * Added Highlightening to buttons.
 * Removed SIGCHLD trapping - its too flakey at the moment.
 *
 * Revision 1.4  1995/05/18  02:11:44  josullvn
 * hehe  - now error messages go to both windows.
 *
 * Revision 1.3  1995/05/15  17:07:08  rich
 * Updated interface to createLineBuffer so you can get partial lines and
 * the delimit character is not replaced.
 * Imporved layout of the X11 window of the console.
 *
 *
 *****************************************************************************/
#include "tca/libc.h"
#include "tca/basics.h"

#include "runConsole.h"
#include "callbacks.h"
#include "messages.h"
#include "messageHandle.h"
#include "history.h"

#include "resource.h"
#include "script.h"

BOOLEAN ACK = FALSE;
char *filterString = NULL;

#define OUTPUT(buf) \
  if (buf[strlen(buf)-1] == '\n') fprintf(stderr, "<- %s", buf); \
  else fprintf(stderr, "<- %s\n", buf);

/*
 * When a process has been tried to be run by the nanny, 
 * this handles the results. If it was executed successfully,
 * we add it to the list of available processes, otherwise add
 * a new report to the error window 
 */
BOOLEAN parseExec(BOOLEAN local, char *remote, int id, char *buf)
{
#if 1
  char console[DEFAULT_LINE_LENGTH];

  fprintf(stderr, "[CONSOLE] Illegal exec message received\n");
  sprintf(console, "[CONSOLE] Illegal exec message received\n");
  MessagePrintf(console);
#else
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);

  deactivateStatus("Ready");
  if (ACK) {
    addProcess(name);
    sprintf(console, "[CONSOLE] %s started sucessfully\n", name); 
  } else {
    sprintf(console, "[CONSOLE] %s failed to start\n", name); 
  }
  fprintf(stderr, console);
  MessagePrintf(console);
  return TRUE;
#endif
}

/*
 * When a process has been been killed
 * this handles the results. If it was killed successfully,
 * we remove it from the list of available processes, 
 * otherwise no change occurs to the error window 
 */
BOOLEAN parseKill(BOOLEAN local, char *remote, int id, char *buf)
{
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];

  buf = messageExtractKey(buf, name);
  messageExtractKey(buf, name);

  deactivateStatus("Ready");
  if (ACK) {
    removeProcess(name);
    sprintf(console, "[CONSOLE] %s killed successfully\n", name); 
  } else {
    sprintf(console, "[CONSOLE] Unable to kill %s\n", name); 
  }
  MessagePrintf(console);
  return TRUE;
}

/*
 * set the program to be executing, and pass it on..
 */
BOOLEAN parseDead(BOOLEAN local, char *remote, int ignore, char *buf)
{
  rcProgramPtr r;
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];
  int   i, id = -1;

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);
  buf++; 

  /* Check to see if we still think this is lying around */
  r = &(rcAvailableProgramList[rcLookupName(name)]);
  for (i=0; i<nConsoleProcs; i++)
  if (!(strncmp(name, consoleProc[i].name, strlen(consoleProc[i].name))))
      id = i; 
  if (id == -1)
    return TRUE;

  /* Kill the dwagon */
  removeProcess(name);
  sprintf(console, "[CONSOLE] %s died\n", name); 
  MessagePrintf(console);

  return TRUE;
}

/*
 * Pretty much the same as Exec parse handling
 * Required in response to a ping of the nanny, or a dependcy 
 * chain firing 
 */
BOOLEAN parseAnnounce(BOOLEAN local, char *remote, int id, char *buf)
{
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];
  int   i;
  BOOLEAN readyP;

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);
  
  readyP = (strlen(buf) == 0);

  deactivateStatus("Ready");
  id = convertProcessNameToId(name);
  if (id == -1) {
    addProcess(name);
    id = convertProcessNameToId(name);
    if (!readyP) {
      sprintf(console, "[CONSOLE] %s executing\n", name); 
      MessagePrintf(console);
      sendQuiet(name);
    }
  }

  if (readyP && !consoleProc[id].ready) {
    sprintf(console, "[CONSOLE] %s ready\n", name); 
    MessagePrintf(console);
    consoleProc[id].ready = TRUE;
    processWhenReadyConditionals(name);
  }

  return TRUE;
}

/*
 * We don't actually have to take any action for each of the 3 following
 * loudness acknowledgements.
 */
BOOLEAN parseQuiet(BOOLEAN local, char *remote, int id, char *buf)
{
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];

  buf = messageExtractKey(buf, name);
  messageExtractKey(buf, name);

  if (ACK)
    sprintf(console, "[CONSOLE] %s set to QUIET successfully\n", name); 
  else
    sprintf(console, "[CONSOLE] Failed to set %s to QUIET\n", name); 
  MessagePrintf(console);

  return TRUE;
}

BOOLEAN parseSilent(BOOLEAN local, char *remote, int id, char *buf)
{
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];

  buf = messageExtractKey(buf, name);
  messageExtractKey(buf, name);

  if (ACK)
    sprintf(console, "[CONSOLE] %s set to SILENT successfully\n", name); 
  else
    sprintf(console, "[CONSOLE] Failed to set %s to SILENT\n", name); 
  MessagePrintf(console);

  return TRUE;
}

BOOLEAN parseVerbose(BOOLEAN local, char *remote, int id, char *buf)
{
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];

  buf = messageExtractKey(buf, name);
  messageExtractKey(buf, name);

  if (ACK)
    sprintf(console, "[CONSOLE] %s set to VERBOSE successfully\n", name); 
  else
    sprintf(console, "[CONSOLE] Failed to set %s to VERBOSE\n", name); 
  MessagePrintf(console);

  return TRUE;
}

BOOLEAN parseStatus(BOOLEAN local, char *remote, int id, char *buf)
{
  char  name[MAXTOKEN];
  char  status[MAXBUF];

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);
  messageExtractKey(buf, status);

  return TRUE;
}

BOOLEAN parseStdin(BOOLEAN local, char *remote, int id, char *buf)
{
  char name[MAXTOKEN];

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);
  buf++; 

  if (!checkForCurrentProc()) return;

  if (!(strncmp(name, consoleProc[currentProc].name, 
		strlen(consoleProc[currentProc].name))))
      EditAppend (buf, strlen(buf));

  WFlush();

  /* Process the script's "When" statements */
  processWhenConditionals(name, buf);

  return TRUE;
}

/*
 * If no filtering is enabled or if the filterString is present
 * return TRUE
 */
static BOOLEAN filtered(char *check)
{
  if (filterString == NULL)
    return TRUE;

  return (strstr(check,filterString) == NULL);
}

BOOLEAN parseError(BOOLEAN local, char *remote, int id, char *buf)
{
  char name[MAXTOKEN], readyProc[MAXTOKEN];
  char console[DEFAULT_LINE_LENGTH];
  int ntmp;

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);
  buf++; 

  if (!checkForCurrentProc()) return;

  bzero(console,DEFAULT_LINE_LENGTH);
  if (filtered(console))
    {
      if ((strncmp(name, consoleProc[currentProc].name, 
		   strlen(consoleProc[currentProc].name))))
	if (!highlightProcess(name))
	  MessagePrintf(console);
    }

  if (!(strncmp(name, consoleProc[currentProc].name, 
		strlen(consoleProc[currentProc].name))))
    { 
      EditAppend (buf, strlen(buf));
      if (!flushing)
	WFlush();
    }

  if (!flushing) {
    /* Process the script's "When" statements, unless just flusing the buffer */
    processWhenConditionals(name, buf);
  }

  return TRUE;
}

/*
 * For acknowledgements, we set the ACK flag corresponding 
 * to the current message, and then parse the remainder
 */
BOOLEAN parseAck(BOOLEAN local, char *remote, int id, char *buf)
{
  char name[MAXTOKEN];

  buf = messageExtractKey(buf, name);
  buf++; 

  ACK = TRUE;
  parseLine(local, remote, id, buf);
  return TRUE;
}

BOOLEAN parseNack(BOOLEAN local, char *remote, int id, char *buf)
{
  char name[MAXTOKEN];

  buf = messageExtractKey(buf, name);
  buf++; 
  
  ACK = FALSE;
  parseLine(local, remote, id, buf);
  return TRUE;
}

/*
 * This should not be received by the console. If they are, something
 * weird has occured.
 */
BOOLEAN parseEnquire(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Illegal enquire message received\n");
  MessagePrintf(console);
  return TRUE;
}

/*
 * Following a flush message, we reenable updates to the edit window
 */
BOOLEAN parseFlush(BOOLEAN local, char *remote, int id, char *buf)
{
  if (flushing) {
    flushing = FALSE;
    unfreeze();
  }
  return TRUE;
}

BOOLEAN parseBreak(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Illegal break message received\n");
  MessagePrintf(console);
  return TRUE;
}

BOOLEAN parsePause(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Illegal pause message received\n");
  MessagePrintf(console);
  return TRUE;
}

BOOLEAN parseContinue(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Illegal continue message received\n");
  MessagePrintf(console);
  return TRUE;
}

BOOLEAN parsePing(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Illegal ping message received\n");
  MessagePrintf(console);
  return TRUE;
}


BOOLEAN parseReload(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Resource reload by nanny completed\n");
  MessagePrintf(console);
  return TRUE;
}

BOOLEAN parseRestart(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Restart illegally received.\n");
  MessagePrintf(console);
  return TRUE;
}

BOOLEAN parseStdout(BOOLEAN local, char *remote, int id, char *buf)
{
  char console[DEFAULT_LINE_LENGTH];

  sprintf(console, "[CONSOLE] Illegal stdout message received from nanny\n");
  MessagePrintf(console);
  return TRUE;
}

/* Handle a heart beat somehow */
BOOLEAN parseBeat(BOOLEAN local, char *remote, int id, char *buf)

{
  char  name[MAXTOKEN];
  char console[DEFAULT_LINE_LENGTH];

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);
  buf++; 

  sprintf(console, "[%s] Thump\n", name);
  MessagePrintf(console);
  return TRUE;
}

BOOLEAN parseList(BOOLEAN local, char *remote, int id, char *buf)
{
  char  item[MAXTOKEN];

  buf = messageExtractKey(buf, item);
  
  cpyList(buf);  

  numPossiblePrograms = 0;
  while (buf[0] != '\0')
    {
      numPossiblePrograms++;
      buf = messageExtractKey(buf, item);
    }

  if (numPossiblePrograms > 0)
    attachProgramMenu();
  deactivateStatus("Ready");
  return TRUE;
}

BOOLEAN parseDestroy(BOOLEAN local, char *remote, int id, char *buf)
{
  char  name[MAXTOKEN];
  char  console[DEFAULT_LINE_LENGTH];

  buf = messageExtractKey(buf, name);
  buf = messageExtractKey(buf, name);

  deactivateStatus("Ready");
  if (ACK) {
    addProcess(name);
    sprintf(console, "[CONSOLE] %s is dead (hee hee)\n", name); 
  } else {
    sprintf(console, "[CONSOLE] %s doesn't seem to exist, or be killable\n",
	    name); 
  }
  MessagePrintf(console);
  return TRUE;
}
