/*****************************************************************************
 * PROJECT: Xavier
 *
 * (c) Copyright 1993 Richard Goodwin & Joseph O'Sullivan. All rights reserved.
 *
 * FILE: callbacks.c
 *
 * ABSTRACT: General callback code (used by both xCallbacks and xfCallbacks).
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/nanny/callbacks.c,v $
 * $Revision: 1.3 $
 * $Date: 1996/07/29 05:02:59 $
 * $Author: josullvn $
 *
 * REVISION HISTORY:
 *
 * $Log: callbacks.c,v $
 * Revision 1.3  1996/07/29  05:02:59  josullvn
 * Bloody hell. This commiting is a bit different than under Xavier. Short
 * story is cleaned up some purify bugs, and also made changes to nanny
 * which should make it a bit better - Improving performance over multiple
 * machines, explict quietening of nondisplayed processes, replacing of
 * runConsole with xfMiniConsole, which is multithreaded, vt102 compilant,
 * adds a uniform emacs-like command line editing feature, better on small
 * screens and otherwise fab.
 *
 * Revision 1.2  1996/06/28  14:07:13  reids
 * Fixed quite a few bugs -- with graphics, interaction with script, and
 *   killing processes
 *
 * Revision 1.1  1996/03/29  16:08:10  reids
 * Consolidated the common code between xCallbacks and xfCallbacks, and
 *    xRunConsole and xfRunConsole.
 *
 ****************************************************************************/
#include "tca/libc.h"
#include "tca/devUtils.h"

#include "runConsole.h"
#include "callbacks.h"
#include "messages.h"
#include "nannyDev.h"
#include "messageHandle.h"
#include "script.h"

/*
 * Forward declarations
 */
BOOLEAN flushing = FALSE;
int flushing_count = 0;
int numPossiblePrograms = 0;
char possiblePrograms[BUFSIZ];
struct timeval pollTime = {0, 1000000/POLLING_PERIOD};

static BOOLEAN nannyConnected = FALSE;
static char buf[DEFAULT_LINE_LENGTH];

/* Status info */
static double posn = 0.0;
static BOOLEAN forward = TRUE;

void safeExit (void)
{
  devShutdown();
  exit(0);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
void cpyList (const char *list)
{
  strcpy(possiblePrograms, list);
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
int convertProcessNameToId (const char *name)
{
  int i;

  for (i=0; i<nConsoleProcs; i++)
    if (!strcmp(consoleProc[i].name, name))
      return i;

  return -1;
}

/*
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 */
BOOLEAN safePassLocalMessage (const char *message)
{
  static char buf[BUFSIZ];

  if (!passLocalMessage((char *)message)) {
    sprintf(buf, "[CONSOLE] Nanny is not ready.\n");
    nannyConnected = FALSE;
    MessagePrintf(buf);
    deactivateStatus("No Nanny");
    return FALSE;
  }
  nannyConnected = TRUE;
  return TRUE;
}

/* don't do anything with this yet... */
void reloadResource (const char *reloadName)
{
  messageMakeReload(buf);
  safePassLocalMessage(buf);
}

void syncWithNanny (void)
{
  /* Announce that we be here */
  messageMakePing(buf); 
  if (safePassLocalMessage(buf)) {
    messageMakeList(buf, ""); 
    safePassLocalMessage(buf);
  }
}

void setFilterString (const char *newFilter)
{
  if (filterString) free(filterString);

  if (strlen(newFilter)) {
    filterString = strdup(newFilter); 
    sprintf(buf, "[CONSOLE] Filtering on occurrences of %s\n", filterString);
  } else {
    filterString = NULL;
    sprintf(buf, "[CONSOLE] Filtering cleared\n");
  }
  MessagePrintf(buf);
}

void killCurrentProcess (void)
{
  runKillProcess(consoleProc[currentProc].name);
}

void restartCurrentProcess (void)
{
  messageMakeKill(buf, consoleProc[currentProc].name);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] %s waiting for nanny kill confirmation\n", 
	    consoleProc[currentProc].name);
    MessagePrintf(buf);
  }
  messageMakeExec(buf, consoleProc[currentProc].name, getenv("DISPLAY"));
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "Waiting for %s...", consoleProc[currentProc].name);
    activateStatus(buf);
    sprintf(buf, "[CONSOLE] %s waiting for confirmation from nanny\n",
	    consoleProc[currentProc].name);
    MessagePrintf(buf);
  }
}

BOOLEAN checkForCurrentProc (void)
{
  if (currentProc == -1) {
    Feep();
    MessagePrintf("[CONSOLE] No process selected\n");
  }
  return (currentProc != -1);
}

void sendBreak (void)
{
  if (!checkForCurrentProc()) return;

  messageMakeBreak(buf, consoleProc[currentProc].name);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] break sent to %s\n", consoleProc[currentProc].name);
    MessagePrintf(buf);
  }
}

void sendQuiet (char *name)
{
  messageMakeQuiet(buf, name);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] Ssshing %s\n", name);
    MessagePrintf(buf);
  }
}
void sendLoud (char *name)
{
  messageMakeVerbose(buf, name);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] setting to verbose %s\n", name);
    MessagePrintf(buf);
  }
}

void getProcessList (const char *action)
{
  /* If we have no items, that means we have no nanny connect yet...  */
  Feep();
  MessagePrintf("[CONSOLE] Checking for nanny\n");
  
  messageMakePing(buf);
  if (safePassLocalMessage(buf))
    MessagePrintf("[CONSOLE] Nanny now found\n");
  else {
    /* An error message will already have been displayed... */
    return;
  }

  messageMakeList(buf, "");
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "[CONSOLE] Programs loaded from nanny, select %s again\n",
	    action);
    MessagePrintf(buf);
  } else {
    MessagePrintf("[CONSOLE] Nanny lost\n");
  }
}

void destroyProcess (const char *procName)
{
  messageMakeDestroy(buf, (char *)procName);
  if (safePassLocalMessage(buf)) {
    sprintf(buf, "Trying to destroy %s...", procName);
    activateStatus(buf);
    sprintf(buf, "[CONSOLE] %s waiting for confirmation from nanny\n", 
	    procName);
    MessagePrintf(buf);
  }
}

void shutdownProcess (void)
{
  int id;
      
  /* for (id=0; id< nConsoleProcs; id++) { */
  /* Doing it in reverse order is safer */
  for (id=nConsoleProcs-1; id>=0; id--) {
    messageMakeKill(buf, consoleProc[id].name);
    safePassLocalMessage(buf);
    sleep(3);
  }
  safeExit();
}

void startFlushing (const char *processName)
{
  flushing = TRUE; flushing_count=0;
  messageMakeFlush(buf, (char *)processName);
  safePassLocalMessage(buf);
}

void addProcess (const char *name)
{
  strcpy(consoleProc[nConsoleProcs].name, name);
  addProcessButton(name);
  nConsoleProcs++;
}

static void clearConsoleProc (consoleProcPtr consoleProc)
{
  bzero(consoleProc->name, DEFAULT_LINE_LENGTH);
  consoleProc->filterValue = FALSE;
  consoleProc->ready = FALSE;
  historyFreeProcess(&(consoleProc->history));
  clearConsoleGraphics(consoleProc);
}

static void copyConsoleProc (consoleProcPtr from, consoleProcPtr to)
{
  bcopy(from->name, to->name, DEFAULT_LINE_LENGTH);
  to->filterValue = from->filterValue;
  to->ready       = from->ready;
  to->history     = from->history;
  copyConsoleGraphics(from, to);
}

void removeProcess (const char *name)
{
  int ntmp, i;

  /* Figure out which of consoleProcs we wish to reset */
  if ((ntmp = convertProcessNameToId(name)) == -1) 
    return;

  /* Destroy it */
  destroyProcessButton(ntmp);
  clearConsoleProc(&consoleProc[ntmp]);

  /* Clean up the hole this may have left in the consoleProc array */
  for (i=ntmp+1; i<nConsoleProcs; i++)
    copyConsoleProc(&(consoleProc[i]), &(consoleProc[i-1]));

  nConsoleProcs--;

  resetProcessButtons(ntmp);

  currentProc = -1;
}

double getPosition (void)
{
  return posn;
}

void initPosition (void)
{
  posn = 0.0; 
  forward = TRUE;
}

void updatePosition (void)
{
  if (forward) { 
    posn += 0.1;
    if (posn >= 0.9)
      forward = FALSE;
  } else { 
    posn -= 0.1;
    if (posn < 0.1)
      forward = TRUE;
  }
}

void rerunScript (void)
{
  if (!scriptRerun()) {
    Feep();
    MessagePrintf("[ERROR] No loaded script to rerun\n");
  } else {
    setSuspendResume(SUSPEND);
  }
}

void suspendResume (const char *currentState)
{
  if (!strcmp(currentState, SUSPEND)) {
    if (!scriptSuspend()) {
      Feep();
      MessagePrintf("[ERROR] No active script to suspend\n");
    } else {
      setSuspendResume(RESUME);
    }
  } else {
    if (!scriptResume()) {
      Feep();
      MessagePrintf("[ERROR] No suspended script to resume\n");
    } else {
      setSuspendResume(SUSPEND);
    }
  }
}

void stopScript (void)
{
  scriptClear();
  setSuspendResume(SUSPEND);
}
