/*****************************************************************************
 * (c) Copyright 1996 Greg Whelan and Reid Simmons.  All rights reserved.
 *
 * FILE: comview.c
 *
 * ABSTRACT: Message visualization tool
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/comview/comview.c,v $ 
 * $Revision: 1.10 $
 * $Date: 1996/07/26 18:22:10 $
 * $Author: rich $
 *
 * REVISION HISTORY
 * $Log: comview.c,v $
 * Revision 1.10  1996/07/26  18:22:10  rich
 * Fixed warnings.
 *
 * Revision 1.9  1996/07/25  18:24:41  whelan
 * Added a "colors" menu that describes what kind of message each of the
 * colors corresponds to, and allows the user to modify the colors.
 *
 * Revision 1.8  1996/07/19  21:04:37  whelan
 * A file can now be opened from within Comview -- does not have to be
 * specified on the command line.
 *
 * Revision 1.7  1996/07/19  14:38:51  reids
 * Check if -f parameter is actually given.
 *
 * Revision 1.6  1996/07/19  14:26:50  whelan
 * Comview directory specified from compilation directory.
 *
 * Revision 1.5  1996/07/18  16:40:01  whelan
 * COMVIEW_DIRECTORY environment variable is no longer neccessary.
 * It is passed to the C code as a define in the makefile (as the directory
 * which the program is made).
 *
 * Revision 1.4  1996/07/18  16:04:30  reids
 * Changes for the New Millennium environment (NMP_IPC) (plus log history)
 *
 ****************************************************************/

#include <stdlib.h>
#ifndef NMP_IPC
#include "tca/libc.h"
#endif
#include "Top.h"
#include "tools/Standard.h"
#include "tools/List.h"
#include "tools/Array.h"
#include "tools/MsgData.h"
#include "tools/TaskTree.h"
#include "tools/MyString.h"
#include "tools/MsgData.h"
#include "tools/Parser.h"
#include <tcl.h>
#include <tk.h>

#include "comview.h"
#include "tcl-framework.h"

#define MAX_LONG_MESSAGE 600

/* -- local prototypes -------------------------------------------------*/

void initialize_modules(void);
int module_lookup(char *, Tcl_Interp *);
int CV_OneStepFromFile(FILE *, char *, MsgData *, Tcl_Interp *);
void update_listbox_vars(char *line, int msg_type);
int local_update_listbox(Tcl_Interp *interp);
void set_time_vars(Tcl_Interp *interp, int hours, int minutes,
		   int seconds, int m_seconds);
int logged_message_type(int msg_type);
int actual_listbox_update(Tcl_Interp *interp);
/*----------------------------------------------------------------------*/
 
/* the number of milliseconds that we will artificially separate messages */
/* which are recorded at the same time in the TCA log file. */
/* A value of 2 says that there's a maximum of 4 messages that can occur */
/* at the same time stamp. */
#define TIME_SEPARATION 0

static Tcl_HashTable *tablePtr;
int module_array[255];
int module_count;
int message_count;
char set_filename[1024];
static int file_selected;
int display_all=1;
static int msg_count=0;
static int lbox_count=0;
int g_mod;
static char listbox_line[MAX_LONG_MESSAGE];
static int listbox_msgt;
int COMVIEW_DEBUG=0;
char result[255];
char zoom[255];
int zoom_found=0;

static void displayVersion(void)
{
  printf("comview %d.%d.%d (log parser %d.%d.%d)\n",
	 COMVIEW_VERSION_MAJOR, COMVIEW_VERSION_MINOR, COMVIEW_VERSION_MICRO,
	 TPARSER_VERSION_MAJOR, TPARSER_VERSION_MINOR, TPARSER_VERSION_MICRO);
  printf(" Released : %s\n", COMVIEW_VERSION_DATE);
  printf(" Commited : %s\n", COMVIEW_COMMIT_DATE);
  printf(" Compiled : %s %s\n", __DATE__, __TIME__);
  fflush(stdout);
}

static void printHelp(void)
{
  printf("-h : Print this message\n");
  printf("-v : Print version information\n");
  printf("-l : Do not display all log file information (only messages)\n");
  printf("-f <logfile> : Use the logfile\n");
  printf("-z <value> : initial zoom value\n");
}

static void parseCommandLineOptions (int argc, char **argv)
{
  int i;

  for (i=1; i<argc; i++) {
    if (!strcmp(argv[i], "-h")) {
      printHelp();
      exit(0);
    } else if (!strcmp(argv[i], "-v")) {
      displayVersion();
      exit(0);
    } else if (!strcmp(argv[i], "-l")) {
      display_all=0;
   } else if (!strcmp(argv[i], "-d")) {
     COMVIEW_DEBUG=1;
   } else if (!strcmp(argv[i], "-z")) {
     i++;
     strcpy(zoom,*(argv+i));
     zoom_found=1;
   } else if (!strcmp(argv[i], "-f")) {
      i++;
      if (!OpenFile(argv[i])) {
	fprintf(stderr, "File %s doesn't exist\n", argv[i]);
	exit(-1);
      } else {
	sprintf(set_filename, "set filename %s", argv[i]);
	file_selected=1;
      }
    } else if (!strcmp(argv[i],"-m")) {
/*      _process_info_msg = TRUE; */
    } else {
      fprintf(stderr, "Unknown option %s\n", argv[i]);
    }
  }
}

void set_time_vars(Tcl_Interp *interp, int hours, int minutes,
		  int seconds, int m_seconds)
{
  char foo[255];

  sprintf(foo, "set_hours %d", hours);
  Tcl_Eval(interp, foo);
  sprintf(foo, "set_mins %d", minutes);
  Tcl_Eval(interp, foo);
  sprintf(foo, "set_secs %d", seconds);
  Tcl_Eval(interp, foo);
  /* this is a bug waiting to happen... gotta fix */
  sprintf(foo, "set_hsecs %d", (m_seconds/10)-1);
  Tcl_Eval(interp, foo);
}

int actual_listbox_update(Tcl_Interp *interp)
{
  char foo[MAX_LONG_MESSAGE+50];

  if (listbox_msgt>=0) {
    
    msg_count++;
    sprintf(foo, ".lbo.lbox insert end \"%d: %s\"", msg_count, listbox_line);
    Tcl_Eval(interp, foo);
      
    sprintf(foo, "add_to_lbox_array %d %d", msg_count, lbox_count);
    Tcl_Eval(interp, foo);
  } else {
    sprintf(foo, ".lbo.lbox insert end \"%s\"", listbox_line);
    Tcl_Eval(interp, foo);  
  }
    
  lbox_count++;
    
  sprintf(foo, "scrollListbox");
  Tcl_Eval(interp, foo);

  return TCL_OK;
}

/* update_listbox will be called from Tcl */

int update_listbox(ClientData clientData, Tcl_Interp *interp,
		 int argc, char *argv[])
{
  if (listbox_msgt && (listbox_msgt != MODULE_CONNECT) && 
      (listbox_msgt != MODULE_DISCONNECT)) {
    actual_listbox_update(interp);
  }
  
  return TCL_OK;
}


/* logged_message_type checks if the given message type a message */
/* which involves inter-module communication. */

int logged_message_type(int msg_type)
{
  switch (msg_type) {
  case QUERY:
  case GOAL:
  case COMMAND:
  case INFORM:
  case REPLY:
  case FAILURE:
  case SUCCESS:
  case BROADCAST:
    return 1;
    break;
  default:
    return 0;
  }
}

/* local_update_listbox is identical to update_listbox except that it */
/* will be called from within this C source file (not Tcl). The only */
/* difference between this local_update_listbox and update_listbox is */
/* the parameters. */

int local_update_listbox(Tcl_Interp *interp)
{

  if (logged_message_type(listbox_msgt)) {
    actual_listbox_update(interp);
  }

  return TCL_OK;
}


void update_listbox_vars(char *line, int msg_type)
{
  int n, x;
  
  x=0;

  for (n=0; line[n]; n++) {
    if (line[n]=='"') {
      listbox_line[x++]='\\';
      listbox_line[x++]='"';
    } else if (line[n]=='[') {
      listbox_line[x++]='\\';
      listbox_line[x++]='[';
    } else {
      listbox_line[x++]=line[n];
    }
  }
  listbox_line[x]='\0';
  
  listbox_msgt=msg_type;
}

void initialize_modules(void)
{
  int i;

  for (i=0; i< 256; i++)
      module_array[i]=0;

  module_count=1;
}

static void CV_Initialize (void)
{
  TaskTreeInitialize(0);
  initialize_modules();
}


/* module_lookup takes a module name (string) and returns the module */
/* number. If the module name isn't already in the data structure then */
/* we have to add it into the data structure. */
int module_lookup(char *module_name, Tcl_Interp *interp)
{
  Tcl_HashEntry *entry;
  int exists;

  entry=Tcl_CreateHashEntry(tablePtr, module_name, &exists);

  if (exists) {
    /* not in hash table */
    Tcl_SetHashValue(entry, module_count);
    create_new_module(interp, module_count, module_name);
    module_count++;
    return (module_count-1);
  } else {
    /* already in hash table */
    return (int)Tcl_GetHashValue(entry);
  }
}

/* These messages are not used by comview. */

static int invalid_msg(msgType type)
{
  int result;

  result = ((type == TYPE_NULL) ||
	    (type == TEMP_CONSTRAINT) ||
	    (type == POINT_CONSTRAINT) ||
	    (type == KILLED));

  return (result);
}

int CV_OneStepFromFile(FILE *file, char *msg, MsgData *msgData, 
		      Tcl_Interp *interp)
{
  int count=0;
  
  if (!file)
    return (0);
  
  *msg = '\0';
  
  /* we loop until a valid message has been read from the log file */

  do {
    if (ReadNlFromFile(file, msg) == EOF) {
      return (0);
    }

    *msgData = ParseMsg(msg);

    if (COMVIEW_DEBUG) {
      fprintf(stderr, "ReadNl Got : %s\n", msg);
      fprintf(stderr, "(*msgData)->type is %d\n", (*msgData)->type);
    }

    if (display_all) {
      /* shouldn't this go after the update_listbox_vars ? */
      if (count) { 
	local_update_listbox(interp);
      }
      update_listbox_vars(msg, (int)(*msgData)->type);
    } else if (((*msgData)->type!=TYPE_NULL) && 
	       ((*msgData)->type!=MODULE_CONNECT) && 
	       ((*msgData)->type!=MODULE_DISCONNECT)) {
      update_listbox_vars(msg, (int)(*msgData)->type);
    } else {
      /* no show */
      listbox_msgt=-1;
    }
    count=1;
  } while (invalid_msg((*msgData)->type));

  return(1);
}

char *uno_step(Tcl_Interp *interp)
{
  char msg[MAX_LONG_MESSAGE];
  MsgData msgData;
  int from=0, to=0, dt=0;
  static int prev_time;
  static int first_run=1;
  char source[128];
  char dest[128];
  int from_resourcep;
  static int time_offset=0;
  Task task;

  /* here's the basic idea:
     (1) Read in a line from the log file
     (2) Check what type of message the line is.
     */

  if (CV_OneStepFromFile(global_log_file, msg, &msgData, interp))
      {
	if (COMVIEW_DEBUG) {
	  fprintf(stderr, "%s\n",msg);
	}

	/* for Success or Failure messages we have to find out which
	   module the message is coming from. */
	if ((msgData->type==FAILURE) || (msgData->type==SUCCESS)) {
	  task = FindTask(msgData->name, msgData->id, task_ACTIVE);
	  if (COMVIEW_DEBUG) {
	    fprintf(stderr, "TASK: %s ", task->creator->name);
	    fprintf(stderr, "%d ", task->creation_time);
	    fprintf(stderr, "%d\n",  (int)task->additional);
	  }
	  from=(int)task->additional;
	}

	/* we only want to include actual data messages in the listbox */
	/* not connection/disconnection messages */
	if ((msgData->type == MODULE_CONNECT) ||
	    (msgData->type == MODULE_DISCONNECT)) {
	  
	  /* the message is a connect or disconnect message */
	  message_count++;
	  
	  /* the destination value doesn't matter */
	  strcpy(dest, "0");
	  strcpy(source, msgData->name);
	  from=module_lookup(source,interp);
	}
	
	from_resourcep=0;

	if ((msgData->type != MODULE_CONNECT) &&
	    (msgData->type != MODULE_DISCONNECT) &&
	    (msgData->type != FAILURE) &&
	    (msgData->type != SUCCESS) &&
	    (msgData->type != TEMP_CONSTRAINT)) {

	  if (strncmp("Resource ", msgData->source, 9)==0) {
	    strncpy(source, msgData->source+9, (strlen(msgData->source))-8);
	    from_resourcep=1;
	  } else strcpy(source, msgData->source);
	  
	  if (strncmp("Resource ", msgData->dest, 9)==0) {
	    strncpy(dest, msgData->dest+9, (strlen(msgData->dest))-8);
	  } else strcpy(dest, msgData->dest);
	  
	  from = module_lookup(source, interp);
	  to = module_lookup(dest, interp);
	}
	
	if ((msgData->type != MODULE_CONNECT) && 
	    (msgData->type != MODULE_DISCONNECT)) {

	  dt=msgData->time - prev_time;

	  if (first_run) {
	    /* the first time through the prev_time is not going to be set */
	    first_run=0;
	    dt=10;
	    /* we also need to tell Tcl what time the first message */
	    /* occurred at (as a point of reference) */
	    set_time_vars(interp, msgData->hours, msgData->minutes,
			  msgData->seconds, msgData->m_seconds);
	  }
	  
	  /* in the event that the time recorded in the TCA log file does */
	  /* not show any change */
	  if (dt==0) {
	    time_offset+=TIME_SEPARATION;
	    dt=time_offset;
	  }
	  else {
	    /* we have to get things back in time-sync */
	    dt-=time_offset;
	    time_offset=0;
	  }

	  prev_time = msgData->time;
	} 

	g_mod = to;

	/* success or failure (completion messages) */
	if ((msgData->type == FAILURE) || (msgData->type == SUCCESS)) {
	  to = 0;
	  strcpy(source, "0");
	  strcpy(dest, "0");
	}

	if (COMVIEW_DEBUG) {
	  fprintf(stderr, "C typ:%d sts:%d nam:%s src:%s dest:%s id:%d\n t:%d h:%d m:%d s:%d msec:%d \n", 
		  msgData->type, msgData->status, msgData->name, 
		  msgData->source, msgData->dest, msgData->id,
		  msgData->time, msgData->hours, msgData->minutes, 
		  msgData->seconds, msgData->m_seconds);
	  
	  fprintf(stderr, "C from: %d  to: %d dt: %d time: %d prev: %d\n", 
		  from, to, dt, msgData->time, prev_time);
	}
	
	sprintf(result, "list %d \"%s\" %d \"%s\" %d %d \"%s\" %d %d",
		from, source, to, dest, msgData->type,
		msgData->status, msgData->name, dt, from_resourcep);

	if (msgData->type != TYPE_NULL) {
	  if (COMVIEW_DEBUG) {
	    fprintf(stderr, "About to ProcessNewMessage %d...\n",
		    msgData->type);
	  }
	  TaskTreeProcessNewMessage(msgData);
	}
      }
  else
      {
	sprintf(result, "perror");
	if (COMVIEW_DEBUG) {
	  fprintf(stderr, "C Could not parse message: %s\n", msg);
	}
      }
  
  /*  free(&msgData); */
  
  return result;
}

static void *CV_AddWidgetData (void)
{
  if (COMVIEW_DEBUG) {
    fprintf(stderr, "add widget data\n");
  }
  return (void *)(g_mod);
}


void main (int argc, char **argv)
{
  char comview_directory[100];
  char *cv_env_var;
  
  bzero(set_filename, 1024);

  cv_env_var=getenv("COMVIEW_DIRECTORY");

  if (!cv_env_var) {
    /* the constant COMVIEW_DIRECTORY is set as a define in the
       makefile. if the user doesn't have an environment variable
       overriding that setting, use that define. */
    strcpy(comview_directory, COMVIEW_DIRECTORY); 
  } else {
    strcpy(comview_directory, cv_env_var);
  }
  
  file_selected=0;

  parseCommandLineOptions(argc, argv);

  setAdditionalTaskDataFn(CV_AddWidgetData);

  tablePtr = (Tcl_HashTable *)malloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(tablePtr, TCL_STRING_KEYS);

  CV_Initialize();

  if (COMVIEW_DEBUG) {
    fprintf(stderr, "About to init_grafx\n");
  }
  
  /* 
  if (!strlen(set_filename)) {
    fprintf(stderr, "Usage: comview -f <filename> [-l] [-z <zoom>]\n");
    exit(-1);
  }
  */

  init_grafx(argc, argv, set_filename, file_selected,
	     zoom, zoom_found, comview_directory);

  if (COMVIEW_DEBUG) {
    fprintf(stderr, "C init_grafx returned\n");
  }
}
