/*****************************************************************************
 * PROJECT: TCA
 *
 * (c) Copyright Richard Goodwin, 1995. All rights reserved.
 *
 * FILE: bridge.c
 *
 * ABSTRACT:
 * 
 * This program provides a bridge between to central servers.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/tools/bridge/bridge.c,v $ 
 * $Revision: 1.5 $
 * $Date: 1996/02/10 16:51:34 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: bridge.c,v $
 * Revision 1.5  1996/02/10  16:51:34  rich
 * Made private functions static.
 *
 * Revision 1.4  1996/01/31  22:54:30  reids
 * Added automatic updating of (micro) version control numbers
 *
 * Revision 1.3  1995/12/17  20:25:07  rich
 * Minor change.
 *
 * Revision 1.2  1995/12/15  01:25:26  rich
 * Fixed the includes.
 *
 * Revision 1.1  1995/10/29  18:29:34  rich
 * Initial version of the message bridge.  It allows messages to be sent
 * from one central server to another.
 *
 *
 * 12-Sept-95 Richard Goodwin Created.
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "tca.h"
#include "tca/devUtils.h"
#include "tca/tcaDev.h"
#include "tca/stdinDev.h"
#include "tca/basics.h"
#include "tca/strList.h"
#include "tca/tcaInternal.h"
#include "tca/centralMsg.h"
#include "bridge.h"

/*****************************************************************************
 * Global constants
 *****************************************************************************/

#define BRIDGE_NAME_PREFIX "bridge_"
#define NEW_MESSAGE_LISTENER "bridge_new_message"
#define NEW_HANDLER_LISTENER "bridge_new_handler"

/*****************************************************************************
 * Global types
 *****************************************************************************/

/*****************************************************************************
 * Global variables
 *****************************************************************************/

TCA_DEV_PTR listeningDev;
TCA_DEV_PTR talkingDev;

/*****************************************************************************
 * Forward declarations.
 *****************************************************************************/

static void crossRegisterMsgs(void);

/*****************************************************************************
 *
 * FUNCTION: void forwardingWaitForCommandHandler(TCA_REF_PTR ref, void *data)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void forwardingWaitForCommandHandler(TCA_REF_PTR ref, void *data)
{
  TCA_RETURN_VALUE_TYPE result;
  
  if (TCA_isConnected(talkingDev)) {
    TCA_setContext(talkingDev);
    if (tcaMessageHandlerRegistered(tcaReferenceName(ref))) {
      result = tcaWaitForCommand(tcaReferenceName(ref), data);
      TCA_setContext(listeningDev);
      if (result == Success) {
	tcaSuccess(ref);
      } else {
	tcaFailure(ref, "Forwarded Command Failed", NULL);
      }
    } else {
      TCA_setContext(listeningDev);
      tcaFailure(ref, "No Handler", NULL);
    }
  } else {
    tcaFailure(ref, "No Connection", NULL);
  }
  tcaFreeData(tcaReferenceName(ref), data);
}

/*****************************************************************************
 *
 * FUNCTION: void forwardingBroadcastHandler(TCA_REF_PTR ref, void *data)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void forwardingBroadcastHandler(TCA_REF_PTR ref, void *data)
{
  TCA_RETURN_VALUE_TYPE result;
  
  if (TCA_isConnected(talkingDev)) {
    TCA_setContext(talkingDev);
    if (tcaMessageHandlerRegistered(tcaReferenceName(ref))) {
      result = tcaBroadcast(tcaReferenceName(ref), data);
    }
  }
  tcaFreeData(tcaReferenceName(ref), data);
}

/*****************************************************************************
 *
 * FUNCTION: void forwardingInformHandler(TCA_REF_PTR ref, void *data)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void forwardingInformHandler(TCA_REF_PTR ref, void *data)
{
  TCA_RETURN_VALUE_TYPE result;
  
  if (TCA_isConnected(talkingDev)) {
    TCA_setContext(talkingDev);
    if (tcaMessageHandlerRegistered(tcaReferenceName(ref))) {
      result = tcaInform(tcaReferenceName(ref), data);
    }
  }
  tcaFreeData(tcaReferenceName(ref), data);
}

/*****************************************************************************
 *
 * FUNCTION: void forwardingQueryHandler(TCA_REF_PTR ref, void *data)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void forwardingQueryHandler(TCA_REF_PTR ref, void *data)
{
  void *reply=NULL;
  
  if (TCA_isConnected(talkingDev)) {
    reply = tcaAllocateReply(tcaReferenceName(ref));
    TCA_setContext(talkingDev);
    if (tcaMessageHandlerRegistered(tcaReferenceName(ref))) {
      tcaQuery(tcaReferenceName(ref), data, reply);
      TCA_setContext(listeningDev);
      tcaReply(ref, reply);
    } else {
      TCA_setContext(listeningDev);
      tcaNullReply(ref);
    }
  } else {
    tcaNullReply(ref);
  }
  tcaFreeData(tcaReferenceName(ref), data);
  tcaFreeReply(tcaReferenceName(ref), reply);
}

/*****************************************************************************
 *
 * FUNCTION: void newMsgHnd(TCA_REF_PTR ref, MSG_REG_DATA_PTR msgRegData)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void newMsgHnd(TCA_REF_PTR ref, MSG_REG_DATA_PTR msgRegData)
{
  
  if ((listeningDev != NULL) && (talkingDev != NULL) &&
      TCA_isConnected(listeningDev) && TCA_isConnected(talkingDev)) {
    TCA_setContext(talkingDev);
    
    switch (msgRegData->msg_class) {
    case UNKNOWN:
    case HandlerRegClass:
    case ExecHndClass:
    case ExceptionClass:
    case PollingMonitorClass:
    case PointMonitorClass:
    case DemonMonitorClass:
    case ReplyClass:
    case SuccessClass:
    case FailureClass:
    case FireDemonClass:
      break;
      
    case QueryClass:
      tcaRegisterQuery(msgRegData->name, 
		       msgRegData->msgFormat, msgRegData->resFormat,
		       forwardingQueryHandler);
      break;
    case CommandClass:
      tcaRegisterCommand(msgRegData->name, msgRegData->msgFormat,
			 forwardingWaitForCommandHandler);
      break;
    case InformClass:
      tcaRegisterInform(msgRegData->name, msgRegData->msgFormat,
			forwardingInformHandler);
    case BroadcastClass:
      tcaRegisterInform(msgRegData->name, msgRegData->msgFormat,
			forwardingBroadcastHandler);
    case MultiQueryClass:
    case GoalClass:
      break;
    }
  }
}

/*****************************************************************************
 *
 * FUNCTION: void newHndHnd(TCA_REF_PTR ref, MSG_REG_DATA_PTR msgRegData)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void newHndHnd(TCA_REF_PTR ref, HND_DATA_PTR hndData)
{
  MSG_INFO_TYPE msgInfo;
  
  if (TCA_isConnected(listeningDev) && TCA_isConnected(talkingDev)) {
    TCA_setContext(listeningDev);
    msgInfo.name = (char *) hndData->msgName;
    tcaGetMsgInfo(&msgInfo);
    TCA_setContext(talkingDev);
    
    tcaGetMsgInfo(&msgInfo);
    
    switch (msgInfo.msg_class) {
    case UNKNOWN:
    case HandlerRegClass:
    case ExecHndClass:
    case ExceptionClass:
    case PollingMonitorClass:
    case PointMonitorClass:
    case DemonMonitorClass:
    case ReplyClass:
    case SuccessClass:
    case FailureClass:
    case FireDemonClass:
      break;
      
    case QueryClass:
      tcaRegisterQuery(msgInfo.name, 
		       msgInfo.msgFormat, msgInfo.resFormat,
		       forwardingQueryHandler);
      break;
    case CommandClass:
      tcaRegisterCommand(msgInfo.name, msgInfo.msgFormat,
			 forwardingWaitForCommandHandler);
      break;
    case InformClass:
      tcaRegisterInform(msgInfo.name, msgInfo.msgFormat,
			forwardingInformHandler);
    case BroadcastClass:
      tcaRegisterInform(msgInfo.name, msgInfo.msgFormat,
			forwardingBroadcastHandler);
    case MultiQueryClass:
    case GoalClass:
      break;
    }
  }
}

/*****************************************************************************
 *
 * FUNCTION: void registerAllListener(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void registerAllListener(void)
{
  /* Need to set up taps to get new messages as they are registered. */
  
  tcaRegisterInform(NEW_MESSAGE_LISTENER,TCA_REGISTER_MSG_INFORM_FORMAT,
		    newMsgHnd);
  tcaTapMessage(WhenSent,TCA_REGISTER_MSG_INFORM, NEW_MESSAGE_LISTENER);
}

/*****************************************************************************
 *
 * FUNCTION: void disconnectTCAListener(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void disconnectTCAListener(void)
{
  /* called when a central disconnect is detected. */
}

/*****************************************************************************
 *
 * FUNCTION: static void reconnectTCAListener(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void reconnectTCAListener(void)
{
  /* called when a central is reconnected. */
  if (TCA_isConnected(talkingDev))
    crossRegisterMsgs();
}

/*****************************************************************************
 *
 * FUNCTION: void registerAllTalker(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void registerAllTalker(void)
{
  tcaRegisterInform(NEW_HANDLER_LISTENER,TCA_REGISTER_HND_INFORM_FORMAT,
		    newHndHnd);
  tcaTapMessage(WhenSent,TCA_REGISTER_HND_INFORM, NEW_HANDLER_LISTENER);
  
}

/*****************************************************************************
 *
 * FUNCTION: void disconnectTCATalker(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void disconnectTCATalker(void)
{
  /* called when a central disconnect is detected. */
}

/*****************************************************************************
 *
 * FUNCTION: static void reconnectTCATalker(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void reconnectTCATalker(void)
{
  /* called when a central is reconnected. */
  if (TCA_isConnected(listeningDev))
    crossRegisterMsgs();
}

/*****************************************************************************
 *
 * FUNCTION: static void crossRegisterMsgs(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void crossRegisterMsgs(void)
{
  char **msgNames;
  char *msgName;
  MSG_INFO_TYPE msgInfo;
  
  if (TCA_isConnected(listeningDev) && TCA_isConnected(talkingDev)) {
    TCA_setContext(listeningDev);
    msgNames = tcaGetRegisteredMsgs();
    for (msgName = msgNames[0]; msgName != NULL; msgName++) {
      TCA_setContext(listeningDev);
      msgInfo.name = msgName;
      tcaGetMsgInfo(&msgInfo);
      TCA_setContext(talkingDev);
      switch (msgInfo.msg_class) {
      case UNKNOWN:
      case HandlerRegClass:
      case ExecHndClass:
      case ExceptionClass:
      case PollingMonitorClass:
      case PointMonitorClass:
      case DemonMonitorClass:
      case ReplyClass:
      case SuccessClass:
      case FailureClass:
      case FireDemonClass:
	break;
	
      case QueryClass:
	tcaRegisterQuery(msgInfo.name, msgInfo.msgFormat, msgInfo.resFormat,
			 forwardingQueryHandler);
	break;
      case CommandClass:
	tcaRegisterCommand(msgInfo.name, msgInfo.msgFormat,
			   forwardingWaitForCommandHandler);
	break;
      case InformClass:
	tcaRegisterInform(msgInfo.name, msgInfo.msgFormat,
			  forwardingInformHandler);
      case BroadcastClass:
	tcaRegisterInform(msgInfo.name, msgInfo.msgFormat,
			  forwardingBroadcastHandler);
      case MultiQueryClass:
      case GoalClass:
	break;
      }
    }
  }
}

/*****************************************************************************
 *
 * FUNCTION: void displayVersion()
 *
 * DESCRIPTION: Displays the version information for the program
 *
 * INPUTS: none.
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

static void displayVersion(void)
{
  printf("TCA Bridge %d.%d.%d \n",
	 BRIDGE_VERSION_MAJOR, BRIDGE_VERSION_MINOR, BRIDGE_VERSION_MICRO);
  printf(" Released : %s\n", BRIDGE_VERSION_DATE);
  printf(" Commited : %s\n", BRIDGE_COMMIT_DATE);
  printf(" Compiled : %s %s\n", __DATE__, __TIME__);
  fflush(stdout);
}

int main(int argc, char **argv)
{ 
  char moduleName[80];
  
  if (argc < 2) {
    printf("Must supply name of second central server machine.\n");
    exit(-1);
  }
  
  displayVersion();

  sprintf(moduleName,"%s%s",BRIDGE_NAME_PREFIX,argv[1]);
  listeningDev = TCA_connect(moduleName,registerAllListener,
			     disconnectTCAListener,reconnectTCAListener,
			     NULL, NULL);
  if (listeningDev) {
    printf("Connected to host %s\n", tcaServerMachine());
  }
  
  TCA_setCentralHost(argv[1]);
  talkingDev = TCA_connect(moduleName,registerAllTalker,
			   disconnectTCATalker,reconnectTCATalker,
			   NULL, NULL);
  if (talkingDev) {
    printf("Connected to host %s\n", argv[1]);
  }

  
  /* May want to limit the crossed registered messages in some way.  
   * For example by providing a list of acceptable messages in a startup 
   * file.
   */
  
  crossRegisterMsgs();
  
  /* connect up the handlers for standard in. */
  stdin_connect(stdin_defaultInputHnd); 
  
  devMainLoop();
  exit(0);
}
