/*
   
   trtest.c:       Tramel performance testing functions.
   
   trtest consists of two functions, testS and
   testP, that are intended to be invoked by
   different Tramel zones (processes, tasks,
   threads) to measure the rate of message
   exchange between the two zones.
   
   testS simply subscribes to messages of
   subjects 1, 2, and 3.  For each message
   that it receives:
   
   If subject is 1, it does nothing.
   
   If subject is 2 (i.e., this is the
   last in a set of messages being
   transmitted by testP), it publishes
   a message of subject 4 to tell testP
   that the last message has been
   received.
   
   If subject is 3, it terminates.
   
   testP takes two arguments, message count C
   and message length L.  If C is zero, it
   simply publishes a message of subject 3 to
   stop testS and terminates.  Otherwise it
   turns off heartbeat monitoring, subscribes
   to messages of subject 4, notes the current
   time (the start time for the test), publishes
   C messages of content length L (L may be zero,
   in which case only message headers are sent),
   and enters Tr_mainLoop; on receipt of a
   message of subject 4 it notes the current
   time, prints calculated performance figures,
   and terminates.
   
   */

#include "tca/libc.h"
#include "tca.h"
#include <sysLib.h>

#define NUM_MSGS  1000
#define CENTRALHOST "als-vw4"

#define STATIC
#define _4_BYTE_MSG_FMT     "[char:4]"
#define _16_BYTE_MSG_FMT    "[char:16]"
#define _64_BYTE_MSG_FMT    "[char:64]"
#define _256_BYTE_MSG_FMT   "[char:256]"
#define _1K_BYTE_MSG_FMT    "[char:1024]"
#define _4K_BYTE_MSG_FMT    "[char:4096]"
#define _16K_BYTE_MSG_FMT   "[char:16384]"
#define _1M_BYTE_MSG_FMT    "[char:1000000]"

#define A4_BYTE_INFORM_MSG   "4ByteInformMsg"
#define A4_BYTE_CMD_MSG         "4ByteCmdMsg"
#define A16_BYTE_INFORM_MSG  "16ByteInformMsg"
#define A16_BYTE_CMD_MSG        "16ByteCmdMsg"
#define A64_BYTE_INFORM_MSG  "64ByteInformMsg"
#define A64_BYTE_CMD_MSG        "64ByteCmdMsg"
#define A256_BYTE_INFORM_MSG "256ByteInformMsg"
#define A256_BYTE_CMD_MSG       "256ByteCmdMsg"
#define A1K_BYTE_INFORM_MSG  "1kByteInformMsg"
#define A1K_BYTE_CMD_MSG        "1kByteCmdMsg"
#define A4K_BYTE_INFORM_MSG  "4kByteInformMsg"
#define A4K_BYTE_CMD_MSG        "4kByteCmdMsg"
#define A16K_BYTE_INFORM_MSG "16kByteInformMsg"
#define A16K_BYTE_CMD_MSG       "16kByteCmdMsg"
#define A1M_BYTE_INFORM_MSG  "1mByteInformMsg"
#define A1M_BYTE_CMD_MSG        "1mByteCmdMsg"


typedef struct
{
  unsigned int    count;
  struct timeval  startTime;
} Timer;

void ignore4(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A4_BYTE_INFORM_MSG, data);
}

void ignore16(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A16_BYTE_INFORM_MSG, data);
}

void ignore64(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A64_BYTE_INFORM_MSG, data);
}

void ignore256(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A256_BYTE_INFORM_MSG, data);
}

void ignore1k(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A1K_BYTE_INFORM_MSG, data);
}

void ignore4k(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A4K_BYTE_INFORM_MSG, data);
}

void ignore16k(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A16K_BYTE_INFORM_MSG, data);
}

void ignore1M(TCA_REF_PTR tcaRef, char *data)
{
  tcaFreeData(A1M_BYTE_INFORM_MSG, data);
}

void ack4(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A4_BYTE_CMD_MSG, data);
}

void ack16(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A16_BYTE_CMD_MSG, data);
}

void ack64(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A64_BYTE_CMD_MSG, data);
}

void ack256(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A256_BYTE_CMD_MSG, data);
}

void ack1k(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A1K_BYTE_CMD_MSG, data);
}

void ack4k(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A4K_BYTE_CMD_MSG, data);
}

void ack16k(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A16K_BYTE_CMD_MSG, data);
}

void ack1M(TCA_REF_PTR tcaRef, char *data)
{
  tcaSuccess(tcaRef);
  tcaFreeData(A1M_BYTE_CMD_MSG, data);
}


void registerMessages(void)
{
  tcaRegisterInformMessage(A4_BYTE_INFORM_MSG, _4_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A4_BYTE_CMD_MSG, _4_BYTE_MSG_FMT);
  
  tcaRegisterInformMessage(A16_BYTE_INFORM_MSG, _16_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A16_BYTE_CMD_MSG, _16_BYTE_MSG_FMT);
  
  tcaRegisterInformMessage(A64_BYTE_INFORM_MSG, _64_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A64_BYTE_CMD_MSG, _64_BYTE_MSG_FMT);
  
  tcaRegisterInformMessage(A256_BYTE_INFORM_MSG, _256_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A256_BYTE_CMD_MSG, _256_BYTE_MSG_FMT);
  
  tcaRegisterInformMessage(A1K_BYTE_INFORM_MSG, _1K_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A1K_BYTE_CMD_MSG, _1K_BYTE_MSG_FMT);
  
  tcaRegisterInformMessage(A4K_BYTE_INFORM_MSG, _4K_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A4K_BYTE_CMD_MSG, _4K_BYTE_MSG_FMT);
  
  tcaRegisterInformMessage(A16K_BYTE_INFORM_MSG, _16K_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A16K_BYTE_CMD_MSG, _16K_BYTE_MSG_FMT);
  
  tcaRegisterInformMessage(A1M_BYTE_INFORM_MSG, _1M_BYTE_MSG_FMT);
  tcaRegisterCommandMessage(A1M_BYTE_CMD_MSG, _1M_BYTE_MSG_FMT);
  
}

void registerHandlers(void)
{
  tcaRegisterHandler(A4_BYTE_INFORM_MSG, "ignore4", ignore4);
  tcaRegisterHandler(A4_BYTE_CMD_MSG, "ack4", ack4);
  
  tcaRegisterHandler(A16_BYTE_INFORM_MSG, "ignore16", ignore16);
  tcaRegisterHandler(A16_BYTE_CMD_MSG, "ack16", ack16);
  
  tcaRegisterHandler(A64_BYTE_INFORM_MSG, "ignore64", ignore64);
  tcaRegisterHandler(A64_BYTE_CMD_MSG, "ack64", ack64);
  
  tcaRegisterHandler(A256_BYTE_INFORM_MSG, "ignore256", ignore256);
  tcaRegisterHandler(A256_BYTE_CMD_MSG, "ack256", ack256);
  
  tcaRegisterHandler(A1K_BYTE_INFORM_MSG, "ignore1k", ignore1k);
  tcaRegisterHandler(A1K_BYTE_CMD_MSG, "ack1k", ack1k);
  
  tcaRegisterHandler(A4K_BYTE_INFORM_MSG, "ignore4k", ignore4k);
  tcaRegisterHandler(A4K_BYTE_CMD_MSG, "ack4k", ack4k);
  
  tcaRegisterHandler(A16K_BYTE_INFORM_MSG, "ignore16k", ignore16k);
  tcaRegisterHandler(A16K_BYTE_CMD_MSG, "ack16k", ack16k);
  
  tcaRegisterHandler(A1M_BYTE_INFORM_MSG, "ignore1M", ignore1M);
  tcaRegisterHandler(A1M_BYTE_CMD_MSG, "ack1M", ack1M);
  
}

int testS(void)
{
  tcaConnectModule("subscriber",CENTRALHOST);
  registerMessages();
  registerHandlers();
  tcaWaitUntilReady();
  tcaRegisterResource("Subscriber Resource", NUM_MSGS);
  tcaModuleListen();
  return 0;
}

int testP(unsigned int count)
{
  register int i;
  register int nmsgs = count-1;
  char *data;
  unsigned long startTicks, endTicks;
  float secs, kbytes;
  
  tcaConnectModule("producer",CENTRALHOST);
  tcaModuleRequires("subscriber", NULL);
  tcaWaitUntilReady();
  data = (char *)malloc(1024*1024);
  bzero(data,1024*1024);


  if (count <= 0) count = 10;
  /* Prime the pump. */
  printf("Priming the pump, please wait....\n");
  tcaInform(A4_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A4_BYTE_CMD_MSG, data);
  tcaInform(A16_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A16_BYTE_CMD_MSG, data);
  tcaInform(A64_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A64_BYTE_CMD_MSG, data);
  tcaInform(A256_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A256_BYTE_CMD_MSG, data);
  tcaInform(A1K_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A1K_BYTE_CMD_MSG, data);
  tcaInform(A4K_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A4K_BYTE_CMD_MSG, data);
  tcaInform(A16K_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A16K_BYTE_CMD_MSG, data);
  tcaInform(A1M_BYTE_INFORM_MSG, data);
  tcaWaitForCommand(A1M_BYTE_CMD_MSG, data);
  
  printf("bytes/msg			usecs/msg			kbytes/sec\n"); fflush(stdout);
  
  /*test four byte messages*/
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A4_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A4_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*4.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 4, (secs*1.0e6)/(float)count, kbytes/secs);
  
  /*test 16 byte messages*/
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A16_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A16_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*16.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 16, (secs*1.0e6)/(float)count, kbytes/secs);
  
  /*test 64 byte messages*/     
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A64_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A64_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*64.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 64, (secs*1.0e6)/(float)count, kbytes/secs);
  
  /*test 256 byte messages*/     
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A256_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A256_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*256.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 256, (secs*1.0e6)/(float)count, kbytes/secs);
  
  /*test 1K byte messages*/     
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A1K_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A1K_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*1024.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 1024, (secs*1.0e6)/(float)count, kbytes/secs);
  
  /*test 4K byte messages*/     
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A4K_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A4K_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*4096.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 4096, (secs*1.0e6)/(float)count, kbytes/secs);
  
  /*test 16K byte messages*/     
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A16K_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A16K_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*16384.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 16384, (secs*1.0e6)/(float)count, kbytes/secs);
  
  /*test 1M byte messages*/     
  startTicks = tickGet();
  for(i=0; i < nmsgs; i++) {
    tcaInform(A1M_BYTE_INFORM_MSG, data);
  }
  tcaWaitForCommandWithConstraints(NULL,A1M_BYTE_CMD_MSG, data,NO_TPLCONSTR);
  endTicks = tickGet();
  
  secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
  kbytes = (float)(count*1000000.0)/1000.0;
  printf("   %7d             %6.3f                   %6.3f\n",
	 1000000, (secs*1.0e6)/(float)count, kbytes/secs);
  
  fflush(stdout);
  return 0;
}

void centralStart(int numModules, int a2, int a3, int a4, int a5, int a6,
		  int a7, int a8, int a9)
{
  central(1,"-u -c -lx\0");
}

void subscriber(int a1, int a2, int a3, int a4, int a5, int a6, int a7, 
		int a8, int a9)
{
  testS();
}

void publisher(int count, int a2, int a3, int a4, int a5, int a6, int a7,
	       int a8, int a9)
{
  testP(count);
}

int trtest(int cnt)
{
  sp(centralStart, 1, 0, 0, 0, 0, 0, 0, 0, 0);
  sp(subscriber, 0, 0, 0, 0, 0, 0, 0, 0, 0);
  sp(publisher, cnt, 0, 0, 0, 0, 0, 0, 0, 0);
  return 0;
}
