/*****************************************************************************
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1996 Reid Simmons.  All rights reserved.
 *
 * MODULE: 
 *
 * FILE: test85.c
 *
 * ABSTRACT: Test the new features of TCA 8.5:
 *		tcaQueryNotify, tcaAddEventHandler, tcaRemoveEventHandler,
 *		enumerated type formats.
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/test/test85.c,v $ 
 * $Revision: 1.1 $
 * $Date: 1996/03/19 02:31:39 $
 * $Author: reids $
 *
 * REVISION HISTORY:
 * $Log: test85.c,v $
 * Revision 1.1  1996/03/19  02:31:39  reids
 * Added test code for new V8.5 features: tcaQueryNotify, tcaAddEventHandler,
 *   tcaRemoveEventHandler, and enumerated formats.
 *
 *****************************************************************************/

#include "tca/libc.h"
#include "tca.h"

typedef enum { WaitVal, SendVal, ReceiveVal, ListenVal } STATUS_ENUM;

typedef struct { int i1;
		 STATUS_ENUM status;
		 double matrix[2][3];
		 double d1;
	       } QUERY_TYPE, *QUERY_PTR;

typedef struct { STATUS_ENUM status;
		 int count;
		 char *str1;
	       } REPLY_TYPE, *REPLY_PTR;

#define QUERY_NOTIFY_MSG  	  "queryNotifyTest"
#define QUERY_NOTIFY_FORMAT       "{int, {enum : 3}, [double:2,3], double}"
#define QUERY_NOTIFY_REPLY_FORMAT "{{enum : 3}, int, string}"

typedef STATUS_ENUM INFORM_TYPE, *INFORM_PTR;

#define INFORM_MSG    "informTest"
#define INFORM_FORMAT "{enum WaitVal, SendVal, ReceiveVal, ListenVal}"

static char *enumString (STATUS_ENUM enum1)
{
  static char *enums[4] = {"Wait", "Send", "Receive", "Listen"};

  return enums[(int)enum1];
}

static void queryHandler (TCA_REF_PTR ref, QUERY_PTR query)
{
  INFORM_TYPE status = WaitVal;
  REPLY_TYPE response;

  printf("queryHandler: Received %s\n", tcaReferenceName(ref));
  printf("  {%d, %s, [%f, %f, %f][%f, %f, %f], %f}\n",
	 query->i1, enumString(query->status), 
	 query->matrix[0][0], query->matrix[0][1], query->matrix[0][2],
	 query->matrix[1][0], query->matrix[1][1], query->matrix[1][2],
	 query->d1);

  tcaInform(INFORM_MSG, &status);

  response.status = ReceiveVal;
  response.count = 1;
  response.str1 = "Hello, World";
  tcaReply(ref, &response);

  tcaFreeData(tcaReferenceName(ref), (void *)query);
}

static void replyHandler (void *data, void *clientData)
{
  REPLY_PTR reply = (REPLY_PTR)data;

  printf("reply Handler: Received response for %s with client data %d\n", 
	 QUERY_NOTIFY_MSG, *(int *)clientData);
  printf("  {%s, %d, %s}\n",
	 enumString(reply->status), reply->count, reply->str1);

  tcaFreeReply(QUERY_NOTIFY_MSG, data);
  free(data);
}

static void informHandler (TCA_REF_PTR ref, INFORM_PTR inform)
{
  printf("informHandler: Received %s with value %s\n", 
	 tcaReferenceName(ref), enumString(*inform));

  tcaFreeData(tcaReferenceName(ref), (void *)inform);
}

static void stdinHnd (int fd, void *clientData)
{
  char inputLine[81];

  fgets(inputLine, 80, stdin);

  switch (inputLine[0]) {
  case 'q': case 'Q': 
    tcaClose();
    exit(0);
  case 'm': case 'M':
    { INFORM_TYPE status = ListenVal;
      tcaInform(INFORM_MSG, &status);
      break;
    }
  case 'r': case 'R':
    { QUERY_TYPE q1 = {666, SendVal, {{0.0, 1.0, 2.0}, {1.0, 2.0, 3.0}}, M_PI};
      static count = 42;
      tcaQueryNotify(QUERY_NOTIFY_MSG, &q1, replyHandler, &count);
      break;
    }
  case 'u': case 'U': 
    tcaRemoveEventHandler(fileno(stdin));
    printf("No longer listening to stdin\n");
    break;
  default: 
    printf("stdinHnd [%s]: Received %s", (char *)clientData, inputLine);
    fflush(stdout);
  }
}

#if defined(VXWORKS)
void a1main(host)
char *host;
#else
void main(int argc, char **argv)
#endif
{
#if !defined(VXWORKS)
  const char *host = tcaServerMachine();
#endif

  printf("Connect ...\n");
  tcaConnectModule("Test 8.5", host);

  tcaRegisterQuery(QUERY_NOTIFY_MSG, QUERY_NOTIFY_FORMAT,
		   QUERY_NOTIFY_REPLY_FORMAT, queryHandler);
  tcaRegisterInform(INFORM_MSG, INFORM_FORMAT, informHandler);
  tcaWaitUntilReady();

  printf("\nType 'm' to send %s; 'r' to send %s; 'u' to stop accepting input; 'q' to quit\n",
	 INFORM_MSG, QUERY_NOTIFY_MSG);

  tcaAddEventHandler(fileno(stdin), stdinHnd, "FD");

  tcaModuleListen();
}
