/*****************************************************************************
 *
 * (c) Copyright 1994. National Aeronautics and Space Administration.
 *     Ames Research Center, Moffett Field, CA 94035-1000 
 *
 * FILE:     sender.c
 *
 * ABSTRACT: simple TCX example (based on an example in the TCX 7.7 manual)
 *
 * REVISION HISTORY:
 *
 * 11-Feb-94 Terry Fong. Created.
 *
 *****************************************************************************/

#include "tca/libc.h"		
#include "tca.h"		/* TCX function declarations */
#include "tca/basics.h"		

#include "common.h"		/* common definitions */

static BOOLEAN done=FALSE;

static void DoneHnd (TCA_REF_PTR ref, void *pData)
{
  /* Just used for synchronization. */
/*  printf("Done\n");*/
  done = TRUE;
}

float sslCalcTimevalDiff (struct timeval *pT1, struct timeval *pT2);


/******************************************************************************
 * main ()
 *****************************************************************************/
int main (int argc, char *argv[])
{
  float *data;
  int i, j, nBytes;
  float secs;
  struct {
    char *msgName; int msgSize;
  } messages[] = {
    { "NullMsg", 	0                    },
    { "Float4Msg", 	sizeof(float)*4      },
    { "Float64Msg",	sizeof(float)*64     },
    { "Float256Msg",	sizeof(float)*256    },
    { "Float1KMsg",	sizeof(float)*1024   },
    { "Float4KMsg",	sizeof(float)*4096   },
    { "Float16KMsg",	sizeof(float)*16384  },
    { "Float64KMsg",	sizeof(float)*65536  },
    { "Float256KMsg",	sizeof(float)*262144 },
    { "", 0 }
  };
#ifndef VXWORKS
  struct timeval sTime, eTime;
#else
  unsigned long startTicks, endTicks;
#endif
  
  printf ("\nStarting Sender 3 %d\n", MAX_DATA_SIZE);
  fflush(stdout);

  data = malloc(MAX_DATA_SIZE);
  
  for (j = 0; j<MAX_DATA_SIZE/sizeof(float); j++)
    data[j] = (float) j;
    
  printf ("\nInitialized\n");
  fflush(stdout);
  
  /*
   * Connect this module to the TCA Server (specified by the CENTRALHOST
   * shell variable).
   */
  tcaConnectModule ("sender", tcaServerMachine());
  tcaDirectResource("sender");
  tcaRegisterInform(DONE_MSG_INFORM,DONE_MSG_INFORM_FORMAT, DoneHnd);
  tcaAddHndToResource(DONE_MSG_INFORM, "sender");

  tcaModuleRequires("receiver", NULL);
  tcaWaitUntilReady();
  
  
  for (j=0; *messages[j].msgName; j++) {
    done = FALSE;
    tcaInform (messages[j].msgName, data);
    while(!done) {
      tcaHandleMessage(WAITFOREVER);
    }
#ifndef VXWORKS
    sleep(1);
#endif
    printf ("\nSending %d <%s> messages...\n", NSEND, messages[j].msgName);
    fflush(stdout);
#ifndef VXWORKS
    gettimeofday(&sTime, NULL);
#else
    startTicks = tickGet();
#endif
    for (i=0; i<NSEND; i++) {
      tcaInform (messages[j].msgName, data);
    }
    done = FALSE;
    while(!done) {
      tcaHandleMessage(WAITFOREVER);
    }
#ifndef VXWORKS
    gettimeofday (&eTime, NULL);
    secs = sslCalcTimevalDiff (&eTime, &sTime);
#else
    endTicks = tickGet();
    secs = (float)(endTicks-startTicks)*1.0/(float)sysClkRateGet();
#endif
    nBytes = messages[j].msgSize*NSEND;
    
    printf ("Sent %d bytes in %.2f seconds (%.2f bytes/sec)\n",
	    nBytes, secs, (float) nBytes / secs);
    printf ("%d messages (%d bytes each) at %.0f messages/sec\n",
	    NSEND, messages[j].msgSize, (float) NSEND / secs);
    fflush(stdout);
  }
  
  /* shut down */
  printf ("\n");
  fflush(stdout);
  exit (0);
}

/******************************************************************************
 *
 * sslCalcTimevalDiff - calculate the difference between two timevals in
 * seconds
 *
 * INPUTS:
 *   pT1		pointer to first timeval
 *   pT2		pointer to first timeval
 *
 * RETURNS:
 *   Difference between timevals in seconds (pT1 - pT2)
 *
 *****************************************************************************/
float
sslCalcTimevalDiff (struct timeval *pT1, struct timeval *pT2)
{
  return ((float) (pT1->tv_sec - pT2->tv_sec) +
          ((float) (pT1->tv_usec - pT2->tv_usec)) / 1000000.0);
}
