/*****************************************************************************
 * PROJECT: Task Control Architecture
 *
 * (c) Copyright 1995 Richard Goodwin.  All rights reserved.
 *
 * FILE: forwarder.c
 *
 * ABSTRACT:
 *
 * Forwarder connects to two central servers and forwards broadcast messages
 * from one server to the other.  It does this by connecting to both centrals
 * and listening to one for a set of broadcast messages.  When it receives
 * a message, it does a broadcast to the other central.
 * 
 * The program takes one argument, the name of the machine with the second
 * central server.
 *
 * Multiple forwarders can be run, forwarding to different central servers.
 *
 * The code does use some of the tca internal directly and may not work with 
 * subsequent releases of tca.  I plan to add the tca hooks needed to be
 * able to have this functionality in a "safer" implementation.
 * 
 * REVISION HISTORY:
 *
 * $Log: forwarder.c,v $
 * Revision 1.12  1996/02/14  22:13:21  rich
 * Fix problems with static declarations.
 *
 * Revision 1.11  1995/12/17  20:23:21  rich
 * Flush stdout to work better with nanny.
 *
 * Revision 1.10  1995/10/29  18:27:47  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.9  1995/07/12  04:56:19  rich
 * Release of 8.0.
 * Added test for new features.
 *
 * Revision 1.8  1995/07/10  16:19:46  rich
 * Interm save.
 *
 * Revision 1.7  1995/04/21  03:54:18  rich
 * Use new tca routines to support multiple central servers.
 *
 * Revision 1.6  1995/04/07  05:04:56  rich
 * Fixed GNUmakefiles to find the release directory.
 * Moved all system includes into libc.h
 * Fixed problems found by sgi cc compiler.  Many type problems.
 *
 * Revision 1.5  1995/04/04  19:44:32  rich
 * Added sgi support.
 *
 * Revision 1.4  1995/03/30  15:49:02  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 *
 * Revision 1.3  1995/03/16  18:05:57  rich
 * Merged in changes to the 7.9 branch.
 * Changed the VERSION_ to TCA_VERSION_
 *
 * Revision 1.2  1995/02/03  03:58:44  rich
 * Added forwarding code and changes needed to GNUmakefile and common.h.
 *
 * Revision 1.1  1995/02/03  03:38:03  rich
 * Adding Forwarder code.
 *
 *
 * $Revision: 1.12 $
 * $Date: 1996/02/14 22:13:21 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "tca/libc.h"		
#include "tca.h"		/* TCA function declarations */

#include "common.h"		/* common definitions */

#define CLEANUP

static TCA_CONTEXT_PTR listenServer = NULL; /* central for listening */
static TCA_CONTEXT_PTR forwardServer = NULL; /* central for forwarding to */


/* The following two routine are used to switch between the two central
 * servers.
 */

/* These routines are handlers for the broadcast message.
 * ordinarily, this module is connected to the central server for listening.
 * Each handler switches centrals, resends the broadcast and switches back.
 *
 * The module should not be sent messages from the second central server.
 * This can be accomplished by not registering any message handlers 
 * with the second central server.
 */

static void NullHnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[0].msgName, pData);
  tcaSetContext(listenServer);
}

static void Float4Hnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[1].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float4Msg", (void *) pData);
  
}

static void Float64Hnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[2].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float64Msg", (void *) pData);
}

static void Float256Hnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[3].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float256Msg", (void *) pData);
}

static void Float1KHnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[4].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float1KMsg", (void *) pData);
}

static void Float4KHnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[5].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float4KMsg", (void *) pData);
}

static void Float16KHnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[6].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float16KMsg", (void *) pData);
}

static void Float64KHnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[7].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float64KMsg", (void *) pData);
}

static void Float256KHnd (TCA_REF_PTR ref, float *pData)
{
  tcaSetContext(forwardServer);
  tcaBroadcast (messages[8].msgName, pData);
  tcaSetContext(listenServer);
  
  /* free the data */
  tcaFreeData ("Float256KMsg", (void *) pData);
}

/******************************************************************************
 * RECEIVER main ()
 ******************************************************************************/
int main (int argc, char *argv[])
{
  int i;
  struct {
    char *msgName; char *hndName; void (*hndProc)(TCA_REF_PTR, float *);
  } hndArray[] = {
    {"NullMsg",      "NullHnd",      NullHnd	  },
    {"Float4Msg",    "Float4Hnd",    Float4Hnd	  },
    {"Float64Msg",   "Float64Hnd",   Float64Hnd   },
    {"Float256Msg",  "Float256Hnd",  Float256Hnd  },
    {"Float1KMsg",   "Float1KHnd",   Float1KHnd   },
    {"Float4KMsg",   "Float4KHnd",   Float4KHnd   },
    {"Float16KMsg",  "Float16KHnd",  Float16KHnd  },
    {"Float64KMsg",  "Float64KHnd",  Float64KHnd  },
    {"Float256KMsg", "Float256KHnd", Float256KHnd },
    { "" }
  };
  
  char moduleName[80];
  
  /*
   * Connect this module to the TCA Server (specified by the CENTRALHOST
   * shell variable).
   */
  if (argc < 2) {
    printf("Must supply name of second central server machine.\n");
    fflush(stdout);
    exit(-1);
  }
  sprintf(moduleName,"forwarder_%s",argv[1]);
  tcaConnectModule(moduleName, tcaServerMachine());
  
  /* register message formats */
  for (i=0; *messageArray[i].msgName; i++) {
    
    tcaRegisterBroadcastMessage (messageArray[i].msgName, 
				 messageArray[i].msgFormat);
  }
  
  /* register handlers */
  for (i=0; *hndArray[i].msgName; i++) {
    tcaRegisterHandler (hndArray[i].msgName, hndArray[i].hndName,
			hndArray[i].hndProc);
#ifdef CLEANUP
    tcaCleanUpAfterAchieved(hndArray[i].msgName);
#endif
  }
  
  tcaLimitPendingResource(moduleName,2);
  
  /* receive & handle messages forever (timeout=NULL) */
  listenServer=tcaGetContext();

  /* Have to connect to the second central server now. */
  tcaConnectModule(moduleName, argv[1]);
  
  forwardServer = tcaGetContext();
  
  /* register message formats */
  /* This is to make sure the second central knows the formats.   */
  for (i=0; *messageArray[i].msgName; i++) {
    
    tcaRegisterBroadcastMessage (messageArray[i].msgName, 
				 messageArray[i].msgFormat);
  }
  
  /* Wait until connections are complete */
  tcaWaitUntilReady();
  
  /* Connected to the second server   */
  printf("Connected to %s\n",argv[1]);
  fflush(stdout);
  
  /* Switch back to the original central server and continue.   */
  tcaSetContext(listenServer);
  tcaWaitUntilReady();
  tcaModuleListen ();
  
  exit(0);
}
