/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: taskTree
 *
 * FILE: tskTreeModule.c
 *
 * ABSTRACT:
 * Modules interface with creation and manipulation of hierarchical trees.
 * 
 * REVISION HISTORY
 *
 * $Log: tskTreeMod.c,v $
 * Revision 1.18  1996/08/22  16:36:07  rich
 * Check the return code on tcaQueryCentral calls.
 *
 * Revision 1.17  1996/06/25  20:51:54  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.16  1995/12/15  01:23:29  rich
 * Moved Makefile to Makefile.generic to encourage people to use
 * GNUmakefile.
 * Fixed a memory leak when a module is closed and some other small fixes.
 *
 * Revision 1.15  1995/11/03  03:04:57  rich
 * Changed msgFind to keep if from going into an infinite loop if there is no
 * central connection.  This only happens when an exit procedure that does
 * not exit is registered.  msgFind can now return NULL, so I added some
 * checks for the return value to keep modules from seg-faulting.
 *
 * Revision 1.14  1995/06/14  03:23:02  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.13  1995/05/31  19:37:20  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.12  1995/04/19  14:29:03  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.11  1995/04/08  02:06:33  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.10  1995/01/18  22:43:09  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.9  1994/05/17  23:17:58  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.8  1994/04/28  16:17:35  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.7  1994/04/16  19:43:20  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.6  1993/11/21  20:19:45  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.5  1993/10/21  16:14:34  rich
 * Fixed compiler warnings.
 *
 * Revision 1.4  1993/08/30  21:54:43  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:17:10  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:32  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:38  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:35  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:26:07  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 22-Dec-90 Christopher Fedor, School of Computer Science, CMU
 * Added tcaQuery NullReply checks to tcaFindParentReference,
 * tcaFindFirstChild,
 * tcaFindLastChild, tcaFindNextChild, tcaFindPreviousChild, 
 * tcaFindTopLevelReference, tcaFindAnteReferenceByName, tcaFindChildByName
 *
 * 16-Nov-90 Christopher Fedor, School of Computer Science, CMU
 * Moved Updated tcaCreateReference back here.
 *
 * 12-Sep-90 Christopher Fedor, School of Computer Science, CMU
 * Updated for new TCA_REF_PTR.
 * Revised to Software Standards.
 *
 * 26-Mar-90 Long-Ji Lin, School of Computer Science, CMU
 * Modified code to allow interval monitors to be created by calling 
 * tcaAddChildReference() or tcaCreateReference().
 *
 * 27-Nov-89 Reid Simmons, School of Computer Science, CMU
 * Added new routines for tracing task tree (find first and last child, 
 * find child by name, get name and data of reference).
 *
 * 14-Aug-89 Reid Simmons, School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.18 $
 * $Date: 1996/08/22 16:36:07 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"
#include "taskTree.h"


/******************************************************************************
 *
 * FUNCTION: int32 creationClass(msgClass)
 *
 * DESCRIPTION: 
 * Returns TRUE if the class of message is one that can be created by calling
 * tcaCreateReference() or tcaAddChildReference() (currently, goals, commands,
 * point-monitors, interval-monitors).
 *
 * INPUTS: TCA_MSG_CLASS_TYPE msgClass;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 creationClass(TCA_MSG_CLASS_TYPE msgClass)
{
  return ((msgClass == GoalClass) || 
	  (msgClass == CommandClass) ||
	  (msgClass == PointMonitorClass) ||
	  (msgClass == PollingMonitorClass) ||
	  (msgClass == DemonMonitorClass));
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaCreateReference(name)
 *
 * DESCRIPTION: 
 * Add a reference as a child of the currently handled message.
 * Add to tcaRootNodeGlobal if no such message.
 * Returns the created reference.
 *
 * INPUTS: const char *name;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaCreateReference(const char *name)
{
  int32 refId;
  MSG_PTR msg;
  TCA_MSG_CLASS_TYPE msg_class;
  
  msg = msgFind(name);
  if (msg == NULL) return NULL;
  msg_class = msg->msgData->msg_class;
  
  if (!creationClass(msg_class)) {
    tcaModError("ERROR: tcaCreateReference: Must be a goal, command or monitor.\n");
    return NULL;
  }
  
  if (tcaQueryCentral(TCA_CREATE_REF_QUERY, (void *)&(msg->msgData->refId),
		      (void *)&refId) == Success) {
    return(tcaRefCreate(msg, name, refId));
  } else {
    return NULL;
  }
}


/******************************************************************************
 *
 * FUNCTION: int32 tcaIsSameReference(ref1, ref2)
 *
 * DESCRIPTION: 
 * Returns TRUE if ref1 and ref2 both represent the same task-tree node.
 *
 * INPUTS: TCA_REF_PTR ref1, ref2;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

int tcaIsSameReference(TCA_REF_PTR ref1, TCA_REF_PTR ref2)
{
  return ((ref1 == ref2) || 
	  (ref1->refId == ref2->refId && ref1->name == ref2->name) ||
	  (ref1->refId == ref2->refId && STREQ(ref1->name, ref2->name)));
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaAddChildReference(parentRef, name)
 *
 * DESCRIPTION: 
 * Add a reference as a child of the parent reference.
 * Returns the reference added.
 *
 * INPUTS:
 * TCA_REF_PTR parentRef;
 * const char *name;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaAddChildReference(TCA_REF_PTR parentRef, const char *name)
{
  int32 refId;
  MSG_PTR msg;
  TCA_MSG_CLASS_TYPE msg_class;
  ADD_CHILD_TYPE addChild;
  
  msg = msgFind(name);
  if (msg == NULL) return NULL;
  msg_class = msg->msgData->msg_class;
  
  if (!creationClass(msg_class)) {
    tcaModError("ERROR: tcaAddChildReference: Must be a goal, command or monitor.\n");
    return NULL;
  }
  
  addChild.msgRef = msg->msgData->refId;
  addChild.parentRefId = parentRef->refId;
  
  if (tcaQueryCentral(TCA_ADD_CHILD_QUERY, (void *)&addChild,
		      (void *)&refId) == Success) {
    return(tcaRefCreate(msg, name, refId));
  } else {
    return NULL;
  }
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindParentReference(childRef)
 *
 * DESCRIPTION: Return the parent reference of the child.
 *
 * INPUTS: TCA_REF_PTR childRef
 *
 * OUTPUTS: TCA_REF_PTR
 *
 * The returned value may be NULL.
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindParentReference(TCA_REF_PTR childRef)
{ 
  TCA_REF_PTR ref=NULL;
  
  if (tcaQueryCentral(TCA_FIND_PARENT_QUERY, (void *)&(childRef->refId),
		      (void *)&ref)
      == Success) {
    return ref;
  } else {
    return NULL;
  }
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindFirstChild(parentRefPtr)
 *
 * DESCRIPTION: 
 * Return the "first" child of the parent node.
 * Since "first" is actually ambiguous, what is returned is one of the 
 * children whose achievement time is no later than the achievement 
 * times of the rest of the children (or NULL if there are no children).
 *
 * INPUTS: TCA_REF_PTR parentRefPtr;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindFirstChild(TCA_REF_PTR parentRefPtr)
{ 
  TCA_REF_PTR ref;
  
  if (tcaQueryCentral(TCA_FIRST_CHILD_QUERY, 
		      (void *)&(parentRefPtr->refId), (void *)&ref)
      == Success)
    return ref;
  else
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindLastChild(parentRefPtr)
 *
 * DESCRIPTION: 
 * Return the "last" child of the parent node.
 * Since "last" is actually ambiguous, what is returned is one of the 
 * children whose achievement time is no earlier than the achievement 
 * times of the rest of the children (or NULL if there are no children).
 *
 * INPUTS: TCA_REF_PTR parentRefPtr;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindLastChild(TCA_REF_PTR parentRefPtr)
{ 
  TCA_REF_PTR ref=NULL;
  
  if (tcaQueryCentral(TCA_LAST_CHILD_QUERY, 
		      (void *)&(parentRefPtr->refId), (void *)&ref)
      == Success)
    return ref;
  else
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindNextChild(childRefPtr)
 *
 * DESCRIPTION: 
 * Return the "next" sibling of the child.
 * "Next" is ambiguous, so what is returned is the earliest sibling
 * whose achievement time is after the end of the child's achievement
 * time (or NULL if there is no such sibling).
 *
 * INPUTS: TCA_REF_PTR childRefPtr;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindNextChild(TCA_REF_PTR childRefPtr)
{ 
  TCA_REF_PTR ref=NULL;
  
  if (tcaQueryCentral(TCA_NEXT_CHILD_QUERY, (void *)&(childRefPtr->refId),
		      (void *)&ref)
      == Success)
    return ref;
  else
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindPreviousChild(childRefPtr)
 *
 * DESCRIPTION: 
 * Return the "previous" sibling of the child.
 * "Previous" is ambiguous, so what is returned is the latest sibling
 * whose achievement time is before the start of the child's achievement
 * time (or NULL if there is no such sibling).
 *
 * INPUTS: TCA_REF_PTR childRefPtr;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindPreviousChild(TCA_REF_PTR childRefPtr)
{ 
  TCA_REF_PTR ref=NULL;
  
  if (tcaQueryCentral(TCA_PREV_CHILD_QUERY, (void *)&(childRefPtr->refId),
		      (void *)&ref)
      == Success)
    return ref;
  else
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindFailedReference(ERefPtr)
 *
 * DESCRIPTION: 
 * Like "FindParentReference", except goes up the tree to find the
 * first node that is not a failure handler (i.e., of ExceptionClass).
 * ERefPtr must be a reference to an exception handler.
 *
 * INPUTS: TCA_REF_PTR ERefPtr;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindFailedReference(TCA_REF_PTR ERefPtr)
{ 
  TCA_REF_PTR ref=NULL;
  
  if (ERefPtr) {
    if (!ERefPtr->msg)
      ERefPtr->msg = msgFind(ERefPtr->name);
    if (ERefPtr->msg == NULL) return NULL;
  }
  else {
    tcaModError("ERROR: tcaFailedReference must be passed an exception class ref.");
    return NULL;
  }
  
  if (ERefPtr->msg->msgData->msg_class != ExceptionClass) {
    tcaModError("ERROR: tcaFailedReference must be passed an exception class ref.");
    return NULL;
  }
  
  if (tcaQueryCentral(TCA_FAILED_CHILD_QUERY,
		      (void *)&(ERefPtr->refId), (void *)&ref) == Success)
    return ref;
  else 
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindTopLevelReference(childRefPtr)
 *
 * DESCRIPTION: 
 * Return the top level reference of the task tree.
 * Return NULL if the given child is root node OR is not a valid task tree
 * node.
 *
 * INPUTS: TCA_REF_PTR childRefPtr;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindTopLevelReference(TCA_REF_PTR childRefPtr)
{
  TCA_REF_PTR ref=NULL;
  
  if (tcaQueryCentral(TCA_FIND_TOPLEVEL_QUERY,
		      (void *)&(childRefPtr->refId), (void *)&ref)
      == Success)
    return ref;
  else
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindAnteReferenceByName(childRefPtr, anteName)
 *
 * DESCRIPTION:
 * Return the child's antecedent node with a specified (message) name.
 *
 * INPUTS:
 * TCA_REF_PTR childRefPtr; 
 * char *anteName;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindAnteReferenceByName(TCA_REF_PTR childRefPtr,
				       const char *anteName)
{
  TCA_REF_PTR ref=NULL;
  FIND_REF_TYPE findAnteRef;
  
  findAnteRef.refId = childRefPtr->refId;
  findAnteRef.name = anteName;
  
  if (tcaQueryCentral(TCA_ANTE_BY_NAME_QUERY, (void *)&findAnteRef,
		      (void *)&ref)
      == Success)
    return ref;
  else
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TCA_REF_PTR tcaFindChildByName(parentRefPtr, childName)
 *
 * DESCRIPTION: 
 * Return the child node of the parent which has the specified (message) 
 * name.  If more than one such child, returns the "first" one it comes
 * across (where "first" is implementation dependent).
 *
 * INPUTS:
 * TCA_REF_PTR parentRefPtr; 
 * char *childName;
 *
 * OUTPUTS: TCA_REF_PTR
 *
 *****************************************************************************/

TCA_REF_PTR tcaFindChildByName(TCA_REF_PTR parentRefPtr,  
			       const char *childName)
{
  TCA_REF_PTR ref=NULL;
  FIND_REF_TYPE findChildRef;
  
  findChildRef.refId = parentRefPtr->refId;
  findChildRef.name = childName;
  
  if (tcaQueryCentral(TCA_CHILD_BY_NAME_QUERY, (void *)&findChildRef,
		      (void *)&ref)
      == Success)
    return ref;
  else
    return NULL;
}


/******************************************************************************
 *
 * FUNCTION: void tcaKillTaskTree(ref)
 *
 * DESCRIPTION: Kill the task tree rooted at the node indicated by ref.
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void tcaKillTaskTree(TCA_REF_PTR ref)
{ 
  (void)tcaInform(TCA_KILL_TREE_INFORM, (void *)&(ref->refId));
}


/******************************************************************************
 *
 * FUNCTION: void tcaKillSubTaskTree(ref)
 *
 * DESCRIPTION: Kill the task trees rooted BELOW the node indicated by ref.
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void tcaKillSubTaskTree(TCA_REF_PTR ref)
{ 
  (void)tcaInform(TCA_KILL_SUBTREE_INFORM, (void *)&(ref->refId));
}


/******************************************************************************
 *
 * FUNCTION: void tcaDisplayTaskTree(ref)
 *
 * DESCRIPTION: Display the task tree rooted at the node indicated by ref.
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void tcaDisplayTaskTree(TCA_REF_PTR ref)
{
  (void)tcaInform(TCA_DISPLAY_TREE_INFORM, (void *)&(ref->refId));
}
