/*****************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 *
 * MODULE: temporal constraints
 *
 * FILE: tplConstr.c
 *
 * ABSTRACT:
 * Partial temporal ordering constraints
 *
 * REVISION HISTORY
 *
 * $Log: tplConstr.c,v $
 * Revision 1.25  1996/07/19  18:14:38  reids
 * Record broadcast messages if handler is registered before message.
 * Transfer any pending messages to the new resource under "addHndToResource"
 * Fixed tcaDelayCommand (wrong time units).
 * Fixed logging of refid's (have to distinguish whether they are part of
 *   a status, message, or "always" log).
 * Sanity check for encoding/decoding messages.
 *
 * Revision 1.24  1996/06/25  20:51:51  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.23  1996/02/10  16:50:49  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.22  1996/01/30  15:05:09  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.21  1996/01/27  21:54:40  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.20  1995/12/17  20:22:33  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.19  1995/10/29  18:27:21  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.18  1995/10/07  19:08:02  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.17  1995/06/14  03:23:00  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.16  1995/05/31  19:37:14  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.15  1995/04/19  14:29:15  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.14  1995/03/19  19:39:50  rich
 * Implemented direct connections using tcaDirectResouce call.
 * Also made the basics.h file a module include.
 * Changed class in the interval structure to be interval_class to avoid a
 * conflict with C++.
 *
 * Revision 1.13  1995/01/25  00:01:53  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Fixed problems with little endian translation.
 *
 * Revision 1.12  1995/01/18  22:43:42  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.11  1994/10/25  17:11:13  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.10  1994/07/26  15:04:19  reids
 * Fixed the printing (spacing) of achievement constraint messages
 *
 * Revision 1.9  1994/05/17  23:18:33  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.8  1994/04/28  16:17:55  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.7  1993/12/14  17:35:44  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.6  1993/11/21  20:20:05  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.5  1993/10/21  16:14:40  rich
 * Fixed compiler warnings.
 *
 * Revision 1.4  1993/08/30  21:55:12  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:17:40  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:57  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:41  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:59  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:26:34  fedor
 * Added Logging.
 *
 *  18-Oct-92, Richard Goodwin, School of Computer Science, CMU
 * Added change to Initialize_TaskTreeNode suggested by Reid.
 *
 *  3-Jul-91, Reid Simmons, School of Computer Science, CMU
 * Freed data used the TplConstrainHandler. 
 *
 * 24-Jun-91, Reid Simmons, School of Computer Science, CMU
 * Changed the "name" of a quantity to be a TCA_TIME_POINT_PTR.  
 * This makes it possible to refer back to the dispatch with which the
 * quantities are associated.
 *
 * 20-Jun-91, Reid Simmons, School of Computer Science, CMU
 * Added rule to explicitly assert that "Now" is greater than the end of
 * achievement or planning intervals when their subtrees are
 * achieved/planned.
 *
 * 30-Nov-89, Reid Simmons, School of Computer Science, CMU
 * Needed to make the handling and achievement intervals of commands
 * distinct in order for exception handling with commands to do the
 * temporally right thing.
 *
 * 28-Nov-89, Long-Ji Lin, School of Computer Science, CMU
 * The code (in AddTplConstraints()) that calls Initialize_TaskTreeNode()
 * to create a new tree node was moved to dispatchServer().
 *
 * 14-Aug-89, Reid Simmons, School of Computer Science, CMU
 * Added routines for killing task trees
 * (Cancel_Activation_Message, FreeInterval, and KillAfterAttending check).
 *
 *  2-Aug-91, Reid Simmons, School of Computer Science, CMU
 * Changed "printf"s to use logging facilities.
 * Made changes so you can now issue commands and goals from a Query.
 *
 * 28-Apr-89, Reid Simmons, School of Computer Science, CMU
 * Created (based on existing QLattice code)
 *
 * $Revision: 1.25 $
 * $Date: 1996/07/19 18:14:38 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalS.h"
#include "tcaMem.h"
#include "recvMsg.h"


/*****
 * The quantity "name" is now a reference back to the dispatch with which it
 *  is associated.  Coerce to a string (char *) to conform to the qlattice
 *   code.
 *****/

static char *createQName(TCA_POINT_CLASS_TYPE point,
			 TCA_INTERVAL_CLASS_TYPE interval,
			 DISPATCH_PTR dispatch)
{ 
  TCA_TIME_POINT_PTR timePoint;
  
  timePoint = NEW(TCA_TIME_POINT_TYPE);
  
  timePoint->point_class = point;
  timePoint->interval.interval_class = interval;
  timePoint->interval.msgRef = dispatch->locId;
  
  return (char *)timePoint;
}

/******
 * This uses a 2D array of characters to avoid having to malloc up
 * space when names are printed.  Cycles through the array so that up
 * to five qName calls can be made before a buffer is overwritten
 ******/

char *qName(QUANTITY_PTR q)
{
  static char name[5][100];
  static int32 last = 0;
  TCA_TIME_POINT_PTR timePoint;
  DISPATCH_PTR dispatch;
  
  if (q == GET_S_GLOBAL(Now)) {
    return "Now";
  } else {
    timePoint = (TCA_TIME_POINT_PTR)(q->name);
    dispatch = DISPATCH_FROM_ID(timePoint->interval.msgRef);
    last = (last+1)%5;
    (void)sprintf(name[last], "%s(%s(%s))", 
		  ((timePoint->point_class == StartPoint) ? "Start" : "End"),
		  ((timePoint->interval.interval_class == HandlingInterval)
		   ? "Handling" 
		   : ((timePoint->interval.interval_class 
		       == AchievementInterval)
		      ? "Achievement" : "Planning")),
		  dispatch->msg->msgData->name);
    return name[last];
  }
}

static DISPATCH_PTR qDispatch(QUANTITY_PTR q)
{
  TCA_TIME_POINT_PTR timePoint;
  DISPATCH_PTR dispatch;
  
  if (q == GET_S_GLOBAL(Now)) {
    return NULL;
  } else {
    timePoint = (TCA_TIME_POINT_PTR)(q->name);
    dispatch = (DISPATCH_PTR)idTableItem(timePoint->interval.msgRef, 
					 GET_S_GLOBAL(dispatchTable));
    return dispatch;
  }
}

static interval_ptr createInterval (TCA_INTERVAL_CLASS_TYPE class,
				    DISPATCH_PTR dispatch)
{ 
  interval_ptr interval;
  
  interval = NEW(interval_type);
  interval->start = createQuantity(createQName(StartPoint, class, dispatch));
  interval->end = createQuantity(createQName(EndPoint, class, dispatch));
  QAssert(interval->start, "<", interval->end, "Physics");
  
  return interval;
}

/***
 * The achievement or planning of a node is detected when "Now" cannot be shown
 *  to be less than end of the associated achievement or planning interval.
 *  However, "Now" cannot actually be inferred to be greater than the end
 *  point.
 *  This causes the temporal constraint mechanism to do a lot of extra work.
 * This rule rectifies the situation by adding the explicit constraint
 *  whenever the interval is achieved.
 *
 * MIGHT INTERACT WITH THE TASK-TREE KILLING CODE
 *
 ***/

/*ARGSUSED*/
static void assertEndOfInterval(const char *ruleName,
				TMS_NODE_PTR relationshipNode)
{
#ifdef applec
#pragma unused(ruleName)
#endif
  if (Less_Or_Equal_Than_Now_Relp(relationshipNode) &&
      Relp_Still_Out(relationshipNode)) {
    QAssert(GET_S_GLOBAL(Now), ">", relation_other_arg(relationshipNode->cell,
						       GET_S_GLOBAL(Now)),
	    "Completed Message");
  }
}

static void addEndOfIntervalRule (interval_ptr interval)
{
  if (QIs_True(GET_S_GLOBAL(Now), "<=", interval->end)) {
    addOutRule(GET_S_GLOBAL(endOfConstraintRule), 
	       relationGetOrMake(GET_S_GLOBAL(Now), interval->end));
  }
}

static void addEndOfConstraints (TASK_TREE_NODE_PTR treeNode)
{
  if (treeNode->dispatch->msg->msgData->msg_class == GoalClass) {
    addEndOfIntervalRule(treeNode->achievementInterval);
    addEndOfIntervalRule(treeNode->planningInterval);
  }
}

static interval_ptr createNodeInterval(TASK_TREE_NODE_PTR node,
				       TCA_INTERVAL_CLASS_TYPE type)
{
  return createInterval(type, node->dispatch);
}

static void add_temporal_intervals_to_node (TASK_TREE_NODE_PTR newTreeNode)
{
  TCA_MSG_CLASS_TYPE nodeClass;
  
  nodeClass = TASK_TREE_NODE_CLASS(newTreeNode);
  
  /* Set up Handling Interval */
  newTreeNode->handlingInterval = createNodeInterval(newTreeNode, 
						     HandlingInterval);
  
  /* Set up Achievement Interval */
  switch (nodeClass) {
  case GoalClass:
  case PollingMonitorClass:
  case DemonMonitorClass:
    newTreeNode->achievementInterval = createNodeInterval(newTreeNode, 
							  AchievementInterval);
    QAssert(newTreeNode->achievementInterval->start, ">=", 
	    newTreeNode->handlingInterval->start, "Task Control");
    QAssert(newTreeNode->achievementInterval->end, ">=", 
	    newTreeNode->handlingInterval->end, "Task Control");
    break;
    
    /******************
     * Need to make the handling and achievement intervals distinct, so that
     * adding an exception handler node under the command node prevents 
     * subsequent messages from firing until the exception is handled.
     ******************/
  case CommandClass:
    newTreeNode->achievementInterval = createNodeInterval(newTreeNode, 
							  AchievementInterval);
    QAssert(newTreeNode->achievementInterval->start, "=",
	    newTreeNode->handlingInterval->start, "Task Control");
    QAssert(newTreeNode->achievementInterval->end, ">=", 
	    newTreeNode->handlingInterval->end, "Task Control");
    break;
    
  case PointMonitorClass:
    newTreeNode->achievementInterval = createNodeInterval(newTreeNode, 
							  AchievementInterval);
    QAssert(newTreeNode->achievementInterval->start, "=", 
	    newTreeNode->handlingInterval->start, "Task Control");
    QAssert(newTreeNode->achievementInterval->end, ">=", 
	    newTreeNode->handlingInterval->end, "Task Control");
    break;
    
  case ExceptionClass:
    newTreeNode->achievementInterval = createNodeInterval(newTreeNode, 
							  AchievementInterval);
    QAssert(newTreeNode->achievementInterval->end, ">=", 
	    newTreeNode->handlingInterval->end, "Task Control");
    break;
  default:
    /* no temporal constraings */
    break;
  }
  
  /* Set up Planning Interval */
  switch (nodeClass) {
  case GoalClass:
  case PollingMonitorClass:
  case DemonMonitorClass:
    newTreeNode->planningInterval = createNodeInterval(newTreeNode, 
						       PlanningInterval);
    QAssert(newTreeNode->planningInterval->start, "=", 
	    newTreeNode->handlingInterval->start, "Task Control");
    QAssert(newTreeNode->planningInterval->end, ">=", 
	    newTreeNode->handlingInterval->end, "Task Control");
    break;
    
  case CommandClass:
    newTreeNode->planningInterval = newTreeNode->handlingInterval;
    break;
    
  case PointMonitorClass:
    newTreeNode->planningInterval = newTreeNode->handlingInterval;
    break;
    
  case ExceptionClass:
    newTreeNode->planningInterval = createNodeInterval(newTreeNode, 
						       PlanningInterval);
    QAssert(newTreeNode->planningInterval->end, ">=", 
	    newTreeNode->handlingInterval->end, "Task Control");
    break;
  default:
    /* no temporal constraings */
    break;
  }
}

/*****************************************************************
 * Set up additional temporal constraints imposed by the user.
 * Constraints can be:
 *   SEQ_ACH        -> End(Achievement(ParentNode->lastChild)) <= 
 *                        Start(Achievement(TreeNode))
 *   SEQ_PLANNING   -> End(Planning(ParentNode->lastChild)) <= 
 *                        Start(Planning(TreeNode))
 *   PLAN_FIRST     -> End(Planning(TreeNode)) <= Start(Achievement(TreeNode))
 *   DELAY_PLANNING -> Start(Planning(TreeNode)) = Start(Achievement(TreeNode))
 *
 * Multiple constraints can be specified by adding the constants together
 *   (e.g. SEQ_ACH+PLAN_FIRST).
 *****************************************************************/

/***********************************************************/

void addTplConstraints(DISPATCH_PTR dispatch, int32 tplConstr)
{ 
  TASK_TREE_NODE_PTR TreeNode, parentNode, lastChild;
  
  TreeNode = dispatch->treeNode;
  parentNode = TreeNode->parent;
  
  if (SEQ_ACH & tplConstr) {
    if (parentNode) {
      lastChild = getLastChild(parentNode, TreeNode->dispatch);
      if (lastChild && !DEAD_NODE(lastChild)) {
	Log_Status("Adding SEQUENTIAL ACHIEVEMENT constraint between %s",
		   TASK_TREE_NODE_MSG_NAME(lastChild));
	Log_RefId(lastChild->dispatch, LOGGING_STATUS);
	Log_Status(" and %s", TASK_TREE_NODE_MSG_NAME(TreeNode));
	Log_RefId(TreeNode->dispatch, LOGGING_STATUS);
	Log_Status("\n");
	QAssert(lastChild->achievementInterval->end, "<=",
		TreeNode->achievementInterval->start, "Temporal Constraint");
      }
    }
  }
  
  if (SEQ_PLANNING & tplConstr) {
    if (parentNode) {
      lastChild = getLastChild(parentNode, TreeNode->dispatch);
      if (lastChild && !DEAD_NODE(lastChild)) {
	Log_Status("Adding SEQUENTIAL PLANNING constraint between %s",
		   TASK_TREE_NODE_MSG_NAME(lastChild));
	Log_RefId(lastChild->dispatch, LOGGING_STATUS);
	Log_Status(" and %s",TASK_TREE_NODE_MSG_NAME(TreeNode));
	Log_RefId(TreeNode->dispatch, LOGGING_STATUS);
	Log_Status("\n");
	QAssert(lastChild->planningInterval->end, "<=", 
		TreeNode->planningInterval->start, "Temporal Constraint");
      }
    }
  }
  
  if (DELAY_PLANNING & tplConstr) {
    Log_Status("Adding DELAY PLANNING constraint to %s",
	       TASK_TREE_NODE_MSG_NAME(TreeNode));
    Log_RefId(dispatch, LOGGING_STATUS);
    Log_Status("\n");
    QAssert(TreeNode->planningInterval->start, "=", 
	    TreeNode->achievementInterval->start, "Temporal Constraint");
  }
  
  if (PLAN_FIRST & tplConstr) {
    Log_Status("Adding PLAN FIRST constraint to %s",
	       TASK_TREE_NODE_MSG_NAME(TreeNode));
    Log_RefId(TreeNode->dispatch, LOGGING_STATUS);
    Log_Status("\n");
    QAssert(TreeNode->planningInterval->end, "<=", 
	    TreeNode->achievementInterval->start, "Temporal Constraint");
  }
}

/***********************************************************/

/* 
   Does the node represent the relationship "Now < X"?
   */
int32 Less_Than_Now_Relp(relationship_node)
     TMS_NODE_PTR relationship_node;
{ 
  return (true_relationship(CV_RELP(relationship_node),
			    (CELL_PTR)relationship_node->cell, GET_S_GLOBAL(Now)) == Less);
}

/* 
   Does the node represent the relationship "Now <= X"?
   */
int32 Less_Or_Equal_Than_Now_Relp(TMS_NODE_PTR relationship_node)
{ 
  return (true_relationship(CV_RELP(relationship_node),
			    (CELL_PTR)relationship_node->cell, GET_S_GLOBAL(Now)) == Less_Or_Equal);
}

int32 Relp_Still_Out(TMS_NODE_PTR relationship_node)
{
  CELL_PTR relation;
  
  relation = (CELL_PTR)relationship_node->cell;
  
  (void)qrelationship1(REL_ARG1(relation), REL_ARG2(relation),
		       CV_RELP(relationship_node));
  
  return IS_OUT(relationship_node);
}

/*ARGSUSED*/
static void Try_To_Activate(const char *rule_name,
			    TMS_NODE_PTR relationship_node)
{ 
#ifdef applec
#pragma unused(rule_name)
#endif
  DISPATCH_PTR dispatch;
  QUANTITY_PTR quantity;
  TCA_TIME_POINT_PTR timePoint;
  
  if (Less_Than_Now_Relp(relationship_node) && 
      Relp_Still_Out(relationship_node)) {
    quantity = relation_other_arg(relationship_node->cell, GET_S_GLOBAL(Now));
    timePoint = (TCA_TIME_POINT_PTR)(quantity->name);
    dispatch = DISPATCH_FROM_ID(timePoint->interval.msgRef);
    processActiveMessage(dispatch);
  }
}

void AssertHoldingConstraint(TASK_TREE_NODE_PTR TreeNode)
{
  QAssert(GET_S_GLOBAL(Now), "<=", TreeNode->handlingInterval->start,
	  "Scheduler");
}

void RetractHoldingConstraint(TASK_TREE_NODE_PTR TreeNode)
{
  QRetract(GET_S_GLOBAL(Now), "<=", TreeNode->handlingInterval->start,
	   "Scheduler");
}

void PendingConstraints(DISPATCH_PTR dispatch)
{
  if (IS_TEMPORAL_DISPATCH(dispatch))
    AssertHoldingConstraint(dispatch->treeNode);
}

void InactiveConstraints(DISPATCH_PTR dispatch)
{ 
  TASK_TREE_NODE_PTR treeNode;
  
  treeNode = dispatch->treeNode;
  
  AssertHoldingConstraint(treeNode);
  addOutRule(GET_S_GLOBAL(activationRule), 
	     relationGetOrMake(GET_S_GLOBAL(Now), treeNode->handlingInterval->start));
}


/******************************************************************************
 *
 * FUNCTION: void Initialize_TaskTreeNode(dispatch, parentId)
 *
 * DESCRIPTION:
 *  Creates a task tree node and sets up the temporal constraints between it
 *  and its parent.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch; 
 * int32 parentId;
 *
 * OUTPUTS: TASK_TREE_NODE_PTR: the newly created task tree node
 *
 * NOTES: 
 * NEED TO CHANGE "removeParentChildRelations" IF THIS FUNCTION EVER CHANGES!!
 *
 *****************************************************************************/

void Initialize_TaskTreeNode(DISPATCH_PTR dispatch, int32 parentId)
{ 
  TASK_TREE_NODE_PTR NewMsgNode, parentNode;
  
  NewMsgNode = taskTreeNodeCreate(dispatch, parentId);
  
  add_temporal_intervals_to_node(NewMsgNode);
  
  /* Set up temporal constraints with parent node */
  parentNode = NewMsgNode->parent;
  if (parentNode && !IS_ROOT_NODE(parentNode)) {
    if (parentNode->achievementInterval != parentNode->handlingInterval) {
      QAssert(parentNode->achievementInterval->start, "<=", 
              NewMsgNode->achievementInterval->start, "Task Control");
      QAssert(parentNode->achievementInterval->end, ">=", 
	      NewMsgNode->achievementInterval->end, "Task Control");
    }
    if ((parentNode->planningInterval != parentNode->handlingInterval) &&
	(NewMsgNode->planningInterval != NewMsgNode->handlingInterval)) {
      QAssert(parentNode->planningInterval->start, "<=", 
	      NewMsgNode->planningInterval->start, "Task Control");
      QAssert(parentNode->planningInterval->end, ">=", 
	      NewMsgNode->planningInterval->end, "Task Control");
    }
  }
}


/******************************************************************************
 *
 * FUNCTION: void removeParentChildRelations(parent, node)
 *
 * DESCRIPTION:
 * Retracts the temporal constraints set up between a parent and a child node.
 *
 * INPUTS: 
 * TASK_TREE_NODE_PTR parent, node;
 *
 * OUTPUTS: void. changes temporal constraints.
 *
 * NOTES:
 * 29-May-91: fedor: should track down what retraction would do if the
 * item was never asserted. should also see what effect this has on memory use.
 *
 *****************************************************************************/

int32 removeParentChildRelations(TASK_TREE_NODE_PTR parentNode,
			       TASK_TREE_NODE_PTR childNode)
{
  if (parentNode && !IS_ROOT_NODE(parentNode)) {
    if (parentNode->achievementInterval != parentNode->handlingInterval) {
      QRetract(parentNode->achievementInterval->start, "<=",
	       childNode->achievementInterval->start, "Task Control");
      QRetract(parentNode->achievementInterval->end, ">=",
	       childNode->achievementInterval->end, "Task Control");
    }
    if ((parentNode->planningInterval != parentNode->handlingInterval) &&
        (childNode->planningInterval != childNode->handlingInterval)) {
      QRetract(parentNode->planningInterval->start, "<=",
	       childNode->planningInterval->start, "Task Control");
      QRetract(parentNode->planningInterval->end, ">=",
	       childNode->planningInterval->end, "Task Control");
    }
  }
  return TRUE;
}

void AttendingConstraints(DISPATCH_PTR dispatch)
{
  TASK_TREE_NODE_PTR treeNode;
  
  if (IS_TEMPORAL_DISPATCH(dispatch)) {
    treeNode = dispatch->treeNode;
    QAssert(GET_S_GLOBAL(Now), "<", treeNode->handlingInterval->end,
	    "Scheduler");
    RetractHoldingConstraint(treeNode);
    
    addEndOfConstraints(treeNode);
    
    QAssert(GET_S_GLOBAL(Now), ">=", treeNode->handlingInterval->start,
	    "Scheduler");
  }
}

void CompletionConstraints(DISPATCH_PTR dispatch)
{
  if (IS_TEMPORAL_DISPATCH(dispatch)) {
    QRetract(GET_S_GLOBAL(Now), "<", dispatch->treeNode->handlingInterval->end,
	     "Scheduler");
    QAssert(GET_S_GLOBAL(Now), ">", dispatch->treeNode->handlingInterval->end, 
	    "Completed Message");
    (void)runRules(-1);
  }
}

/*****************************************************************************
 * Free the interval and its associated start and end points.
 *****************************************************************************/

void FreeInterval(interval_ptr *Interval)
{
  if (! *Interval) return;

  FreeQuantity(&((*Interval)->start));
  FreeQuantity(&((*Interval)->end));
  tcaFree((char *)(*Interval));
  *Interval = NULL;
}


/*
   void print_reason(tms_node) 
   TMS_NODE_PTR tms_node;
   { 
   JUSTIFICATION_PTR just;
   cons_ptr supporters;
   
   just = tms_node->support_justification;
   supporters = just->supporters;
   if (supporters) {
   while (supporters) {
   print_reason((TMS_NODE_PTR)Car(supporters));
   supporters = Cdr(supporters);
   }
   }
   else {
   say_node_value(tms_node);
   tcaModWarning(
   " (%s)\n", Just_Informant(Node_Support_Justification(tms_node)));
   }
   }
   
   void Why_Rel(q1, q2) 
   QUANTITY_PTR q1, q2;
   {   
   Print_Relationship(q1,q2,TRUE);
   print_reason(Current_Value_Supporter(relationGetOrMake(q1,q2)));
   }
   */

static interval_ptr Interval_Of(TCA_INTERVAL_PTR Interval)
{ 
  DISPATCH_PTR dispatch;
  
  if (Interval->interval_class == NoInterval) {
    return NULL;
  } else {
    dispatch = (DISPATCH_PTR)idTableItem(Interval->msgRef,
					 GET_S_GLOBAL(dispatchTable));
    if (!IS_TEMPORAL_DISPATCH(dispatch)) {
      Log("ERROR!!! Interval_Of: No interval associated with dispatch\n");
      return NULL;
    } else {
      switch (Interval->interval_class) {
      case HandlingInterval: 
	return dispatch->treeNode->handlingInterval;
      case PlanningInterval: 
	return dispatch->treeNode->planningInterval;
      case AchievementInterval: 
	return dispatch->treeNode->achievementInterval;
      case NoInterval:  /* should never reach here */
      default:
	return NULL;
      }
    }
  }
  return NULL;
}

QUANTITY_PTR Time_Of(TCA_TIME_POINT_PTR timePoint)
{ 
  interval_ptr interval;
  
  if (timePoint->point_class == NoTime)
    return NULL;
  else {
    interval = Interval_Of(&(timePoint->interval));
    
    if (interval) 
      switch (timePoint->point_class) {
      case StartPoint: 
	return interval->start;
      case EndPoint: 
	return interval->end;
      case NoTime: /* should not reach here */
#ifndef TEST_CASE_COVERAGE
      default:
	return NULL;
#endif
      }
    else
      return NULL;
  }
  return NULL;
}

static void TplConstrainHandler(DISPATCH_PTR dispatch,
				TCA_TPL_CONSTRAINT_PTR TplConstraint)
{
  QUANTITY_PTR q1, q2;
  
  q1 = Time_Of(&(TplConstraint->timePoint1));
  q2 = Time_Of(&(TplConstraint->timePoint2));
  
  if (q1 && q2) {
    Log("   Asserting %s ", qName(q1));
    Log_RefId(qDispatch(q1), LOGGING_ALWAYS);
    Log(" %s %s", TplConstraint->relationship, qName(q2));
    Log_RefId(qDispatch(q2), LOGGING_ALWAYS);
    Log("\n");
    
    if (QIs(q1, TplConstraint->relationship, q2) == Is_False) {
      tcaError("TplConstrainHandler: Inconsistent Relationship Being Asserted\n");
    } else {
      QAssert(q1, TplConstraint->relationship, q2, "Temporal Constraint");
    }
    /* A bit more efficient than using tcaFreeData */
    freeDataStructure(dispatch->msg->msgData->msgFormat, (char *)TplConstraint);
  }
}

void Tpl_Initialize(void)
{
  qlatticeInit();
  
  GET_S_GLOBAL(Now) = createQuantity("Now");
  
  GET_S_GLOBAL(activationRule) = ruleCreateOut("Activate Message", 
					       (NODE_RULE_FN) Try_To_Activate, 
					       3, (char *)NULL);
  GET_S_GLOBAL(activationRule)->used = 1000; /* Never free, Reid: 10-Jul-91 */
  
  GET_S_GLOBAL(endOfConstraintRule) = ruleCreateOut("EndOf Rule", 
						    (NODE_RULE_FN) assertEndOfInterval, 
						    3, (char *)NULL);
  /* Never free, Reid: 10-Jul-91 */
  GET_S_GLOBAL(endOfConstraintRule)->used = 1000;
  
  /* "{Class, msgRef}" */
  centralRegisterNamedFormatter("Interval", "{int, int}"); 
  
  /* "{Class, Interval}" */
  centralRegisterNamedFormatter("TimePoint", "{int, Interval}"); 
  centralRegisterNamedFormatter("TimeList", "*{TimePoint, *!}"); 
  
  centralRegisterInform(TCA_CONTRAINT_INFORM,
			TCA_CONTRAINT_INFORM_FORMAT,
			TplConstrainHandler);
  
  centralRegisterInform(TCA_CONTRAINT_INFORM_OLD,
			TCA_CONTRAINT_INFORM_FORMAT,
			TplConstrainHandler);
}
