/*****************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 *
 * MODULE: tms
 *
 * FILE: tms.c
 *
 * ABSTRACT:
 * This file contains a very simple monotonic TMS with retraction
 * A node is in only iff its support is wellfounded, i.e., contains no cycles.
 *   
 * tmsContraNodeGlobal is the tms_node equated with "false".
 *   
 * tmsAssumpJustificationGlobal is the justification used 
 * to support all assumption nodes.
 *
 * Important external calls made by the TMS (the user must supply these -- 
 * note that defaults are already defined in file "cells").
 *   
 *
 *
 * EXPORTS:
 *     
 *		      USER INTERFACE FUNCTIONS:
 *   
 * tmsInit()
 * Create a new TMS. NOTE: Must be called before using the TMS the FIRST TIME. 
 *   
 * tmsCreateNode(datum)
 * Creates a tms_node (initially OUT) with the appropriate "datum".
 *
 * tmsFreeNode(datum)
 * Frees a tms_node, removing all rules, justifications, and consequents.
 *
 * tmsCreateJustification(informant, supporters)
 * Creates a tms justification.
 * Informant" is a symbol, "supporters" is a list of tms_nodes.
 * Copies both the informant and the supporters (so they can be
 * freed later with impunity).
 *
 * tmsCreateJustification1(informant, supporters)
 * Like Create_Justification, except it doesn't copy the supporters 
 * list (so the list should not be used elsewhere if the justification
 *  might be freed at some later time).
 *   
 * tmsJustifyNode(node justification)
 * Adds a justification that the tms_node is IN if all the supporters of the
 * justification are in.  If there are no supporters then the justification
 * is a premise (or assumption -- but it is better to use Assume_Node for that)
 *
 * tmsAssumeNode(node)
 * Makes node an assumption, forcing it IN.
 *   
 * tmsUnassumeNode(node)
 * Removes node as an assumption (it might still remain IN, 
 * however, if it has other justifications).
 *   
 * tmsAssertNode(node, reason)
 * Makes node a premise, forcing it IN.
 *
 * tmsUnassertNode(node, reason)
 * Removes node as a premise (where the informant of the premise is "reason").
 *
 * tmsSupport(node)
 * Returns the support nodes for the current support of the TMS node.
 *
 * tmsIsPremise(node)
 * Is the node a premise?
 *
 * tmsIsDerived(node)
 * Is the node IN but not a premise or assumption?
 * Returs the supporters of the assumption.
 *
 * IS_IN(node) 
 * Is the node supported for any reason? (returns TRUE or FALSE)
 *   
 * IS_OUT(node)
 * Is there no current support for the node? (returns TRUE or FALSE)
 *   
 * NODE_DATUM(node)
 * Accessor for datum of node.
 *   
 * NODE_JUSTIFICATIONS(node) 
 * Returns a list of justifications which could support belief in NODE.
 *
 * NODE_SUPPORT_JUSTIFICATION(node)
 * Returns current justification which supports belief in NODE.
 *
 * IS_ASSUMED(node)
 * Is the node an assumption?
 *   
 * tmsContradictory(nodes, reason)
 * The set of nodes are mutually contradictory
 * (they support "false").  "Reason" is a symbol.
 *
 * tmsInNode(node, do_resupport)
 * Notifies that the node is going in.
 * "do_resupport" is TRUE if it's just going in under a different context.
 *
 * tmsOutNode(node) 
 * Notifies that the node is going out.
 *
 * REVISION HISTORY:
 *
 * $Log: tms.c,v $
 * Revision 1.13  1995/12/17  20:22:28  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.12  1995/08/05  17:16:26  reids
 * Several important bug fixes:
 *   a) Found a memory leak in the tms (when nodes are unasserted)
 *   b) Fixed a problem with direct connections that would cause TCA to crash
 *      when a command or goal message was sent from an inform or a query.
 *      As part of that fix, all command and goal messages that are sent from
 *      informs or queries are now added to the root node of the task tree.
 *
 * Revision 1.11  1995/07/06  21:17:48  rich
 * Solaris and Linux changes.
 *
 * Revision 1.10  1995/06/14  03:22:58  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.9  1995/01/18  22:43:37  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.8  1994/04/28  16:17:52  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.7  1993/12/14  17:35:38  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.6  1993/11/21  20:20:01  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.5  1993/10/21  16:14:37  rich
 * Fixed compiler warnings.
 *
 * Revision 1.4  1993/08/30  21:55:07  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:17:35  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:51  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:40  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:54  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:26:29  fedor
 * Added Logging.
 *
 * $Revision: 1.13 $
 * $Date: 1995/12/17 20:22:28 $
 * $Author: rich $
 *
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 25-Jun-91  Reid Simmons (reids) at CMU
 * Fixed tmsFreeNode to work correctly.
 *
 *  6-Nov-89  Christopher Fedor at  School of Computer Science, CMU
 * Revised to Software Standards.
 *
 * Ancient    Reid Simmons at School of Computer Science, CMU
 * (c) Copyright 1988 Reid G. Simmons.  All rights reserved.
 * Adapted from LISP version used for the Quantity Lattice
 * (see Simmons, AAAI-86).   
 * Original code taken from Brian Williams
 *
 *****************************************************************************/

#include "globalS.h"
#include "tcaMem.h"

char *String_Copy(const char *string_to_copy)
{ 
  char *copy;
  
  copy = (char *)tcaMalloc((u_int32)(1+strlen(string_to_copy)));
  (void)strcpy(copy, string_to_copy);
  return copy;
}

/***********************************************************************/

static void tmsTrace(char *message, TMS_NODE_PTR node)
{ 
  if (GET_S_GLOBAL(tmsTraceGlobal)) {
    tcaModWarning( "%s Node ", message);
    sayNodeValue(node);	/* found in file "cells" */
    tcaModWarning( "\n");
  }
}

/***********************************************************************
 ***   TMS NODES   ***
 ************************************************************************/


/***********************************************************************
 *
 * FUNCTION: TMS_NODE_PTR tmsCreateNode(datum)
 *
 * DESCRIPTION: Create and return a new tms node for datum.
 *
 * INPUTS: char *datum
 *
 * OUTPUTS: TMS_NODE_PTR - a newly created tms node.
 *
 * DESIGN:
 * Initialize consquents, justifications and rules to an empty list.
 *
 ***********************************************************************/

TMS_NODE_PTR tmsCreateNode(const void *datum)
{ 
  TMS_NODE_PTR newNode;
  
  newNode = NEW(TMS_NODE_TYPE);
  GET_S_GLOBAL(Node_Counter)++;
  
  newNode->datum = datum;
  newNode->consequents = listCreate();
  newNode->justifications = listCreate();
  newNode->rules = listCreate();
  
  newNode->supportJustification = NULL;
  
  newNode->cell = NULL;
  
  return newNode;
}


/***********************************************************************
 *
 * FUNCTION: void freeJustification(justification) 
 *
 * DESCRIPTION: If the justification is not the tmsAssumpJustificationGlobal
 * free the justification and all of its supporters.
 *
 * INPUTS: JUSTIFICATION_PTR justification;
 *
 * OUTPUTS: none.
 *
 ***********************************************************************/

void freeJustification(JUSTIFICATION_PTR *justification)
{
  if ((*justification) == NULL) return;

  if ((*justification) != GET_S_GLOBAL(tmsAssumpJustificationGlobal)) {
    tcaFree((char *)(*justification)->informant);
    listFree(&((*justification)->supporters));
    tcaFree((char *)(*justification));
    *justification = NULL;
  }
}


/*
 * This could be dangerous.  
 * Should only be used if the nodes it is supporting and
 * the nodes it is supported by are also being freed. 
 * SHOULD BE USED WITH CARE!
 */
static int32 sameConsequent (CONSEQUENT_PTR consequent1,
			   CONSEQUENT_PTR consequent2)
{ 
  return ((consequent1->node == consequent2->node) &&
	  (consequent1->justification == consequent2->justification));
}

/* defined below */
static void tryOutingNodeAndItsConsequents(TMS_NODE_PTR node,
					   JUSTIFICATION_PTR just);

void tmsFreeNode(TMS_NODE_PTR *node)
{
  JUSTIFICATION_PTR justifier;
  CONSEQUENT_PTR consequent;
  CONSEQUENT_TYPE consequent1;
  TMS_NODE_PTR supporter;
  
  if ((*node) == NULL) return;

  rulesFree((*node)->rules);
  
  justifier = (JUSTIFICATION_PTR)listFirst((*node)->justifications);
  while (justifier) {
    supporter = (TMS_NODE_PTR)listFirst(justifier->supporters);
    if (supporter) {
      consequent1.node = (*node);
      consequent1.justification = justifier;
      consequent = (CONSEQUENT_PTR)
	listMemReturnItem((LIST_ITER_FN) sameConsequent, 
			  (char *)&consequent1,
			  supporter->consequents);
      while (supporter) {
	listDeleteItem((char *)consequent, supporter->consequents);
	supporter = (TMS_NODE_PTR)listNext(justifier->supporters);
      }
      tcaFree((char *)consequent);
    }
    freeJustification(&justifier);
    justifier = (JUSTIFICATION_PTR)listNext((*node)->justifications);
  }
  
  consequent = (CONSEQUENT_PTR)listFirst((*node)->consequents);
  while (consequent) {
    listDeleteItem((char *)consequent->justification, 
		   consequent->node->justifications);
    supporter = (TMS_NODE_PTR)listFirst(consequent->justification->supporters);
    while (supporter) {
      if (supporter != (*node)) {
	listDeleteItem((char *)consequent, supporter->consequents);
      }
      supporter = (TMS_NODE_PTR)
	listNext(consequent->justification->supporters);
    }
    if (consequent->node->supportJustification == consequent->justification) {
      tryOutingNodeAndItsConsequents(consequent->node,
				     consequent->justification);
    }
    freeJustification(&(consequent->justification));
    tcaFree((char *)consequent);
    consequent = (CONSEQUENT_PTR)listNext((*node)->consequents);
  }
  listFree(&((*node)->justifications));
  listFree(&((*node)->consequents));
  tcaFree((char *)(*node));
  *node = NULL;
}


/***********************************************************************
 * 
 * FUNCTION: CONSEQUENT_PTR createConsequent(node, justification)
 *
 * DESCRIPTION: Create and initialize a consequent.
 *
 * INPUTS: 
 * TMS_NODE_PTR node;
 * JUSTIFICATION_PTR justification;
 *
 * OUTPUTS: CONSEQUENT_PTR - the newly created consequent
 *
 * DESIGN:
 *
 ***********************************************************************/

static CONSEQUENT_PTR createConsequent(TMS_NODE_PTR node,
				       JUSTIFICATION_PTR justification)
{ 
  CONSEQUENT_PTR newConsequent;
  
  newConsequent = NEW(CONSEQUENT_TYPE);
  newConsequent->node = node;
  newConsequent->justification = justification;
  
  return newConsequent;
}


/**********************************************************************
 ***   JUSTIFICATIONS ***
 ***********************************************************************/


/***********************************************************************
 *
 * FUNCTION: JUSTIFICATION_PTR tmsCreateJustification1(informant, supporters)
 *
 * DESCRIPTION: Create a justification.
 *
 * INPUTS:
 * char *informant; 
 * LIST_PTR supporters;
 *
 * OUTPUTS: JUSTIFICATION_PTR - the newly created justification.
 *
 * NOTES:
 *
 * There are three types of justifications: 
 *
 * Assumptions:  = {"ASSUMPTION", NULL}
 *
 * Premises: = {informant, NULL} 
 * --  where informant is a string, but not "ASSUMPTION"
 *
 * Derived = {informant, antecedent-nodes}
 ***********************************************************************/

JUSTIFICATION_PTR tmsCreateJustification1(const char *informant,
					  LIST_PTR supporters)
{ 
  JUSTIFICATION_PTR newJustification;
  
  newJustification = NEW(JUSTIFICATION_TYPE);
  newJustification->informant = String_Copy(informant);
  newJustification->supporters = supporters;
  
  return newJustification;
}


/***********************************************************************
 *
 * FUNCTION: JUSTIFICATION_PTR tmsCreateJustification(informant, supporters)
 *
 * DESCRIPTION: 
 * Calls tmsCreateJustification1 to create a justification with a copy of
 * the supporters list.
 *
 ***********************************************************************/

JUSTIFICATION_PTR tmsCreateJustification(const char *informant,
					 LIST_PTR supporters)
{ 
  return tmsCreateJustification1(informant, listCopy(supporters));
}


/***********************************************************************
 *
 * FUNCTION: LIST_PTR tmsSupport(node)
 *
 * DESCRIPTION: Return the support justifications list for this node 
 * NULL is returned if there are none.
 *
 * INPUTS: TMS_NODE_PTR node.
 *
 * OUTPUTS: LIST_PTR
 *
 ***********************************************************************/

LIST_PTR tmsSupport(TMS_NODE_PTR node)
{ 
  JUSTIFICATION_PTR justification;
  
  justification = node->supportJustification;
  return (justification) ? justification->supporters : NULL;
}


/***********************************************************************
 * 
 * FUNCTION: int32 isPremiseJust(just)
 *
 * DESCRIPTION:
 * Returns TRUE if the justification is a premise (i.e no antecedant nodes, 
 * but not tmsAssumpJustificationGlobal. 
 *
 ***********************************************************************/

static int32 isPremiseJust(JUSTIFICATION_PTR just)
{ 
  return (!listLength(just->supporters) && !IS_ASSUMPTION_JUST(just));
}


/* 
 * Returns TRUE if the node is currently supported by a premise justification
 */
int32 tmsIsPremise(TMS_NODE_PTR node)
{ 
  return ((NODE_SUPPORT_JUSTIFICATION(node) != NULL) &&
	  isPremiseJust(NODE_SUPPORT_JUSTIFICATION(node)));
}

/*
 * Returns TRUE if the node is supported by a non-premise, 
 * non-assumption justification. 
 */
/* Currently not used (RGS: 11/11/92)
   int32 tmsIsDerived(node) 
   TMS_NODE_PTR node;
   {
   return (node && node->supportJustification &&
   !listLength(node->supportJustification->supporters));
   }
   */

/***********************************************************************
 ***   ADDING JUSTIFICATIONS   ***
 ***********************************************************************/

/*
 * Justify "node" using the assumption justification.  
 * "node" becomes an assumption.
 */
void tmsAssumeNode(TMS_NODE_PTR node)
{ 
  tmsJustifyNode(node, GET_S_GLOBAL(tmsAssumpJustificationGlobal)); 
}

/*
 * Create a new node holding DATUM and justify it as an assumption 
 */
/* Currently not used (RGS: 11/11/92)
   TMS_NODE_PTR assume_datum(datum)
   char *datum;
   { 
   TMS_NODE_PTR new_node;
   
   new_node = tmsCreateNode(datum);
   tmsAssumeNode(new_node);
   return new_node;
   }
   */

/*
 * tmsAssertNode(node, reason) -- Makes node a premise, forcing it IN.
 */
void tmsAssertNode(TMS_NODE_PTR node, const char *reason)
{ 
  tmsJustifyNode(node, tmsCreateJustification(reason, (LIST_PTR)NULL));
}

/*
 * Use "justification" as the current support for "node" and call the 
 * user code "in-node" (the default is in the file "cells").
 * "do_resupport" is TRUE if this is a node which was in, but now needs 
 * new support due to a retraction.
 *
 * Returns the justification.
 */
static void makeInAndNotify(TMS_NODE_PTR node, JUSTIFICATION_PTR justification,
			    int32 doResupport)
{
  node->supportJustification = justification;
  tmsInNode(node, doResupport);
}


/***********************************************************************
 *
 * FUNCTION: int32 trySupportingNodeWithJustification(node, just, doResupport)
 *
 * DESCRIPTION:
 * If all of "just"'s supporters are IN, use "just" as "node"'s current
 * support. Returns TRUE if able to resupport the node.
 *
 * INPUTS:
 * TMS_NODE_PTR node; 
 * JUSTIFICATION_PTR just; 
 * int32 doResupport;
 *
 * OUTPUTS: Returns TRUE if able to resupport the node.
 *
 ***********************************************************************/

/*ARGSUSED*/
static int32 testIn(void *param, TMS_NODE_PTR node)
{
#ifdef applec
#pragma unused(param)
#endif
  return(IS_IN(node));
}

static int32 trySupportingNodeWithJustification(TMS_NODE_PTR node, 
					      JUSTIFICATION_PTR just, 
					      int32 doResupport)
{ 
  int32 justIsIn;
  
  justIsIn = listIterate((LIST_ITER_FN)testIn, (char *)NULL, just->supporters);
  
  if (justIsIn)
    makeInAndNotify(node, just, doResupport); 
  
  return justIsIn;
}


/***********************************************************************
 *
 * FUNCTION: void trySupportingConsequents(firstNode, doResupport)
 *
 * DESCRITPION:
 * Propagate effects of a justification forward through the network of
 * justifications.
 *
 * INPUTS:
 * TMS_NODE_PTR firstNode; 
 * int32 doResupport;
 *
 * OUTPUTS: none.
 *
 * DESIGN: Iterate through the list of consequents, any consequent node 
 * that was OUT and successfully supported with justifications are aded to 
 * a queue to be further expanded. This continues until the queue is empty.
 *
 ***********************************************************************/

static int32 propagateJustification(int32 *doResupport, CONSEQUENT_PTR consequent)
{
  TMS_NODE_PTR cnode;
  
  cnode = consequent->node;
  if (IS_OUT(cnode) && 
      trySupportingNodeWithJustification(cnode, consequent->justification, 
					 *doResupport)) {
    Enqueue((char *)cnode, GET_S_GLOBAL(In_Queue));
    tmsTrace("     And allows support for", cnode);
  }
  
  return TRUE;
}

static void trySupportingConsequents(TMS_NODE_PTR firstNode, int32 doResupport)
{ 
  while (firstNode != NULL) {
    (void)listIterate((LIST_ITER_FN)propagateJustification, 
		      (char *)&doResupport,
		      firstNode->consequents);
    firstNode = (TMS_NODE_PTR)Dequeue(GET_S_GLOBAL(In_Queue));
  }
}


/***********************************************************************
 *
 * FUNCTION: void justifyNodeInternal(node, just)
 *
 * DESCRIPTION: Add "justification" to list of potential supporters of "node".
 *
 * INPUTS:
 * TMS_NODE_PTR node; 
 * JUSTIFICATION_PTR just; 
 *
 * OUTPUTS: none.
 *
 ***********************************************************************/

static int32 insertConsequent(void *item, TMS_NODE_PTR node)
{
  listInsertItem((char *)item, node->consequents);
  return TRUE;
}

static void justifyNodeInternal(TMS_NODE_PTR node, JUSTIFICATION_PTR just)
{ 
  LIST_PTR antecedents;
  CONSEQUENT_PTR consequent;
  
  antecedents = just->supporters;
  
  listInsertItem((char *)just, node->justifications);
  if (listLength(antecedents) != 0) {
    consequent = createConsequent(node, just);
    (void)listIterate((LIST_ITER_FN)insertConsequent, 
		      (char *)consequent, antecedents);
  }
  
  if (IS_IN(node)) {
    /* making node an assumption or premise, it is currently neither */
    if (!(listLength(antecedents) || tmsIsPremise(node)))
      makeInAndNotify(node, just, TRUE);
  }
  else if (trySupportingNodeWithJustification(node, just, FALSE)) {
    tmsTrace("    New justification supports", node);
    trySupportingConsequents(node, FALSE);
  }
}


/***************************************************************************
 *
 * FUNCTION: int32 sameJustification(just1, just2)
 *
 * DESCRIPTION: Are "just1" and "just2" the same justifications.
 *
 * INPUTS: JUSTIFICATION_PTR just1, just2;
 *
 * OUTPUTS: int32 TRUE or FALSE
 *
 ***************************************************************************/

static int32 sameJustification(JUSTIFICATION_PTR just1, JUSTIFICATION_PTR just2)
{ 
  int32  samep;
  
  if (just1 == just2) 
    return TRUE;
  else if (!just1 || !just2) 
    return FALSE;
  else {
    samep = STREQ(just1->informant, just2->informant);
    if (samep)
      samep = listEqual(just1->supporters, just2->supporters);
    return samep;
  }
}


/***************************************************************************
 * 
 * FUNCTION: void tmsJustifyNode(node, just)
 *
 * DESCRIPTION:
 * Add "justification" to list of potential supporters of "node" if
 * appropriate.
 *
 * INPUTS:
 * TMS_NODE_PTR node;
 * JUSTIFICATION_PTR just;
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:
 *
 * DESIGN
 *
 * Basic problem solving operation is to justify one node in terms
 * of others.  It does not return a useful value.
 *
 * Loops are okay however a node will only be IN if it has a
 * well-founded support (i.e., Has a support chain that doesn't depend on
 * itself).  Note that the same consequent node may be re-examined many
 * times.
 *
 ***************************************************************************/

void tmsJustifyNode(TMS_NODE_PTR node, JUSTIFICATION_PTR just)
{
  if (just) {
    if (listMemReturnItem((LIST_ITER_FN) sameJustification,
			  (char *)just, node->justifications)) {
      tmsTrace("Ignoring useless duplicate justification for", node);
      if (just != GET_S_GLOBAL(tmsAssumpJustificationGlobal))
	freeJustification(&just);
    } else {
      justifyNodeInternal(node, just);
    }
  }
}


/**********************************************************************
 ***   REMOVING JUSTIFICATIONS   ***
 ***********************************************************************/

/***************************************************************************
 *
 * FUNCTION:
 * JUSTIFICATION_PTR tryResupportingNodeByPremiseOrAssumption(node)
 *
 * DESCRIPTION:
 * Try resupporting node with another premise or an assumption justification.
 * A premise justification is preferred over an assumption.
 *
 * INPUT: TMS_NODE_PTR node
 *
 * OUTPUTS:
 * Returns the new justification if rejustified, else NULL
 *
 * EXCEPTIONS: 
 *
 * DESIGN:
 * listMemReturnItem looks through the list for a justification that is a
 * premise and returns it. As a side effect the func passed to 
 * listMemReturnItem sets hasAssumptionJust if an item in the list
 * passes the is_assumption_just test.
 *
 * NOTES:
 *
 ***************************************************************************/

/*ARGSUSED*/
static int32 isAssumpOrPremiseTest(void *param, JUSTIFICATION_PTR just)
{
#ifdef applec
#pragma unused(param)
#endif
  if (IS_ASSUMPTION_JUST(just)) {
    GET_S_GLOBAL(hasAssumptionJust) = TRUE;
    return FALSE;
  }
  else if (isPremiseJust(just))
    return TRUE;
  else
    return FALSE;
}

static JUSTIFICATION_PTR
tryResupportingNodeByPremiseOrAssumption(TMS_NODE_PTR node)
{ 
  JUSTIFICATION_PTR just;
  
  GET_S_GLOBAL(hasAssumptionJust) = FALSE;
  
  just = (JUSTIFICATION_PTR)
    listMemReturnItem((LIST_ITER_FN) isAssumpOrPremiseTest,
		      (char *)NULL, node->justifications);
  
  if (just) {
    makeInAndNotify(node, just, TRUE);
    return just;
  }
  
  if (GET_S_GLOBAL(hasAssumptionJust)) {
    makeInAndNotify(node, GET_S_GLOBAL(tmsAssumpJustificationGlobal), TRUE);
    return GET_S_GLOBAL(tmsAssumpJustificationGlobal);
  }
  else 
    return NULL;
}


/***************************************************************************
 *
 * FUNCTION: void outNodeAndItsConsequents(onode, justification)
 *
 * INPUTS:
 * TMS_NODE_PTR onode; 
 * JUSTIFICATION_PTR justification;
 *
 * OUTPUTS: none.
 *
 * DESCRIPTION:
 * Make node and its consequents OUT, if the node is supported by
 * justification.
 * Justification is the justification just removed.
 *
 * DESIGN:
 *
 * NOTES:
 *
 ***************************************************************************/

/*ARGSUSED*/
static int32 stackSameJust(void *param, CONSEQUENT_PTR consequent)
{
#ifdef applec
#pragma unused(param)
#endif
  TMS_NODE_PTR cnode;
  JUSTIFICATION_PTR cjust;
  
  cnode = consequent->node;
  cjust = consequent->justification;
  if (sameJustification(cnode->supportJustification, cjust))
    Push_Stack((char *)cnode, GET_S_GLOBAL(Out_Stack));
  
  return TRUE;
}

static void outNodeAndItsConsequents(TMS_NODE_PTR onode, 
				     JUSTIFICATION_PTR justification)
{ 
  TMS_NODE_PTR node;
  
  if (sameJustification(onode->supportJustification, justification)) {
    node = onode;
    while (node) {
      if (IS_IN(node)) {
	/**** Makes a node OUT (not known to be true) ***/
	node->supportJustification = NULL;
	tmsTrace("   losing support of", node);
	Remember((char *)node, GET_S_GLOBAL(Temp_Out_Memory));
	
	(void)listIterate((LIST_ITER_FN)stackSameJust, 
			  (char *)NULL, node->consequents);
      }
      node = (TMS_NODE_PTR)Pop_Stack(GET_S_GLOBAL(Out_Stack));
    }
  }
}


/***************************************************************************
 *
 * FUNCTION: void tryResupportingOuttedNode(node) 
 *
 * DESCRIPTION:
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 ***************************************************************************/

static int32 supportNodeWithJust(TMS_NODE_PTR node, JUSTIFICATION_PTR just)
{
  if (trySupportingNodeWithJustification(node, just, TRUE)) {
    tmsTrace("   Found new support trying consequents for", node);
    trySupportingConsequents(node, TRUE);
  }
  
  return TRUE;
}

static void tryResupportingOuttedNode(TMS_NODE_PTR node)
{ 
  if (IS_OUT(node)) {
    (void)listIterate((LIST_ITER_FN)supportNodeWithJust, 
		      (char *)node, node->justifications);
  }
}


/***************************************************************************
 *
 * FUNCTION: void tryResupportingOuttedNodes(memoryOfOutNodes)
 *
 * DESCRIPTION:
 * Look around for alternative justification for nodes made OUT, resupport 
 * if possible
 *
 * Note: we don't need to run "In_Node" since any node going in is one that 
 * was IN at the beginning of truth maintenance but momentarily became
 * unsupported.
 *
 * INPUTS: memory_ptr memoryOfOutNodes;
 *
 * OUTPUTS: none.
 *
 ***************************************************************************/

static void tryResupportingOuttedNodes(memory_ptr memoryOfOutNodes)
{
  Do_Memory(memoryOfOutNodes, (DO_MEMORY_FN) tryResupportingOuttedNode); 
}


/***************************************************************************
 *
 * FUNCTION: void trulyOutNodeFn(node) 
 *
 * DESCRIPTION:
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 ***************************************************************************/

static void trulyOutNodeFn(TMS_NODE_PTR node)
{
  if (IS_OUT(node)) {
    tmsTrace("  Staying out:", node);
    tmsOutNode(node);
  }
}


/***************************************************************************
 * 
 * FUNCTION: void check_truly_out_nodes(memoryOfOutNodes)
 *
 * DESCRIPTION: Run OUT_NODE (interface to problem solver) for each OUT node.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 ***************************************************************************/

static void check_truly_out_nodes(memory_ptr memoryOfOutNodes)
{ 
  Do_Memory(memoryOfOutNodes, (DO_MEMORY_FN) trulyOutNodeFn);
}


/***************************************************************************
 *
 * FUNCTION: void tryOutingNodeAndItsConsequents(node, just)
 *
 * DESCRIPTION:
 * 1st, make OUT node and any consequents whose support justification depends 
 * on node.  Then for every node that went out try to find another support 
 * justification  for it.  Given that there are no loops in support 
 * justifications. This rejustification process can be performed in the same 
 * order that the nodes were made out.
 *
 * INPUTS:
 * TMS_NODE_PTR node; 
 * JUSTIFICATION_PTR just;
 *
 * OUTPUTS: none.
 *
 ***************************************************************************/

static void tryOutingNodeAndItsConsequents(TMS_NODE_PTR node, 
					   JUSTIFICATION_PTR just)
{ 
  JUSTIFICATION_PTR supp_just;
  
  supp_just = node->supportJustification;
  if (!((just == supp_just) &&
	(IS_ASSUMPTION_JUST(just) || isPremiseJust(just)) &&
	tryResupportingNodeByPremiseOrAssumption(node))) {
    tmsTrace("  Removing support of node and consequents -", node);
    outNodeAndItsConsequents(node, just);
    tryResupportingOuttedNodes(GET_S_GLOBAL(Temp_Out_Memory));
    check_truly_out_nodes(GET_S_GLOBAL(Temp_Out_Memory));
    Clear_Memory(GET_S_GLOBAL(Temp_Out_Memory));
  }
}


/***************************************************************************
 *
 * FUNCTION: void tmsUnassumeNode(node) 
 *
 * DESCRIPTION:
 * Remove the assumption justification from this node, and see if it goes OUT.
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 ***************************************************************************/

void tmsUnassumeNode(TMS_NODE_PTR node)
{ 
  if (listMemberItem((char *)GET_S_GLOBAL(tmsAssumpJustificationGlobal),
		     node->justifications)) {
    listDeleteItem((char *)GET_S_GLOBAL(tmsAssumpJustificationGlobal),
		   node->justifications);
    tryOutingNodeAndItsConsequents(node,
				   GET_S_GLOBAL(tmsAssumpJustificationGlobal));
  } else {
    tcaModWarning( "Can't unassume node %s since it is not an assumption\n", 
		  node->datum);
  }
}

/***************************************************************************
 *
 * FUNCTION: void tmsUnassertNode(node, reason)
 *
 * DESCRIPTION:
 * Remove any PREMISE assertions from this node, and see if it goes OUT 
 *
 * INPUTS: 
 * TMS_NODE_PTR node;
 * char *reason;
 *
 * OUTPUTS: none.
 *
 ***************************************************************************/

int32 sameReason(const char *reason, JUSTIFICATION_PTR just)
{ 
  return STREQ(reason, just->informant);
}

void tmsUnassertNode(TMS_NODE_PTR node, const char *reason)
{ 
  JUSTIFICATION_PTR premiseJustifier;
  
  premiseJustifier = 
    (JUSTIFICATION_PTR)listMemReturnItem((LIST_ITER_FN)sameReason,
					 (char *)reason, node->justifications);
  if (premiseJustifier) {
    listDeleteItem((char *)premiseJustifier, node->justifications);
    tryOutingNodeAndItsConsequents(node, premiseJustifier);
    /* Found the memory leak! RGS, 7/26/95 */
    freeJustification(&premiseJustifier);
  } else {
    tcaModWarning( "Can't unassert node %s since it is not a premise\n", 
		  node->datum);
  }
}

/**********************************************************************
 ***   Contradictions   ***
 ***********************************************************************/

/*** Assert that the TMS is in a contradictory state because of JUST ***/
static void contradiction(JUSTIFICATION_PTR just)
{ 
  tmsJustifyNode(GET_S_GLOBAL(tmsContraNodeGlobal), just); 
}

/*
 * Assert that the TMS is contradictory because of "reason" for nodes "nodes" 
 *
 * NOTE: Uses "tmsCreateJustification1" which does *not* copy the list of 
 *       supporters.  Therefore, the supporters cannot be used elsewhere 
 *       otherwise memory management will screw up.  Currently, all users of
 *       tmsContradictory make a new list.
 */
void tmsContradictory(LIST_PTR nodes_list, const char *reason)
{ 
  contradiction(tmsCreateJustification1(reason, nodes_list));
}

/*** Assert that the TMS is contradictory because of "reason" for "node" ***/
/* Currently not used (RGS: 11/11/92)
   static void contradictory_node(TMS_NODE_PTR node, char *reason)
   { 
   tmsContradictory(listMake1((char *)node), reason);
   }
   */

/**********************************************************************
 ***   Initializing the TMS   ***
 **********************************************************************/

void tmsInit()
{
  GET_S_GLOBAL(In_Queue) = Create_Queue();
  GET_S_GLOBAL(Out_Stack) = Create_Stack();
  GET_S_GLOBAL(Temp_Out_Memory) = Create_Memory();
  
  GET_S_GLOBAL(Node_Counter) = 0;
  GET_S_GLOBAL(tmsContraNodeGlobal) = tmsCreateNode((char *)"A contradiction");
  GET_S_GLOBAL(tmsAssumpJustificationGlobal) = 
    tmsCreateJustification("Assumption", 
			   (LIST_PTR)NULL);
}

void tmsPrintTruth(TMS_NODE_PTR node)
{ 
  tcaModWarning( "Node %s is %s\n",
		node->datum, (IS_OUT(node) ? "out" : "in"));
}
