/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * MODULE: tcaRef
 *
 * FILE: tcaRef.c
 *
 * ABSTRACT:
 * 
 * Implements TCA_REF_PTR
 *
 * REVISION HISTORY
 *
 * $Log: tcaRef.c,v $
 * Revision 1.26  1996/06/25  21:27:47  rich
 * Forgot brace for lisp version.
 *
 * Revision 1.25  1996/06/25  20:51:48  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.24  1996/05/09  18:31:43  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.23  1996/03/05  05:05:02  reids
 * Changes (mainly delineated by NMP_IPC conditionals) to support the
 *   New Millennium IPC.
 *
 * Revision 1.22  1996/03/02  03:22:06  rich
 * Fixed memory leaks found using purify.
 *
 * Revision 1.21  1996/02/10  16:50:47  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.20  1996/02/06  19:05:18  rich
 * Changes for VXWORKS pipes.  Note: the read and write sockets descriptors
 * can be different.
 *
 * Revision 1.19  1995/11/03  03:04:54  rich
 * Changed msgFind to keep if from going into an infinite loop if there is no
 * central connection.  This only happens when an exit procedure that does
 * not exit is registered.  msgFind can now return NULL, so I added some
 * checks for the return value to keep modules from seg-faulting.
 *
 * Revision 1.18  1995/10/29  18:27:20  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.17  1995/10/25  22:49:09  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.16  1995/06/14  03:22:54  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.15  1995/04/19  14:29:12  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.14  1995/04/08  02:06:41  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.13  1995/01/18  22:43:28  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.12  1994/10/25  17:11:06  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.11  1994/05/17  23:18:20  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.10  1994/05/11  01:57:42  rich
 * Now set an invalid tcaServerGlobal (a socket fd) to -1 rather than 0
 * which is stdout.
 * Added checks to make sure tcaServerGlobal is a valid socket before
 * sending messages or waiting for messages.
 *
 * Revision 1.9  1994/05/05  00:46:32  rich
 * Added a gmake makefile GNUmakefile so that the system can be easily
 * compiled on different machines.
 * Can now create the targets: tarfile and ftp for creating versions for
 * export.
 *
 * Fixed a number of places were tcaExitHnd was not expected to return.
 * Set the tcaSeverGlobal to 0 when the socket is closed.
 *
 * Revision 1.8  1994/04/28  16:17:47  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.7  1994/04/16  19:43:33  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.6  1993/12/14  17:35:35  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.5  1993/11/21  20:19:58  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.4  1993/08/30  21:55:01  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:17:27  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:48  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:40  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:50  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:26:24  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 14-Dec-91 Christopher Fedor, School of Computer Science, CMU
 * Addded Rich Goodwin's tcaReferenceStatus.
 *
 * 14-Mar-91 Christopher Fedor, School of Computer Science, CMU
 * Added tcaReferenceId for lisp interface to tpl constraints.
 *
 * 13-Aug-90 Christopher Fedor, School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.26 $
 * $Date: 1996/06/25 21:27:47 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"
#if defined(LISP)
/******************************************************************************
 * Forward Declarations
 *****************************************************************************/
int32 tcaRefDataLisp(TCA_REF_PTR ref);

#endif

void tcaRefInitialize(void)
{
  GET_C_GLOBAL(tcaRefFreeList) = listCreate();
}

/* 26-Jul-91: Reid: There was a memory leak here -- sometimes the name is
   newly malloc'd (see primFmtters) and sometimes an old copy is passed in.
   In either case, the name is not freed.  Solution is to always copy the
   string name in tcaRefCreate and free it in tcaRefFree
   */

TCA_REF_PTR tcaRefCreate(MSG_PTR msg, const char *name, int32 refId)
{
  TCA_REF_PTR tcaRef;
  int32 length;
  
  if (!listLength(GET_C_GLOBAL(tcaRefFreeList)))
    tcaRef = NEW(TCA_REF_TYPE);
  else 
    tcaRef = (TCA_REF_PTR)listPopItem(GET_C_GLOBAL(tcaRefFreeList));
  
  tcaRef->refId = refId;    
  tcaRef->msg = msg;
  
  if (!name) {
    tcaRef->name = NULL;
  } else {
    char *namePtr;
    length = strlen(name);
    namePtr = tcaMalloc((unsigned)(length+1));
    BCOPY(name, namePtr, length);
    namePtr[length] = '\0';
    tcaRef->name = namePtr;
  }
#ifdef NMP_IPC
  tcaRef->responded = FALSE;
#endif
  
  return tcaRef;
}

void tcaRefFree(TCA_REF_PTR tcaRef)
{
  if (tcaRef) {
    tcaRef->refId = 0;
    tcaRef->msg = NULL;
    
    tcaFree((char *)tcaRef->name);
    tcaRef->name = NULL;
    
    listInsertItemFirst((char *)tcaRef, GET_C_GLOBAL(tcaRefFreeList));
  }
}

/******************************************************************************
 *
 * FUNCTION: void tcaReferenceRelease(ref)
 *
 * DESCRIPTION: 
 * Frees the reference and 
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: none
 *
 *****************************************************************************/
void tcaReferenceRelease(TCA_REF_PTR ref)
{
  if (ref) {
    (void)tcaInform(TCA_REF_RELEASE_INFORM, (void *)&(ref->refId));
    
    tcaRefFree(ref);
  }
}

/******************************************************************************
 *
 * FUNCTION: char *tcaReferenceName(ref)
 *
 * DESCRIPTION: 
 * Returns the name of the message associated with the reference.
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: char *
 *
 *****************************************************************************/

const char *tcaReferenceName(TCA_REF_PTR ref)
{
  if (ref->name == NULL) {
    tcaModError("ERROR: tcaReferenceName: invalid tca reference pointer.\n");
    return NULL;
  } else {
    return ref->name;
  }
}


/******************************************************************************
 *
 * FUNCTION: int32 tcaReferenceId(ref)
 *
 * DESCRIPTION: 
 * Done primarily for lisp.
 * Returns the msg id of the message associated with the reference.
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

int tcaReferenceId(TCA_REF_PTR ref)
{
  if (ref)
    return ref->refId;
  else
    return NO_REF;
}


/******************************************************************************
 *
 * FUNCTION: void *tcaReferenceData(ref)
 *
 * DESCRIPTION: 
 * Returns a pointer to the data that was sent when the message
 * that is associated with the reference was sent.
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: Pointer to data.
 *
 *****************************************************************************/

void *tcaReferenceData(TCA_REF_PTR ref)
{
  int32 refId;
  MSG_PTR msg;
  char *msgData;
  TCA_REF_PTR waitRef;
  TCA_RETURN_VALUE_TYPE returnValue;
  
  msg = msgFind(TCA_REF_DATA_QUERY);
  if (msg == NULL) return NULL;
  
  if (!ref->msg) {
    if (!ref->name) {
      /* 17-Jun-91: fedor: start enforcing correct refs */
      tcaModError("ERROR: tcaReferenceData: Badly Formed Reference: %d\n",
		  ref->refId);
      return NULL;
    }
    ref->msg = msgFind(ref->name);
    if (ref->msg == NULL) return NULL;
  }
  
  /* 17-Jun-91: fedor: check if any message form */
  if (!ref->msg->msgData->msgFormat)
    return NULL;
  
  refId = nextSendMessageRef();
  returnValue = sendMessage((TCA_REF_PTR)NULL, msg, 
			    (char *)&ref->refId, (char *)NULL);
  
  if (returnValue != Success) {
    tcaModError("ERROR: tcaReferenceData: sendMessage Failed.\n");
    return NULL;
  }
  
  waitRef = tcaRefCreate(ref->msg, ref->name, refId);
  
  msgData = tcaMalloc((unsigned)dataStructureSize(ref->msg->msgData->msgFormat));
  
  returnValue = waitForReplyFrom(waitRef, msgData, TRUE, WAITFOREVER,
				 GET_C_GLOBAL(tcaServerReadGlobal));
  
  tcaRefFree(waitRef);
  
  if (returnValue == NullReply) {
    /* 17-Jun-91: fedor: if NullReply then nothing else was malloced. */
#if defined(THINK_C) || defined(applec)
    free(msgData);
#else
    /*    cfree(msgData);*/
    free(msgData);
#endif /* THINK_C || applec */
    return NULL;
  }
  else
    return msgData;
}

#if defined(LISP)
/* 25-Jun-91: fedor: this is really a cheap shot at making ref data work
   for lisp - lets hope it works. */
int32 tcaRefDataLisp(TCA_REF_PTR ref)
{
  int32 refId;
  MSG_PTR msg;
  TCA_REF_PTR waitRef;
  TCA_RETURN_VALUE_TYPE returnValue;
  
  msg = msgFind(TCA_REF_DATA_QUERY);
  if (msg == NULL) return 0;
  
  if (!ref->msg) {
    if (!ref->name) {
      /* 17-Jun-91: fedor: start enforcing correct refs */
      tcaModError("ERROR: tcaReferenceData: Badly Formed Reference: %d\n",
		  ref->refId);
      return 0;
    }
    ref->msg = msgFind(ref->name);
    if (ref->msg == NULL) return 0;
  }
  
  /* 17-Jun-91: fedor: check if any message form */
  if (!ref->msg->msgData->msgFormat)
    return 0;
  
  refId = nextSendMessageRef();
  returnValue = sendMessage((TCA_REF_PTR)NULL, msg, 
			    (char *)&ref->refId, (char *)NULL);
  
  if (returnValue != Success) {
    tcaModError("ERROR: tcaReferenceData: sendMessage Failed.\n");
    return 0;
  }
  waitRef = tcaRefCreate(ref->msg, ref->name, refId);
  
  returnValue = waitForReplyFrom(waitRef,
				 &(GET_M_GLOBAL(lispFlagGlobal)),
				 TRUE, WAITFOREVER,
				 GET_C_GLOBAL(tcaServerReadGlobal));
  
  tcaRefFree(waitRef);
  
  if (returnValue == NullReply) {
    /* 17-Jun-91: fedor: if NullReply then nothing else was malloced. */
    return 0;
  }
  else
    return 1;
}

#endif


/******************************************************************************
 *
 * FUNCTION:  TCA_REF_STATUS_TYPE tcaReferenceStatus(ref)
 *
 * DESCRIPTION: 
 * Returns the msg status of the message associated with the reference.
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: TCA_REF_STATUS_TYPE
 *
 *****************************************************************************/
#if 0
/* New version: install after Rich finishes testing with the Hero simulator */
TCA_REF_STATUS_TYPE tcaReferenceStatus(TCA_REF_PTR ref)
{
  TCA_REF_STATUS_TYPE status;
  
  (void)tcaQueryCentral(TCA_REF_STATUS_QUERY, (void *)&(ref->refId),
			(void *)&status);
  
  return status;
}
#endif

int tcaReferenceStatus(TCA_REF_PTR ref)
{
  int status = -1;
  (void)tcaQueryCentral(TCA_REF_STATUS_QUERY, &(ref->refId), &status);
  return status;
}
