/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: memory
 *
 * FILE: tcaMem.c
 *
 * ABSTRACT:
 * 
 * Interface to memory management routines.
 *
 * REVISION HISTORY
 *
 * $Log: tcaMem.c,v $
 * Revision 1.21  1996/07/23  19:32:26  reids
 * Changes to support Windows 95.
 *
 * Revision 1.20  1996/06/25  20:51:46  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.19  1996/02/10  16:50:46  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.18  1995/07/06  21:17:38  rich
 * Solaris and Linux changes.
 *
 * Revision 1.17  1995/06/15  14:58:44  rich
 * Fixed definition of the malloc routine.
 *
 * Revision 1.16  1995/06/14  03:22:48  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.15  1995/06/05  23:59:20  rich
 * Improve support of detecting broken pipes.  Add support for OSF 2.
 * Add return types to the global variable routines.
 *
 * Revision 1.14  1995/05/31  19:37:08  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.13  1995/04/08  02:06:39  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.12  1995/01/18  22:43:25  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.11  1994/10/25  17:11:02  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.10  1994/05/31  03:24:03  rich
 * Removed CFLAGS_sun4.
 * Removed cfree and bzero from mem routines.
 * Set zero wait on both sides of the pipe.  Can connect to host using inet
 * number.
 *
 * Revision 1.9  1994/05/17  23:18:12  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.8  1994/04/28  16:17:45  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.7  1994/04/16  19:43:29  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.6  1993/12/14  17:35:32  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.5  1993/11/21  20:19:53  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.4  1993/08/30  21:54:57  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:17:23  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:46  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:39  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:47  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:26:21  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 7-oct-92 Richard Goodwin : Changed free() to cfree().  Try to make it 
 * easier to track down memory problems.  Also added an 
 *
 *  4-Jul-91 Christopher Fedor, School of Computer Science, CMU
 * Added tcaRegisterMallocHnd to complete the routines so that modules
 * can handle all memory allocation.
 *
 *  4-Jun-91 Christopher Fedor, School of Computer Science, CMU
 * Added tcaRegisterFreeMemHnd so that modules can be called to free
 * memory to satisfy a malloc request.
 *
 * 25-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Created.
 *
 * $Revision: 1.21 $
 * $Date: 1996/07/23 19:32:26 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalM.h"

/******************************************************************************
 * Forward Declarations
 *****************************************************************************/

#if !defined(DBMALLOC)
#define SIZETYPE int
void *tcaDBMalloc(const char* file, int line, unsigned long amount);
void tcaDBFree(const char *file, int line,char *item);
#else
#undef tcaMalloc
void *tcaMalloc(unsigned long amount);
#endif


/******************************************************************************
 *
 * FUNCTION: void tcaRegisterFreeMemHnd(void *(*func)(int), int retry)
 *
 * DESCRIPTION: 
 * Sets a function for freeing memory when malloc returns NULL.
 * The function is passed an unsigned int amount of the memory requested.
 * The number of times this routine is called is set by retry.
 * The default is 1.
 *
 * INPUTS: 
 * VOID_FN func;
 * int retry;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaRegisterFreeMemHnd(void (*func)(unsigned int), int retry)
{
  GET_M_GLOBAL(freeMemRetryAmount) = retry;
  
  if (GET_M_GLOBAL(freeMemRetryAmount) < 1)
    GET_M_GLOBAL(freeMemRetryAmount) = 1;
  
  GET_M_GLOBAL(tcaFreeMemoryHnd) = func;
}


/******************************************************************************
 *
 * FUNCTION: void tcaRegisterMallocHnd(func, retry)
 *
 * DESCRIPTION: 
 * Registers a function to call in place of malloc.
 * The routine will be passed an unsigned int of the amount of storage needed.
 * The routine will be called a max of retry times if NULL is returned.
 * The default retry amount is 1.
 *
 * INPUTS: 
 * void *(*func)();
 * int32 retry;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

#if defined(DBMALLOC)
void tcaRegisterMallocHnd(void *(*func)(const char*, int, SIZETYPE),
			  int retry)
#elif defined(__sgi)
     void tcaRegisterMallocHnd(void *(*func)(unsigned int), int retry)
#elif defined(__TURBOC__)
     void tcaRegisterMallocHnd(void *(*func)(unsigned long), long retry)
#else
     void tcaRegisterMallocHnd(void *(*func)(unsigned long), int retry)
#endif
{
  GET_M_GLOBAL(mallocMemRetryAmount) = retry;
  
  if (GET_M_GLOBAL(mallocMemRetryAmount) < 1)
    GET_M_GLOBAL(mallocMemRetryAmount) = 1;
  
  if (func)
    GET_M_GLOBAL(tcaMallocMemHnd) = func;
}


/******************************************************************************
 *
 * FUNCTION: void tcaFree(item)
 *
 * DESCRIPTION: 
 * An interface to free - should use tcaFreeData or tcaFreeReply in most cases.
 *
 * INPUTS: char *item;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

#if defined(DBMALLOC)
#undef tcaFree
void tcaFree(char *item)
{
  tcaDBFree("Unknown", 9999, item);
}
void tcaDBFree(const char* file, int line, char *item)
#else
void tcaDBFree(const char* file, int line, char *item)
{
  tcaFree(item);
}
void tcaFree(char *item)
#endif
{
  if (item == NULL) return;

#if defined(THINK_C) || defined(applec)
  free(item);
#elif defined(DBMALLOC)
  debug_free(file,line,item);
#elif defined(__TURBOC__)
  farfree(item);
#elif defined(DEBUG)
  cfree(item);
#else 
  free(item);
#endif /* THINK_C || applec */
}


/******************************************************************************
 *
 * FUNCTION: char *tcaMalloc(amount)
 *
 * DESCRIPTION: Interface to malloc requests from tca.
 *
 * INPUTS: u_int32 amount;
 *
 * OUTPUTS: char * - generic pointr to the memory
 *
 * NOTES: 
 * Stops everything if we have run out of memory. 
 * Note there may not be enough memory to print that we have run out.
 *
 *****************************************************************************/

#if defined(DBMALLOC)
void *tcaMalloc(unsigned long amount)
{
  return tcaDBMalloc("Unknown", 9999, amount);
}
void *tcaDBMalloc(const char* file, int line, SIZETYPE amount)
#elif defined(__TURBOC__)
     void *tcaDBMalloc(const char* file, int line, unsigned long amount)
{
  return tcaMalloc(amount);
}
void *tcaMalloc(unsigned long amount)
#else
     void *tcaDBMalloc(const char* file, int line, unsigned long amount)
{
  return tcaMalloc(amount);
}
void *tcaMalloc(size_t amount)
#endif
{
  int i, j;
  char *mem;
  
  mem = NULL;
  
  if (GET_M_GLOBAL(tcaMallocMemHnd)) 
    for(i=0;!mem && i < GET_M_GLOBAL(mallocMemRetryAmount);i++) {
#if defined(DBMALLOC)
      mem = (*GET_M_GLOBAL(tcaMallocMemHnd))(file,line,amount);
#else
      mem = (*GET_M_GLOBAL(tcaMallocMemHnd))(amount);
#endif
    }
  
  if (mem) {
    GET_M_GLOBAL(totalMemRequest) += amount;
    /* RTG: make sure the new memory is "clean" */
    /* bzero(mem, (int)amount);  */
    return(mem);
  }
  
  if (GET_M_GLOBAL(tcaFreeMemoryHnd))
    for(j=0;!mem && j < GET_M_GLOBAL(freeMemRetryAmount);j++) {
      (*GET_M_GLOBAL(tcaFreeMemoryHnd))(amount);
      
      if (GET_M_GLOBAL(tcaMallocMemHnd)) 
	for(i=0;!mem && i < GET_M_GLOBAL(mallocMemRetryAmount);i++) {
#if defined(DBMALLOC)
	  mem = (*GET_M_GLOBAL(tcaMallocMemHnd))(file,line,amount);
#else
	  mem = (*GET_M_GLOBAL(tcaMallocMemHnd))(amount);
#endif
	}
    }
  
  if (mem) {
    GET_M_GLOBAL(totalMemRequest) += amount;
    /* RTG: make sure the new memory is "clean" */
    /* bzero(mem, (int)amount);  */
    return(mem);
  }
  
  tcaModError("tcaMalloc: NULL returned from malloc for request: %ld\n", amount);
  return NULL;
}


/******************************************************************************
 *
 * FUNCTION: void tcaStats(stream)
 *
 * DESCRIPTION: Quick hack to display some memory stats.
 *
 * INPUTS: none.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaStats(FILE *stream)
{
  if (!mGlobalp())
    return;
  
  (void)fprintf(stream,"\nTotal Memory Requests Filled: %ld\n", 
		GET_M_GLOBAL(totalMemRequest));
  (void)fprintf(stream,"\n");
  dataMsgDisplayStats(stream);
  (void)fprintf(stream,"\n");
  FLUSH_IF_NEEDED(stream);
}

