/*****************************************************************************
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: matrix
 *
 * FILE: tcaMatrix.c
 *
 * ABSTRACT:
 *
 * matrix creation routines for matrix package used in GIL
 * Modified from existing matrix package.
 *
 * REVISION HISTORY:
 *
 * $Log: tcaMatrix.c,v $
 * Revision 1.10  1995/06/14  03:22:46  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.9  1995/04/21  03:53:39  rich
 * Added central commands to kill the task tree and close a module.
 * Added tcaGetContext and tcaSetContext to support connections to multiple
 * central servers.  tcaConnectModules can be called multiple times.
 * Fixed a bug in the resource limit pending.
 * Created seperate routines to print help and option messages.
 *
 * Revision 1.8  1995/04/19  14:29:08  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.7  1995/04/07  05:03:35  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.6  1995/04/05  19:11:16  rich
 * Adding Reid's Changes.
 * Consolidating the matrix code.
 *
 * Revision 1.5  1995/01/18  22:43:20  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.4  1993/08/30  21:54:55  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:17:19  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:42  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:39  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:44  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:26:17  fedor
 * Added Logging.
 *
 * 17-Apr-90 Christopher Fedor at School of Computer Science, CMU
 * Added newucmat2.
 *
 *  3-Apr-90 Christopher Fedor at School of Computer Science, CMU
 * Created from matrix.c to avoid perception link conflicts.
 *
 * $Revision: 1.10 $
 * $Date: 1995/06/14 03:22:46 $
 * $Author: rich $
 *
 *****************************************************************************/

/* matrix.c -- library routines for constructing dynamic matrices
 * with arbitrary bounds using Iliffe vectors
 ****************************************************************
 * HISTORY
 * 25-Nov-80  David Smith (drs) at Carnegie-Mellon University
 * Changed virtual base address name to "el" for all data
 * types (Previously vali, vald, ...)  This was possible due to the
 * compiler enhancement which keeps different structure declarations
 * separate.
 *
 * 30-Oct-80  David Smith (drs) at Carnegie-Mellon University
 *	Rewritten for record-style matrices
 *
 * 28-Oct-80  David Smith (drs) at Carnegie-Mellon University
 *	Written.
 *
 */

#include "globalM.h"

genericMatrix newmat(int32 elementSize, int32 rs, int32 re, int32 cs,
		     int32 ce, int32 *error)
{
  register char *p, **b;
  register int32 r, rows, cols;
  genericMatrix matrix;
  
  *error=0;
  rows = re-rs+1;
  cols = ce-cs+1;
  
  if (rows<=0 || cols<=0) {*error=1; return matrix;}
  
  matrix.lb1 = rs;
  matrix.ub1 = re;
  matrix.lb2 = cs;
  matrix.ub2 = ce;
  b = (char **)tcaMalloc((unsigned)(rows*sizeof(char *) + 
				    rows*cols*elementSize));
  if (b==0) {*error=1; return(matrix);}
  matrix.mat_sto = (char *) b;
  p = ((char *) &b[rows]) - (cs * elementSize);
  b -= rs;
  matrix.el = b;
  for (r=rs; r<=re; r++)
    {
      b[r] = p;
      p += cols*elementSize;
    }
  return matrix;
}

ucmat newucmat2(int32 rs, int32 re, int32 cs, int32 ce, int32 *error)
{
  ucmat mat;
  genericMatrix matrix;
  
  matrix = newmat(sizeof(unsigned char),rs, re, cs, ce, error);
  mat = *((ucmat *) &matrix);
  return mat;
}

cmat newcmat2(int32 rs, int32 re, int32 cs, int32 ce, int32 *error)
{
  cmat mat;
  genericMatrix matrix;
  
  matrix = newmat(sizeof(char),rs, re, cs, ce, error);
  mat = *((cmat *) &matrix);
  return mat;
}

smat newsmat2(int32 rs, int32 re, int32 cs, int32 ce, int32 *error)
{
  smat mat;
  genericMatrix matrix;
  
  matrix = newmat(sizeof(int16),rs, re, cs, ce, error);
  mat = *((smat *) &matrix);
  return mat;
}

imat newimat2(int32 rs, int32 re, int32 cs, int32 ce, int32 *error)
{
  imat mat;
  genericMatrix matrix;
  
  matrix = newmat(sizeof(int32),rs, re, cs, ce, error);
  mat = *((imat *) &matrix);
  return mat;
}

lmat newlmat2(int32 rs, int32 re, int32 cs, int32 ce, int32 *error)
{
  lmat mat;
  genericMatrix matrix;
  
  matrix = newmat(sizeof(long),rs, re, cs, ce, error);
  mat = *((lmat *) &matrix);
  return mat;
}

fmat newfmat2(int32 rs, int32 re, int32 cs, int32 ce, int32 *error)
{
  fmat mat;
  genericMatrix matrix;
  
  matrix = newmat(sizeof(float),rs, re, cs, ce, error);
  mat = *((fmat *) &matrix);
  return mat;
}

dmat newdmat2(int32 rs, int32 re, int32 cs, int32 ce, int32 *error)
{
  dmat mat;
  genericMatrix matrix;
  
  matrix = newmat(sizeof(double),rs, re, cs, ce, error);
  mat = *((dmat *) &matrix);
  return mat;
}
