/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: taskTree
 *
 * FILE: taskTree.h
 *
 * ABSTRACT:
 * 
 * Creation and manipulation of hierarchical task trees
 * (containing goal, command, and monitor message nodes).
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/src/taskTree.h,v $ 
 * $Revision: 1.16 $
 * $Date: 1996/01/30 15:04:59 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: taskTree.h,v $
 * Revision 1.16  1996/01/30  15:04:59  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.15  1996/01/27  21:54:30  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.14  1995/07/19  14:26:41  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.13  1995/07/06  21:17:29  rich
 * Solaris and Linux changes.
 *
 * Revision 1.12  1995/05/31  19:36:57  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.11  1995/04/21  03:53:31  rich
 * Added central commands to kill the task tree and close a module.
 * Added tcaGetContext and tcaSetContext to support connections to multiple
 * central servers.  tcaConnectModules can be called multiple times.
 * Fixed a bug in the resource limit pending.
 * Created seperate routines to print help and option messages.
 *
 * Revision 1.10  1995/04/19  14:29:01  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.9  1995/03/30  15:44:16  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.8  1995/01/18  22:43:06  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.7  1994/05/17  23:17:56  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.6  1994/04/28  16:17:30  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.5  1993/12/01  18:04:34  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.4  1993/08/30  21:54:38  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/23  17:40:44  rich
 * Fixed the type definitions for function pointers. Added const
 * declarations.  Removed definitions VOID_FN and INT_FN.
 *
 * Revision 1.2  1993/05/27  22:20:50  rich
 * Added automatic logging.
 *
 * 25-May-90 Christopher Fedor, School of Computer Science, CMU
 * Revised to Software Standards.
 *
 * 28-Nov-89 Long-Ji Lin, School of Computer Science, CMU
 * Added 'ECNode' & 'excHnds' slots to tree node
 *
 *  2-Oct-89 Reid Simmons, School of Computer Science, CMU
 * Added status slot to task tree node.
 *
 * 19-Sep-89 Reid Simmons, School of Computer Science, CMU
 * Added a global task tree root node.
 *
 * 14-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Added routines to kill task trees, get parent and children references.
 *
 * 11-Aug-89 Reid Simmons, School of Computer Science, CMU
 *  Changed terms from "goal tree" to "task tree"
 *
 * 15-May-89 Christopher Fedor, School of Computer Science, CMU
 * Changed interval type not to make use of quantity_ptr typedef to 
 * minimize include files needed in modules that do not use qlattice.
 *
 * 28-Apr-89 Reid Simmons, School of Computer Science, CMU
 * First attempt at goal structure. Added slots for handling temporal 
 * constraints.
 *
 *****************************************************************************/

#ifndef INCtaskTree
#define INCtaskTree

/* For displaying task trees */
#define TAB 14
#define pr_space(n) { int32 i; i=(n); for(; i>0; --i) Log(" "); }

typedef enum {
  ActiveNode=0, AchievedNode=1, KilledNode=2, ToBeKilledNode=3, 
  SuspendedNode=4, ToBeSuspendedNode=5
#ifdef FORCE_32BIT_ENUM
    , dummyTaskTreeStatus = 0x7FFFFFFF
#endif
} TASK_TREE_STATUS_TYPE;

#define DEAD_NODE(node) \
(((node)->status == KilledNode) || ((node)->status == ToBeKilledNode))

typedef struct {
  struct _TREE_NODE *ENode, *CNode;
} ECNODE_TYPE, *ECNODE_PTR;

typedef struct _TREE_NODE {
  DISPATCH_PTR dispatch;
  struct _LIST *children;
  struct _TREE_NODE *parent;
  struct _TREE_NODE *lastChild; /* used for doing SEQ_ACH and SEQ_PLANNING */
  struct _QUANTITY_TYPE *priority;
  struct _interval_type *handlingInterval;
  struct _interval_type *achievementInterval;
  struct _interval_type *planningInterval;
  TASK_TREE_STATUS_TYPE status;
  
  int32 retries;
  ECNODE_PTR ECNode;
  struct _LIST *excHndList;
  
} TASK_TREE_NODE_TYPE, *TASK_TREE_NODE_PTR;

typedef struct {
  MODULE_PTR module;
  TASK_TREE_NODE_PTR lastChild;
} MOD_LAST_CHILD_TYPE, *MOD_LAST_CHILD_PTR;

typedef struct {
  int32 msgRef;
  int32 parentRefId;
} ADD_CHILD_TYPE, *ADD_CHILD_PTR;

typedef struct {
  int32 refId;
  const char *name;
} FIND_REF_TYPE, *FIND_REF_PTR;

typedef struct { 
  const char *name;
  BOOLEAN (* hndProc)(DISPATCH_PTR);
  MODULE_PTR hndOrg;
} ABORT_MSG_TYPE, *ABORT_MSG_PTR;

#define TASK_TREE_NODE_CLASS(taskTreeNode) \
(taskTreeNode)->dispatch->msg->msgData->msg_class

#define TASK_TREE_NODE_MSG_NAME(taskTreeNode) \
(taskTreeNode)->dispatch->msg->msgData->name

#define IS_ROOT_NODE(taskTreeNode) ((taskTreeNode) == GET_S_GLOBAL(taskTreeRootGlobal))

DISPATCH_PTR findParentDispatch(DISPATCH_PTR child);
TASK_TREE_NODE_PTR taskTreeNodeCreate(DISPATCH_PTR dispatch,
				      int32 parentRef);
int32 taskTreeClass(TCA_MSG_CLASS_TYPE msgClass);
int32 Sequencing_Class(TCA_MSG_CLASS_TYPE msgClass);

void removeModLastChild(TASK_TREE_NODE_PTR rootNode, MODULE_PTR module);
int32 Update_Last_Child(TASK_TREE_NODE_PTR childNode);
TASK_TREE_NODE_PTR getLastChild(TASK_TREE_NODE_PTR treeNode,
				DISPATCH_PTR dispatch);

ABORT_MSG_PTR addAbortHandler(const char *name, 
			      BOOLEAN (* hndProc)(DISPATCH_PTR));
void killTaskTree(int32 killRefId);
void killSubTaskTree(TASK_TREE_NODE_PTR node);
void clearTaskTree(void);
int32 isToBeKilled(DISPATCH_PTR dispatch);
void HandleKillAfterAttendingNodes(DISPATCH_PTR dispatch);
int32 NeedToKillMessage(DISPATCH_PTR dispatch);

/* DISPATCH_PTR CreateReferenceDispatch(int32 msgRef, int32 parentId,
   MODULE_PTR dispatchOrg); */
int32 retractEndOfConstraints (TASK_TREE_NODE_PTR node);
void restoreEndOfRelations(TASK_TREE_NODE_PTR node);

void taskTreeInitialize(void);

void showTaskTree(void);
void displayNonTaskTree(void);

#endif /* INCtaskTree */
