/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: taskTree
 *
 * FILE: taskTree.c
 *
 * ABSTRACT:
 * 
 * Creation and manipulation of hierarchical task trees
 * (containing goal, command, and monitor message nodes).
 *
 * REVISION HISTORY
 *
 * $Log: taskTree.c,v $
 * Revision 1.37  1996/07/19  18:14:32  reids
 * Record broadcast messages if handler is registered before message.
 * Transfer any pending messages to the new resource under "addHndToResource"
 * Fixed tcaDelayCommand (wrong time units).
 * Fixed logging of refid's (have to distinguish whether they are part of
 *   a status, message, or "always" log).
 * Sanity check for encoding/decoding messages.
 *
 * Revision 1.36  1996/07/12  13:56:40  reids
 * Fixed how "last child" is reset -- don't add monitors as the last child!
 *
 * Revision 1.35  1996/06/25  20:51:39  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.34  1996/03/15  21:24:00  reids
 * Fixed a "free memory read" bug when killing task trees.  Now, memory safe
 *
 * Revision 1.33  1996/02/10  16:50:36  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.32  1996/01/27  21:54:27  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.31  1996/01/10  03:16:31  rich
 * Fixed libtca_lisp.a to work with dbmalloc.  Added central commands to
 * show resource state and to unlock locked resouces.  Fixed a bug where
 * dispatches were not freed when handlers were cleared. Reset errno variable.
 *
 * Revision 1.30  1995/12/17  20:22:21  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.29  1995/10/29  18:27:08  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.28  1995/10/25  22:48:58  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.27  1995/10/10  00:43:10  rich
 * Added more system messages to ignore.
 *
 * Revision 1.26  1995/10/07  19:07:51  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.25  1995/08/05  17:16:20  reids
 * Several important bug fixes:
 *   a) Found a memory leak in the tms (when nodes are unasserted)
 *   b) Fixed a problem with direct connections that would cause TCA to crash
 *      when a command or goal message was sent from an inform or a query.
 *      As part of that fix, all command and goal messages that are sent from
 *      informs or queries are now added to the root node of the task tree.
 *
 * Revision 1.24  1995/07/19  14:26:37  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.23  1995/07/10  16:18:49  rich
 * Interm save.
 *
 * Revision 1.22  1995/07/06  21:17:26  rich
 * Solaris and Linux changes.
 *
 * Revision 1.21  1995/06/14  03:22:29  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.20  1995/05/31  19:36:52  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.19  1995/04/21  03:53:29  rich
 * Added central commands to kill the task tree and close a module.
 * Added tcaGetContext and tcaSetContext to support connections to multiple
 * central servers.  tcaConnectModules can be called multiple times.
 * Fixed a bug in the resource limit pending.
 * Created seperate routines to print help and option messages.
 *
 * Revision 1.18  1995/04/19  14:28:58  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.17  1995/03/30  15:44:11  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.16  1995/01/25  00:01:46  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Fixed problems with little endian translation.
 *
 * Revision 1.15  1995/01/18  22:43:01  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.14  1994/11/02  21:34:36  rich
 * Now works for linux machines (i486).
 * Got afs to work on alpha (and hopefully other vendor OS's)
 * Added generic Makefile.
 * Made libc.h and tcaMatrix.h module includes.
 * Reduced the size of libc.h by using more system includes.
 *
 * Revision 1.13  1994/10/25  17:10:52  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.12  1994/05/17  23:17:50  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.11  1994/04/28  16:17:25  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.10  1994/04/16  19:43:16  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.9  1994/01/31  18:28:59  reids
 * Several major changes (and some minor ones)
 * 1. tcaFreeData and tcaFreeReply now work even if the data or message format
 *    is NULL
 * 2. Using the "-t" option in central, message taps are added as a child of
 *    the task tree node that was tapped.
 * 3. Named formatters are now expanded only when needed
 * For more details, see ../doc/tca-7-4.release.notes
 *
 * Revision 1.8  1993/12/14  17:35:21  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.7  1993/11/21  20:19:38  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.6  1993/10/21  16:14:30  rich
 * Fixed compiler warnings.
 *
 * Revision 1.5  1993/08/30  21:54:32  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.4  1993/08/27  07:17:05  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.3  1993/08/23  16:01:34  rich
 * Changed the global include files so that they don't double include
 * system files.  This was causing problems on the mach machines.
 *
 * Revision 1.2  1993/05/26  23:19:25  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:37  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:27  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:26:02  fedor
 * Added Logging.
 *
 * 29-Sept-92 RTG : Modified FreeTaskTreeNode to Remove reference to the
 * node from the parent node.  This was causing problems when when doing
 * findFirstChild when some of the children of a node had been killed.
 *
 * 15-Oct-92 Reid Simmons, School of Computer Science, CMU
 * Converted "childrenList" to use the new "list" data structures.
 * Changed the return values for "tcaReferenceStatus".
 *
 * 31-Dec-91 Reid Simmons, School of Computer Science, CMU
 * Added call to add all exception handlers to node that are associated
 *  with the node's message in general.
 *
 * 14-Dec-91 Christopher Fedor, School of Computer Science, CMU
 * Added Rich Goodwin's referenceStatusHnd
 *
 * 16-Oct-91 Christopher Fedor, School of Computer Science, CMU
 * Moved the code to remove children from a killed node to happen
 * before deleteTaskTree call for that node in both killTaskTree and
 * killSubTaskTree.
 *
 * 21-Aug-91 Christopher Fedor, School of Computer Science, CMU
 * Added changes from Reid includeing:
 * Check that msgRef != NO_REF in referenceReleaseHnd.
 * Tests on dispatch refCount and status for HandleKilAfterAttendingNodes
 *
 * 26-Jul-91, Reid Simmons, School of Computer Science, CMU
 * Task tree killing had a bug -- did not correctly remove root node of kill
 * tree from the children of its parent.
 *
 * 23-Jul-91, Reid Simmons, School of Computer Science, CMU
 * Adding a new node to a goal node might nullify the "end of interval"
 * completion constraints.  Need to retract those constraints before
 * proceeding.
 *
 *  3-Jul-91, Reid Simmons, School of Computer Science, CMU
 * Pre-remove *all* the rules from *all* the relations of the quantities 
 * associated with the nodes to be killed.
 * Also freed data used by the centrally registered handlers. 
 *
 * 13-Jun-91, Reid Simmons, School of Computer Science, CMU
 * Added code to remove wiretap rules when task tree nodes are killed.
 * Also, free dispatch when corresponding nodes are killed.
 *
 * 24-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Spliced in the new resource routines were needed.
 *
 * 25-May-90 Christopher Fedor, School of Computer Science, CMU
 * Revised to Software Standards.
 * Removed addTreeNodeIfNecessary test because with the new flow and control,
 * the message class is known, so only the existance of a tree node is needed
 * and can be done easier elsewhere, i.e. in recvMessage.
 *
 * 30-Nov-89 Reid Simmons, School of Computer Science, CMU
 * Display task tree now prints out status of node.
 *
 * 28-Nov-89 Long-Ji Lin, School of Computer Science, CMU
 * Added addTreeNodeIfNecessary (dispatch). Added two slots (excHnds & ECNode)
 * to tree node and updated "FreeTaskTreeNode()" and "CreateTaskTreeNode" 
 * correspondingly.
 *
 * 27-Nov-89 Reid Simmons, School of Computer Science, CMU
 * Added new routines for tracing task tree (find first and last child, 
 * find child by name, get data of reference).
 *
 * 10-Nov-89 Reid Simmons, School of Computer Science, CMU
 * First, primitive attempt at adding a mechanism to abort tasks that are 
 * currently attending. Used now only for aborting a tcaDelay command that is
 * to-be-killed (otherwise, the delay could linger for a long while).
 *
 * 10-Nov-89 Reid Simmons, School of Computer Science, CMU
 * Bug fixes -- task tree tracing functions didn't work right for the 
 * root node; displaying and re-killing a killed node didn't work; couldn't
 * kill an inactive interval monitor.
 *
 * 18-Oct-89 Reid Simmons, School of Computer Science, CMU
 * Added tracing code to track down "intermittent" task tree killing bug.
 * (running unknown rule type).
 *
 *  2-Oct-89 Reid Simmons, School of Computer Science, CMU
 * Moved "Kill" statuses to task tree nodes.
 *
 * 19-Sep-89 Reid Simmons, School of Computer Science, CMU
 * Added a global task tree root node.
 *
 * 29-Aug-89 Long-Ji Lin, School of Computer Science, CMU
 * Changed "DoList" in list.h and replaced "DoList_Star" with "DoList".
 *
 * 23-Aug-89 Long-Ji Lin, School of Computer Science, CMU
 * Fixed a few bugs, including replacing DoList with DoList_Star.
 * DoList may leave the "Cdr" pointer dangling, if the body of the 
 * do loop involves destructive operations to the "Car" element.
 *
 * 23-Aug-89 Long-Ji Lin, School of Computer Science, CMU
 * Added routines to get root and antecedent references, 
 * and to display task trees.
 *
 * 14-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Added routines to kill task trees, get parent and
 * children references, message handlers.
 *
 * 11-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Changed terms from "goal tree" to "task tree"
 *
 * 4-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Added warning if Query or tcaAddConstraint issue a
 * goal, command, or monitor message.
 *
 * 2-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Added "lastChild" slot to nodes, to make sequentiality constraints easier
 * to add. Also, changed so that CreateReference no longer makes the created 
 * node the last child (this is done only when responding to an actual
 *  message).
 *
 * 28-Apr-89 Reid Simmons, School of Computer Science, CMU
 * Created.
 *
 * $Revision: 1.37 $
 * $Date: 1996/07/19 18:14:32 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalS.h"


/******************************************************************************
 *
 * FUNCTION TASK_TREE_NODE_PTR taskTreeNodeFromRef(ref)
 *
 * DESCRIPTION: 
 * Returns the task tree node associated with this dispatch ref.
 * If NO_REF is passed, tcaTaskTreeRootGlobal is returned.
 * If the dispatch reffered to is NULL, then NULL is returned.
 *
 * INPUTS: int32 ref
 *
 * OUTPUTS: TASK_TREE_NODE_PTR
 *
 *****************************************************************************/

static TASK_TREE_NODE_PTR taskTreeNodeFromRef(int32 ref)
{ 
  DISPATCH_PTR dispatch;
  
  if (ref == NO_REF) {
    return GET_S_GLOBAL(taskTreeRootGlobal);
  } else {
    dispatch = DISPATCH_FROM_ID(ref);
    if (dispatch) {
      return dispatch->treeNode;
    } else {
      return NULL;
    }
  }
}


/******************************************************************************
 *
 * FUNCTION: DISPATCH_PTR findParentDispatch(child)
 *
 * DESCRIPTION:
 * Given the child dispatch, return the dispatch of the 
 * parent node of the child.
 * Return NULL if no susch node exists.
 *
 * INPUTS: DISPATCH_PTR child;
 *
 * OUTPUTS: DISPATCH_PTR
 *
 *****************************************************************************/

DISPATCH_PTR findParentDispatch(DISPATCH_PTR child)
{ 
  TASK_TREE_NODE_PTR childNode, parentNode;
  
  if (child) {
    childNode = child->treeNode;
    if (childNode) {
      parentNode = childNode->parent;
      if (parentNode)
	return parentNode->dispatch;
    }
  }
  /* default case -- no parent */
  return NULL;
}


/******************************************************************************
 *
 * FUNCTION: TASK_TREE_NODE_PTR taskTreeNodeCreate(dispatch, parentRef)
 *
 * DESCRIPTION: Create a task tree node and attach it to its parent.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * int32 parentRef;
 *
 * OUTPUTS: TASK_TREE_NODE_PTR
 *
 *****************************************************************************/

TASK_TREE_NODE_PTR taskTreeNodeCreate(DISPATCH_PTR dispatch, int32 parentRef)
{ 
  DISPATCH_PTR parentDispatch;
  TASK_TREE_NODE_PTR treeNode, parentNode;
  
  treeNode = NEW(TASK_TREE_NODE_TYPE);
  
  treeNode->dispatch = dispatch;
  treeNode->priority = NULL;
  treeNode->children = NULL;
  treeNode->lastChild = NULL;
  treeNode->handlingInterval = NULL;
  treeNode->achievementInterval = NULL;
  treeNode->planningInterval = NULL;
  treeNode->status = ActiveNode;
  
  treeNode->excHndList = NULL;
  
  treeNode->ECNode = NULL;
  treeNode->retries = 0;
  
  /* 12-May-91: fedor: important that root has a NULL parent for exceptions */
  
  /* No dispatch means no parent node */
  if (!dispatch) {
    treeNode->parent = NULL; 
  } else {
    parentNode = taskTreeNodeFromRef(parentRef);
    if (!parentNode) {
      parentNode = GET_S_GLOBAL(taskTreeRootGlobal);

      parentDispatch = DISPATCH_FROM_ID(parentRef);
      if (!taskTreeClass(parentDispatch->msg->msgData->msg_class)) {
	Log("\nWARNING!! Issuing a %s class message (%s) from \n",
	    messageClassName(dispatch->msg->msgData->msg_class),
	    dispatch->msg->msgData->name);
	Log("  a %s class message (%s);\n  Adding under task tree root node.\n",
	    messageClassName(parentDispatch->msg->msgData->msg_class),
	    parentDispatch->msg->msgData->name);
      } else {
	tcaError("ERROR: Could not find a task tree node for message %s.\n",
		 parentDispatch->msg->msgData->name);
      }
    }

    treeNode->parent = parentNode;
    dispatch->treeNode = treeNode;
    if (!parentNode->children)
      parentNode->children = listCreate();
    listInsertItem((char *)treeNode, parentNode->children);
    
    /* Added 12/31/91: reids */
    if (dispatch->msg) {
      addMsgExceptions(dispatch->msg, treeNode);
    }
  }
  
  return treeNode;
}


/******************************************************************************
 *
 * FUNCTION: void FreeTaskTreeNode(TreeNode)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void FreeTaskTreeNode(TASK_TREE_NODE_PTR TreeNode)
{ 
  struct _interval_type *handlingInterval;
  
  if (!TreeNode) return;

  /* 29-Sept-92 RTG : Remove reference from the parent to be safe */
  if ((TreeNode->parent != NULL) && listLength(TreeNode->parent->children) > 0)
    {
      /* RTG this is not quit right, so I'm commenting it out for now */
      /* resetLastChild(TreeNode->parent, TreeNode); */
      listDeleteItem((char *)TreeNode, TreeNode->parent->children);
    }

  listFree(&(TreeNode->children));
  TreeNode->children=NULL;
  if (TreeNode->priority) {
    FreeQuantity(&(TreeNode->priority));
    TreeNode->priority = NULL;
  }
  handlingInterval = TreeNode->handlingInterval;
  if (handlingInterval) {
    if (TreeNode->planningInterval != handlingInterval)
      FreeInterval(&(TreeNode->planningInterval));
    TreeNode->planningInterval = NULL;
    if (TreeNode->achievementInterval != handlingInterval) 
      FreeInterval(&(TreeNode->achievementInterval));
    TreeNode->achievementInterval = NULL;
    FreeInterval(&handlingInterval);
    TreeNode->handlingInterval = NULL;
  }
  /* 29-Nov-89: ljl - free ECNode & excHnds here? */
  if (TreeNode->ECNode)  {
    tcaFree((char *)TreeNode->ECNode);
    TreeNode->ECNode = NULL;
  }
  
  if (TreeNode->excHndList) {
    listFree(&(TreeNode->excHndList));
    TreeNode->excHndList = NULL;
  }
  
  if (TreeNode->dispatch) {
    dispatchFree(TreeNode->dispatch);
    TreeNode->dispatch = NULL;
  }
  
  tcaFree((char *)TreeNode);
}

/*ARGSUSED*/
static int32 freeTaskTreeNode1(void *dummy, TASK_TREE_NODE_PTR node)
{
#ifdef applec
#pragma unused(dummy)
#endif
  FreeTaskTreeNode(node);
  return TRUE;
}

/******************************************************************************
 *
 * FUNCTION: int32 taskTreeClass(msgClass)
 *
 * DESCRIPTION:
 * Returns TRUE if the class of message is one that is added to the task 
 * tree (currently, goals, commands, exceptions, point-monitors,
 * interval-monitors).
 *
 * INPUTS: TCA_MSG_CLASS_TYPE msgClass;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

int32 taskTreeClass(TCA_MSG_CLASS_TYPE msgClass)
{
  return ((msgClass == GoalClass) || 
	  (msgClass == CommandClass) ||
	  (msgClass == ExceptionClass) ||
	  (msgClass == PointMonitorClass) ||
	  (msgClass == PollingMonitorClass) ||
	  (msgClass == DemonMonitorClass));
}


/******************************************************************************
 *
 * FUNCTION: int32 Sequencing_Class(msgClass)
 *
 * DESCRIPTION:
 * Returns TRUE if the class of message is one that is used for the
 * sequential temporal constraints (SEQ_ACH, SEQ_PLANNING).
 * These are the class of messages that can be a "lastChild"
 * (currently, goals, commands, and point-monitors).
 *
 * INPUTS: TCA_MSG_CLASS_TYPE msgClass;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

int32 Sequencing_Class(TCA_MSG_CLASS_TYPE msgClass)
{
  return ((msgClass == GoalClass) || 
	  (msgClass == CommandClass) ||
	  (msgClass == PointMonitorClass));
}


/******************************************************************************
 *
 * FUNCTION: int sameModule(module, modLastChild)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static int32 sameModule(MODULE_PTR module, MOD_LAST_CHILD_PTR modLastChild)
{
  return module == modLastChild->module;
}


/******************************************************************************
 *
 * FUNCTION: MOD_LAST_CHILD_PTR getModLastChild(rootNode, module)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static MOD_LAST_CHILD_PTR getModLastChild(TASK_TREE_NODE_PTR rootNode,
					  MODULE_PTR module)
{
  if (!IS_ROOT_NODE(rootNode)) {
    tcaError("Screwed up: getModLastChild");
  } 
  
  return (MOD_LAST_CHILD_PTR)listMemReturnItem((LIST_ITER_FN) sameModule,
					       (char *)module,
					       (LIST_PTR)(rootNode->lastChild));
}


/******************************************************************************
 *
 * FUNCTION: void removeModLastChild(rootNode, module)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

void removeModLastChild(TASK_TREE_NODE_PTR rootNode, MODULE_PTR module)
{
  MOD_LAST_CHILD_PTR modLastChild;
  
  if (rootNode == NULL) return;

  modLastChild = 
    (MOD_LAST_CHILD_PTR)listMemReturnItem((LIST_ITER_FN) sameModule,
					  (char *)module,
					  (LIST_PTR)rootNode->lastChild);
  if (modLastChild) {
    listDeleteItem((char *)modLastChild, (LIST_PTR)rootNode->lastChild);
    tcaFree((char *)modLastChild);
  }
}


/******************************************************************************
 *
 * FUNCTION: void setLastChild(parentNode, childNode, childModule)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void setLastChild(TASK_TREE_NODE_PTR parentNode,
			 TASK_TREE_NODE_PTR childNode, MODULE_PTR childModule)
{
  MOD_LAST_CHILD_PTR modLastChild;
  
  if (!childNode || !IS_LISTENING(childNode->dispatch)) {
    if (!IS_ROOT_NODE(parentNode)) {
      parentNode->lastChild = childNode;
    } else {
      /* root node has set of last children, one for each module */
      modLastChild = getModLastChild(parentNode, childModule);
      if (!modLastChild) {
	modLastChild = NEW(MOD_LAST_CHILD_TYPE);
	modLastChild->module = childModule;
	listInsertItem((char *)modLastChild, (LIST_PTR)parentNode->lastChild);
      }
      modLastChild->lastChild = childNode;
    }
  }
}

static int32 setLastChildIterate (TASK_TREE_NODE_PTR parent,
				TASK_TREE_NODE_PTR child)
{
  setLastChild(parent, (TASK_TREE_NODE_PTR)NULL, child->dispatch->org);
  return TRUE;
}

/******************************************************************************
 *
 * FUNCTION: int32 Update_Last_Child(childNode)
 *
 * DESCRIPTION:
 * If the childNode is goal, command, or point-monitor, it becomes
 * the new "lastChild" of its parent.
 * Returns TRUE if an update occurred.
 *
 * The root node (tcaTaskTreeRootGlobal), has multiple last
 * children, one for each module.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 * NOTES: 25-Jul-91: Reid: Added the check for "DEAD_NODE" -- one process could
 *        kill a node that another has created a reference to.
 *****************************************************************************/

int32 Update_Last_Child(TASK_TREE_NODE_PTR childNode)
{ 
  TASK_TREE_NODE_PTR parentNode;
  
  parentNode = childNode->parent;
  if (parentNode && !DEAD_NODE(childNode)) {
    if (Sequencing_Class(TASK_TREE_NODE_CLASS(childNode))) {
      setLastChild(parentNode, childNode, childNode->dispatch->org);
      return TRUE;
    }
  }
  return FALSE;
}


/******************************************************************************
 *
 * FUNCTION: TASK_TREE_NODE_PTR getLastChild(treeNode, dispatch)
 *
 * DESCRIPTION:
 * Returns the last goal, command, or point-monitor message added
 * to the tree_node.
 *
 * The root node (tcaTaskTreeRootGlobal), has multiple last
 * children, one for each module.  Returns the last child that
 * matches the module of the dispatch's handler.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

TASK_TREE_NODE_PTR getLastChild(TASK_TREE_NODE_PTR treeNode,
				DISPATCH_PTR dispatch)
{
  MOD_LAST_CHILD_PTR modLastChild;
  
  if (IS_ROOT_NODE(treeNode)) {
    modLastChild = getModLastChild(treeNode, dispatch->org);
    if (modLastChild) {
      return modLastChild->lastChild;
    } else {
      return NULL;
    }
  } else {
    return treeNode->lastChild;
  }
}


/******************************************************************************
 *
 * FUNCTION: TASK_TREE_NODE_PTR getFirstChildAfter(treeNode, childrenList)
 *
 * DESCRIPTION:
 * Return the "first" child in the childrenList after the treeNode.
 * This is actually fairly ambiguous, so we return one of the (possibly
 * many) earliest nodes on the list whose achievement start time is
 * greater than the end of the achievement time of the treeNode.
 * If "treeNode" is NULL, the child with the earliest achievement
 * time is returned.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static TASK_TREE_NODE_PTR getFirstChildAfter(TASK_TREE_NODE_PTR treeNode,
					     LIST_PTR childrenList)
{ 
  TASK_TREE_NODE_PTR child, nextChild = NULL;
  struct _QUANTITY_TYPE *treeNodeEnd=NULL, *nextChildStart=NULL, *childStart;
  
  if (treeNode) treeNodeEnd = treeNode->achievementInterval->end;
  child = (TASK_TREE_NODE_PTR)listFirst(childrenList);
  while (child) {
    if (child != treeNode && taskTreeClass(TASK_TREE_NODE_CLASS(child))
	&& !IS_LISTENING(child->dispatch)) {
      childStart = child->achievementInterval->start;
      if ((!treeNode || QIs_True(childStart, ">=", treeNodeEnd)) &&
	  (!nextChild || QIs_True(childStart, "<=", nextChildStart))) {
	nextChild = child;
	nextChildStart = childStart;
      }
    }
    child = (TASK_TREE_NODE_PTR)listNext(childrenList);
  }
  return nextChild;
}

/******************************************************************************
 *
 * FUNCTION: TASK_TREE_NODE_PTR getLastChildBefore(treeNode, childrenList)
 *
 * DESCRIPTION:
 * Return the "last" child in the childrenList before the treeNode.
 * This is actually fairly ambiguous, so we return one of the (possibly
 * many) latest nodes in the list whose achievement end time is less than
 * the start of the achievement time of the treeNode.
 * If "treeNode" is NULL, the node with the latest achievement time is
 * returned.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static TASK_TREE_NODE_PTR getLastChildBefore(TASK_TREE_NODE_PTR treeNode,
					     LIST_PTR childrenList)
{ 
  TASK_TREE_NODE_PTR child, previousChild = NULL;
  struct _QUANTITY_TYPE *treeNodeStart=NULL, *previousChildEnd=NULL, *childEnd;
  
  if (treeNode) treeNodeStart = treeNode->achievementInterval->start;
  child = (TASK_TREE_NODE_PTR)listFirst(childrenList);
  while (child) {
    if (child != treeNode && Sequencing_Class(TASK_TREE_NODE_CLASS(child))
	&& !IS_LISTENING(child->dispatch)) {
      childEnd = child->achievementInterval->end;
      if ((!treeNode || QIs_True(childEnd, "<=", treeNodeStart)) &&
	  (!previousChild || QIs_True(childEnd, ">=", previousChildEnd))) {
	previousChild = child;
	previousChildEnd = childEnd;
      }
    }
    child = (TASK_TREE_NODE_PTR)listNext(childrenList);
  }
  return previousChild;
}

/******************************************************************************
 *
 * FUNCTION: Remove_TaskTree_Rules(TreeNode)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS: Returns TRUE (needed by listIterate)
 *
 * NOTES: Changed (3-Jul-91) to free *all* rules associated with the
 *        quantities to be killed in order to remove spurious firings.
 *
 *****************************************************************************/

/*ARGSUSED*/
static int32 removeTaskTreeRules1 (void* dummy, TASK_TREE_NODE_PTR treeNode)
{ 
#ifdef applec
#pragma unused(dummy)
#endif
  DISPATCH_PTR dispatch;
  
  if (!DEAD_NODE(treeNode)) {
    
    dispatch = treeNode->dispatch;
    
    /* 16-Nov-90: fedor: there could still be problems with this! see mon.c */
    cancelIfAttendingIntervalMonitor(dispatch);
    
    removeQuantityRules(treeNode->handlingInterval->start);
    removeQuantityRules(treeNode->handlingInterval->end);
    if (treeNode->achievementInterval != treeNode->handlingInterval) {
      removeQuantityRules(treeNode->achievementInterval->start);
      removeQuantityRules(treeNode->achievementInterval->end);
    }
    if (treeNode->planningInterval != treeNode->handlingInterval) {
      removeQuantityRules(treeNode->planningInterval->start);
      removeQuantityRules(treeNode->planningInterval->end);
    }
    
    (void)listIterate((LIST_ITER_FN)removeTaskTreeRules1, 
		      (char *)NULL, treeNode->children);
  }
  return TRUE;
}

static void Remove_TaskTree_Rules(TASK_TREE_NODE_PTR treeNode)
{
  (void)removeTaskTreeRules1((void *)NULL, treeNode);
}

/******************************************************************************
 *
 * FUNCTION: void resetLastChild(parentNode, currentChildNode)
 *
 * DESCRIPTION: If this is the "last" child node of the parent, need to
 *    reset in order for future sequencing constraints to work correctly.
 *    Some hair because the root node has multiple "last" children, one for
 *    each module.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void resetLastChild(TASK_TREE_NODE_PTR parentNode,
			   TASK_TREE_NODE_PTR childNode)
{      
  TASK_TREE_NODE_PTR nextChild, lastChild;
  MODULE_PTR module;
  struct _QUANTITY_TYPE *lastChildEnd=NULL, *nextChildEnd;
  
  if (getLastChild(parentNode, childNode->dispatch) == childNode) {
    module = childNode->dispatch->org;
    if (IS_ROOT_NODE(parentNode)) {
      /* Get the last node before the "childNode",
	 but make sure it's from the same module */
      lastChild = NULL;
      nextChild = (TASK_TREE_NODE_PTR)listFirst(parentNode->children);
      while (nextChild) {
	if (nextChild != childNode && nextChild->dispatch->org == module &&
	    Sequencing_Class(TASK_TREE_NODE_CLASS(nextChild))) {
	  nextChildEnd = nextChild->achievementInterval->end;
	  if (!lastChild || QIs_True(nextChildEnd, ">=", lastChildEnd)) {
	    lastChild = nextChild;
	    lastChildEnd = nextChildEnd;
	  }
	}
	nextChild = (TASK_TREE_NODE_PTR)listNext(parentNode->children);
      }
    } else {
      lastChild = getLastChildBefore(childNode, parentNode->children);
    }
    setLastChild(parentNode, lastChild, module);
  }
}

/******************************************************************************
 *
 * FUNCTION: void try_to_delete_node(TreeNode, firstCall)
 *
 * DESCRIPTION:
 * Free the children nodes if they are all marked "KilledNode" and the TreeNode
 * is marked "ToBeKilledNode".
 * If children are freed, try to delete the parent of the TreeNode,
 * and so on up the tree.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void try_to_delete_node(TASK_TREE_NODE_PTR TreeNode, int32 firstCall)
{ 
  TASK_TREE_NODE_PTR child, parentNode;
  
  if (TreeNode->status == ToBeKilledNode) {
    child = (TASK_TREE_NODE_PTR)listFirst(TreeNode->children);
    while (child) {
      if (child->status != KilledNode) {
	if (firstCall) {
	  Log_Status("Will kill %s ", TASK_TREE_NODE_MSG_NAME(TreeNode));
	  Log_RefId(TreeNode->dispatch, LOGGING_STATUS); 
	  Log_Status("when submessages complete\n");
	}
	return;
      }
      child = (TASK_TREE_NODE_PTR)listNext(TreeNode->children);
    }
    
    /* All children are marked "Killed" */
    if (TreeNode->dispatch->refCount <= 0) {
      (void)listIterate((LIST_ITER_FN)freeTaskTreeNode1, 
			(char *)NULL, TreeNode->children);
      TreeNode->status = KilledNode;
      Log_Status("Killed %s", TASK_TREE_NODE_MSG_NAME(TreeNode));
      Log_RefId(TreeNode->dispatch, LOGGING_STATUS);
      Log_Time(1); 
      Log_Status("\n");
      parentNode = TreeNode->parent;
      if (parentNode) {
	if (parentNode->status != ToBeKilledNode) {
	  /* Top of kill tree */
	  FreeTaskTreeNode(TreeNode);
	} else if (!parentNode->dispatch ||
		   parentNode->dispatch->status != AttendingDispatch) {
	  try_to_delete_node(parentNode, FALSE);
	}
      }
    } else if (firstCall) {
      Log_Status("Will kill %s ", TASK_TREE_NODE_MSG_NAME(TreeNode));
      Log_RefId(TreeNode->dispatch, LOGGING_STATUS);
      Log_Status(" when all references to it are released\n");
    }
  }
}

/******************************************************************************
 *
 * FUNCTION: 
 *
 * DESCRIPTION:
 * If the message has an abort handler registered, call the abort
 * handler, and assume that the node is killed.
 * 
 * Abort handler must return TRUE (1) if the abort was successful.
 *
 * CURRENTLY ONLY WORKS FOR INTERNAL GOALS AND COMMANDS -- EXTERNAL
 * ABORTS ARE HARDER BECAUSE OF TIMING PROBLEMS.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static int32 tryToAbort(TASK_TREE_NODE_PTR treeNode)
{ 
  ABORT_MSG_PTR abortHnd;
  DISPATCH_PTR abortDispatch;
  
  abortDispatch = treeNode->dispatch;
  abortHnd = (ABORT_MSG_PTR)hashTableFind(abortDispatch->msg->msgData->name,
					  GET_S_GLOBAL(tcaAbortTable));
  if (abortHnd) {
    if ((abortHnd->hndProc)(abortDispatch)) {
      CompletionConstraints(abortDispatch);
      try_to_delete_node(treeNode, TRUE);
      return TRUE;
    } else {
      return FALSE;
    }
  } else {
    return FALSE;
  }
}


/******************************************************************************
 *
 * FUNCTION: ABORT_MSG_PTR addAbortHandler(name, hndProc)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

ABORT_MSG_PTR addAbortHandler(const char *name, 
			      BOOLEAN (* hndProc)(DISPATCH_PTR))
{ 
  ABORT_MSG_PTR abortItem;
  
  abortItem = (ABORT_MSG_PTR)hashTableFind(name, 
					   GET_S_GLOBAL(tcaAbortTable));
  if (!abortItem) {
    abortItem = NEW(ABORT_MSG_TYPE);
    abortItem->name = name;
    abortItem->hndProc = hndProc;
    abortItem->hndOrg = GET_S_GLOBAL(tcaServerModGlobal);
    (void)hashTableInsert(name, strlen(name)+1, (char *)abortItem,
			  GET_S_GLOBAL(tcaAbortTable));
  } else {
    abortItem->hndProc = hndProc;
  }
  return abortItem;
}


/******************************************************************************
 *
 * FUNCTION: void deleteTaskTree(TreeNode)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

/*ARGSUSED*/
static int32 deleteTaskTree1 (void* dummy, TASK_TREE_NODE_PTR TreeNode)
{ 
#ifdef applec
#pragma unused(dummy)
#endif
  DISPATCH_PTR dispatch;   
  DISPATCH_STATUS_TYPE CurrentStatus;
  
  if (!DEAD_NODE(TreeNode)) {
    
    dispatch = TreeNode->dispatch;   
    CurrentStatus = ((dispatch) ? dispatch->status : HandledDispatch);
    
    switch (CurrentStatus) {
    case AttendingDispatch:
    case ReplyDispatch:
    case HandledDispatch:
    case FailureDispatch:
      break;
    case InactiveDispatch:
      RetractHoldingConstraint(TreeNode);
      break;
    case PendingDispatch:
      RetractHoldingConstraint(TreeNode);
      resourceRemovePending(dispatch);
      break;
    case AllocatedDispatch:
      RetractHoldingConstraint(TreeNode);
      break;
    default:
      tcaError("ERROR: Cannot handle killing node with status %d",
	       CurrentStatus);
    }
    
    /*
     * RGS: 11/30/93: If this is a blocking command, and the current status is
     * pending or inactive, need to issue a reply in order to prevent deadlock
     */
    if ((CurrentStatus == InactiveDispatch || CurrentStatus == PendingDispatch)
	&& dispatch->blockCom && dispatch->blockCom->waitFlag &&
	dispatch->msg && dispatch->msg->msgData->msg_class == CommandClass) {
      blockingCommandReply(dispatch, FailureDispatch);
    }
    
    (void)listIterate((LIST_ITER_FN)deleteTaskTree1, 
		      (char *)NULL, TreeNode->children);
    TreeNode->status = ToBeKilledNode;
    if (CurrentStatus != AttendingDispatch) {
      try_to_delete_node(TreeNode, TRUE);
    } else if (!tryToAbort(TreeNode)) {
      Log_Status("Will kill %s ", TASK_TREE_NODE_MSG_NAME(TreeNode));
      Log_RefId(TreeNode->dispatch, LOGGING_STATUS);
      Log_Status(" when it completes\n");
    }
  }
  return TRUE; /* Needed by listIterate */
}

static void deleteTaskTree(TASK_TREE_NODE_PTR TreeNode)
{
  (void)deleteTaskTree1((void *)NULL, TreeNode);
}

/******************************************************************************
 *
 * FUNCTION: void killTaskTreeHnd(dispatch, KillRefId)
 *
 * DESCRIPTION:
 * Stop planning and execution of the task tree node and all its sub-nodes.
 *
 * The status "KilledNode" indicates that all the children of the node have
 * been killed.
 * The status "ToBeKilledNode" indicates that at least one child is not 
 * yet "Killed".
 *
 * Algorithm has three phases:
 *  1. Remove all activation/inactivation rules from the nodes to be killed.
 *  2. Mark all the nodes as to be killed, remove from the pending/inactive
 *     sets, and retract any necessary temporal assertions.
 *  3. Try to delete all nodes whose children are already deleted (i.e.,
 *     marked "Killed").
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

/* 3-July-91: Reid: also called by wire-tapping code */
void killTaskTree(int32 killRefId)
{
  TASK_TREE_NODE_PTR killNode, parentNode;
  
  killNode = taskTreeNodeFromRef(killRefId);
  if (killNode && !DEAD_NODE(killNode)) {
    if (IS_ROOT_NODE(killNode)) {
      tcaError("Cannot kill the root node");
    } else {
      (void)runRules(-1);
      Remove_TaskTree_Rules(killNode);
      
      /* 26-Jul-91: Reid: Remove node from children list of parent */
      parentNode = killNode->parent;
      if (parentNode) {
	resetLastChild(parentNode, killNode);
	listDeleteItem((char *)killNode, parentNode->children);
      }
      
      /* Delay running rules until all retractions are done.
       * Efficiency hack 
       */
      DELAY_RULES(deleteTaskTree(killNode));
    }
  }
}

/*ARGSUSED*/
static void killTaskTreeHnd(DISPATCH_PTR dispatch, int32 *KillRefIdPtr)
{ 
#ifdef applec
#pragma unused(dispatch)
#endif
  killTaskTree(*KillRefIdPtr);
  
  tcaFree((char *)KillRefIdPtr); /* Use simple free: only 1 int32 */
}


/******************************************************************************
 *
 * FUNCTION: void killSubTaskTreeHnd(dispatch, KillRefId)
 *
 * DESCRIPTION:
 * Like "KillTaskTreeHnd", except does not kill the task tree node itself.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

/* 20-May-91: fedor: also called by exception handling code */
void killSubTaskTree(TASK_TREE_NODE_PTR node)
{
  if (!DEAD_NODE(node)) {
    /* Delay running rules until all retractions are done.  Efficiency hack */
    (void)runRules(-1);
    (void)listIterate((LIST_ITER_FN)removeTaskTreeRules1, 
		      (char *)NULL, node->children);
    
    /* 26-Jul-91: Reid: Remove all children nodes */
    if (IS_ROOT_NODE(node)) {
      (void)listIterate((LIST_ITER_FN)setLastChildIterate, 
			(char *)node, node->children);
    } else {
      setLastChild(node, (TASK_TREE_NODE_PTR)NULL, (MODULE_PTR)NULL);
    }
    
    DELAY_RULES( (void)listIterate((LIST_ITER_FN)deleteTaskTree1,
				   (char *)NULL, node->children) );
    
    listFree(&(node->children));
    node->children = NULL;
  }
}

/*ARGSUSED*/
static void killSubTaskTreeHnd(DISPATCH_PTR dispatch, int32 *KillRefId)
{ 
#ifdef applec
#pragma unused(dispatch)
#endif
  TASK_TREE_NODE_PTR Kill_TreeNode;
  
  Kill_TreeNode = taskTreeNodeFromRef(*KillRefId);
  if (Kill_TreeNode)
    killSubTaskTree(Kill_TreeNode);
  
  tcaFree((char *)KillRefId); /* Use simple free: only 1 int32 */
}


/******************************************************************************
 *
 * FUNCTION: int32 isToBeKilled(dispatch)
 *
 * DESCRIPTION:
 * Return TRUE (1) if the status of the task tree node associated with
 * the dispatch is "ToBeKilled".
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

int32 isToBeKilled(DISPATCH_PTR dispatch)
{
  return (dispatch->treeNode && dispatch->treeNode->status == ToBeKilledNode);
}


/******************************************************************************
 *
 * FUNCTION: void HandleKillAfterAttendingNodes(dispatch)
 *
 * DESCRIPTION:
 * If the status of the task tree node associated with the dispatch is
 * "ToBeKilled", then try to kill the task tree node and all its subnodes.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

void HandleKillAfterAttendingNodes(DISPATCH_PTR dispatch)
{ 
  if (isToBeKilled(dispatch) &&
      (dispatch->refCount <= 0) && (dispatch->status != AttendingDispatch)) {
    try_to_delete_node(dispatch->treeNode, FALSE);
  }
}


/******************************************************************************
 *
 * FUNCTION: int32 NeedToKillMessage(dispatch)
 *
 * DESCRIPTION:
 * Return TRUE (1) if the message is non-blocking and the parent is
 * to be killed after it finishes attending.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

int32 NeedToKillMessage(DISPATCH_PTR dispatch)
{ 
  DISPATCH_PTR parent;
  
  if (!ONE_WAY_MSG(dispatch->msg) && !TWO_WAY_MSG(dispatch->msg)) {
    /* non-blocking */
    parent = findParentDispatch(dispatch);
    
    return(parent && isToBeKilled(parent));
  } else {
    return FALSE;
  }
}


/******************************************************************************
 *
 * FUNCTION: void referenceDataHnd(dispatch, msgRef)
 *
 * DESCRIPTION:
 * Send the data that is associated with the reference.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void referenceDataHnd(DISPATCH_PTR dispatch, int32 *msgRef)
{
  MSG_PTR msg;
  DISPATCH_PTR refDispatch;
  
  refDispatch = DISPATCH_FROM_ID(*msgRef);
  
  msg = dispatch->msg;
  
  msg->msgData->resFormat = refDispatch->msg->msgData->resFormat;
  
  dispatchSetResData(DISPATCH_MSG_DATA(refDispatch), dispatch);
  
  recvMessage(dispatch, ReplyClass, (char *)NULL);
  
  /* 9-Jul-91: Reid: Set the dispatch to ReplyClass so it will be freed */
  dispatch->msg_class = ReplyClass;
  
  msg->msgData->resFormat = NULL;
  
  tcaFree((char *)msgRef); /* Use simple free: only 1 int32 */
}

/******************************************************************************
 *
 * FUNCTION: void referenceReleaseHnd(dispatch, msgRef)
 *
 * DESCRIPTION:
 * Release one reference to the associated dispatch.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/
/*ARGSUSED*/
static void referenceReleaseHnd(DISPATCH_PTR dispatch, int32 *msgRef)
{
#ifdef applec
#pragma unused(dispatch)
#endif
  DISPATCH_PTR refDispatch;
  
  if (*msgRef != NO_REF) {
    refDispatch = DISPATCH_FROM_ID(*msgRef);
    releaseDispatch(refDispatch);
  }
}

/******************************************************************************
 *
 * FUNCTION: DISPATCH_PTR CreateReferenceDispatch(msgRef, parentId,
 *                                                dispatchOrg)
 *
 * DESCRIPTION:
 * Add a new reference to message msgRef to the parent node 
 * ParentId, where the dispatch originated at DispatchOrg.
 * Return the newly created dispatch.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static DISPATCH_PTR CreateReferenceDispatch(int32 msgRef, int32 parentId,
					    MODULE_PTR dispatchOrg)
{ 
  DISPATCH_PTR refDispatch;
  
  refDispatch = dispatchAllocate();
  
  refDispatch->msg = (MSG_PTR)idTableItem(msgRef, GET_C_GLOBAL(msgIdTable));
  /* "DispatchOrg" needed by CreateTaskNode, within Initialize_TaskTreeNode */
  refDispatch->org = dispatchOrg;
  refDispatch->pRef = parentId;
  
  switch (refDispatch->msg->msgData->msg_class) {
  case GoalClass:
  case CommandClass:
  case PointMonitorClass:
  case PollingMonitorClass:
  case DemonMonitorClass:
    Initialize_TaskTreeNode(refDispatch, parentId);
    break;
  default:
    break;
  }
  return refDispatch;
}


/******************************************************************************
 *
 * FUNCTION: int32 retractEndOfConstraintsForNewChild
 *
 * DESCRIPTION: Adding a new child to a goal node might nullify the
 *              "end of interval" completion constraints.  Need to retract
 *              these constraints for *all* ancestor goal nodes.
 *
 * INPUTS: TASK_TREE_NODE_PTR childNode
 *
 * OUTPUTS: Returns TRUE if any of the relations were retracted.
 *
 *****************************************************************************/
static int32 retractEndOfConstraintsForNewChild (TASK_TREE_NODE_PTR childNode)
{
  TASK_TREE_NODE_PTR parentNode;
  int32 retracted = FALSE;
  
  parentNode = childNode->parent;
  if (parentNode->dispatch && (TASK_TREE_NODE_CLASS(parentNode) == GoalClass)) 
    {
      if (QIs_True(GET_S_GLOBAL(Now), ">", 
		   parentNode->achievementInterval->end)) {
	QRetract(GET_S_GLOBAL(Now), ">", 
		 parentNode->achievementInterval->end, "Completed Message");
	retracted = TRUE;
      }
      if ((TASK_TREE_NODE_CLASS(childNode) == GoalClass) &&
	QIs_True(GET_S_GLOBAL(Now), ">", parentNode->planningInterval->end)) {
      QRetract(GET_S_GLOBAL(Now), ">", parentNode->planningInterval->end,
	       "Completed Message");
      retracted = TRUE;
    }
    
    if (retracted && !IS_ROOT_NODE(parentNode->parent)) {
      (void)retractEndOfConstraintsForNewChild(parentNode);
    }
  }
  
  return retracted;
}

/******************************************************************************
 *
 * FUNCTION: int32 retractEndOfConstraints
 *
 * DESCRIPTION: When certain task tree changes occur (such as retrying a
 *              message), the assumptions that are needed to fire the
 *              "end of interval" rules get violated.  Need to retract
 *              these constraints for the node and all ancestor nodes.
 *
 * INPUTS: TASK_TREE_NODE_PTR node
 *
 * OUTPUTS: Returns TRUE if any of the relations were retracted.
 *
 *****************************************************************************/
int32 retractEndOfConstraints (TASK_TREE_NODE_PTR node)
{
  TASK_TREE_NODE_PTR parentNode;
  int32 retracted = FALSE;
  
  if (QIs_True(GET_S_GLOBAL(Now), ">", node->achievementInterval->end)) {
    QRetract(GET_S_GLOBAL(Now), ">", node->achievementInterval->end,
	     "Completed Message");
    retracted = TRUE;
  }
  if ((TASK_TREE_NODE_CLASS(node) == GoalClass) &&
      QIs_True(GET_S_GLOBAL(Now), ">", node->planningInterval->end)) {
    QRetract(GET_S_GLOBAL(Now), ">", node->planningInterval->end,
	     "Completed Message");
    retracted = TRUE;
  }
  
  parentNode = node->parent;
  if (retracted && !IS_ROOT_NODE(parentNode)) {
    (void)retractEndOfConstraints(parentNode);
  }
  
  return retracted;
}

/******************************************************************************
 *
 * FUNCTION: void restoreEndOfRelations
 *
 * DESCRIPTION: After retracting the "end of" completion constraints, need to
 *              re-establish (i.e., cache) the relationship between "Now" and
 *              the end of the appropriate intervals, so that the "end of"
 *              rules will fire when needed.
 *
 * INPUTS: TASK_TREE_NODE_PTR node
 *
 * OUTPUTS: none
 *
 *****************************************************************************/
void restoreEndOfRelations (TASK_TREE_NODE_PTR node)
{
  TASK_TREE_NODE_PTR parentNode;
  
  (void)QRelationship(GET_S_GLOBAL(Now), node->planningInterval->end);
  (void)QRelationship(GET_S_GLOBAL(Now), node->achievementInterval->end);
  
  parentNode = node->parent;
  if (!IS_ROOT_NODE(parentNode)) {
    restoreEndOfRelations(parentNode);
  }
}

/******************************************************************************
 *
 * FUNCTION: void AddChildRefHnd(dispatch, AddChild)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void logAddedReference(DISPATCH_PTR childDispatch)
{
  DISPATCH_PTR parentDispatch;
  
  Log("   Adding message %s", childDispatch->msg->msgData->name);
  Log_RefId(childDispatch, LOGGING_ALWAYS);
  if (childDispatch->pRef == NO_REF) {
    Log(" as child of Root Node");
  } else {
    parentDispatch = DISPATCH_FROM_ID(childDispatch->pRef);
    Log(" as child of %s", parentDispatch->msg->msgData->name);
    Log_RefId(parentDispatch, LOGGING_ALWAYS);
  }
  Log("\n");
}

static void addChildRefHnd(DISPATCH_PTR dispatch, ADD_CHILD_PTR addChild)
{ 
  DISPATCH_PTR refDispatch;
  int32 retracted;
  
  refDispatch = CreateReferenceDispatch(addChild->msgRef, 
					addChild->parentRefId, dispatch->org);
  reserveDispatch(refDispatch);
  
  logAddedReference(refDispatch);
  
  /* 23-Jul-91: Reid: Nullify the "end of interval" completion constraints */
  retracted = retractEndOfConstraintsForNewChild(refDispatch->treeNode);
  
  PendingConstraints(refDispatch);
  
  /* 23-Jul-91: Reid: Now need to re-establish the "end of interval"
     relations */
  if (retracted) restoreEndOfRelations(refDispatch->treeNode->parent);
  
  tcaFree((char *)addChild); /* Use simple free: only 2 ints */
  
  centralReply(dispatch, (char *)&(refDispatch->locId));
}


/******************************************************************************
 *
 * FUNCTION: void createRefHnd(dispatch, msgRef)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void createRefHnd(DISPATCH_PTR dispatch, int32 *msgRef)
{ 
  DISPATCH_PTR refDispatch;
  
  refDispatch = CreateReferenceDispatch(*msgRef, dispatch->pRef,
					dispatch->org);
  reserveDispatch(refDispatch);
  
  logAddedReference(refDispatch);
  
  PendingConstraints(refDispatch);
  
  tcaFree((char *)msgRef);
  centralReply(dispatch, (char *)&(refDispatch->locId));
}


/******************************************************************************
 *
 * FUNCTION: void findParentRefHnd(dispatch, childRefId)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findParentRefHnd(DISPATCH_PTR dispatch, int32 *childRefId)
{ 
  TCA_REF_PTR ref = NULL;
  DISPATCH_PTR parentDispatch;
  TASK_TREE_NODE_PTR childNode;
  
  childNode = taskTreeNodeFromRef(*childRefId);
  if (childNode) {
    if (IS_ROOT_NODE(childNode->parent)) {
      ref = CREATE_NULL_REF();
    } else {
      parentDispatch = findParentDispatch(childNode->dispatch);
      if (parentDispatch) {
	ref = tcaRefCreate((MSG_PTR)NULL, parentDispatch->msg->msgData->name,
			   parentDispatch->locId);
	reserveDispatch(parentDispatch);
      }
    }
  }
  
  tcaFree((char *)childRefId); /* Use simple free: only 1 int32 */
  
  if (ref) {
    centralReply(dispatch, (char *)&ref);
    tcaRefFree(ref);
  } else {
    centralNullReply(dispatch);
  }
}

/******************************************************************************
 *
 * FUNCTION: void findChildReply(dispatch, child)
 *
 * DESCRIPTION: Generic reply routine for the various task-tree traversal
 *              message handlers.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/
static void findChildReply (DISPATCH_PTR dispatch, TASK_TREE_NODE_PTR child)
{ 
  TCA_REF_PTR ref = NULL;
  
  if (child) {
    ref = tcaRefCreate((MSG_PTR)NULL, TASK_TREE_NODE_MSG_NAME(child),
		       child->dispatch->locId);
    reserveDispatch(child->dispatch);
  }
  
  if (ref) {
    centralReply(dispatch, (char *)&ref);
    tcaRefFree(ref);
  } else {
    centralNullReply(dispatch);
  }
}

/******************************************************************************
 *
 * FUNCTION: void findFirstChildHnd(dispatch, parentRefId)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findFirstChildHnd(DISPATCH_PTR dispatch, int32 *parentRefId)
{ 
  TASK_TREE_NODE_PTR parentNode;
  
  parentNode = taskTreeNodeFromRef(*parentRefId);
  findChildReply(dispatch, (!parentNode ? (TASK_TREE_NODE_PTR)NULL : 
			    getFirstChildAfter((TASK_TREE_NODE_PTR)NULL,
					       parentNode->children)));
  
  tcaFree((char *)parentRefId); /* Use simple free: only 1 int32 */
}

/******************************************************************************
 *
 * FUNCTION: void findLastChildHnd(dispatch, parentRefId)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findLastChildHnd(DISPATCH_PTR dispatch, int32 *parentRefId)
{ 
  TASK_TREE_NODE_PTR parentNode;
  
  parentNode = taskTreeNodeFromRef(*parentRefId);
  findChildReply(dispatch, (!parentNode ? (TASK_TREE_NODE_PTR)NULL :
			    getLastChildBefore((TASK_TREE_NODE_PTR)NULL, 
					       parentNode->children)));
  
  tcaFree((char *)parentRefId); /* Use simple free: only 1 int32 */
}

/******************************************************************************
 *
 * FUNCTION: void findNextChildHnd(dispatch, childRefId)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findNextChildHnd(DISPATCH_PTR dispatch, int32 *childRefId)
{ 
  TASK_TREE_NODE_PTR childNode;
  
  childNode = taskTreeNodeFromRef(*childRefId);
  findChildReply(dispatch, (!(childNode && childNode->parent) ? NULL :
			    getFirstChildAfter(childNode, 
					       childNode->parent->children)));
  
  tcaFree((char *)childRefId); /* Use simple free: only 1 int32 */
}

/******************************************************************************
 *
 * FUNCTION: void findPrevChildHnd(dispatch, childRefId)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findPrevChildHnd(DISPATCH_PTR dispatch, int32 *childRefId)
{ 
  TASK_TREE_NODE_PTR childNode;
  
  childNode = taskTreeNodeFromRef(*childRefId);
  findChildReply(dispatch, (!(childNode && childNode->parent) ? NULL :
			    getLastChildBefore(childNode, 
					       childNode->parent->children)));
  
  tcaFree((char *)childRefId); /* Use simple free: only 1 int32 */
}

/******************************************************************************
 *
 * FUNCTION: void findFailedRefHnd(dispatch, failedRefId)
 *
 * DESCRIPTION:
 * Like "FindParentRefHnd", except goes up the tree to find the
 * first node that is not of ExceptionClass.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findFailedRefHnd(DISPATCH_PTR dispatch, int32 *failedRefId)
{ 
  TCA_REF_PTR ref = NULL;
  TASK_TREE_NODE_PTR childNode, parentNode;
  
  childNode = taskTreeNodeFromRef(*failedRefId);
  
  if (childNode) 
    for(;;) {
      parentNode = childNode->parent;
      if (!parentNode)
	break;
      
      if (IS_ROOT_NODE(parentNode)) {
	ref = CREATE_NULL_REF();
	break;
      }
      
      if (TASK_TREE_NODE_CLASS(parentNode) != ExceptionClass) {
	ref = tcaRefCreate((MSG_PTR)NULL, TASK_TREE_NODE_MSG_NAME(parentNode),
			   parentNode->dispatch->locId);
	reserveDispatch(parentNode->dispatch);
	break;
      }
      
      childNode = parentNode;
    }
  
  tcaFree((char *)failedRefId); /* Use simple free: only 1 int32 */
  
  if (ref) {
    centralReply(dispatch, (char *)&ref);
    tcaRefFree(ref);
  } else {
    centralNullReply(dispatch);
  }
}

/******************************************************************************
 *
 * FUNCTION: void findTopLevelRefHnd(dispatch, childRefId)
 *
 * DESCRIPTION:
 * Find the top level node (the node whose parent is tcaRootNodeGlobal
 * or NULL) of the task tree in which the node of ChildRefId resides.
 * In case tcaRootNodeGlobal is given as the child node, NULL
 * is used as the answer.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findTopLevelRefHnd(DISPATCH_PTR dispatch, int32 *childRefId)
{
  TASK_TREE_NODE_PTR childNode;
  DISPATCH_PTR parentDispatch, childDispatch = NULL;
  
  childNode = taskTreeNodeFromRef(*childRefId);
  if (childNode)
    childDispatch = childNode->dispatch;
  
  parentDispatch = childDispatch;
  while (parentDispatch && (!IS_ROOT_NODE(parentDispatch->treeNode))) {
    childDispatch = parentDispatch;
    parentDispatch = findParentDispatch(childDispatch);
  }
  
  findChildReply(dispatch, (childDispatch ? childDispatch->treeNode : NULL));
  
  tcaFree((char *)childRefId); /* Use simple free: only 1 int32 */
}

/******************************************************************************
 *
 * FUNCTION: void findAnteRefByNameHnd(dispatch, findAnteRef)
 *
 * DESCRIPTION:
 * Find the child's antecedent node with a specified (message) name.
 * If there are more than one antecedents with the name, get the one
 * closest to the child.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findAnteRefByNameHnd(DISPATCH_PTR dispatch,
				 FIND_REF_PTR findAnteRef)
{
  const char *anteName;
  DISPATCH_PTR childDispatch=NULL;
  TASK_TREE_NODE_PTR childNode;
  
  childNode = taskTreeNodeFromRef(findAnteRef->refId);
  if (childNode)
    childDispatch = childNode->dispatch;
  
  anteName = findAnteRef->name;
  while (childDispatch &&
	 !strKeyEqFunc(childDispatch->msg->msgData->name, (char *)anteName)) {
    childDispatch = findParentDispatch(childDispatch);
  }
  
  findChildReply(dispatch, (childDispatch ? childDispatch->treeNode : NULL));
  
  /* A bit more efficient than using tcaFreeData */
  freeDataStructure(dispatch->msg->msgData->msgFormat, (char *)findAnteRef);
}

/******************************************************************************
 *
 * FUNCTION: void findChildByNameHnd(dispatch, findChildRef)
 *
 * DESCRIPTION:
 * Find the child node of the parent reference which has the given message 
 * name.  If more than one such child, returns the "first" one it comes
 * across (where "first" is implementation dependent).
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void findChildByNameHnd(DISPATCH_PTR dispatch,
			       FIND_REF_PTR findChildRef)
{  
  const char *childName;
  TCA_REF_PTR ref = NULL;
  TASK_TREE_NODE_PTR parentNode, child;
  DISPATCH_PTR childDispatch;
  
  childName = findChildRef->name;
  
  parentNode = taskTreeNodeFromRef(findChildRef->refId);
  if (parentNode) {
    child = (TASK_TREE_NODE_PTR)listFirst(parentNode->children);
    while (child) {
      childDispatch = child->dispatch;
      if (childDispatch && 
	  strKeyEqFunc(childDispatch->msg->msgData->name, (char *)childName)) {
	ref = tcaRefCreate((MSG_PTR)NULL, childName, childDispatch->locId);
	reserveDispatch(childDispatch);
	centralReply(dispatch, (char *)&ref);
	freeDataStructure(dispatch->msg->msgData->msgFormat, (char *)findChildRef);
	tcaRefFree(ref);
	return; 
      }
      child = (TASK_TREE_NODE_PTR)listNext(parentNode->children);
    }
  }
  /* Send a NULL reply on failure to find a message by that name */
  centralNullReply(dispatch);
  freeDataStructure(dispatch->msg->msgData->msgFormat, (char *)findChildRef);
}


/******************************************************************************
 *
 * FUNCTION: int32 displayTaskTree(position, treeNode)
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS: Returns TRUE (needed by listIterate)
 *
 *****************************************************************************/

static int32 displayTaskTree(int32 *position, TASK_TREE_NODE_PTR treeNode)
{
  const char *nodeName, *status;
  
  if (TRUE) /*(!DEAD_NODE(treeNode))*/ {
    
    if (GET_S_GLOBAL(cursor) > *position) {
      Log("\n");
      GET_S_GLOBAL(cursor) = 0;
    }
    pr_space(*position - GET_S_GLOBAL(cursor));
    GET_S_GLOBAL(cursor) = *position;
    if (treeNode->dispatch || IS_ROOT_NODE(treeNode)) {
      if (IS_ROOT_NODE(treeNode)) {
	nodeName = "Root Node";
	status = "";
      } else {
	nodeName = TASK_TREE_NODE_MSG_NAME(treeNode);
	status = GET_S_GLOBAL(nodeStatusNames)[(int32)treeNode->dispatch->status];
      }
      
      Log("(%s%s%s",
	  (treeNode->dispatch && IS_LISTENING(treeNode->dispatch) ? "TAP: " : ""),
	  status, nodeName);
      Log_RefId(treeNode->dispatch, LOGGING_ALWAYS);
      if (treeNode->children) Log("  ");
      
      GET_S_GLOBAL(cursor) += (2 + strlen(nodeName) + strlen(status));
      if (treeNode->dispatch && IS_LISTENING(treeNode->dispatch)) 
	GET_S_GLOBAL(cursor) += 5;
      
      (void)listIterateFromFirst((LIST_ITER_FN)displayTaskTree, 
				 (void *)&GET_S_GLOBAL(cursor),
				 treeNode->children);
      Log(")");
      ++(GET_S_GLOBAL(cursor));
    } else {
      Log("(*FREED*)");
      GET_S_GLOBAL(cursor) += 9;
    }
  }
  return TRUE;
}


/******************************************************************************
 *
 * FUNCTION: void displayTaskTreeHnd(dispatch, dispRefId)
 *
 * DESCRIPTION:
 * Display the task tree rooted at DispRefId
 *
 * For each node, prints out its message name and its status:
 *  "I"nactive 
 *  "A"ttending
 *  "C"ompleted (Handled)
 *  "P"ending
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

/*ARGSUSED*/
static void displayTaskTreeHnd(DISPATCH_PTR dispatch, int32 *dispRefId)
{ 
#ifdef applec
#pragma unused(dispatch)
#endif
  TASK_TREE_NODE_PTR dispNode;
  int32 indent = 0;
  
  dispNode = taskTreeNodeFromRef(*dispRefId);
  if (dispNode) {
    (void)displayTaskTree (&indent, dispNode);
    Log("\n");
  } else {
    Log("DisplayTaskTreeHnd: Invalid Task Tree Node\n");
  }
  
  tcaFree((char *)dispRefId); /* Use simple free: only 1 int32 */
}


/******************************************************************************
 *
 * FUNCTION: int32 displayNonTaskTree()
 *
 * DESCRIPTION:
 *
 * INPUTS: 
 *
 * OUTPUTS: Returns TRUE (needed by listIterate)
 *
 *****************************************************************************/

void displayNonTaskTree(void)
{
  int32 i;
  DISPATCH_PTR dispatch;
  
  for (i=0; i<GET_S_GLOBAL(dispatchTable)->currentSize; i++) {
    dispatch = (DISPATCH_PTR)idTableItem(i, GET_S_GLOBAL(dispatchTable));
    if (dispatch->status != UnallocatedDispatch) {
      if ((dispatch->msg_class == InformClass) ||
	  (dispatch->msg_class == QueryClass) ||
	  (dispatch->msg_class == BroadcastClass) ||
	  (dispatch->msg_class == MultiQueryClass)
	  )
	Log("(%s : %s %d)\n", dispatch->msg->msgData->name,
	    GET_S_GLOBAL(nodeStatusNames)[(int32)dispatch->status],
	    dispatch->locId);
    }
  }
}


/******************************************************************************
 *
 * FUNCTION: void referenceStatusHnd(dispatch, msgRef)
 *
 * DESCRIPTION: Send the status of the message associated with the reference.
 *              Possible status: UnknownRefStatus, InactiveRef, PendingRef,
 *                               ActiveRef, HandledRef, PlannedRef,
 *                               AchievedRef, KilledRef
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

#if 0
/* New version: install after Rich finishes testing with the Hero simulator */
static void referenceStatusHnd(DISPATCH_PTR dispatch, int32 *msgRef)
{
  TASK_TREE_NODE_PTR node;
  TCA_REF_STATUS_TYPE status;
  
  status = UnknownRefStatus;
  node = taskTreeNodeFromRef(*msgRef);
  if (node) {
    switch (node->dispatch->status) {
    case AllocatedDispatch:
    case UnallocatedDispatch:
    case InactiveDispatch: status = InactiveRef; break;
      
    case AttendingDispatch: status = ActiveRef; break;
      
    case PendingDispatch: status = PendingRef; break;
      
    case KilledDispatch: status = KilledRef; break;
      
    case HandledDispatch: 
    case SuccessDispatch: 
    case FailureDispatch: 
      if (QIs_True(node->achievementInterval->end, "<=", GET_S_GLOBAL(Now)))
	status = AchievedRef;
      else if (QIs_True(node->planningInterval->end, "<=", GET_S_GLOBAL(Now)))
	status = PlannedRef;
      else
	status = HandledRef;
      break;
#ifndef TEST_CASE_COVERAGE
    default:
      tcaModError("Unknown referenceStatusHnd Status %d",
		  node->dispatch->status);
      break;
#endif
    }
  }
  centralReply(dispatch, (char *)&status);
}
#endif

static void referenceStatusHnd(DISPATCH_PTR dispatch,
			       int32 *msgRef)
{
  TASK_TREE_NODE_PTR node;
  int32 status;
  
  node = taskTreeNodeFromRef(*msgRef);
  if (node) {
    status = (int32) node->dispatch->status;
  } else {
    status = -1;
  }
  centralReply(dispatch, &status);
}


/******************************************************************************
 *
 * FUNCTION: void taskTreeInitialize()
 *
 * DESCRIPTION:
 * Initialize the messages and static variables for the task tree
 * and temporal constraints.
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

void taskTreeInitialize(void)
{
  Tpl_Initialize();
  
  GET_S_GLOBAL(tcaAbortTable) = 
    hashTableCreate(11,
		    (HASH_FN)strHashFunc, 
		    (EQ_HASH_FN)strKeyEqFunc);
  
  GET_S_GLOBAL(taskTreeRootGlobal) = taskTreeNodeCreate((DISPATCH_PTR)NULL,
							NO_REF);
  GET_S_GLOBAL(taskTreeRootGlobal)->lastChild = 
    (TASK_TREE_NODE_PTR)listCreate();
  
  GET_S_GLOBAL(nodeStatusNames)[(int32)AllocatedDispatch] = "X:";
  GET_S_GLOBAL(nodeStatusNames)[(int32)AttendingDispatch] = "A:";
  GET_S_GLOBAL(nodeStatusNames)[(int32)PendingDispatch] = "P:";
  GET_S_GLOBAL(nodeStatusNames)[(int32)InactiveDispatch] = "I:";
  /* For "completed" */
  GET_S_GLOBAL(nodeStatusNames)[(int32)HandledDispatch] = "C:";
  
  centralRegisterQuery(TCA_CREATE_REF_QUERY, 
		       TCA_CREATE_REF_QUERY_FORMAT,
		       TCA_CREATE_REF_QUERY_REPLY,
		       createRefHnd);
  
  centralRegisterQuery(TCA_CREATE_REF_QUERY_OLD, 
		       TCA_CREATE_REF_QUERY_FORMAT,
		       TCA_CREATE_REF_QUERY_REPLY,
		       createRefHnd);
  
  centralRegisterQuery(TCA_ADD_CHILD_QUERY, 
		       TCA_ADD_CHILD_QUERY_FORMAT,
		       TCA_ADD_CHILD_QUERY_REPLY,
		       addChildRefHnd);
  
  centralRegisterQuery(TCA_ADD_CHILD_QUERY_OLD, 
		       TCA_ADD_CHILD_QUERY_FORMAT,
		       TCA_ADD_CHILD_QUERY_REPLY,
		       addChildRefHnd);
  
  centralRegisterQuery(TCA_FIND_PARENT_QUERY, 
		       TCA_FIND_PARENT_QUERY_FORMAT,
		       TCA_FIND_PARENT_QUERY_REPLY,
		       findParentRefHnd);
  
  centralRegisterQuery(TCA_FIND_PARENT_QUERY_OLD, 
		       TCA_FIND_PARENT_QUERY_FORMAT,
		       TCA_FIND_PARENT_QUERY_REPLY,
		       findParentRefHnd);
  
  centralRegisterQuery(TCA_FIRST_CHILD_QUERY, 
		       TCA_FIRST_CHILD_QUERY_FORMAT,
		       TCA_FIRST_CHILD_QUERY_REPLY,
		       findFirstChildHnd);
  
  centralRegisterQuery(TCA_FIRST_CHILD_QUERY_OLD, 
		       TCA_FIRST_CHILD_QUERY_FORMAT,
		       TCA_FIRST_CHILD_QUERY_REPLY,
		       findFirstChildHnd);
  
  centralRegisterQuery(TCA_LAST_CHILD_QUERY,
		       TCA_LAST_CHILD_QUERY_FORMAT, 
		       TCA_LAST_CHILD_QUERY_REPLY,
		       findLastChildHnd);
  
  centralRegisterQuery(TCA_LAST_CHILD_QUERY_OLD,
		       TCA_LAST_CHILD_QUERY_FORMAT, 
		       TCA_LAST_CHILD_QUERY_REPLY,
		       findLastChildHnd);
  
  centralRegisterQuery(TCA_NEXT_CHILD_QUERY, 
		       TCA_NEXT_CHILD_QUERY_FORMAT,
		       TCA_NEXT_CHILD_QUERY_REPLY,
		       findNextChildHnd);
  
  centralRegisterQuery(TCA_NEXT_CHILD_QUERY_OLD, 
		       TCA_NEXT_CHILD_QUERY_FORMAT,
		       TCA_NEXT_CHILD_QUERY_REPLY,
		       findNextChildHnd);
  
  centralRegisterQuery(TCA_PREV_CHILD_QUERY, 
		       TCA_PREV_CHILD_QUERY_FORMAT,
		       TCA_PREV_CHILD_QUERY_REPLY,
		       findPrevChildHnd);
  
  centralRegisterQuery(TCA_PREV_CHILD_QUERY_OLD, 
		       TCA_PREV_CHILD_QUERY_FORMAT,
		       TCA_PREV_CHILD_QUERY_REPLY,
		       findPrevChildHnd);
  
  centralRegisterQuery(TCA_FAILED_CHILD_QUERY, 
		       TCA_FAILED_CHILD_QUERY_FORMAT,
		       TCA_FAILED_CHILD_QUERY_REPLY,
		       findFailedRefHnd);
  
  centralRegisterQuery(TCA_FAILED_CHILD_QUERY_OLD, 
		       TCA_FAILED_CHILD_QUERY_FORMAT,
		       TCA_FAILED_CHILD_QUERY_REPLY,
		       findFailedRefHnd);
  
  centralRegisterQuery(TCA_FIND_TOPLEVEL_QUERY, 
		       TCA_FIND_TOPLEVEL_QUERY_FORMAT,
		       TCA_FIND_TOPLEVEL_QUERY_REPLY,
		       findTopLevelRefHnd);
  
  centralRegisterQuery(TCA_FIND_TOPLEVEL_QUERY_OLD, 
		       TCA_FIND_TOPLEVEL_QUERY_FORMAT,
		       TCA_FIND_TOPLEVEL_QUERY_REPLY,
		       findTopLevelRefHnd);
  
  centralRegisterQuery(TCA_ANTE_BY_NAME_QUERY,
		       TCA_ANTE_BY_NAME_QUERY_FORMAT,
		       TCA_ANTE_BY_NAME_QUERY_REPLY,
		       findAnteRefByNameHnd);
  
  centralRegisterQuery(TCA_ANTE_BY_NAME_QUERY_OLD,
		       TCA_ANTE_BY_NAME_QUERY_FORMAT,
		       TCA_ANTE_BY_NAME_QUERY_REPLY,
		       findAnteRefByNameHnd);
  
  centralRegisterQuery(TCA_CHILD_BY_NAME_QUERY, 
		       TCA_CHILD_BY_NAME_QUERY_FORMAT,
		       TCA_CHILD_BY_NAME_QUERY_REPLY,
		       findChildByNameHnd);
  
  centralRegisterQuery(TCA_CHILD_BY_NAME_QUERY_OLD,
		       TCA_CHILD_BY_NAME_QUERY_FORMAT,
		       TCA_CHILD_BY_NAME_QUERY_REPLY,
		       findChildByNameHnd);
  
  centralRegisterInform(TCA_KILL_TREE_INFORM,
			TCA_KILL_TREE_INFORM_FORMAT,
			killTaskTreeHnd);
  
  centralRegisterInform(TCA_KILL_TREE_INFORM_OLD,
			TCA_KILL_TREE_INFORM_FORMAT,
			killTaskTreeHnd);
  
  centralRegisterInform(TCA_KILL_SUBTREE_INFORM,
			TCA_KILL_SUBTREE_INFORM_FORMAT,
			killSubTaskTreeHnd);
  
  centralRegisterInform(TCA_KILL_SUBTREE_INFORM_OLD,
			TCA_KILL_SUBTREE_INFORM_FORMAT,
			killSubTaskTreeHnd);
  
  centralRegisterInform(TCA_DISPLAY_TREE_INFORM,
			TCA_DISPLAY_TREE_INFORM_FORMAT,
			displayTaskTreeHnd);
  
  centralRegisterInform(TCA_DISPLAY_TREE_INFORM_OLD,
			TCA_DISPLAY_TREE_INFORM_FORMAT,
			displayTaskTreeHnd);
  
  centralRegisterQuery(TCA_REF_DATA_QUERY,
		       TCA_REF_DATA_QUERY_FORMAT,
		       TCA_REF_DATA_QUERY_REPLY,
		       referenceDataHnd);
  
  centralRegisterQuery(TCA_REF_DATA_QUERY_OLD,
		       TCA_REF_DATA_QUERY_FORMAT,
		       TCA_REF_DATA_QUERY_REPLY,
		       referenceDataHnd);
  
  centralRegisterInform(TCA_REF_RELEASE_INFORM,
			TCA_REF_RELEASE_INFORM_FORMAT,
			referenceReleaseHnd);
  Add_Message_To_Ignore(TCA_REF_RELEASE_INFORM);
  
  centralRegisterInform(TCA_REF_RELEASE_INFORM_OLD,
			TCA_REF_RELEASE_INFORM_FORMAT,
			referenceReleaseHnd);
  Add_Message_To_Ignore(TCA_REF_RELEASE_INFORM_OLD);
  
  centralRegisterQuery(TCA_REF_STATUS_QUERY,
		       TCA_REF_STATUS_QUERY_FORMAT,
		       TCA_REF_STATUS_QUERY_REPLY,
		       referenceStatusHnd);
  
  centralRegisterQuery(TCA_REF_STATUS_QUERY_OLD,
		       TCA_REF_STATUS_QUERY_FORMAT,
		       TCA_REF_STATUS_QUERY_REPLY,
		       referenceStatusHnd);
}


/******************************************************************************
 *
 * FUNCTION: void showTaskTree(void)
 *
 * DESCRIPTION: Clears (or kills) the entire task tree.
 * 
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

void showTaskTree(void)
{
  int32 position=0;
  Log("The task tree\n");
  displayTaskTree(&position,taskTreeNodeFromRef(NO_REF));
  Log("\n");
  Log("Non task tree messages\n");
  displayNonTaskTree();
  Log("\n");
}


/******************************************************************************
 *
 * FUNCTION: void clearTaskTree(void)
 *
 * DESCRIPTION: Clears (or kills) the entire task tree.
 * 
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 *****************************************************************************/

void clearTaskTree(void)
{
  int position=0;
  Log("Killing the task tree\n");
  displayTaskTree(&position,taskTreeNodeFromRef(NO_REF));
  killSubTaskTree(taskTreeNodeFromRef(NO_REF));
  Log("\n Task tree killed.\n");
}
