/**************************************************************************
 * 
 * PROJECT: 
 *
 * (c) Copyright 1995 Richard Goodwin.  All rights reserved.
 * 
 * MODULE: strList
 *
 * FILE: strList.c
 *
 * ABSTRACT:
 * The list module provides basic list creation and manipulation
 * routines for lists of strings.
 *
 * EXPORTS:
 *
 * STR_LIST_PTR strListCreate()
 * Creates an empty strList.
 *
 * void strListFree(strList)
 * STR_LIST_PTR strList;
 * Frees storage associated with a strList.
 *
 * void strListPush(item, strList)
 * char *item;
 * STR_LIST_PTR strList;
 * Adds item as the first item in the strList.
 *
 * int strListMemberItem(item, strList)
 * char *item;
 * STR_LIST_PTR strList;
 * Returns 1 (TRUE) if item is in the strList, otherwise 0 (FALSE) is returned.
 *
 * char *strListPopItem(strList)
 * STR_LIST_PTR strList
 * Removes and returns first item from strList.
 *
 * REVISION HISTORY:
 *
 * $Log: strList.c,v $
 * Revision 1.11  1996/04/04  18:18:11  rich
 * Fixed problems found by the SGI compiler.
 *
 * Revision 1.10  1996/02/10  16:50:32  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.9  1996/01/23  00:06:39  rich
 * Fixed memory leak when a module connects and disconnects.  Also fixed a
 * problem with using the direct connection flag.  This was introduced when
 * we added contexts for keeping track of the central server.
 *
 * Revision 1.8  1996/01/05  16:31:38  rich
 * Added windows NT port.
 *
 * Revision 1.7  1995/12/17  20:22:13  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.6  1995/10/07  19:07:46  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.5  1995/07/19  14:26:33  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.4  1995/07/12  04:55:30  rich
 * Release of 8.0.
 * Fixed problems with sending between machines of different endien.
 *
 * Revision 1.3  1995/07/10  16:18:43  rich
 * Interm save.
 *
 * Revision 1.2  1995/04/04  19:43:03  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.1  1995/03/30  15:44:07  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 *
 * $Revision: 1.11 $
 * $Date: 1996/04/04 18:18:11 $
 * $Author: rich $
 *
 **************************************************************************/

#include "globalM.h"

/******************************************************************************
 * Forward Declarations
 *****************************************************************************/

#if !defined(DBMALLOC)
STR_LIST_PTR strListDBCreate(const char* file, int line);
void strListDBPush(const char* file, int line,
		   const char *item, STR_LIST_PTR strList);
void strListDBPushUnique(const char* file, int line,
			 const char *item, STR_LIST_PTR strList);
#else
#undef strListPush
void strListPush(const char *item, STR_LIST_PTR strList);
#undef strListPushUnique
void strListPushUnique(const char *item, STR_LIST_PTR strList);
#undef strListCreate
STR_LIST_PTR strListCreate(void);
#endif


/**************************************************************************
 *
 * FUNCTION: void strListFree(strList)
 *
 * DESCRIPTION: Frees a strList.
 *
 * INPUTS: STR_LIST_PTR strList.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: Call strListFreeCell on each strList element.
 *
 * NOTES: 
 *
 **************************************************************************/

void strListFree(STR_LIST_PTR *strList, BOOLEAN freeString)
{
  STR_LIST_PTR tmpA, tmpB;
  
  if (!(*strList))
    return;
  
  tmpA = (*strList);

  while (tmpA) {
    tmpB = tmpA;
    tmpA = tmpA->next;
    
    if (freeString && (tmpB->item))
      free(tmpB->item);
    free(tmpB);
  }
  *strList = NULL;
}    


/**************************************************************************
 *
 * FUNCTION: void strListPush(item, strList)
 *
 * DESCRIPTION: Adds item as the first item in the strList.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * STR_LIST_PTR strList - a pointer to a strList.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the strList is NULL, return.
 * If the item is NULL, return.
 *
 * DESIGN: 
 * Create a strList element to store item and insert the element as the 
 * first item in the strList.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

#if defined(DBMALLOC)
void strListPush(const char *item, STR_LIST_PTR strList)
{strListDBPush("Unknown",-1,item,strList);}

void strListDBPush(const char* file, int line,
		   const char *item, STR_LIST_PTR strList)
{
  STR_LIST_PTR element;
  
  element = strListDBCreate(file,line);
  element->item = (char *)item;
  element->next = strList->next;

  strList->next = element;

}

#else
void strListDBPush(const char* file, int line,
		   const char *item, STR_LIST_PTR strList)
{ strListPush(item,strList);}

void strListPush(const char *item, STR_LIST_PTR strList)
{
  STR_LIST_PTR element;
  
  element = strListCreate();
  element->item = (char *)item;
  element->next = strList->next;
  
  strList->next = element;
  
}
#endif


/**************************************************************************
 *
 * FUNCTION: void strListPushUnique(item, strList)
 *
 * DESCRIPTION: Adds item as the first item in the strList.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * STR_LIST_PTR strList - a pointer to a strList.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the strList is NULL, return.
 * If the item is NULL, return.
 *
 * DESIGN: 
 * Create a strList element to store item and insert the element as the 
 * first item in the strList.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

#if defined(DBMALLOC)
void strListPushUnique(const char *item, STR_LIST_PTR strList)
{strListDBPushUnique("Unknown",-1,item,strList);}

void strListDBPushUnique(const char* file, int line,
			 const char *item, STR_LIST_PTR strList)
{
  if (!strListMemberItem(item,strList))
    strListDBPush(file, line, item, strList);
}

#else
void strListDBPushUnique(const char* file, int line,
			 const char *item, STR_LIST_PTR strList)
{ strListPushUnique(item,strList);}

void strListPushUnique(const char *item, STR_LIST_PTR strList)
{
  if (!strListMemberItem(item,strList))
    strListPush(item, strList);
}
#endif


/**************************************************************************
 *
 * FUNCTION: char *strListPopItem(strList)
 *
 * DESCRIPTION: Removes and returns first item from strList.
 *
 * INPUTS: 
 * STR_LIST_PTR strList - a pointer to a strList.
 *
 * OUTPUTS:
 * The first item or NULL
 *
 * EXCEPTIONS:  If the strList is NULL or the strList is empty, return NULL.
 *
 * DESIGN: 
 * Remove the first strList element, return the item and free the element.
 *
 * NOTES:
 * might want to start a free strList of strList elements.
 *
 **************************************************************************/

const char *strListPopItem(STR_LIST_PTR strList)
{
  const char *item;
  STR_LIST_PTR oldElement;
  
  if (strList->next == NULL)
    return NULL;

  oldElement = strList->next;
  strList->next = strList->next->next;

  item = oldElement->item;
  tcaFree((char *)oldElement);
  
  return item;
}


/**************************************************************************
 *
 * FUNCTION: char *strListRemoveItem(strList,item)
 *
 * DESCRIPTION: Removes an item from strList.
 *
 * INPUTS: 
 * STR_LIST_PTR strList - a pointer to a strList.
 *
 * OUTPUTS:
 * The first item or NULL
 *
 * EXCEPTIONS:  If the strList is NULL or the strList is empty, return NULL.
 *
 * DESIGN: 
 * Remove the first strList element, return the item and free the element.
 *
 * NOTES:
 * might want to start a free strList of strList elements.
 *
 **************************************************************************/

void strListRemoveItem(const char *item, STR_LIST_PTR strList)
{
  STR_LIST_PTR tmp, last;
  
  if (strList->next == NULL) return;

  if (item == NULL) return;
  
  last = strList;
  tmp = strList->next;
  
  while (tmp) {
    if ((tmp->item!= NULL) && (strcmp(tmp->item,item) == 0)) {
      /* Found it, can now delete it. */
      last->next = tmp->next;
      tcaFree((char *)tmp);
      return;
    }
    last = tmp;
    tmp = tmp->next;
  }
  
  return;
}


/**************************************************************************
 *
 * FUNCTION: int strListMemberItem(item, strList)
 *
 * DESCRIPTION: Tests if item is an element of strList.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * STR_LIST_PTR strList - a pointer to a strList.
 *
 * OUTPUTS: 
 * 0 - FALSE 
 * 1 - TRUE
 *
 * EXCEPTIONS:  If the strList is NULL or the strList is empty, return;
 *
 * DESIGN: 
 * Linearly search the strList for a strList element containing item. 
 * If found the value 1 is returned, else the value 0 is returned.
 *
 * NOTES:
 *
 **************************************************************************/

BOOLEAN strListMemberItem(const char *item, STR_LIST_PTR strList)
{
  STR_LIST_PTR tmp;
  
  if (strList == NULL) return FALSE;

  if (item == NULL) return FALSE;
  
  tmp = strList->next;
  
  while (tmp) {
    if ((tmp->item!= NULL) && (strcmp(tmp->item,item) == 0))
      return TRUE; /* TRUE */
    tmp = tmp->next;
  }
  
  return FALSE; /* False */
}


/**************************************************************************/

/* StrList interface routines */

int strListLength(STR_LIST_PTR strList)
{
  if (strList == NULL)
    return -1;
  
  return(1 + strListLength(strList->next));
}

int strListEmpty(STR_LIST_PTR strList)
{
  if (strList == NULL)
    return TRUE;
  return FALSE;
}


int strListEqual(STR_LIST_PTR strList1, STR_LIST_PTR strList2)
{
  
  if (strList1 == strList2)
    return TRUE;
  
  if ((strList1 == NULL) ||(strList1 == NULL))
    return FALSE;

  if ((((strList1->item == NULL) || (strList2->item == NULL)) && 
       (strList1->item != strList2->item))
      || (strcmp(strList1->item,strList2->item) != 0))
    return FALSE;
  
  return(strListEqual(strList1->next, strList2->next));
}


const char *strListFirst(STR_LIST_PTR strList)
{
  
  if (strList == NULL)
    return NULL;
  
  if (strList->next == NULL)
    return NULL;
  
  return strList->next->item;
}

const char *strListLast(STR_LIST_PTR strList)
{
  STR_LIST_PTR tmp;

  if (strList == NULL)
    return NULL;

  if (strList->next == NULL)
    return NULL;

  tmp = strList->next;
  while (tmp->next != NULL)
    tmp = tmp->next;

  return tmp->item;
}


/**************************************************************************
 *
 * FUNCTION: int strListDeleteItem(item, strList)
 *
 * DESCRIPTION: Tests if item is an element of strList.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * STR_LIST_PTR strList - a pointer to a strList.
 *
 * OUTPUTS: 
 * 0 - FALSE 
 * 1 - TRUE
 *
 * EXCEPTIONS:  If the strList is NULL or the strList is empty, return;
 *
 * DESIGN: 
 * Linearly search the strList for a strList element containing item. 
 * If found the value 1 is returned, else the value 0 is returned.
 *
 * NOTES:
 *
 **************************************************************************/

int strListDeleteItem(const char *item,
		      STR_LIST_PTR strList,
		      BOOLEAN freeString)
{
  STR_LIST_PTR tmp;
  STR_LIST_PTR tmp2;
  
  if (!strList)
    return FALSE; /* False */
  
  tmp = strList;
  
  while (tmp->next) {
    if ((tmp->next->item != NULL) &&
	(strcmp(tmp->next->item,item) == 0)) {
      if (freeString)
	free(tmp->next->item);
      tmp2 = tmp->next;
      tmp->next = tmp->next->next;
      free(tmp2);
      return TRUE; /* TRUE */
    }
    tmp = tmp->next;
  }
  
  return FALSE; /* False */
}


/**************************************************************************
 *
 * FUNCTION: STR_LIST_PTR strListCreate()
 *
 * DESCRIPTION: Creates an empty strList.
 *
 * INPUTS: none.
 *
 * OUTPUTS: A pointer to the newly created empty strList of type STR_LIST_PTR.
 * If there is an error NULL is returned.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: malloc up storage for the strList, checking to see if space was
 * successfully allocated, and initialize the strList structure.
 *
 * NOTES:
 *
 **************************************************************************/
#if defined(DBMALLOC)
STR_LIST_PTR strListCreate(void) 
{return strListDBCreate("Unknown",-1);}

STR_LIST_PTR strListDBCreate(const char* file, int line)
{
  STR_LIST_PTR newList;
  
  newList = tcaDBMalloc(file,line,sizeof(STR_LIST_TYPE));
  newList->item = NULL;
  newList->next = NULL;

  return newList;
}

#else
STR_LIST_PTR strListDBCreate(const char* file, int line)
{ return strListCreate();}

STR_LIST_PTR strListCreate(void)
{
  STR_LIST_PTR newList;
  
  newList = tcaMalloc(sizeof(STR_LIST_TYPE));
  newList->item = NULL;
  newList->next = NULL;

  return newList;
}
#endif
